/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.TreeSet;

import org.apache.xerces.dom.ElementNSImpl;
import org.odftoolkit.odfdom.dom.element.office.OfficeFormsElement;
import org.odftoolkit.odfdom.dom.element.table.TableShapesElement;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.ElementNS;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.odf.UnknownContentHandler;

public class SheetHandler extends SaxContextHandler {

	final private Sheet sheet;

	private GroupStack colGroupStack = new GroupStack();
	private GroupStack rowGroupStack = new GroupStack();

	public SheetHandler(ContentHandler parentContextHandler, Sheet sheet)
		throws SAXException {

		super(parentContextHandler);
		this.sheet = sheet;
	}

	public Sheet getSheet() {
		return sheet;
	}

	@Override
	public SaxContextHandler startElement(Attributes attributes, String uri, String localName, String qName)
       	throws SAXException {

    	if(localName.equals("table-column")) {
    		final TreeSet<Column> columns = sheet.getColumns();
    		final int nextCol = columns.isEmpty() ? 0 : columns.last().getMax() + 1;
    		String columnStyle = null;
    		for(int i=0; i<attributes.getLength(); i++) {
    			if(attributes.getLocalName(i).equals("style-name")) {
					columnStyle = attributes.getValue(i);
					break;
				}
    		}
    		final Column column = new Column(nextCol);
    		column.setStyleName(columnStyle);
    		column.setGroupLevel(colGroupStack.getLevel());
    		column.setGroupCollapse(colGroupStack.pullCollapse());

    		if(columns.isEmpty()) {
    			// adding column placeholder only once
    			sheet.getChilds().add(column);
    			sheet.columnPlaceholderAvailable = true;
    		}
    		columns.add(column);
    		return new ColumnHandler(this, attributes, column);
    	}
    	else if(localName.equals("table-row")) {
    		final TreeSet<Row> rows = sheet.getRows();
    		int currentRow = 0;
    		if(!rows.isEmpty()) {
    			final Row lastRow = rows.last();
    			currentRow = lastRow.getMax()+1;
    		}
    		String rowStyle = null;
    		for(int i=0; i<attributes.getLength(); i++) {
				if(attributes.getLocalName(i).equals("style-name")) {
					rowStyle = attributes.getValue(i);
					break;
				}
    		}
    		final Row row = new Row(currentRow, 1);
    		row.setStyleName(rowStyle);
    		row.setGroupLevel(rowGroupStack.getLevel());
    		row.setGroupCollapse(rowGroupStack.pullCollapse());

    		// for the first row we are inserting a placeholder node, at this point
    		// the all rows are saved in ascending order
    		if(rows.isEmpty()) {
        		sheet.getChilds().add(row);
        		sheet.rowPlaceholderAvailable = true;
    		}
    		rows.add(row);
    		return new RowHandler(this, attributes, row);
    	}
    	else if(localName.equals("table-column-group")) {
    		final AttributesImpl attrsImpl = new AttributesImpl(attributes);
    		final boolean collapsed = !attrsImpl.getBooleanValue("table:display", true);
    		colGroupStack.pushLevel(collapsed);
    	}
    	else if(localName.equals("table-row-group")) {
    		final AttributesImpl attrsImpl = new AttributesImpl(attributes);
    		final boolean collapsed = !attrsImpl.getBooleanValue("table:display", true);
    		rowGroupStack.pushLevel(collapsed);
    	}
    	else if(localName.equals("shapes")&&uri.equals(TableShapesElement.ELEMENT_NAME.getUri())) {
    		return new ShapesHandler(this, sheet.getDrawings());
    	}
    	else if(localName.equals("forms")&&uri.equals(OfficeFormsElement.ELEMENT_NAME.getUri())) {
    		final ElementNSImpl officeForms = new ElementNS(getFileDom(), attributes, uri, qName);
    		sheet.setOfficeForms(officeForms);
    		return new UnknownContentHandler(this, officeForms);
    	}
    	else if(qName.equals("table:named-expressions")) {
    		return new NamedExpressionsHandler(this, sheet.getNamedExpressions(true));
    	}
    	else if(qName.equals("calcext:conditional-formats")) {
    		return new ConditionalFormatsHandler(this, sheet.getConditionalFormats(true));
    	}
    	else {
    		if(qName.equals("table:table-source")) {
    			sheet.setIsLinkedTable(true);
    		}
    		final ElementNSImpl element = new ElementNS(getFileDom(), attributes, uri, qName);
    		sheet.getChilds().add(element);
    		return new UnknownContentHandler(this, element);
    	}
    	return this;
	}

    @Override
    public void endElement(String localName, String qName)
    	throws SAXException {

    	if(localName.equals("table-column-group")) {
    		colGroupStack.popLevel();
    	}
    	else if(localName.equals("table-row-group")) {
    		rowGroupStack.popLevel();
    	}
    }

	@Override
	public void endContext(String qName, String characters) {
		super.endContext(qName, characters);

		if(qName.equals("table:table")) {
    		final TreeSet<Row> rows = sheet.getRows();
    		int emptyRows = Sheet.getMaxRowCount();
			if(!rows.isEmpty()) {
	    		emptyRows -= rows.last().getMax()+1;
			}
	    	if(emptyRows>0) {
	    		final Row emptyRow = new Row(1048576 - emptyRows, emptyRows);
				emptyRow.setDefaultCellStyle("Default");
				rows.add(emptyRow);
				final Cell emptyCell = new Cell(0);
				emptyCell.setRepeated(sheet.getMaxColCount());
				emptyRow.getCells().add(emptyCell);
			}
		}
	}
}
