/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;
import org.apache.commons.lang.mutable.MutableBoolean;
import org.apache.commons.lang.mutable.MutableInt;
import org.apache.xerces.dom.ElementNSImpl;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.attribute.table.TableFieldNumberAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableOperatorAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableProtectedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableValueAttribute;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import com.openexchange.office.odf.ConfigItem;
import com.openexchange.office.odf.ConfigItemMapEntry;
import com.openexchange.office.odf.ConfigItemSet;
import com.openexchange.office.odf.Settings;
import com.openexchange.office.odf.properties.TableProperties;
import com.openexchange.office.odf.styles.DocumentStyles;
import com.openexchange.office.odf.styles.StyleBase;
import com.openexchange.office.odf.styles.StyleManager;
import com.openexchange.office.odf.styles.StyleTable;
import com.openexchange.office.ods.dom.SmlUtils.CellRefRange;

/**
 *
 * @author sven.jacobi@open-xchange.com
 */
public class JsonOperationProducer {

    private JSONArray operationQueue;

    private final OdfSpreadsheetDocument doc;
    private final DocumentStyles styles;
    private final Content content;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationProducer(OdfOperationDocument opsDoc)
    	throws SAXException {

    	doc = (OdfSpreadsheetDocument)opsDoc.getDocument();
        styles = (DocumentStyles)doc.getStylesDom();
        content = (Content)doc.getContentDom();
        styleManager = content.getStyleManager();
        settings = (Settings)doc.getSettingsDom();
    }

    public JSONObject getDocumentOperations()
    	throws JSONException, SAXException {

    	operationQueue = new JSONArray();
    	createSetDocumentattributesOperation();
    	styleManager.createInsertStyleOperations(styleManager.getStyles(), "spreadsheet", operationQueue);
    	styleManager.createInsertStyleOperations(styleManager.getAutomaticStyles(true), "spreadsheet", operationQueue);

//    	styles.createInsertStyleOperations(operationQueue, styles.getOfficeStyles(), false, false);
//    	styles.createInsertStyleOperations(operationQueue, content.getAutomaticStyles_DEPRECATED(), true, false);

    	for(int i=0; i<content.getSheets().size();i++) {
    		createSheetOperations(content.getSheets().get(i), i);
    	}
    	createNamedExpressions(content.getNamedExpressions(false), null);
    	createDatabaseRanges(content.getDatabaseRanges(false));
        final JSONObject operations = new JSONObject(1);
        operations.put("operations", operationQueue);
        return operations;
    }

    private void createSetDocumentattributesOperation()
       	throws JSONException {

    	final JSONObject documentPropsObject = new JSONObject(2);
    	final JSONObject docPropsJson = new JSONObject(4);
        docPropsJson.putOpt("fileFormat", "odf");
    	docPropsJson.put("cols", content.getMaxColumnCount());
    	final int activeSheet = content.getSheetIndex(settings.getActiveSheet());
    	if(activeSheet>=0) {
    		docPropsJson.put("activeSheet", activeSheet);
    	}
    	final ConfigItemSet configItemSet = settings.getConfigItemSet("ooo:configuration-settings", false);
    	if(configItemSet!=null) {
    		final HashMap<String, IElementWriter> configItems = configItemSet.getItems();
    		final IElementWriter autoCalculate = configItems.get("AutoCalculate");
    		if(autoCalculate instanceof ConfigItem) {
    			final String val = ((ConfigItem)autoCalculate).getValue();
    			if(val!=null&&val.equals("true")) {
    				docPropsJson.put("calcOnLoad", true);
    			}
    		}
    	}
        documentPropsObject.put("document", docPropsJson);
        final JSONObject characterProps = new JSONObject(1);
        characterProps.put("fontSize", 10);
        documentPropsObject.put("character", characterProps);
        final JSONObject insertComponentObject = new JSONObject(2);
        insertComponentObject.put("name", "setDocumentAttributes");
        insertComponentObject.put("attrs", documentPropsObject);
        operationQueue.put(insertComponentObject);
    }

    private void createConditionalFormats(ConditionalFormats conditionalFormats, Integer sheetIndex)
    	throws JSONException, SAXException {

    	if(conditionalFormats!=null) {
    		final Iterator<Condition> conditionalFormatIter = conditionalFormats.getConditionalFormatList().values().iterator();
    		while(conditionalFormatIter.hasNext()) {
				operationQueue.put(conditionalFormatIter.next().createCondFormatRuleOperation(doc, sheetIndex));
    		}
    	}
    }

    private void createNamedExpressions(NamedExpressions namedExpressions, Integer sheetIndex)
    	throws JSONException {

    	if(namedExpressions!=null) {
    		final Iterator<Entry<String, NamedExpression>> namedExpressionIter = namedExpressions.getExpressionList().entrySet().iterator();
    		while(namedExpressionIter.hasNext()) {
    			final NamedExpression namedExpression = namedExpressionIter.next().getValue();
	        	final JSONObject insertComponentObject = new JSONObject(3);
	        	insertComponentObject.put("name", "insertName");
	        	insertComponentObject.put("label", namedExpression.getName());
	        	if(namedExpression.getExpression()!=null) {
	        		insertComponentObject.put("formula", namedExpression.getExpression());
        			insertComponentObject.put("isExpr", true);
	        	}
	        	else {
	        		String cellRange = namedExpression.getCellRangeAddress();
	        		if(cellRange==null) {
	        			cellRange = namedExpression.getBaseCellAddress();
	        		}
        			insertComponentObject.put("formula", (cellRange == null) ? "" : ("[" + cellRange + "]"));
    			}
	        	if(namedExpression.getBaseCellAddress()!=null) {
	        		insertComponentObject.put("ref", "[" + namedExpression.getBaseCellAddress() + "]");
	        	}
        		if(sheetIndex!=null) {
        			insertComponentObject.put("sheet", sheetIndex);
        		}
	        	operationQueue.put(insertComponentObject);
    		}
    	}
    }

    private void createDatabaseRanges(DatabaseRanges databaseRanges)
    	throws JSONException {

    	if(databaseRanges==null) {
    		return;
    	}
    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
    	while(databaseRangeIter.hasNext()) {
    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
    		if(!databaseRange.getName().startsWith("__Anonymous_Sheet_DB__") || databaseRange.isDisplayFilterButtons()) {
    			final String tableName = !databaseRange.getName().isEmpty()&&!databaseRange.getName().startsWith("__Anonymous_Sheet_DB__")
    					? databaseRange.getName()
    					: null;
    			final Range sheetRange = databaseRange.getRange();
    			if(sheetRange.getSheetIndex()>=0) {
	    			JSONObject insertComponentObject = new JSONObject(5);
	    			insertComponentObject.put("name", "insertTable");
	    			final int sheetIndex = sheetRange.getSheetIndex();
	    			insertComponentObject.put("sheet", sheetIndex);
	    			if(tableName!=null) {
	    				insertComponentObject.put("table", tableName);
	    			}
	    			final CellRefRange cellRefRange = sheetRange.getCellRefRange();
	    			final JSONArray start = cellRefRange.getStart().getJSONArray();
	    			final JSONArray end = cellRefRange.getEnd().getJSONArray();
	    			insertComponentObject.put("start", start);
	    			insertComponentObject.put("end", end);
	    			final JSONObject tableAttrs = databaseRange.createTableAttrs();
	    			if(tableAttrs!=null) {
		    			final JSONObject attrs = new JSONObject(1);
		    			attrs.put("table", tableAttrs);
		    			insertComponentObject.put("attrs", attrs);
	    			}
	    			operationQueue.put(insertComponentObject);

	    			final List<ElementNSImpl> databaseRangeChildList = databaseRange.getChilds();
	    			for(ElementNSImpl databaseRangeChild:databaseRangeChildList) {
	    				createDatabaseRangesImpl(databaseRangeChild, insertComponentObject, sheetIndex, tableName);
	    			}
    			}
    		}
    	}
    }

    private void createDatabaseRangesImpl(Node child, JSONObject insertComponentObject, int sheetIndex, String tableName)
    	throws JSONException {

    	while(child!=null) {
			final Node nextChild = child.getNextSibling();
			final MutableInt column = new MutableInt(-1);
			final JSONArray entries = new JSONArray();
			if(child.getLocalName().equals("filter-condition")) {
				createDatabaseColumnEntries(sheetIndex, child, column, new MutableBoolean(false), entries);
    			if(column.intValue()>=0&&entries.length()>0) {
	    			insertComponentObject = new JSONObject(4);
	    			insertComponentObject.put("name", "changeTableColumn");
	    			insertComponentObject.put("sheet", sheetIndex);
	    			if(tableName!=null) {
	    				insertComponentObject.put("table", tableName);
	    			}
                    insertComponentObject.put("col", column.intValue());
                    final JSONObject attrs = new JSONObject(1);
                    final JSONObject filterAttrs = new JSONObject(2);
                    filterAttrs.put("type", "discrete");
                    filterAttrs.put("entries", entries);
                    attrs.put("filter", filterAttrs);
                    insertComponentObject.put("attrs", attrs);
	    			operationQueue.put(insertComponentObject);
    			}
			}
			else {
        		final Node childChild = child.getFirstChild();
        		if(childChild!=null) {
        			createDatabaseRangesImpl(childChild, insertComponentObject, sheetIndex, tableName);
        		}
			}
			child = nextChild;
		}
    }

    private void createDatabaseColumnEntries(int sheetIndex, Node node, MutableInt column, MutableBoolean setItemFound, JSONArray entries)
    	throws JSONException {

    	if(node instanceof ElementNSImpl) {
    		final String elementName = ((ElementNSImpl)node).getLocalName();
    		if(elementName.equals("filter-condition")) {
                final String fieldNumber = ((ElementNSImpl) node).getAttributeNS(TableFieldNumberAttribute.ATTRIBUTE_NAME.getUri(), "field-number");
                if(!fieldNumber.isEmpty()&&column.intValue()==-1) {
                	column.setValue(Integer.parseInt(fieldNumber));
                }
                final String value = ((ElementNSImpl) node).getAttributeNS(TableValueAttribute.ATTRIBUTE_NAME.getUri(), "value");
                final String operator = ((ElementNSImpl) node).getAttributeNS(TableOperatorAttribute.ATTRIBUTE_NAME.getUri(), "operator");
                if(operator.equals("=")&&!value.isEmpty()) {
                	convertColumnChangeValue(entries, value);
                }
    		}
    		else if(elementName.equals("filter-set-item")) {
    			if(!setItemFound.booleanValue()) {
    				// if set items are used, then we will reset the entries table the first time so we remove eventual values set by the filter-condition
    				setItemFound.setValue(Boolean.TRUE);
    				entries.reset();
    			}
                final String value = ((ElementNSImpl) node).getAttributeNS(TableValueAttribute.ATTRIBUTE_NAME.getUri(), "value");
                if(!value.isEmpty()) {
                	convertColumnChangeValue(entries, value);
                }
    		}
    		final NodeList childNodes = ((ElementNSImpl)node).getChildNodes();
    		for(int i=0; i<childNodes.getLength(); i++) {
	    		createDatabaseColumnEntries(sheetIndex, childNodes.item(i), column, setItemFound, entries);
    		}
    	}
    }

    private void convertColumnChangeValue(JSONArray entries, String value)
    	throws JSONException {

    	try {
    		entries.put(Integer.parseInt(value));
    	}
    	catch (NumberFormatException e) {};
    	try {
    		entries.put(Double.parseDouble(value));
    	}
    	catch (NumberFormatException e) {};

    	entries.put(value);
    }

    private void createSheetOperations(Sheet sheet, int sheetIndex)
    	throws JSONException, SAXException {

    	final JSONObject insertComponentObject = new JSONObject(3);
        insertComponentObject.put("name", "insertSheet");
        insertComponentObject.put("sheet", sheetIndex);
        String tableName = sheet.getSheetName();
        if(tableName==null||tableName.isEmpty()) {
        	tableName = "table" + String.valueOf(sheetIndex+1);
        }
        insertComponentObject.put("sheetName", tableName);

        // creating and writing default attr styles...
        final JSONObject attrs = new JSONObject(4);
        createSheetProperties(sheet, attrs);
    	if(!attrs.isEmpty()) {
    		insertComponentObject.put("attrs", attrs);
    	}
        operationQueue.put(insertComponentObject);

        final ColumnStyles columnStyles = new ColumnStyles(sheet);
        createColumnOperations(columnStyles, sheet.getColumns(), sheetIndex);
        createRowOperations(sheet, sheetIndex, columnStyles);
        createMergeOperations(sheetIndex, sheet.getMergeCells());
        createHyperlinkOperations(sheetIndex, sheet.getHyperlinks());
        createSheetDrawingOperations(sheet, sheetIndex);
        createConditionalFormats(sheet.getConditionalFormats(false), sheetIndex);
        createNamedExpressions(sheet.getNamedExpressions(false), sheetIndex);
    }

    private void createSheetProperties(Sheet sheet, JSONObject attrs)
    	throws JSONException, SAXException {

    	final JSONObject sheetProperties = new JSONObject(10);
        final String sheetProtection = sheet.getAttributeNS(TableProtectedAttribute.ATTRIBUTE_NAME.getUri(), TableProtectedAttribute.ATTRIBUTE_NAME.getLocalName());
        if(sheetProtection.equals("true")) {
        	sheetProperties.put("locked", true);
        }
        final String sheetStyle = sheet.getStyleName();
        if(sheetStyle!=null&&!sheetStyle.isEmpty()) {
        	final StyleBase styleBase = styleManager.getStyle(sheetStyle, "table", true);
        	if(styleBase instanceof StyleTable) {
        		final TableProperties tableProperties = ((StyleTable)styleBase).getTableProperties();
        		final String display = tableProperties.getAttribute("table:display");
        		if(display!=null&&display.equals("false")) {
        			sheetProperties.put("visible", false);
        		}
        	}
        }
        final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(false);
        if(globalViewSettings!=null) {
			final ConfigItemMapEntry sheetViewSettings = settings.getViewSettings(globalViewSettings, sheet.getSheetName(), false);
			final Integer cursorPositionX = getConfigValueInt(globalViewSettings, sheetViewSettings, "CursorPositionX");
			final Integer cursorPositionY = getConfigValueInt(globalViewSettings, sheetViewSettings, "CursorPositionY");
			if(cursorPositionX!=null&&cursorPositionY!=null) {
				final SmlUtils.CellRef cellRef = new SmlUtils.CellRef(cursorPositionX, cursorPositionY);
    			final JSONArray ranges = new JSONArray();
    			final JSONObject range = new JSONObject(2);
	            range.put("start", cellRef.getJSONArray());
				range.put("end", cellRef.getJSONArray());
				ranges.put(range);
				sheetProperties.put("selectedRanges", ranges);
				sheetProperties.put("activeIndex", 0);
			}
			final Integer zoomValue = getConfigValueInt(globalViewSettings, sheetViewSettings, "ZoomValue");
			if(zoomValue!=null) {
				sheetProperties.put("zoom", zoomValue / 100.0);
			}
			final Integer positionX1 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionLeft");
			if(positionX1!=null) {
				sheetProperties.put("scrollLeft", positionX1);
			}
			final Integer positionY1 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionTop");
			if(positionY1!=null) {
				sheetProperties.put("scrollBottom", positionY1);
			}
			final Integer positionX2 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionRight");
			if(positionX2!=null) {
				sheetProperties.put("scrollRight", positionX2);
			}
			final Integer positionY2 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionBottom");
			if(positionY2!=null) {
				sheetProperties.put("scrollTop", positionY2);
			}
			final String showGrid = getConfigValue(globalViewSettings, sheetViewSettings, "ShowGrid");
			if(showGrid!=null&&showGrid.equals("false")) {
				sheetProperties.put("showGrid", false);
			}
        }
        if(!sheetProperties.isEmpty()) {
            attrs.put("sheet", sheetProperties);
        }
        if(!sheet.getColumns().isEmpty()) {
            sheet.getColumns().last().createAttributes(attrs, doc);
        }
        final AttrsHash<Row> defaultRowAttrs = getDefaultAttrs(sheet.getRows());
        if(defaultRowAttrs!=null) {
            defaultRowAttrs.getObject().createAttributes(attrs, doc);
        }
    }

    private Integer getConfigValueInt(ConfigItemMapEntry global, ConfigItemMapEntry local, String name) {
    	Integer val = null;
    	final String stringVal = getConfigValue(global, local, name);
    	if(stringVal!=null) {
    		try {
    			val = Integer.valueOf(stringVal);
    		}
    		catch(NumberFormatException e) {
    			return null;
    		}
    	}
    	return val;
    }
    private String getConfigValue(ConfigItemMapEntry global, ConfigItemMapEntry local, String name) {
    	String val = null;
    	if(local!=null) {
    		final IElementWriter item = local.getItems().get(name);
    		if(item instanceof ConfigItem) {
    			val = ((ConfigItem)item).getValue();
    		}
    	}
    	if(val==null&&global!=null) {
    		final IElementWriter item = global.getItems().get(name);
    		if(item instanceof ConfigItem) {
    			val = ((ConfigItem)item).getValue();
    		}
    	}
    	return val;
    }

    private <T> AttrsHash<T> getDefaultAttrs(TreeSet<? extends IAttrs<T>> objs) {
    	// retrieving the default style (most used styles)
    	AttrsHash<T> maxUsedAttrs = null;
    	final HashMap<AttrsHash<T>, AttrsHash<T>> attrMap = new HashMap<AttrsHash<T>, AttrsHash<T>>();
    	for(IAttrs<T> obj:objs) {

    		final AttrsHash<T> attrHash = obj.createAttrHash();
    		AttrsHash<T> hashEntry = attrMap.get(attrHash);
    		if(hashEntry==null) {
    			attrMap.put(attrHash, attrHash);
    			hashEntry = attrHash;
    		}
    		else {
    			hashEntry.setCount(hashEntry.getCount()+attrHash.getCount());
    		}
			if(maxUsedAttrs==null) {
				maxUsedAttrs = hashEntry;
			}
			else if(hashEntry.getCount()>maxUsedAttrs.getCount()) {
				maxUsedAttrs = hashEntry;
			}
    	}
    	return maxUsedAttrs;
    }

    private void createMergeOperations(int sheetIndex, List<MergeCell> mergeCells)
    	throws JSONException {

    	for(MergeCell mergeCell:mergeCells) {
	    	final JSONObject addMergeCellsObject = new JSONObject(5);
	        addMergeCellsObject.put("name", "mergeCells");
	        addMergeCellsObject.put("sheet", sheetIndex);
	        addMergeCellsObject.put("start", mergeCell.getCellRefRange().getStart().getJSONArray());
	        addMergeCellsObject.put("end", mergeCell.getCellRefRange().getEnd().getJSONArray());
	        addMergeCellsObject.putOpt("type", "merge");
	        operationQueue.put(addMergeCellsObject);
    	}
    }

    private void createHyperlinkOperations(int sheetIndex, List<Hyperlink> hyperlinks)
    	throws JSONException {

    	for(Hyperlink hyperlink:hyperlinks) {
	    	final JSONObject insertHyperlinkObject = new JSONObject(5);
	    	insertHyperlinkObject.put("name", "insertHyperlink");
	    	insertHyperlinkObject.put("sheet", sheetIndex);
	    	final CellRefRange cellRefRange = hyperlink.getCellRefRange();
	    	insertHyperlinkObject.put("start", cellRefRange.getStart().getJSONArray());
	    	if(!cellRefRange.getStart().equals(cellRefRange.getEnd())) {
	    		insertHyperlinkObject.put("end", cellRefRange.getEnd().getJSONArray());
	    	}
	    	insertHyperlinkObject.put("url", hyperlink.getUrl());
	        operationQueue.put(insertHyperlinkObject);
    	}
    }

    private void createSheetDrawingOperations(Sheet sheet, int sheetIndex)
    	throws JSONException, SAXException {

    	final Drawings drawings = sheet.getDrawings();
    	for(int i=0; i<drawings.getCount(); i++) {
    		final Drawing drawing = drawings.getDrawing(i);
    		final JSONObject setInsertDrawingOperation = new JSONObject();
    		setInsertDrawingOperation.put("name", "insertDrawing");
    		final JSONArray start = new JSONArray(2);
    		start.put(sheetIndex);
    		start.put(i);
    		setInsertDrawingOperation.put("start", start);
    		setInsertDrawingOperation.put("type", drawing.getType());
    		final JSONObject attrs = drawing.createAttributes(sheet, null);
    		if(!attrs.isEmpty()) {
    			setInsertDrawingOperation.put("attrs", attrs);
    		}
    		operationQueue.put(setInsertDrawingOperation);
    	}
    }

    private void createColumnOperations(ColumnStyles columnStyles, TreeSet<Column> columns, int sheetIndex)
    	throws JSONException, SAXException {

    	final Iterator<ColumnStyle> columnStyleIter = columnStyles.iterator();
    	while(columnStyleIter.hasNext()) {
    		ColumnStyle columnStyle = columnStyleIter.next();
    		final Iterator<Column> columnIter = columns.subSet(new Column(columnStyle.getMin()), true, new Column(columnStyle.getMax()), true).iterator();
    		while(columnIter.hasNext()) {
    			final Column column = columnIter.next();
        		final JSONObject attrs = new JSONObject();
    			column.createAttributes(attrs, doc);
    			if(!attrs.isEmpty()||columnStyle.getMaxUsedStyle()!=null) {
    		        final int start = column.getMin();
    		        final int end = column.getMax();
    		        final JSONObject operation = new JSONObject(6);
    		        operation.put("name", "changeColumns");
    		        operation.put("sheet", sheetIndex);
    		        operation.put("start", start);
    		        if(end>start) {
    		        	operation.put("end", end);
    		        }
    		        if(!attrs.isEmpty()) {
    		        	operation.put("attrs", attrs);
    		        }
    		        if(columnStyle.getMaxUsedStyle()!=null) {
    		        	operation.put("s", columnStyle.getMaxUsedStyle());
    		        }
    		        operationQueue.put(operation);
    			}
        	}
    	}
    }

    private class CellContentProducer {

        final JSONArray cellContents = new JSONArray();

        JSONObject lastCellData = null;
        int lastCellRepeat = 0;

        public CellContentProducer() {
        }

        public void addCellData(JSONObject cellData, int cellRepeat) throws JSONException {
            if(lastCellData!=null&&lastCellData.isEqualTo(cellData)) {
                lastCellRepeat += cellRepeat;
            }
            else {
                if(lastCellData!=null) {
                    if(lastCellRepeat>1) {
                        lastCellData.put("r", lastCellRepeat);
                    }
                    cellContents.put(lastCellData);
                }
                lastCellData = cellData;
                lastCellRepeat = cellRepeat;
            }
        }

        public void writeCellContent(JSONObject rowData) throws JSONException {
            if(lastCellData!=null) {
                if(lastCellRepeat>1) {
                    lastCellData.put("r", lastCellRepeat);
                }
                cellContents.put(lastCellData);
            }
            for(int i = cellContents.length() - 1; i >= 0; i--) {
                final JSONObject rowContent = cellContents.getJSONObject(i);
                final Set<Entry<String, Object>> rowContentEntrySet = rowContent.entrySet();
                if(rowContentEntrySet.size()==0||(rowContentEntrySet.size()==1&&rowContent.has("r"))) {
                    cellContents.remove(i);
                }
                else {
                    break;
                }
            }
            if(!cellContents.isEmpty()) {
                rowData.put("c", cellContents);
            }
        }
    }

    public void createRowOperations(Sheet sheet, int sheetIndex, ColumnStyles columnStyles)
    	throws JSONException, SAXException {

    	final TreeSet<Row> rows = sheet.getRows();
    	if(rows.isEmpty()) {
    		return;
    	}

    	JSONObject	lastRowObject = null;
    	int 		lastRowEnd = 0;
    	String 		lastRowCellStyle = null;
    	JSONObject	lastRowAttrs = null;
    	for(Row row:rows) {
			final JSONObject attrs = new JSONObject();
    		row.createAttributes(attrs, doc);
    		if(!attrs.isEmpty()||row.getDefaultCellStyle()!=null) {
    	        final JSONObject operation = new JSONObject(6);
    	        final int start = row.getRow();
    	        final int end = (start + row.getRepeated())-1;
    	        operation.put("name", "changeRows");
    	        operation.put("sheet", sheetIndex);
    	        operation.put("start", start);
    	        if(end!=start) {
    	        	operation.put("end", end);
    	        }
    	        if(attrs!=null)	{
    	        	operation.put("attrs", attrs);
    	        }
    	        final String defaultCellStyle = row.getDefaultCellStyle();
    	        if(defaultCellStyle!=null) {
    	        	operation.put("s", defaultCellStyle);
    	        }
    	        boolean identicalRowAttrs = false;
    	        if(lastRowObject!=null&&lastRowCellStyle==defaultCellStyle&&lastRowEnd+1==start) {
    	        	identicalRowAttrs = attrs==null&&lastRowAttrs==null;
    	        	if(attrs!=null&&lastRowAttrs!=null) {
    	        		identicalRowAttrs = attrs.isEqualTo(lastRowAttrs);
    	        	}
    	        }
    	        if(identicalRowAttrs) {
    	        	lastRowObject.put("end", end);
    	        }
    	        else {
    	        	lastRowObject = operation;
        	        operationQueue.put(operation);
    	        }
	        	lastRowEnd = end;
	        	lastRowCellStyle = defaultCellStyle;
	        	lastRowAttrs = attrs;
    		}
    	}
    	final TreeSet<Column> columns = sheet.getColumns();
    	final JSONObject changeCellOperation = new JSONObject(4);
    	changeCellOperation.put("name", "changeCells");
    	changeCellOperation.put("sheet", sheetIndex);
    	changeCellOperation.put("start", createPosition(0, 0));
    	final JSONArray rowContents = new JSONArray(rows.size());
    	changeCellOperation.put("contents", rowContents);
    	operationQueue.put(changeCellOperation);
    	for(Row row:rows) {
    		final JSONObject rowData = new JSONObject(2);
    		rowContents.put(rowData);
    		if(row.getRepeated()>1) {
    			rowData.put("r", row.getRepeated());
    		}
    		final TreeSet<Cell> cells = row.getCells();
    		final CellContentProducer cellContentProducer = new CellContentProducer();
    		for(Cell cell:cells) {
    			if(row.getDefaultCellStyle()!=null) {
    				createChangeCellOperation(cellContentProducer, cell, row.getRow(), cell.getColumn(), cell.getRepeated(), columnStyles, row.getDefaultCellStyle());
    			}
    			else if(!columns.isEmpty()) {
    				if(cell.getCellStyle()!=null) {
    					createChangeCellOperation(cellContentProducer, cell, row.getRow(), cell.getColumn(), cell.getRepeated(), columnStyles, null);
    				}
    				else {
    					String defaultCellStyle = null;
    					// no row and no cell style ... the column default cell style is necessary
    					final int max = (cell.getColumn()+cell.getRepeated())-1;
    					for(int min = cell.getColumn(); min<=max; ) {
    						final Column col = columns.floor(new Column(min));
        					if(col.getMax()<min) {
        						// there are no more columns, the last used default cell style is used
        						createChangeCellOperation(cellContentProducer, cell, row.getRow(), min, (max-min)+1, columnStyles, defaultCellStyle);
        						break;
        					}
        					else {
        						if(col.getDefaultCellStyle()!=null) {
        							defaultCellStyle = col.getDefaultCellStyle();
        						}
        						if(col.getMax()>=max) {
        							createChangeCellOperation(cellContentProducer, cell, row.getRow(), min, (max-min)+1, columnStyles, defaultCellStyle);
	        						break;
        						}
        						else {
        							createChangeCellOperation(cellContentProducer, cell, row.getRow(), min, (col.getMax()-min)+1, columnStyles, defaultCellStyle);
        							min = col.getMax()+1;
        						}
        					}
    					}
    				}
    			}
        		else {
        			createChangeCellOperation(cellContentProducer, cell, row.getRow(), cell.getColumn(), cell.getRepeated(), columnStyles, null);
    			}
    		}
    		cellContentProducer.writeCellContent(rowData);
    	}
		for(int i = rowContents.length() - 1; i >= 0; i--) {
			final JSONObject rowContent = rowContents.getJSONObject(i);
			final Set<Entry<String, Object>> rowContentEntrySet = rowContent.entrySet();
			if(rowContentEntrySet.size()==0||(rowContentEntrySet.size()==1&&rowContent.has("r"))) {
				rowContents.remove(i);
			}
			else {
				break;
			}
		}
    }

    public void createChangeCellOperation(CellContentProducer cellContentProducer, Cell cell, int row, int column, int repeated, ColumnStyles columnStyles, String defaultCellStyle)
    	throws JSONException {

    	final int min = column;
    	final int max = (column + repeated) - 1;

    	columnStyles.getColumnStyle(min, true, true, false);
    	columnStyles.getColumnStyle(max, true, false, true);

        final Iterator<ColumnStyle> columnIter = columnStyles.subSet(new ColumnStyle(min), true, new ColumnStyle(max), true).iterator();
    	while(columnIter.hasNext()) {

    		final ColumnStyle columnStyle = columnIter.next();

    		final JSONObject cellData = new JSONObject();

    		int cMin = columnStyle.getMin();
    		int cMax = columnStyle.getMax();

    		if(cMin<min) {
    			cMin = min;
    		}
    		if(cMax>max) {
    			cMax = max;
    		}

    		int cellRepeat = (cMax-cMin>0) ? (cMax-cMin)+1 : 1;

    		boolean isEmptyCell = true;
	    	if(cell.getCellFormula()!=null) {
	    		isEmptyCell = false;
	    		cellData.put("f", cell.getCellFormula());
	    	}
	    	final Object content = cell.getCellContent();
	    	if(content instanceof Cell.ErrorCode) {
	    		isEmptyCell = false;
	    		final String errorCode = ((Cell.ErrorCode)content).getError();
	    		cellData.put("e", errorCode!=null ? errorCode : "");
	    	}
	    	else if(content!=null) {
	    		isEmptyCell = false;
    			cellData.put("v", content);
	    	}
	    	if(isEmptyCell) {
	    		String s = null;
		    	if(cell.getCellStyle()!=null) {
		    		s = cell.getCellStyle();
		    	}
		    	else if(defaultCellStyle!=null) {
		    		s = defaultCellStyle;
		    	}
		    	if(s!=null&&!s.equals(columnStyle.getMaxUsedStyle())) {
		    		cellData.put("s", s);
		    	}
	    	}
	    	else {
	    		String s = null;
		    	if(cell.getCellStyle()!=null) {
		    		s = cell.getCellStyle();
		    	}
		    	else if(defaultCellStyle!=null) {
		    		s = defaultCellStyle;
		    	}
		    	else {
		    		s = columnStyle.getMaxUsedStyle();
		    	}
		    	if(s!=null) {
		    		cellData.put("s", s);
		    	}
	    	}
	    	final CellAttributesEnhanced enhancedCellAttributes = cell.getCellAttributesEnhanced(false);
	    	if(enhancedCellAttributes!=null) {
	    	    final String matrixColumnsSpanned = enhancedCellAttributes.getNumberMatrixColumnsSpanned();
	    	    final String matrixRowsSpanned = enhancedCellAttributes.getNumberMatrixRowsSpanned();
	    	    if(matrixColumnsSpanned!=null||matrixRowsSpanned!=null) {
	    	        final int matrixColumns = matrixColumnsSpanned != null ? Integer.parseInt(matrixColumnsSpanned) : 1;
                    final int matrixRows = matrixRowsSpanned != null ? Integer.parseInt(matrixRowsSpanned) : 1;
                    cellData.put("mr", SmlUtils.getCellRefRange(cell.getColumn(), row, (cell.getColumn() + matrixColumns) - 1, (row + matrixRows) - 1));
	    	    }
	    	}
	    	cellContentProducer.addCellData(cellData, cellRepeat);
    	}
    }

    public static JSONArray createPosition(int columnNumber, int rowNumber) {

    	final JSONArray position = new JSONArray(2);
    	position.put(columnNumber);
    	position.put(rowNumber);
    	return position;
    }

    public void insertAutoStyle(String family, String styleId, Map<String, Object> componentProps, String parentStyle, boolean isDefaultStyle)
    	throws JSONException {

    	final JSONObject insertAutoStyleOperation = new JSONObject(5);
    	insertAutoStyleOperation.put("name", "insertAutoStyle");
    	insertAutoStyleOperation.put("type", family);
    	insertAutoStyleOperation.put("styleId", styleId);
    	if(parentStyle!=null) {
    		componentProps.put("styleId", parentStyle);
    	}
    	insertAutoStyleOperation.put("attrs", componentProps);
    	if(isDefaultStyle) {
    		insertAutoStyleOperation.put("default", true);
    	}
    	operationQueue.put(insertAutoStyleOperation);
    }
}
