/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odf.styles;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.odf.properties.ParagraphProperties;
import com.openexchange.office.odf.properties.TableCellProperties;
import com.openexchange.office.odf.properties.TextProperties;

final public class StyleTableCell extends StyleBase implements ITextProperties, IParagraphProperties {

	private TableCellProperties tableCellProperties = new TableCellProperties(new AttributesImpl());
	private ParagraphProperties paragraphProperties = new ParagraphProperties(new AttributesImpl());
	private TextProperties textProperties = new TextProperties(new AttributesImpl());

	public StyleTableCell(String name, boolean automaticStyle, boolean contentStyle) {
		super("table-cell", name, automaticStyle, contentStyle);
	}

	public StyleTableCell(String name, AttributesImpl attributesImpl, boolean defaultStyle, boolean automaticStyle, boolean contentStyle) {
		super(name, attributesImpl, defaultStyle, automaticStyle, contentStyle);
	}

	@Override
    public String getFamily() {
		return "table-cell";
	}

	public TableCellProperties getTableCellProperties() {
		return tableCellProperties;
	}

	@Override
    public ParagraphProperties getParagraphProperties() {
		return paragraphProperties;
	}

	@Override
    public TextProperties getTextProperties() {
		return textProperties;
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		SaxContextHandler.startElement(output, getNamespace(), getLocalName(), getQName());
		writeAttributes(output);
		tableCellProperties.writeObject(output);
		paragraphProperties.writeObject(output);
		textProperties.writeObject(output);
		writeMapStyleList(output);
		SaxContextHandler.endElement(output, getNamespace(), getLocalName(), getQName());
	}

	@Override
	public void mergeAttrs(StyleBase styleBase) {
		if(styleBase instanceof IParagraphProperties) {
			getParagraphProperties().mergeAttrs(((IParagraphProperties)styleBase).getParagraphProperties());
		}
		if(styleBase instanceof ITextProperties) {
			getTextProperties().mergeAttrs(((ITextProperties)styleBase).getTextProperties());
		}
		if(styleBase instanceof StyleTableCell) {
			getTableCellProperties().mergeAttrs(((StyleTableCell)styleBase).getTableCellProperties());
		}
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs)
		throws JSONException {

		getParagraphProperties().applyAttrs(styleManager, attrs);
		getTextProperties().applyAttrs(styleManager, attrs);
		getTableCellProperties().applyAttrs(styleManager, attrs);

		final JSONObject cellAttrs = attrs.optJSONObject("cell");
        if(cellAttrs!=null) {
            final String formatCode = cellAttrs.optString("formatCode");
            if(!formatCode.isEmpty()) {
            	attributes.setValue(Namespaces.STYLE, "data-style-name", "style:data-style-name",
            		styleManager.applyDataStyle(formatCode, cellAttrs.optInt("formatId", -1), false, isAutoStyle(), isContentStyle()));
            }
            else {
            	attributes.remove("style:date-style-name");
            }
        }
	}

	@Override
	public void createAttrs(StyleManager styleManager, OpAttrs attrs) {
		Object alignment = null;
		paragraphProperties.createAttrs(styleManager, isContentStyle(), attrs);
		alignment = attrs.remove("paragraph", "alignment");
		textProperties.createAttrs(styleManager, isContentStyle(), attrs);
		if(alignment!=null) {
			attrs.getMap("cell", true).put("alignHor", alignment);
		}
		tableCellProperties.createAttrs(styleManager, isContentStyle(), attrs);
		final String dataStyleName = attributes.getValue("style:data-style-name");
		if(dataStyleName!=null) {
			String formatCode = "";
			final StyleBase dataStyle = styleManager.getStyle(dataStyleName, "data-style", isContentStyle());
			if(dataStyle instanceof NumberStyleBase) {
				formatCode = ((NumberStyleBase)dataStyle).getFormat(styleManager, isContentStyle());
			}
			attrs.getMap("cell", true).put("formatCode", formatCode);
		}
	}

	@Override
	protected int _hashCode() {

		int result = 0;

		result += tableCellProperties.hashCode();
		result += paragraphProperties.hashCode();
		result += textProperties.hashCode();

		return result;
	}

	@Override
	protected boolean _equals(StyleBase e) {
		final StyleTableCell other = (StyleTableCell)e;
		if(!tableCellProperties.equals(other.tableCellProperties)) {
			return false;
		}
		if(!paragraphProperties.equals(other.paragraphProperties)) {
			return false;
		}
		if(!textProperties.equals(other.textProperties)) {
			return false;
		}
		return true;
	}

	@Override
	public StyleTableCell clone() {
		final StyleTableCell clone = (StyleTableCell)_clone();
		clone.tableCellProperties = tableCellProperties.clone();
		clone.paragraphProperties = paragraphProperties.clone();
		clone.textProperties = textProperties.clone();
		return clone;
	}
}
