/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odf.properties;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.odf.AttributeImpl;
import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.Length;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.styles.StyleManager;

final public class TextProperties extends StylePropertiesBase {

	public TextProperties(AttributesImpl attributesImpl) {
		super(attributesImpl);
	}

	@Override
	public String getQName() {
		return "style:text-properties";
	}

	@Override
	public String getLocalName() {
		return "text-properties";
	}

	@Override
	public String getNamespace() {
		return Namespaces.STYLE;
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs)
		throws JSONException {

		final JSONObject characterAttributes = attrs.optJSONObject("character");
		if(characterAttributes!=null) {

	    	Object language = null;
	    	Object noProof = null;

			final Iterator<Entry<String, Object>> characterIter = characterAttributes.entrySet().iterator();
			while(characterIter.hasNext()) {
				final Entry<String, Object> characterEntry = characterIter.next();
				final Object value = characterEntry.getValue();
				switch(characterEntry.getKey()) {
					case "color": {
						if (value==JSONObject.NULL) {
							attributes.remove("fo:color");
						}
						else {
	                        final JSONObject color = (JSONObject)value;
	                        if (color.hasAndNotNull("type")) {
	                            String type = color.optString("type", "");
	                            if(!type.equals("AUTO")) {
	                            	attributes.setValue(Namespaces.FO, "color", "fo:color", PropertyHelper.getColor(color, null));
	                            	attributes.remove("style:use-window-font-color");
	                            } else {
	                            	attributes.setValue(Namespaces.STYLE, "use-window-font-color", "style:use-window-font-color", "true");
	                            }
	                        } else {
	                            attributes.setValue(Namespaces.STYLE, "use-window-font-color", "style:use-window-font-color", "true");
	                        }
						}
						break;
					}
					case "bold": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("fo:font-weight");
		                }
		                else {
	                        attributes.setValue(Namespaces.FO, "font-weight", "fo:font-weight", ((Boolean)value).booleanValue() ? "bold" : "normal");
		                }
		                break;
					}
					case "boldAsian": {
						if(value==JSONObject.NULL) {
							attributes.remove("style:font-weight-asian");
						} else {
							attributes.setValue(Namespaces.STYLE, "font-weight-asian", "style:font-weight-asian", ((Boolean)value).booleanValue() ? "bold" : "normal");
						}
						break;
					}
					case "boldComplex": {
						if(value==JSONObject.NULL) {
							attributes.remove("style:font-weight-complex");
						} else {
							attributes.setValue(Namespaces.STYLE, "font-weight-complex", "style:font-weight-complex", ((Boolean)value).booleanValue() ? "bold" : "normal");
						}
						break;
					}
					case "underline": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("style:text-underline-style");
		                    attributes.remove("style:text-underline-width");
		                    attributes.remove("style:text-underline-color");
		                    attributes.remove("style:text-underline-type");
		                } else {
							attributes.setValue(Namespaces.STYLE, "text-underline-style", "style:text-underline-style", ((Boolean)value).booleanValue() ? "solid" : "none");
		                }
		                break;
					}
					case "italic": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("fo:font-style");
		                } else {
							attributes.setValue(Namespaces.FO, "font-style", "fo:font-style", ((Boolean)value).booleanValue() ? "italic" : "normal");
		                }
		                break;
					}
					case "italicAsian": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("style:font-style-asian");
		                } else {
							attributes.setValue(Namespaces.STYLE, "font-style-asian", "style:font-style-asian", ((Boolean)value).booleanValue() ? "italic" : "normal");
		                }
		                break;
	                }
					case "italicComplex": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("style:font-style-complex");
		                } else {
							attributes.setValue(Namespaces.STYLE, "font-style-complex", "style:font-style-complex", ((Boolean)value).booleanValue() ? "italic" : "normal");
		                }
		                break;
	                }
					case "fillColor": {
		                if(value==JSONObject.NULL) {
		                	attributes.remove("fo:background-color");
		                } else {
		                    attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", PropertyHelper.getColor((JSONObject)value, "transparent"));
		                }
		                break;
					}
					case "fontSize": {
						if(value==JSONObject.NULL) {
							attributes.remove("fo:font-size");
						} else {
							attributes.setValue(Namespaces.FO, "font-size", "fo:font-size", value.toString() + "pt");
						}
						break;
					}
					case "fontSizeAsian": {
						if(value==JSONObject.NULL) {
							attributes.remove("style:font-size-asian");
						} else {
							attributes.setValue(Namespaces.STYLE, "font-size-asian", "style:font-size-asian", value.toString() + "pt");
						}
						break;
					}
					case "fontSizeComplex": {
						if(value==JSONObject.NULL) {
							attributes.remove("style:font-size-complex");
						} else {
							attributes.setValue(Namespaces.STYLE, "font-size-complex", "style:font-size-complex", value.toString() + "pt");
						}
						break;
					}
					case "fontName": {
		                if (value==JSONObject.NULL) {
		                	attributes.remove("style:font-name");
		                } else {
		                	attributes.setValue(Namespaces.STYLE, "font-name", "style:font-name", (String)value);
		                	styleManager.addFontToDocument((String)value);
		                }
		                break;
					}
					case "fontNameAsian": {
		                if (value==JSONObject.NULL) {
		                	attributes.remove("style:font-name-asian");
		                } else {
		                	attributes.setValue(Namespaces.STYLE, "font-name-asian", "style:font-name-asian", (String)value);
		                	styleManager.addFontToDocument((String)value);
		                }
		                break;
					}
					case "fontNameComplex": {
		                if (value==JSONObject.NULL) {
		                	attributes.remove("style:font-name-complex");
		                } else {
		                	attributes.setValue(Namespaces.STYLE, "font-name-complex", "style:font-name-complex", (String)value);
		                	styleManager.addFontToDocument((String)value);
		                }
		                break;
					}
					case "vertAlign": {
		                if (value == null || value.equals(JSONObject.NULL)) {
		                	attributes.remove("style:text-position");
		                } else {
		                	final String alignment = (String)value;
		                    if (alignment.equals("sub")) {
		                    	attributes.setValue(Namespaces.STYLE, "text-position", "style:text-position", "sub");
		                    } else if (alignment.equals("super")) {
		                    	attributes.setValue(Namespaces.STYLE, "text-position", "style:text-position", "super");
		                    } else { //baseline
		                    	attributes.setValue(Namespaces.STYLE, "text-position", "style:text-position", "0% 100%");
		                    }
		                }
		                break;
					}
					case "strike": {
		                if(value==JSONObject.NULL) {
		                    attributes.remove("style:text-line-through-color");
		                    attributes.remove("style:text-line-through-mode");
		                    attributes.remove("style:text-line-through-style");
		                    attributes.remove("style:text-line-through-text");
		                    attributes.remove("style:text-line-through-text-style");
		                    attributes.remove("style:text-line-through-type");
		                    attributes.remove("style:text-line-through-width");
		                }
		                else {
		                    String strikeType = (String)value;
		                    if(strikeType.equals("none")) {
		                        attributes.setValue(Namespaces.STYLE, "text-line-through-style", "style:text-line-through-style", "none");
		                    }
		                    else {
		                        attributes.setValue(Namespaces.STYLE, "text-line-through-style", "style:text-line-through-style", "solid");
		                    	attributes.setValue(Namespaces.STYLE, "text-line-through-type", "style:text-line-through-type", strikeType.equals("single") ? "single" : "double");
		                        attributes.setValue(Namespaces.STYLE, "text-line-through-mode", "style:text-line-through-mode", "continuous");
		                        attributes.setValue(Namespaces.STYLE, "text-underline-mode", "style:text-underline-mode", "continuous");
		                        attributes.setValue(Namespaces.STYLE, "text-overline-mode", "style:text-overline-mode", "continuous");
		                    }
		                }
		                break;
					}
					case "language": {
						language = value;
						break;
					}
					case "noProof": {
						noProof = value;
						break;
					}
					case "letterSpacing": {
		                if (value==JSONObject.NULL) {
		                	attributes.remove("fo:letter-spacing");
		                }
		                else {
		                	attributes.setValue(Namespaces.FO, "letter-spacing", "fo:letter-spacing", ((String)value).equals("normal") ? "normal" : ((Number)value).longValue() / 100 + "mm");
		                }
		                break;
					}
				}
			}
			
	        if(noProof!=null || language!=null) {
	        	Object newLanguage = language;
	        	if((noProof instanceof Boolean && ((Boolean)noProof).booleanValue()) || ((language instanceof String) && ((String)language).equals("none"))) {
	    			attributes.setValue(Namespaces.FO, "language", "fo:language", "zxx");
	    			attributes.setValue(Namespaces.STYLE, "language-asian", "style:language-asian", "zxx");
	    			attributes.setValue(Namespaces.STYLE, "language-complex", "style:language-complex", "zxx");
	    			attributes.setValue(Namespaces.FO, "country", "fo:country", "none");
	    			attributes.setValue(Namespaces.STYLE, "country-asian", "style:country-asian", "none");
	    			attributes.setValue(Namespaces.STYLE, "country-complex", "style:country-complex", "none");
	        	}
	        	else if (newLanguage == null || newLanguage.equals(JSONObject.NULL)) {
	        		attributes.remove("fo:country");
	        		attributes.remove("fo:language");
	        		attributes.remove("style:country-asian");
	        		attributes.remove("style:language-asian");
	        		attributes.remove("style:country-complex");
	        		attributes.remove("style:language-complex");
	            }
	        	else {
	                String locale = (String)newLanguage;
	                if (!locale.isEmpty()) {
	                    int delimiterPos = locale.indexOf('-');
	                    if (delimiterPos > -1) {
	                    	attributes.setValue(Namespaces.FO, "language", "fo:language", locale.substring(0, delimiterPos));
	                    	attributes.setValue(Namespaces.FO, "country", "fo:country", locale.substring(delimiterPos + 1, locale.length()));
	                    } else {
	                    	attributes.setValue(Namespaces.FO, "language", "fo:language", locale);
	                    }
	                }
	            }
	        }
		}
	}	

	@Override
	public void createAttrs(StyleManager styleManager, boolean contentAutoStyle, OpAttrs attrs) {
		final HashMap<String, Object> characterAttrs = attrs.getMap("character", true);
		final Iterator<Entry<String, AttributeImpl>> propIter = attributes.getUnmodifiableMap().entrySet().iterator();
        while(propIter.hasNext()) {
        	final Entry<String, AttributeImpl> propEntry = propIter.next();
        	final String propName = propEntry.getKey();
        	final String propValue = propEntry.getValue().getValue();
        	switch(propName) {
        		case "fo:letter-spacing": {
	        	    if (propValue.equals("normal")) {
	        	    	characterAttrs.put("letterSpacing", propValue);
	        	    } else {
	        	        characterAttrs.put("letterSpacing", AttributesImpl.normalizeLength(propValue));
	        	    }
	        	    break;
        		}
        		case "fo:font-size": {
	        	    if (propValue.contains("%")) {
	        	    	//
	        	    } else {
	        	        characterAttrs.put("fontSize", new Length(propValue).getPoint());
	        	    }
	        	    break;
        		}
        		case "style:font-size-asian": {
	        	    if (propValue.contains("%")) {
	        	    	//
	        	    } else {
	        	        characterAttrs.put("fontSizeAsian", new Length(propValue).getPoint());
	        	    }
	        	    break;
        		}
        		case "style:font-size-complex": {
	        	    if (propValue.contains("%")) {
	        	    	//
	        	    } else {
	        	        characterAttrs.put("fontSizeComplex", new Length(propValue).getPoint());
	        	    }
	        	    break;
        		}
        		case "style:font-name": {
        			characterAttrs.put("fontName", propValue);
        			break;
        		}
        		case "style:font-name-asian": {
        			characterAttrs.put("fontNameAsian", propValue);
        			break;
        		}
        		case "style:font-name-complex": {
        			characterAttrs.put("fontNameComplex", propValue);
        			break;
        		}
        		case "style:text-position": {
	        	    if (propValue.contains("sub")) {
	        	        characterAttrs.put("vertAlign", "sub");
	        	    } else if (propValue.contains("super")) {
	        	        characterAttrs.put("vertAlign", "super");
	        	    } else if (propValue.equals("0%") || propValue.equals("0% 100%")) {
	        	        characterAttrs.put("vertAlign", "baseline");
	        	    }
	        	    break;
        		}
        		case "fo:language": {
	        	    if (!propValue.equals("none")) {
	        			final String country = attributes.getValue("fo:country");
	        			String language = propValue;
	        	        if (country!=null && !country.isEmpty()&&!country.equals("none")) {
	        	        	language = propValue + '-' + country;
	        	        }
	        	        characterAttrs.put("language", language);
	        	    }
	        	    break;
        		}
        		case "fo:font-weight": {
    		    	characterAttrs.put("bold", !propValue.equals("normal"));
        		    break;
        		}
        		case"style:font-weight-asian": {
    		    	characterAttrs.put("boldAsian", !propValue.equals("normal"));
        		    break;
        		}
        		case "style:font-weight-complex": {
        	    	characterAttrs.put("boldComplex", !propValue.equals("normal"));
	        	    break;
        		}
        		case "style:text-underline-style": {
    		    	characterAttrs.put("underline", !propValue.equals("none"));
        		    break;
        		}
        		case "fo:font-style": {
        	    	characterAttrs.put("italic", !propValue.equals("normal"));
	        	    break;
        		}
        		case "style:font-style-asian": {
        	    	characterAttrs.put("italicAsian", !propValue.equals("normal"));
	        	    break;
        		}
        		case "style:font-style-complex": {
        	    	characterAttrs.put("italicComplex", !propValue.equals("normal"));
	        	    break;
        		}
        		case "fo:color": {
        		    if (!characterAttrs.containsKey("color")) {
        		    	final Map<String, String> color = PropertyHelper.createColorMap(propValue);
        		    	if(color!=null) {
        		    		characterAttrs.put("color", color);
        		    	}
        		    }
        		    break;
        		}
        		case "style:use-window-font-color": {
	        	    if (propValue.equals("true")) {
	        	    	characterAttrs.put("color", PropertyHelper.createColorMap("auto"));
	        	    }
	        	    break;
        		}
        		case "fo:background-color": {
        			final Map<String, String> color = PropertyHelper.createColorMap(propValue);
        			if(color!=null) {
        				characterAttrs.put("fillColor", color);
        			}
	        	    break;
        		}
        		case "style:width": {
        			final Integer width = attributes.getLength100thmm(propValue, true);
        			if(width!=null) {
	        	    	characterAttrs.put("width", width);
	        	    }
	        	    break;
        		}
        		case "style:writing-mode": {
	        	    characterAttrs.put("writingMode", propValue);
	        	    break;
        		}
        		case "style:text-line-through-style": {
	        	    if (propValue.equals("none")) {
	        	    	characterAttrs.put("strike", "none");
	        	    } else {

	        	    	final String lineType = attributes.getValue("style:text-line-through-type");
	        	    	if(lineType!=null) {
        	            	characterAttrs.put("strike", lineType.equals("double") ? "double" : "single");
	        	        }
	        	    	else {
	        	        	characterAttrs.put("strike", "single");
	        	        }
	        	    }
	        	    break;
        		}
        	}
        }
        if(characterAttrs.isEmpty()) {
        	attrs.remove("character");
        }
	}

	public String getColorFromElement() {
        //contains e.g. fo:color
        String ret = "";
        final String color = attributes.getValue("fo:color");
        if(color != null) {
            if(color.equalsIgnoreCase("#0000ff")) {
                ret = "[BLUE]";
            } else if(color.equalsIgnoreCase("#00FF00")) {
                ret = "[GREEN]";
            } else if(color.equalsIgnoreCase("#FF0000")) {
                ret = "[RED]";
            } else if(color.equalsIgnoreCase("#FFFFFF")) {
                ret = "[WHITE]";
            } else if(color.equalsIgnoreCase("#FF00FF")) {
                ret = "[MAGENTA]";
            } else if(color.equalsIgnoreCase("#FFFF00")) {
                ret = "[YELLOW]";
            } else if(color.equalsIgnoreCase("#00FFFF")) {
                ret = "[CYAN]";
            } else if(color.equalsIgnoreCase("#000000")) {
                ret = "[BLACK]";
            }
        }
        return ret;
    }

	@Override
	public TextProperties clone() {
		return (TextProperties)_clone();
	}
}
