/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odf.properties;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;

import com.openexchange.office.odf.AttributeImpl;
import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.Border;
import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.styles.StrokeDashStyle;
import com.openexchange.office.odf.styles.StyleManager;

final public class GraphicProperties extends StylePropertiesBase {

	private DLList<IElementWriter> content;
	private Boolean isFrameOrGraphic = null;

	public GraphicProperties(AttributesImpl attributesImpl) {
		super(attributesImpl);
	}

	@Override
	public DLList<IElementWriter> getContent() {
		if(content==null) {
			content = new DLList<IElementWriter>();
		}
		return content;
	}

	public Boolean getIsFrameOrGraphic() {
		return isFrameOrGraphic;
	}

	public void setIsFrameOrGraphic(boolean isFrameOrGraphic) {
		this.isFrameOrGraphic = isFrameOrGraphic;
	}

	@Override
	public String getQName() {
		return "style:graphic-properties";
	}

	@Override
	public String getLocalName() {
		return "graphic-properties";
	}

	@Override
	public String getNamespace() {
		return Namespaces.STYLE;
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs)
		throws JSONException {

		final JSONObject fillAttributes = attrs.optJSONObject("fill");
		if(fillAttributes!=null) {
			final Object type = fillAttributes.opt("type");
			if(type!=null) {
				if(type==JSONObject.NULL) {
					attributes.remove("draw:fill");
				}
				else if(type instanceof String) {
					if(((String)type).equals("none")) {
						attributes.setValue(Namespaces.DRAW, "fill", "draw:fill", "none");
					}
					else {
						attributes.setValue(Namespaces.DRAW, "fill", "draw:fill", "solid");
					}
				}
			}
			final JSONObject color = fillAttributes.optJSONObject("color");
			if(color!=null) {
				if (color==JSONObject.NULL) {
					attributes.remove("fo:color");
					attributes.remove("fo:background-color");
				}
				else {
					final String colorType = color.optString("type", null);
					if(colorType!=null) {
	                    if(colorType.equals("auto")) {
							attributes.setValue(Namespaces.DRAW, "fill-color", "draw:fill-color", "#ffffff");
							attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", "#ffffff");
	                    }
	                    else {
	                    	attributes.setValue(Namespaces.DRAW, "fill-color", "draw:fill-color", PropertyHelper.getColor(color, null));
	                    	attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", PropertyHelper.getColor(color, null));
	                    }
					}
				}
			}
		}
		final JSONObject lineAttributes = attrs.optJSONObject("line");
		if(lineAttributes!=null) {
			if(isFrameOrGraphic) {
				final Border border = new Border();
				border.applyFoBorder(attributes.getValue("fo:border"));
				border.applyJsonLine(lineAttributes);
				final String foBorder = border.toString();
				if(foBorder!=null) {
					attributes.setValue(Namespaces.FO, "border", "fo:border", foBorder);
				}
			}
			else {
				final Object type = lineAttributes.opt("type");
				if(type!=null) {
					if(type==JSONObject.NULL) {
						attributes.remove("draw:stroke");
					}
					else if (type instanceof String) {
						if(((String)type).equals("none")) {
							attributes.setValue(Namespaces.DRAW, "stroke", "draw:stroke", "none");
						}
						else if(((String)type).equals("solid")) {
							final Object lineStyle = lineAttributes.opt("style");
							if(lineStyle!=null) {
								if(lineStyle==JSONObject.NULL||((String)lineStyle).equals("solid")) {
									attributes.setValue(Namespaces.DRAW, "stroke", "draw:stroke", "solid");
								}
								else {
									attributes.setValue(Namespaces.DRAW, "stroke", "draw:stroke", "dash");
									attributes.setValue(Namespaces.DRAW, "stroke-dash", "draw:stroke-dash", StrokeDashStyle.getStrokeDash(styleManager, (String)lineStyle));
								}
							}
							else {
								final String stroke = attributes.getValue("draw:stroke");
								if(stroke==null||stroke.isEmpty()) {
									attributes.setValue(Namespaces.DRAW, "stroke", "draw:stroke", "solid");
								}
							}
						}
					}
				}
				final JSONObject color = lineAttributes.optJSONObject("color");
				if(color!=null) {
					if (color==JSONObject.NULL) {
						attributes.remove("svg:stroke-color");
					}
					else {
						final String colorType = color.optString("type", null);
						if(colorType!=null) {
		                    if(colorType.equals("auto")) {
								attributes.setValue(Namespaces.SVG, "stroke-color", "svg:stroke-color", "#000000");
		                    }
		                    else {
		                    	attributes.setValue(Namespaces.SVG, "stroke-color", "svg:stroke-color", PropertyHelper.getColor(color, null));
		                    }
						}
					}
				}
				final Object strokeWidth = lineAttributes.opt("width");
				if(strokeWidth!=null) {
					if(strokeWidth==JSONObject.NULL) {
						attributes.remove("svg:stroke-width");
					}
					else if(strokeWidth instanceof Number) {
	                    attributes.setValue(Namespaces.SVG, "stroke-width", "svg:stroke-width", (((Number)strokeWidth).doubleValue() / 100) + "mm");
					}
				}
			}
		}
		final JSONObject drawingAttributes = attrs.optJSONObject("drawing");
		if(drawingAttributes!=null) {
			final Iterator<Entry<String, Object>> drawingIter = drawingAttributes.entrySet().iterator();
			while(drawingIter.hasNext()) {
				final Entry<String, Object> drawingEntry = drawingIter.next();
				final Object value = drawingEntry.getValue();
				switch(drawingEntry.getKey()) {
					case "marginBottom": {
			            if(value==JSONObject.NULL) {
			            	attributes.remove("fo:margin-bottom");
			            }
			            else {
			                attributes.setValue(Namespaces.FO, "margin-bottom", "fo:margin-bottom", (((Number)value).doubleValue() / 1000) + "cm");
			            }
			            break;
					}
					case "marginLeft": {
			            if(value==JSONObject.NULL) {
			            	attributes.remove("fo:margin-left");
			            }
			            else {
			                attributes.setValue(Namespaces.FO, "margin-left", "fo:margin-left", (((Number)value).doubleValue() / 1000) + "cm");
			            }
			            break;
					}
					case "marginRight": { 
			            if(value==JSONObject.NULL) {
			            	attributes.remove("fo:margin-right");
			            }
			            else {
			                attributes.setValue(Namespaces.FO, "margin-right", "fo:margin-right", (((Number)value).doubleValue() / 1000) + "cm");
			            }
			            break;
					}
					case "marginTop": {
			            if(value==JSONObject.NULL) {
			            	attributes.remove("fo:margin-top");
			            }
			            else {
			                attributes.setValue(Namespaces.FO, "margin-top", "fo:margin-top", (((Number)value).doubleValue() / 1000) + "cm");
			            }
			            break;
					}
		    		case "anchorHorAlign": {
		    			if(value==JSONObject.NULL) {
		    				attributes.remove("style:horizontal-pos");
		    			}
		    			else if(value instanceof String) {
		    				String align = null;
		    				if(value.equals("center")) {
		    					align = "center";
		    				}
		    				else if(value.equals("offset")) {
		    					align = "from-left";
		    				}
		    				else if(value.equals("left")) {
		    					align = "left";
		    				}
		    				else if(value.equals("right")) {
		    					align = "right";
		    				}
		    				else if(value.equals("inside")) {
		    					align = "inside";
		    				}
		    				else if(value.equals("outside")) {
		    					align = "outside";
		    				}
		    				if(align!=null) {
		    					attributes.setValue(Namespaces.STYLE, "horizontal-pos", "style:horizontal-pos", align);
		    				}
		    			}
		    			break;
		    		}
		    		case "anchorVertAlign": {
		    			if(value==JSONObject.NULL) {
		    				attributes.remove("style:vertical-pos");
		    			}
		    			else if(value instanceof String) {
		    				String align = null;
		    				if(value.equals("center")) {
		    					align = "center";
		    				}
		    				else if(value.equals("offset")) {
		    					align = "from-top";
		    				}
		    				else if(value.equals("top")) {
		    					align = "top";
		    				}
		    				else if(value.equals("bottom")) {
		    					align = "bottom";
		    				}
		    				else if(value.equals("inside")) {
		    					align = "inside";
		    				}
		    				else if(value.equals("outside")) {
		    					align = "outside";
		    				}
		    				if(align!=null) {
		    					attributes.setValue(Namespaces.STYLE, "vertical-pos", "style:vertical-pos", align);
		    				}
		    			}
		                break;
		    		}
		    		case "anchorHorBase": {
		    			if(value==JSONObject.NULL) {
		    				attributes.remove("style:horizontal-rel");
		    			}
		    			else if(value instanceof String) {
		    				String base = null;
			                if(value.equals("character")) {
			                	base = "char";
			                }
			                else if(value.equals("margin")) {
			                	base = "page-content";
			                }
			                else if(value.equals("leftMargin")) {
			                	base = "page-start-margin";
			                }
			                else if(value.equals("rightMargin")) {
			                	base = "page-end-margin";
			                }
			                else if(value.equals("column")) {
			                	base = "paragraph";
			                }
			                else if(value.equals("rightMargin")) {
			                	base = "paragraph-end-margin";
			                }
			                else if(value.equals("leftMargin")) {
			                	base = "paragraph-start-margin";
			                }
			                if(base!=null) {
			                	attributes.setValue(Namespaces.STYLE, "horizontal-rel", "style:horizontal-rel", base);
			                }
			                break;
		    			}
		    		}
		    		case "anchorVertBase": {
		    			if(value==JSONObject.NULL) {
		    				attributes.remove("style:vertical-rel");
		    			}
		    			else if(value instanceof String) {
		    				String base = null;
		    				if(value.equals("line")) {
		    					base = "line";
		    				}
		    				else if(value.equals("paragraph")) {
		    					base = "paragraph";
		    				}
		    				else if(value.equals("page")) {
		    					base = "page";
		    				}
		    				else if(value.equals("margin")) {
		    					base = "page-content";
		    				}
		    				else if(value.equals("topMargin")) {
		    					base = "page-content";
		    				}
		    				else if(value.equals("bottomMargin")) {
		    					base = "page-content";
		    				}
		    				if(base!=null) {
		    					attributes.setValue(Namespaces.STYLE, "vertical-rel", "style:vertical-rel", base);
		    				}
		    			}
		    			break;
		    		}
		    		case "anchorHorOffset": 
	    			// PATHTHROUGH INTENDED
		    		case "left": {
		    			if(value instanceof Number) {
		    				attributes.setValue(Namespaces.SVG, "x", "svg:x", (((Number)value).intValue() / 100) + "mm");
		    			}
		    			break;
		    		}
		    		case "anchorVertOffset":
		    		//PATHTHROUGH INTENDED
		    		case "top": {
		    			if(value instanceof Number) {
		    				attributes.setValue(Namespaces.SVG, "y", "svg:y", (((Number)value).intValue() / 100) + "mm");
		                }
		                break;
		    		}
		    		case "textWrapMode" : {
                        if(value==null||value.equals(JSONObject.NULL)) {
                            attributes.remove("style:wrap");
                            attributes.remove("style:wrap-contour");
                        } else {
                        	String textWrapMode = (String) value;
                            if (textWrapMode.equals("topAndBottom")) {
                                attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "none");
                                attributes.setValue(Namespaces.STYLE, "wrap-contour", "style:wrap-contour", "false");
                            } else {
                                String textWrapSide = attrs.optString("textWrapSide");
                                if(textWrapSide!=null) {
                                    if (textWrapMode.equals("square")) {
                                        if (textWrapSide.equals("largest")) {
                                            attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "biggest");
                                        } else if (textWrapSide.equals("left")) {
                                            attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "left");
                                        } else if (textWrapSide.equals("both")) {
                                        	attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "parallel");
                                        } else if (textWrapSide.equals("right")) {
                                        	attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "right");
                                        }
                                        attributes.setValue(Namespaces.STYLE, "wrap-contour", "style:wrap-contour", "false");
                                    } else if (textWrapMode.equals("through")) {
                                    	attributes.setValue(Namespaces.STYLE, "wrap", "style:wrap", "run-through");
                                    	attributes.setValue(Namespaces.STYLE, "wrap-contour", "style:wrap-contour", "false");
                                    }
                                }
                            }
                        }
		    			break;
		    		}
				}
			}
		}
		final JSONObject shapeAttributes = attrs.optJSONObject("shape");
		if(shapeAttributes!=null) {
			final Iterator<Entry<String, Object>> shapeIter = shapeAttributes.entrySet().iterator();
			while(shapeIter.hasNext()) {
				final Entry<String, Object> shapeEntry = shapeIter.next();
				final Object value = shapeEntry.getValue();
				switch(shapeEntry.getKey()) {
					case "paddingBottom": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-bottom");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-bottom", "fo:padding-bottom", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingLeft": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-left");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-left", "fo:padding-left", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingRight": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-right");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-right", "fo:padding-right", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingTop": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-top");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-top", "fo:padding-top", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
				}
			}
		}
		final JSONObject imageAttrs = attrs.optJSONObject("image");
		if(imageAttrs!=null) {
		/* TODO: clip
			case "fo:clip": {
	            final int start = propValue.indexOf("rect(");
	            final int end = propValue.indexOf(")");
	            final String clip = (start > -1 && end > -1) ? propValue.substring(start + 5, end) : propValue;
	            String clips[] = clip.split(", ");
	            if (clips.length != 4) {
	                clips = clip.split(" ");
	            }
	            imageAttrs.put("cropTop", AttributesImpl.normalizeLength(clips[0]));
	            imageAttrs.put("cropRight", AttributesImpl.normalizeLength(clips[1]));
	            imageAttrs.put("cropBottom", AttributesImpl.normalizeLength(clips[2]));
	            imageAttrs.put("cropLeft", AttributesImpl.normalizeLength(clips[3]));
	    		break;
			}
		*/
		}
	}

	@Override
	public void createAttrs(StyleManager styleManager, boolean contentAutoStyle, OpAttrs attrs) {
		final HashMap<String, Object> drawingAttrs = attrs.getMap("drawing", true);
		final HashMap<String, Object> shapeAttrs = attrs.getMap("shape", true);
		final HashMap<String, Object> imageAttrs = attrs.getMap("image", true);
		final HashMap<String, Object> lineAttrs = attrs.getMap("line", true);
		final HashMap<String, Object> fillAttrs = attrs.getMap("fill",  true);
		PropertyHelper.createDefaultMarginAttrs(drawingAttrs, attributes);
        final Integer defaultPadding = PropertyHelper.createDefaultPaddingAttrs(shapeAttrs, attributes);
        Border.createDefaultBorderMapAttrs(drawingAttrs, defaultPadding, attributes);
        final Iterator<Entry<String, AttributeImpl>> propIter = attributes.getUnmodifiableMap().entrySet().iterator();
        while(propIter.hasNext()) {
        	final Entry<String, AttributeImpl> propEntry = propIter.next();
        	final String propName = propEntry.getKey();
        	final String propValue = propEntry.getValue().getValue();
        	switch(propName) {
	    		case "fo:margin-left": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	                if(margin!=null) {
	                	drawingAttrs.put("marginLeft", margin);
	                	drawingAttrs.put("indentLeft", margin);
	                }
	    			break;
	    		}
	    		case "fo:margin-top": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	                if(margin!=null) {
	                	drawingAttrs.put("marginTop", margin);
	                }
	    			break;
	    		}
	    		case "fo:margin-right": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	    			if(margin!=null) {
	    				drawingAttrs.put("marginRight", margin);
	    				drawingAttrs.put("indentRight", margin);
	    			}
	    			break;
	    		}
	    		case "fo:margin-bottom": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	    			if(margin!=null) {
	    				drawingAttrs.put("marginBottom", margin);
	    			}
	    			break;
	    		}
	    		case "fo:padding-left": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				shapeAttrs.put("paddingLeft", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-top": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				shapeAttrs.put("paddingTop", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-right": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				shapeAttrs.put("paddingRight", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-bottom": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				shapeAttrs.put("paddingBottom", padding);
	    			}
	    			break;
	    		}
	    		case "fo:border-left": {
	    			final Map<String, Object> border = Border.createBorderMap(propValue, defaultPadding);
	                if(border!=null) {
	                	drawingAttrs.put("borderLeft", border);
	                }
	    			break;
	    		}
	    		case "fo:border-top": {
	    			final Map<String, Object> border = Border.createBorderMap(propValue, defaultPadding);
	                if(border!=null) {
	                	drawingAttrs.put("borderTop", border);
	                }
	    			break;
	    		}
	    		case "fo:border-right": {
	    			final Map<String, Object> border = Border.createBorderMap(propValue, defaultPadding);
	                if(border!=null) {
	                	drawingAttrs.put("borderRight", border);
	                }
	    			break;
	    		}
	    		case "fo:border-bottom": {
	    			final Map<String, Object> border = Border.createBorderMap(propValue, defaultPadding);
	                if(border!=null) {
	                	drawingAttrs.put("borderBottom", border);
	                }
	    			break;
	    		}
	    		case "draw:fill": {
	    			if(propValue.equals("none")) {
	    				fillAttrs.put("type", "solid");
	    			}
	    			else if(propValue.equals("bitmap")) {
	    				fillAttrs.put("type", "bitmap");
	    			}
	    			else if(propValue.equals("gradient")) {
	    				fillAttrs.put("type", "gradient");
	    			}
	    			else {
	    				fillAttrs.put("type", "solid");
	    			}
	    			break;
	    		}
	    		case "fo:background-color": {
	                fillAttrs.put("color", PropertyHelper.createColorMap(propValue));
	                break;
	    		}
	    		case "draw:fill-color": {
	                fillAttrs.put("color", PropertyHelper.createColorMap(propValue));
	                break;
	    		}
	    		case "style:mirror": {
	                if (propValue.contains("horizontal") && !propValue.contains("-on-")) {
	                	drawingAttrs.put("flipH", Boolean.TRUE);
	                }
	                if (propValue.contains("vertical")) {
	                	drawingAttrs.put("flipV", Boolean.TRUE);
	                }
	                if (propValue.equals("none")) {
	                	drawingAttrs.put("flipV", Boolean.FALSE);
	                	drawingAttrs.put("flipH", Boolean.FALSE);
	                }
	                break;
	    		}
	    		case "style:horizontal-pos": {
	                if (propValue.equals("center")) {
	                	drawingAttrs.put("anchorHorAlign", "center");
	                } else if (propValue.equals("from-left")) {
	                	drawingAttrs.put("anchorHorAlign", "offset");
	                } else if (propValue.equals("left")) {
	                	drawingAttrs.put("anchorHorAlign", "left");
	                } else if (propValue.equals("right")) {
	                	drawingAttrs.put("anchorHorAlign", "right");
	                } else if (propValue.equals("inside")) {
	                	drawingAttrs.put("anchorHorAlign", "inside");
	                } else if (propValue.equals("outside")) {
	                	drawingAttrs.put("anchorHorAlign", "outside");
	                }
	                break;
	    		}
	    		case "style:vertical-pos": {
	                if (propValue.equals("center")) {
	                	drawingAttrs.put("anchorVertAlign", "center");
	                } else if (propValue.equals("from-top")) {
	                	drawingAttrs.put("anchorVertAlign", "offset");
	                } else if (propValue.equals("top")) {
	                	drawingAttrs.put("anchorVertAlign", "top");
	                } else if (propValue.equals("bottom")) {
	                	drawingAttrs.put("anchorVertAlign", "bottom");
	                } else if (propValue.equals("inside")) {
	                	drawingAttrs.put("anchorVertAlign", "inside");
	                } else if (propValue.equals("outside")) {
	                	drawingAttrs.put("anchorVertAlign", "outside");
	                }
	                break;
	    		}
	    		case "style:horizontal-rel": {
	                if(propValue.equals("char"))
	                	drawingAttrs.put("anchorHorBase", "character");
	                else if(propValue.equals("page-content"))
	                	drawingAttrs.put("anchorHorBase", "margin");
	                else if(propValue.equals("page-start-margin"))
	                	drawingAttrs.put("anchorHorBase", "leftMargin");
	                else if(propValue.equals("page-end-margin"))
	                	drawingAttrs.put("anchorHorBase", "rightMargin");
	                else if(propValue.equals("frame"))
	                	drawingAttrs.put("anchorHorBase", "column");
	                else if(propValue.equals("frame-content") || propValue.equals("frame-end-margin") || propValue.equals("frame-start-margin"))
	                	drawingAttrs.put("anchorHorBase", "column");
	                else if(propValue.equals("paragraph") || propValue.equals("paragraph-content"))
	                	drawingAttrs.put("anchorHorBase", "column");
	                else if(propValue.equals("paragraph-end-margin"))
	                	drawingAttrs.put("anchorHorBase", "rightMargin");
	                else if(propValue.equals("paragraph-start-margin"))
	                	drawingAttrs.put("anchorHorBase", "leftMargin");
	                break;
	    		}
	    		case "style:vertical-rel": {
	                if(propValue.equals("char"))
	                	drawingAttrs.put("anchorVertBase", "line");
	                else if(propValue.equals("frame"))
	                	drawingAttrs.put("anchorVertBase", "paragraph");
	                else if(propValue.equals("frame-content"))
	                	drawingAttrs.put("anchorVertBase", "paragraph");
	                else if(propValue.equals("line"))
	                	drawingAttrs.put("anchorVertBase", "line");
	                else if(propValue.equals("page"))
	                	drawingAttrs.put("anchorVertBase", "page");
	                else if(propValue.equals("page-content"))
	                	drawingAttrs.put("anchorVertBase", "margin");
	                else if(propValue.equals("paragraph"))
	                	drawingAttrs.put("anchorVertBase", "paragraph");
	                else if(propValue.equals("paragraph-content"))
	                	drawingAttrs.put("anchorVertBase", "paragraph");
	                else if(propValue.equals("text"))
	                	drawingAttrs.put("anchorVertBase", "line");
	                break;
	    		}
	    		case "svg:x": {
	    			final int x = AttributesImpl.normalizeLength(propValue);
	                if (x != 0) {
	                	drawingAttrs.put("anchorHorOffset", x);
	                	drawingAttrs.put("left", x);
	                }
	                break;
	    		}
	    		case "svg:y": {
	    			final int y = AttributesImpl.normalizeLength(propValue);
	                if (y != 0) {
	                	drawingAttrs.put("anchorVertOffset", y);
	                    drawingAttrs.put("top", y);
	                }
	                break;
	    		}
	    		case "fo:clip": {
	                final int start = propValue.indexOf("rect(");
	                final int end = propValue.indexOf(")");
	                final String clip = (start > -1 && end > -1) ? propValue.substring(start + 5, end) : propValue;
	                String clips[] = clip.split(", ");
	                if (clips.length != 4) {
	                    clips = clip.split(" ");
	                }
	                imageAttrs.put("cropTop", AttributesImpl.normalizeLength(clips[0]));
	                imageAttrs.put("cropRight", AttributesImpl.normalizeLength(clips[1]));
	                imageAttrs.put("cropBottom", AttributesImpl.normalizeLength(clips[2]));
	                imageAttrs.put("cropLeft", AttributesImpl.normalizeLength(clips[3]));
		    		break;
	    		}
	    		case "style:wrap": {
	                if (propValue.equals("biggest")) {
	                	drawingAttrs.put("textWrapMode", "square");
	                	drawingAttrs.put("textWrapSide", "largest");
	                } else if (propValue.equals("left")) {
	                	drawingAttrs.put("textWrapMode", "square");
	                	drawingAttrs.put("textWrapSide", "left");
	                } else if (propValue.equals("none")) {
	                	drawingAttrs.put("textWrapMode", "topAndBottom");
	                } else if (propValue.equals("parallel")) {
	                	drawingAttrs.put("textWrapMode", "square");
	                	drawingAttrs.put("textWrapSide", "both");
	                } else if (propValue.equals("right")) {
	                	drawingAttrs.put("textWrapMode", "square");
	                	drawingAttrs.put("textWrapSide", "right");
	                } else if (propValue.equals("run-through")) {
	                	drawingAttrs.put("textWrapMode", "through");
	                	drawingAttrs.put("textWrapSide", "both");
	                }
	                break;
	    		}
	    		case "draw:stroke": {
	                if(propValue.equals("none")) {
	                    lineAttrs.put("type", "none");
	                }
	                else if(propValue.equals("solid")) {
	                	lineAttrs.put("type", "solid");
	                	lineAttrs.put("style", "solid");
	                }
	                else if(propValue.equals("dash")) {
	                	lineAttrs.put("type", "solid");
	                	lineAttrs.put("style", StrokeDashStyle.getStrokeDashEnumFromStyleName(styleManager, attributes.getUnmodifiableMap().get("draw:stroke-dash")));
	                }
	                break;
	    		}
	    		case "svg:stroke-color": {
	                lineAttrs.put("color", PropertyHelper.createColorMap(propValue));
	                break;
	    		}
	    		case "svg:stroke-width": {
	    			lineAttrs.put("width", AttributesImpl.normalizeLength(propValue));
	    			break;
	    		}
        	}
        }
        //border handling: text frames have a border attribute, shapes _can_ have stroke
        if(drawingAttrs.containsKey("border") || drawingAttrs.containsKey("borderTop")) {
            // convert to "line"
        	@SuppressWarnings("unchecked")
			final HashMap<String, Object> currentBorder = drawingAttrs.containsKey("border") ? (HashMap<String, Object>)drawingAttrs.get("border") : (HashMap<String, Object>)drawingAttrs.get("borderTop");
            if(currentBorder.containsKey("style") ) {
                final String style = (String)currentBorder.get("style");
                if(style.equals("none")) {
                    lineAttrs.put("type", "none");
                } else {
                    if(style.equals("dashed")){
                        lineAttrs.put("style", "dashed");
                    } else if(style.equals("dotted")) {
                        lineAttrs.put("style", "dotted");
                    } else {
                        lineAttrs.put("style", "solid");
                    }
                    lineAttrs.put("type", "solid");
                }
            }
            if(currentBorder.containsKey("width") ) {
                lineAttrs.put("width", currentBorder.get("width"));
            }
            if( currentBorder.containsKey("color") ) {
                lineAttrs.put("color", currentBorder.get("color"));
            }
        }
       	drawingAttrs.remove("borderTop");
    	drawingAttrs.remove("borderBottom");
    	drawingAttrs.remove("borderLeft");
    	drawingAttrs.remove("borderRight");
    	drawingAttrs.remove("border");

        if(drawingAttrs.isEmpty()) {
        	attrs.remove("drawing");
        }
        if(shapeAttrs.isEmpty()) {
        	attrs.remove("shape");
        }
        if(imageAttrs.isEmpty()) {
        	attrs.remove("image");
        }
        if(lineAttrs.isEmpty()) {
        	attrs.remove("line");
        }
        if(fillAttrs.isEmpty()) {
        	attrs.remove("fill");
        }
	}

	@Override
	public GraphicProperties clone() {
		return (GraphicProperties)_clone();
	}
}
