/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.hazelcast.cleanup;

import java.net.InetSocketAddress;
import java.util.Collection;

import org.apache.commons.logging.Log;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.core.MemberAttributeEvent;
import com.hazelcast.core.MembershipEvent;
import com.hazelcast.core.MembershipListener;
import com.openexchange.exception.OXException;
import com.openexchange.office.hazelcast.access.HazelcastAccess;
import com.openexchange.office.hazelcast.doc.HazelcastDocumentDirectory;
import com.openexchange.office.hazelcast.doc.HazelcastDocumentResourcesDirectory;
import com.openexchange.office.hazelcast.serialization.document.PortableCleanupStatus;
import com.openexchange.office.hazelcast.utils.DocRestoreIDMap;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentState;

/**
 * {@link CleanupMembershipListener} - Reacts to a member removal by cleaning up resources that
 * have been located on that member.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since 7.8.1
 *
 */
public class CleanupMembershipListener implements MembershipListener {
    @SuppressWarnings("deprecation")
    private final Log LOG = com.openexchange.log.Log.loggerFor(CleanupMembershipListener.class);

    private final String cleanupLockMapName;
    private final HazelcastDocumentDirectory directory;
    private final HazelcastDocumentResourcesDirectory resourceDir;

    /**
     *
     * @param cleanupLockMapName
     * @param directory
     */
    public CleanupMembershipListener(final String cleanupLockMapName, final HazelcastDocumentDirectory directory, final HazelcastDocumentResourcesDirectory resourceDir) {
        super();
        this.cleanupLockMapName = cleanupLockMapName;
        this.directory = directory;
        this.resourceDir = resourceDir;
    }

    @Override
    public void memberAdded(MembershipEvent membershipEvent) {
        // nothing to do
    }

    @Override
    public void memberRemoved(MembershipEvent membershipEvent) {
        final Member memberToClean = membershipEvent.getMember();
        final String uuid = memberToClean.getUuid();
        final InetSocketAddress socketAddr = memberToClean.getSocketAddress();

        try {
            IMap<String, PortableCleanupStatus> cleanupMapping = getCleanupMapping();

            cleanupMapping.lock(uuid);
            try {
                PortableCleanupStatus cleanupStatus = cleanupMapping.get(uuid);
                // is somebody already cleaning up for him?
                if (!cleanupDone(cleanupStatus)) {
                    final long nStartTime = System.currentTimeMillis();
                    LOG.info("RT connection: Starting cleanup for member " + uuid + " with IP " + socketAddr);
                    cleanupStatus = new PortableCleanupStatus(HazelcastAccess.getLocalMember(), memberToClean);

                    // do actual cleanup
                    DocRestoreIDMap<DocumentState> docsOfMember = directory.getDocumentsOfMember(memberToClean);
                    LOG.debug("RT connection: Found the following resources to clean up: " + docsOfMember.toString());

                    // just remove entries from both maps (document and resource state)
                    Collection<DocRestoreID> ids = docsOfMember.keySet();
                    directory.remove(ids);
                    resourceDir.remove(ids);

                    // update status and put to map
                    final long nEndTime = System.currentTimeMillis();
                    cleanupStatus.setCleaningFinishTime(nEndTime);
                    cleanupMapping.put(uuid, cleanupStatus);
                    LOG.debug("RT connection: CleanupMapping after cleanup: " + cleanupMapping.entrySet());
                    LOG.debug("RT connection: Cleanup needed " + (nEndTime - nStartTime) + "ms");
                } else {
                    LOG.debug("RT connection: Cleanup was already started: " + cleanupStatus);
                }
            } catch (Exception e) {
                LOG.error("RT connection: Failed to start cleanup after member " + uuid + " with IP " + socketAddr + " left the cluster", e);
            } finally {
                cleanupMapping.unlock(uuid);
            }
        } catch (OXException oxe) {
            LOG.error("RT connection: Failed to start cleanup after member " + uuid + " with IP " + socketAddr + " left the cluster", oxe);
        }
    }

    @Override
    public void memberAttributeChanged(MemberAttributeEvent memberAttributeEvent) {
        // nothing to do
    }

    private IMap<String, PortableCleanupStatus> getCleanupMapping() throws OXException {
        HazelcastInstance hazelcast = HazelcastAccess.getHazelcastInstance();
        return hazelcast.getMap(cleanupLockMapName);
    }

    /**
     * Check the distributed memberCleanup map for a status indicating that some other member has already finished cleaning up.
     *
     * @param member may be null
     * @return true if member is not null, the cleanup was finished during the last run and the last run was not more than 5 minutes ago
     */
    private boolean cleanupDone(PortableCleanupStatus cleanupStatus) {
        if(cleanupStatus!= null) {
            long cleaningFinishTime = cleanupStatus.getCleaningFinishTime();
            if(cleaningFinishTime == -1) {
                //the cleanup wasn't finished
                return false;
            }
            long diff = System.currentTimeMillis() - cleaningFinishTime;
            if(diff > 300000) {
                //the last cleanup was more than 5 minutes ago, clean again!
                return false;
            } else {
                return true;
            }
        }
        return false;
    }

}
