/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.backup.manager.impl;

import java.lang.ref.WeakReference;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;

import com.openexchange.log.Log;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.message.MessageChunk;

/**
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since 7.10.0
 */
public class AsyncBackupWorker {
	@SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(AsyncBackupWorker.class);

    static private final long WRITEBACK_OPERATIONS_TIMER_PERIOD = 10000;
    static private final long WRITEBACK_ACTIVE_TIME             = 2000;

    private final Queue<OperationDocData> pendingDocData = new ConcurrentLinkedQueue<OperationDocData>();
    private Timer timer = new Timer("com.openexchange.office.backup.DocumentBackupManager");
    private TimerTask writeBackTimerTask = null;
    private final Object sync = new Object();
    private final WeakReference<StoreOperationData> storeOperationDataProcessor;

    /**
     *
     * @param services
     * @param storeOperationData
     */
	public AsyncBackupWorker(final StoreOperationData storeOpsData) {
		this.storeOperationDataProcessor = new WeakReference<StoreOperationData>(storeOpsData);

        // create the task to touch the held resources at at specified interval
		writeBackTimerTask = new TimerTask() {

            @Override
            public void run() {
                boolean timeout = false;
                long timeStart = System.currentTimeMillis();

                while (!timeout) {
                    try {
                        final OperationDocData currOpsDocData = pendingDocData.poll();

                        if (null == currOpsDocData)
                            break;

                        final StoreOperationData storeOperationData = storeOperationDataProcessor.get();
                        if (null != storeOperationData) {

	                        // store the pending operations to hazelcast/distributed managed file
	                        storeOperationData.storePendingOperationsToDocumentState(currOpsDocData.getID(), currOpsDocData.getOperations(), currOpsDocData.getOSN());
                        }
                    } catch (Exception e) {
                        LOG.warn("RT connection: Exception caught while trying to save operations to backup file", e);
                    }

                    timeout = (System.currentTimeMillis() - timeStart) > WRITEBACK_ACTIVE_TIME;
                }
            }
        };

        // start the periodic timer with the specified timeout
        timer.schedule(writeBackTimerTask, WRITEBACK_OPERATIONS_TIMER_PERIOD, WRITEBACK_OPERATIONS_TIMER_PERIOD);
	}

	/**
	 *
	 * @param docResId
	 * @param actionChunk
	 */
	void appendOperations(final DocRestoreID docResId, final MessageChunk actionChunk, int newOSN) {
        final OperationDocData opsData = new OperationDocData(docResId, actionChunk, newOSN);

        pendingDocData.add(opsData);
	}

	/**
	 *
	 */
	void dispose() {
	    final Timer aTimer = timer;
	    if (aTimer != null) {
	        aTimer.cancel();
	        aTimer.purge();
	        timer = null;
	    }
		writeBackTimerTask = null;
		pendingDocData.clear();
	}
}
