/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.backup.distributed;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicReference;

import com.openexchange.log.Log;
import com.openexchange.office.tools.directory.IStreamIDCollection;
import com.openexchange.server.ServiceLookup;

public class DistributedManagedFileLocker {

    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(DistributedDocumentStreamManager.class);
    static final long TOUCH_MANAGEDFILE_TIMER_PERIOD = 90000;
    private static final AtomicReference<DistributedManagedFileLocker> REF = new AtomicReference<DistributedManagedFileLocker>();
    private Timer timer = new Timer("com.openexchange.office.backup.DistributedManagedFileLocker");
    private final Map<String, WeakReference<IDistributedManageFileCollection>> lockCollections = new HashMap<String, WeakReference<IDistributedManageFileCollection>>();
    private TimerTask timerTask = null;

    /**
     * Initializes a new {@link ResourceLocker}.
     */
    private DistributedManagedFileLocker(final ServiceLookup services) {
        super();

        // create the task to touch the held resources at at specified interval
        timerTask = new TimerTask() {

            @Override
            public void run() {
                try {
                    final List<IDistributedManageFileCollection> collections = new ArrayList<IDistributedManageFileCollection>();

                    synchronized(lockCollections) {
                        for (WeakReference<IDistributedManageFileCollection> weak : lockCollections.values()) {
                            final IDistributedManageFileCollection collection = weak.get();

                            if (null != collection) {
                                collections.add(collection);
                            }
                        }
                    }

                    for (final IDistributedManageFileCollection aCollection : collections) {
                        final Set<IStreamIDCollection> streamIDCollections = aCollection.getStreamIDCollectionCollection();

                        if (null != streamIDCollections) {

                            // touch each collected document streams
                            for (final Iterator<IStreamIDCollection> streamIDCollectionIter = streamIDCollections.iterator(); streamIDCollectionIter.hasNext();) {
                                final IStreamIDCollection streamIDCollection = streamIDCollectionIter.next();
                                final long timeStamp = new Date().getTime();
                                final Set<String> streamIDs = streamIDCollection.getStreamIDs();

                                if (null != streamIDs) {
                                    // touch the managed files using the stream id
                                    for (final Iterator<String> streamsIDsIter = streamIDs.iterator(); streamsIDsIter.hasNext();) {
                                        aCollection.touch(streamIDCollection.getUniqueCollectionID(), streamsIDsIter.next());
                                    }
                                }

                                // set new time stamp for the collection
                                streamIDCollection.touch(timeStamp);
                            }

                            // update the state of the stream ID collections
                            aCollection.updateStates(streamIDCollections);
                        }
                    }
                } catch (Throwable e) {
                    LOG.warn("RT connection: Exception caught while trying to touch active document resources", e);
                }
            }
        };

        // start the periodic timer with the specified timeout
        timer.schedule(timerTask, TOUCH_MANAGEDFILE_TIMER_PERIOD, TOUCH_MANAGEDFILE_TIMER_PERIOD);
    }

    private void dispose() {
        final Timer aTimer = timer;

        if (aTimer != null) {
            aTimer.cancel();
            aTimer.purge();
            timer = null;
        }

        synchronized(lockCollections) {
            lockCollections.clear();
        }
    }

    /**
     * Initializes the singleton.
     *
     * @param serviceLookup
     */
    public static void init(final ServiceLookup serviceLookup) {
        REF.compareAndSet(null, new DistributedManagedFileLocker(serviceLookup));
    }

    /**
     * Disposes the singleton, which resets memory consuming members.
     */
    public static void destroy() {
        final DistributedManagedFileLocker locker = REF.getAndSet(null);

        if (null != locker) {
            locker.dispose();
        }
    }

    /**
     * Returns the one and only DocumentDataManager instance.
     *
     * @return the one and only DocumentDataManager instance or null, if not
     * initialized.
     */
    public static final DistributedManagedFileLocker get() {
        return REF.get();
    }

    /**
     *
     * @param collection
     */
    public void addCollectionToLock(final IDistributedManageFileCollection collection) {
        synchronized(lockCollections) {
            if (null != collection) {
                final String uniqueID = collection.getUniqueID();
                lockCollections.put(uniqueID, new WeakReference<IDistributedManageFileCollection>(collection));
            }
        }
    }

    public void removeCollectionToLock(final IDistributedManageFileCollection collection) {
        synchronized(lockCollections) {
            if (null != collection) {
                final String uniqueID = collection.getUniqueID();
                lockCollections.remove(uniqueID);
            }
        }
    }

}
