/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.backup.distributed;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.log.Log;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.IStreamIDCollection;
import com.openexchange.server.ServiceLookup;

/**
 * Manages the document specific streams in a distributed managed file manager.
 * It refreshes the managed files to keep them alive.
 *
 * @author Carsten Driesner
 * @since 7.8.1
 */
public class DistributedDocumentStreamManager implements IDistributedManageFileCollection {

    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(DistributedDocumentStreamManager.class);
    private static final String DOC_STREAM_PREFIX = "office:doc-stream:";
    private static final String OPS_STREAM_PREFIX = "office:ops-stream:";
    private final String uniqueSingletonID;
    private final ManagedFileManagement fileManager;
    private final HashMap<DocRestoreID, DocStreams> docStreamsMap = new HashMap<DocRestoreID, DocStreams>();
    private final DistributedManagedFileLocker locker;
    private final String uniqueID;

    /**
     * Initializes a DistributedDocumentStreamManager
     *
     * @param services the service lookup to be used for this instance
     */
    public DistributedDocumentStreamManager(final ServiceLookup services, final String uniqueSingletonID) {
        fileManager = services.getService(ManagedFileManagement.class);
        locker = DistributedManagedFileLocker.get();
        this.uniqueSingletonID = uniqueSingletonID;
        this.uniqueID = UUID.randomUUID().toString();

        locker.addCollectionToLock(this);
    }

    /**
     * Adds/sets the provided document data as distributed managed file.
     *
     * @param docResourceID the unique document id
     * @param docData the document content to be stored
     * @return
     * @throws OXException
     */
    public String addDocumentStreams(final DocRestoreID docRestoreID, final byte[] docData) throws OXException {
         return setDocumentStream(docRestoreID, docData, DOC_STREAM_PREFIX);
    }

    /**
     * Adds/sets the provided document operation data as distributed managed file.
     *
     * @param docResourceID the unique document id
     * @param opsData the document operations applied to the document stream
     * @return
     * @throws OXException
     */
    public String addOperationStream(final DocRestoreID docRestoreID, final byte[] opsData) throws OXException {
        return setDocumentStream(docRestoreID, opsData, OPS_STREAM_PREFIX);
    }

    /**
     * Retrieves the stream of a unique stream ID.
     *
     * @param uniqueStreamID the unique stream ID
     * @return the stream or null if no stream is associated to the stream ID.
     */
    public InputStream getStream(final String uniqueStreamID) {
        InputStream stream = null;

        try {
            final ManagedFile managedFile = fileManager.getByID(uniqueStreamID);
            stream = managedFile.getInputStream();
        } catch (OXException e) {
            // nothing to do - it's possible that the managed file doesn't exists
        }

        return stream;
    }

    /**
     * Removes the document related distributed managed files from this
     * manager and the responsible structures.
     *
     * @param docResourceID the document unqiue ID
     */
    public void removeDocumentStreams(final DocRestoreID docRestoreID) {
        DocStreams docStreams = null;

        synchronized(docStreamsMap) {
            docStreams = docStreamsMap.get(docRestoreID);
            docStreams = (docStreams != null) ? docStreams.clone() : null;
        }

        if (null != docStreams) {
            final String docStreamID = docStreams.getDocStreamID();
            final String opsStreamID = docStreams.getOpsStreamID();

            try {
                if (null != docStreamID) { fileManager.removeByID(docStreamID); }
            } catch (OXException e) {
                LOG.debug("RT connection: distributed managed file " + docStreamID + "cannot be removed!", e);
            }

            try {
                if (null != opsStreamID) { fileManager.removeByID(opsStreamID); }
            } catch (OXException e) {
                LOG.debug("RT connection: distributed managed file " + opsStreamID + "cannot be removed!", e);
            }

            synchronized(docStreamsMap) {
                docStreamsMap.remove(docRestoreID);
            }
        }
    }

    /**
     * Determines if a stream ID is known to the distributed document stream
     * manager.
     *
     * @param uniqueStreamID
     * @return
     */
    public boolean hasStream(final String uniqueStreamID) {
        return (StringUtils.isNotEmpty(uniqueStreamID)) ? fileManager.contains(uniqueStreamID) : false;
    }

    @Override
    public Set<IStreamIDCollection> getStreamIDCollectionCollection() {
        Set<IStreamIDCollection> result = null;

        synchronized(docStreamsMap) {
            if (docStreamsMap.size() > 0) {
                final HashSet<IStreamIDCollection> copy = new HashSet<IStreamIDCollection>();
                final Collection<DocStreams> docStreams = docStreamsMap.values();

                // make a deep copy of the map to enable parallel access
                for (DocStreams doc : docStreams) {
                    copy.add(doc.clone());
                }

                result = copy;
            }
        }

        return result;
    }

    @Override
    public void touch(final DocRestoreID uniqueCollectionID, String streamID) {
        boolean success = fileManager.contains(streamID);
        if (!success) {
            LOG.warn("RT connection: Trying to touch managed file: " + streamID + ", but without success!");
        }
    }

    @Override
    public void updateStates(Set<IStreamIDCollection> updatedCollections) {
        synchronized(docStreamsMap) {

            for (final IStreamIDCollection updated : updatedCollections) {
                final DocStreams docStream = docStreamsMap.get(updated.getUniqueCollectionID());
                if (null != docStream) {
                    docStream.touch(updated.getTimeStamp());
                }
            }
        }
    }

    @Override
    public String getUniqueID() {
        return uniqueID;
    }

    /**
     * Disposes this instance which means that the internal data structures
     * remove state data and frees as much memory as possible.
     */
    public void dispose() {
        DistributedManagedFileLocker.destroy();
        synchronized(docStreamsMap) {
            docStreamsMap.clear();
        }
    }

    /**
     *
     * @param docResourceID
     * @param data
     * @param PREFIX
     * @return
     * @throws OXException
     */
    private String setDocumentStream(final DocRestoreID docRestoreID, final byte[] data, final String PREFIX) throws OXException {
        Validate.notNull(docRestoreID);

        String oldStreamID = null;
        synchronized(docStreamsMap) {
            oldStreamID = getStreamID(PREFIX, docStreamsMap.get(docRestoreID));
        }

        if (StringUtils.isNotEmpty(oldStreamID) && hasStream(oldStreamID)) {
            fileManager.removeByID(oldStreamID);
        }

        final String uniqueStreamID = createUniqueStreamID(docRestoreID, PREFIX);
        final InputStream inStream = new ByteArrayInputStream(data);
        fileManager.createManagedFile(uniqueStreamID, inStream, true);

        synchronized(docStreamsMap) {
            DocStreams docStreams = docStreamsMap.get(docRestoreID);
            if (null == docStreams) {
                docStreams = new DocStreams(docRestoreID);
            }
            setStreamID(PREFIX, docStreams, uniqueStreamID);
            docStreamsMap.put(docRestoreID, docStreams);
        }
        IOUtils.closeQuietly(inStream);

        return uniqueStreamID;
    }

    /**
     * Creates a unique stream ID to be used to reference the document/operation
     * stream.
     *
     * @param docResourceID the document resource id to be used to associate the stream
     * @param PREFIX a prefix to be used for the unique id
     * @return a unique stream ID
     */
    private static String createUniqueStreamID(final DocRestoreID docRestoreID, final String PREFIX) {
        Validate.notNull(docRestoreID);

        final StringBuffer strBuf = new StringBuffer(PREFIX);
        strBuf.append(docRestoreID.toString());
        strBuf.append(":");
        strBuf.append(UUID.randomUUID());
        return strBuf.toString();
    }

    /**
     *
     * @param PREFIX
     * @param docStreams
     * @return
     */
    private static String getStreamID(final String PREFIX, final DocStreams docStreams) {
        String result = null;

        if (null != docStreams) {
            if (DOC_STREAM_PREFIX.equals(PREFIX)) {
                result = docStreams.getDocStreamID();
            } else if (OPS_STREAM_PREFIX.equals(PREFIX)) {
                result = docStreams.getOpsStreamID();
            }
        }

        return result;
    }

    /**
     *
     * @param PREFIX
     * @param docStreams
     * @param streamID
     */
    private static void setStreamID(final String PREFIX, final DocStreams docStreams, final String streamID) {
        if (null != docStreams) {
            if (DOC_STREAM_PREFIX.equals(PREFIX)) {
                docStreams.setDocStreamID(streamID);
            } else if (OPS_STREAM_PREFIX.equals(PREFIX)) {
                docStreams.setOpsStreamID(streamID);
            }
        }
    }

}
