/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.spellchecker.hunspell;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.regex.Pattern;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.spellchecker.ISpellChecker;
import com.openexchange.spellchecker.hunspell.jna.Hunspell;

/**
 * {@link SpellChecker}
 *
 * @author <a href="mailto:oliver.specht@open-xchange.com">Oliver Specht</a>
 */
public class SpellChecker implements ISpellChecker {
    private Hunspell hunspell = null;
    private DictionaryContainer dictionaries = null;
    
    private static final String SEPARATOR_PATTERN = "[\t\n\u00a0\f\r \\.,;\\?!\":(){}\\[\\] ]";
    private static final String THE_DOT = ".";

    /**
     * Initializes a new {@link SpellChecker}.
     */
    public SpellChecker( final ConfigurationService configService ){
        String hunspelLibrary = "/usr/lib/x86_64-linux-gnu/libhunspell-1.3.so.0";
        String dictionaryPath = "/usr/share/hunspell";
        String emptyString = "";
        String curStringEntry;
        if (null != configService) {

            // hunspell library
            if ((curStringEntry = configService.getProperty( "com.openexchange.spellchecker.hunspell.library", emptyString)) != emptyString) {
                hunspelLibrary = curStringEntry;
            }
            // dictionaries
            if ((curStringEntry = configService.getProperty( "com.openexchange.spellchecker.hunspell.dictionaries", emptyString)) != emptyString) {
                dictionaryPath = curStringEntry;
            }
        }
        try{
            if( new File(hunspelLibrary).exists() && new File(dictionaryPath).exists()){
                hunspell = Hunspell.getInstance(hunspelLibrary);
                dictionaries = new DictionaryContainer(hunspell, dictionaryPath);
            }
        }
        catch(Throwable t) {
            System.err.println( t.getMessage());
        }
    }
    /* (non-Javadoc)
     * @see com.openexchange.oxospellchecker.ISpellChecker#checkSpelling(java.lang.String, java.lang.String)
     */
    @Override
    public JSONArray checkSpelling(String text, String locale, int offset) {

        text = text + ' ';
        JSONArray result = new JSONArray();
        Hunspell.Dictionary dict = dictionaries != null ? dictionaries.getDictionary( locale ) : null;
        if( dict != null ){
            int startPos = 0;
            int len = 0;
            for( int pos = 0; pos < text.length(); ++pos){
                String substring = text.substring(pos, pos + 1);
                if( Pattern.matches(SEPARATOR_PATTERN, substring) ){
                    if( len > 0 ){
                    	// #40631# A dot might also be used for abbreviations.
                    	int extra = substring.equals(THE_DOT) ? 1 : 0;
                    	String word = text.substring(startPos, startPos + len + extra);
                        if(dict.misspelled(word)){
                            JSONObject error = new JSONObject();
                            try {
                                error.put("start", startPos + offset);
                                error.put("length", len);
                                error.put("locale", dict.getLocale());
                                error.put("word", word);
                                ListIterator<String> it = dict.suggest(word).listIterator();
                                if(it.hasNext()){
                                    JSONArray replacements = new JSONArray();
                                    while( it.hasNext() ){
                                        replacements.put( it.next() );
                                    }
                                    error.put("replacements", replacements);
                                }

                            } catch (JSONException e) {
                                ;
                            }
                            result.put( error );
                        }
                        len = 0;
                    }
                    startPos = pos + 1;

                } else {
                    ++len;
                }
            }
        }
        return result;
    }
    /**
     * @param paragraph contains the parts of the paragraph combined to it's locale.
     * @param locale
     * @return array containing the suggestions. If the word is detected as correct the array contains the correct word, only
     */
    @Override
    public JSONArray checkParagraphSpelling( JSONArray paragraph ){
        JSONArray  result = new JSONArray ();
        int pos = 0;
        int offset = 0;
        StringBuilder text = new StringBuilder();
        String locale = null;
        try {
        while(pos < paragraph.length() ){
            JSONObject element = paragraph.getJSONObject(pos);
            if( (locale != null && locale.compareTo( element.getString("locale")) != 0) ){
                JSONArray check = checkSpelling( text.toString(), locale, offset );
                for( int i = 0; i < check.length(); ++i)
                    result.put( check.get(i) );
                offset += text.length();
                text.setLength(0);
                locale = null;
            }
            locale = element.getString("locale");
            text.append(element.getString("word"));
            ++pos;
        }
        if( text.length() > 0 ){
            JSONArray check = checkSpelling( text.toString(), locale, offset );
            for( int i = 0; i < check.length(); ++i)
                result.put( check.get(i) );
            }
        } catch (JSONException e) {
            ;
        }
        return result;
    }
    /* (non-Javadoc)
     * @param word
     * @param locale
     * @return a list of suggestions.
     * @see com.openexchange.office.spellchecker.ISpellChecker#suggestReplacement(java.lang.String, java.lang.String)
     */
    @Override
    public List<String> suggestReplacements( String word, String locale ){
        List<String>  result = new ArrayList<String>();
        Hunspell.Dictionary dict = dictionaries != null ? dictionaries.getDictionary( locale ) : null;
        if( dict != null ){
            ListIterator<String> it = dict.suggest(word).listIterator();
            while( it.hasNext() ){
                result.add( it.next() );
            }
        }
        return result;
    }
    /* (non-Javadoc)
     * @param word the word that has to be checked
     * @param locale the related locale
     * @return true if the word is detected as wrong. false if no dictionary for this locale is available
     * @see com.openexchange.office.spellchecker.ISpellChecker#isMisspelled(java.lang.String, java.lang.String)
     */
    @Override
    public boolean isMisspelled( String word, String locale ){
        Hunspell.Dictionary dict = dictionaries != null ? dictionaries.getDictionary( locale ) : null;
        return dict != null ? dict.misspelled(word) : false;
    }
    /* (non-Javadoc)
     * @return the list of supported locales
     * @see com.openexchange.office.spellchecker.ISpellChecker#getSupportedLocales()
     */
    @Override
    public List<String> getSupportedLocales(){
        return dictionaries != null ? dictionaries.getLocales() : new ArrayList<String>();
    }
}


