/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.resource;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.CRC32;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.server.ServiceLookup;

/**
 * {@link Resource}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class Resource {

    /**
     * Initializes a new {@link Resource}.
     * @param services
     * @param buffer
     */
    public Resource(ServiceLookup services, byte[] buffer) {
        this(services, getCRC32(buffer), buffer);
    }

    /**
     * Initializes a new {@link Resource}.
     * @param services If set, a <link>ManagedFile</Link> will be created to hold the content
     * @param _buffer The content of this resource as a byte Array
     */
    public Resource(ServiceLookup services, int crc32, byte[] buffer) {
        if (null != buffer) {
            m_uid = Resource.getUIDFromData(crc32, buffer);

            if (null == services) {
                m_buffer = buffer;
            } else {
                final ManagedFileManagement fileMgr = services.getService(ManagedFileManagement.class);

                if (null != fileMgr) {
                    // first try to find out, if we already have the managed file
                    try {
                        final String managedId = getManagedIdFromUID(m_uid);
                        m_managedFile = fileMgr.getByID(managedId);
                    } catch (OXException e) {
                        // nothing to do - exception means managed file is not known
                    }

                    if (null == m_managedFile) {
                        // create a new managed file
                        try {
                            final InputStream inputStream = new ByteArrayInputStream(buffer);
                            m_managedFile = fileMgr.createManagedFile(getManagedIdFromUID(m_uid), inputStream, true);
                            IOUtils.closeQuietly(inputStream);
                        } catch (OXException e) {
                            LOG.error(e.getMessage(), e);
                        }
                    }
                }
            }
        }
    }

    /**
     * Initializes a new {@link Resource}.
     * @param services
     * @param id The id of the distributed managed file
     */
    public Resource(ServiceLookup services, String id) {
        if (null != services) {
            final ManagedFileManagement fileMgr = services.getService(ManagedFileManagement.class);
            final long uid = Resource.getUIDFromManagedId(id);

            if ((null != fileMgr) && (0 != uid)) {
                try {
                    if (null != (m_managedFile = fileMgr.getByID(id))) {
                        m_uid = Resource.getUIDFromManagedId(id);
                    }
                } catch (OXException e) {
                    LOG.error(e.getMessage(), e);
                }
            }
        }
    }

    /**
     * @return The content of this resource as a byte array
     */
    public byte[] getBuffer() {
        byte[] ret = null;

        if (null == m_managedFile) {
            ret = m_buffer;
        } else {
            try {
                final InputStream inputStream = m_managedFile.getInputStream();

                if (null != inputStream) {
                    try {
                        ret = IOUtils.toByteArray(inputStream);
                    } catch (IOException e) {
                        LOG.error(e.getMessage(), e);
                    } finally {
                        IOUtils.closeQuietly(inputStream);
                    }
                }
            } catch (OXException e) {
                LOG.error(e.getMessage());
            }
        }

        return ret;
    }

    /**
     * @return
     */
    public int getCRC32() {
        return Resource.getCRC32FromUID(m_uid);
    }

    /**
     * @return The UID of the contained data
     */
    public long getUID() {
        return m_uid;
    }

    /**
     * @return The id of the distributed managed file or null if the id is not valid
     */
    public String getManagedId() {
        return (0 != m_uid) ? getManagedIdFromUID(getUID()) : null;
    }

    /**
     * @return The resource identifier of this resource as a string according to the following schema: "uid" + HexValueString
     */
    public String getResourceName() {
        return getResourceNameFromUID(m_uid);
    }

    /**
     *
     */
    public void touch() {
        if (null != m_managedFile) {
            m_managedFile.touch();
        }
    }

    /**
     *
     */
    public void delete() {
        if (null != m_managedFile) {
            m_managedFile.delete();
        }
    }

    /**
     * @param data
     * @return
     */
    static public int getCRC32(byte[] data) {
        CRC32 crc32 = new CRC32();

        if (null != data) {
            crc32.update(data);
        }

        return (int) crc32.getValue();
    }

    /**
     * @param crc32Key
     * @return
     */
    static public String getManagedIdFromUID(long uid) {
        return (idPrefix + Long.toString(uid, 16));
    }

    /**
     * @param id
     * @return
     */
    static public long getUIDFromManagedId(String id) {
        long uid = 0;

        if ((null != id) && id.startsWith(idPrefix)) {
            final String uidHexString = id.substring(idPrefix.length());

            if (uidHexString.length() > 0) {
                try {
                    uid = Long.parseLong(uidHexString, 16);
                } catch (NumberFormatException e ) {
                    LOG.error(e.getMessage(), e);
                }
            }
        }

        return uid;
    }

    /**
     * @param services
     * @param id
     * @return The distributed managed file resource or null if no resource with this id exists
     */
    static public Resource createFrom(ServiceLookup services, String id) {
        Resource ret = new Resource(services, id);

        if (0 == ret.getUID()) {
            ret = null;
        }

        return ret;
    }

    /**
     * @param uid The uid of a resource
     * @return The resource identifier for the given uid as a string according to the
     * following schema: "uid" + HexValueString, or null in case of an invalid uid
     */
    static public String getResourceNameFromUID(long uid) {
        return ((0 != uid) ? ("uid" + Long.toString(uid, 16)) : null);
    }

    /**
     * @param resourceName A resource name with the folllowing schema: "uid" + HexValueString
     * @return The uid of a resource name, or 0 in case of an invalid resource name
     */
    static public long getUIDFromResourceName(String resourceName) {
        return ((null != resourceName) ? Long.parseLong(resourceName.substring(3), 16) : 0);
    }

    /**
     * @param crc32
     * @param resdata
     * @return
     */
    static public long getUIDFromData(int crc32, byte[] resdata) {
        return (((long) resdata.length) << 32) | (crc32 & 0xffffffffL);
    }

    /**
     * @param uid
     * @return
     */
    static public int getCRC32FromUID(long uid) {
        return (int) (uid & 0xffffffffL);
    }

    // - Members ---------------------------------------------------------------

    static protected Logger LOG = LoggerFactory.getLogger(Resource.class);

    static protected String idPrefix = "office.resource0x";

    private ManagedFile m_managedFile = null;

    private byte[] m_buffer = null;

    private long m_uid = 0;
}
