package com.openexchange.office.tools.memory;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;

import org.apache.commons.logging.Log;

import com.openexchange.java.ConcurrentList;

/**
 * Singleton memory observer class which tracks the heap usage of the
 * greatest heap memory pool.
 *
 * @author Carsten Driesner
 *
 */
public class MemoryObserver implements NotificationListener
{
    private static final Log LOG = com.openexchange.log.Log.loggerFor(MemoryObserver.class);
    private static final MemoryPoolMXBean thresholdMemPool = getThresholdMemPool();
    private static final MemoryObserver memoryObserver = new MemoryObserver();
    private final ConcurrentList<MemoryListener> registeredListeners = new ConcurrentList<MemoryListener>();
    private final double maxPercentageLimit = 0.8; // 80%
    private final long   minFreeHeapMemory  = 131072000; // 128MB minimum free heap space
    private long threshold = 0;

    public interface MemoryListener {
        void memoryTresholdExceeded(long usedMemory, long maxMemory);
    }

    private MemoryObserver() {
        MemoryMXBean mbean = ManagementFactory.getMemoryMXBean();
        NotificationEmitter emitter = (NotificationEmitter) mbean;
        emitter.addNotificationListener(this, null, null);

        setUsageThreshold();
    }

    public static synchronized MemoryObserver getMemoryObserver() {
        return memoryObserver;
    }

    public boolean addListener(MemoryListener listener) {
        return registeredListeners.add(listener);
    }

    public boolean removeListener(MemoryListener listener) {
        return registeredListeners.remove(listener);
    }

    public static boolean isUsageThresholdExceeded() {
        boolean exceeded = thresholdMemPool.isCollectionUsageThresholdExceeded();
        if (exceeded) {
            // check memory usage manually as the call is true although
            // memory consumption has fallen below threshold
            final MemoryUsage memUsage = thresholdMemPool.getUsage();
            long memThreshold = getMemoryObserver().getMemPoolThreshold();
            exceeded = (memUsage.getUsed() > memThreshold);
            if (exceeded && LOG.isInfoEnabled()) {
                final long maxMemory = memUsage.getMax();
                final long usedMemory = memUsage.getUsed();
                LOG.debug("RT connection: Memory observer detects exceeding memory threshold! Max memory: " + maxMemory + ", limit: " + memThreshold + ", used memory: " + usedMemory);
            }
            thresholdMemPool.setCollectionUsageThreshold(memThreshold);
        }

        return exceeded;
    }

    private void setUsageThreshold() {
        long maxMemory = thresholdMemPool.getUsage().getMax();
        long thresh1 = (long)(maxMemory * maxPercentageLimit);
        long thresh2 = maxMemory - minFreeHeapMemory;

        // we only want to be notified after a comprehensive gc run
        threshold = (thresh2 > thresh1) ? thresh2 : thresh1;

        thresholdMemPool.setCollectionUsageThreshold(threshold);
        LOG.debug("RT connection: Memory observer uses memory pool: " + thresholdMemPool.getName()+ " with maxMemory: " + maxMemory + ", threshold: " + threshold);
    }

    private static MemoryPoolMXBean getThresholdMemPool() {
        MemoryPoolMXBean thresholdMemPool = null;
        for (MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
            if (pool.getType() == MemoryType.HEAP && pool.isUsageThresholdSupported()) {
                thresholdMemPool = pool;
                break;
            }
        }
        return thresholdMemPool;
    }

    private long getMemPoolThreshold() {
        return threshold;
    }

    /**
     * Invoked when a JMX notification occurs. The registered listeners are
     * called to react on exceeding the memory threshold.
     */
    @Override
    public void handleNotification(Notification n, Object jb) {
        if (n.getType().equals(MemoryNotificationInfo.MEMORY_COLLECTION_THRESHOLD_EXCEEDED)) {
            final CompositeData cd = (CompositeData) n.getUserData();
            final MemoryNotificationInfo info = MemoryNotificationInfo.from(cd);
            final MemoryUsage memUsage = info.getUsage();

            // retrieve memory data from the notification
            final long maxMemory = memUsage.getMax();
            final long usedMemory = memUsage.getUsed();
            final long memLimit = threshold;

            // Check limits manually, there are situations were the notification is sent
            // although memory threshold is NOT reached.
            LOG.debug("RT connection: Memory observer notified due to exceeding memory threshold! Max memory: " + maxMemory + ", limit: " + memLimit + ", used memory: " + usedMemory);

            final java.util.List<MemoryListener> listeners = registeredListeners.getSnapshot();
            for (MemoryListener listener : listeners) {
                try {
                    listener.memoryTresholdExceeded(usedMemory, maxMemory);
                } catch (Throwable e) {
                    // nothing can be done
                }
            }
        }
    }
}
