/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import java.lang.ref.WeakReference;
import java.util.LinkedList;
import java.util.List;
import com.openexchange.config.ConfigurationService;
import com.openexchange.realtime.Component;
import com.openexchange.realtime.LoadFactorCalculator;
import com.openexchange.server.ServiceLookup;

/**
 * The class checks the load factor history from RT to determine, if it is
 * save to create new component instances or not.
 *
 * {@link LoadFactorLimitManager}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.8.0
 */
public class LoadFactorLimitManager {
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(LoadFactorLimitManager.class);
    private static final int CHECK_LOADFACTOR_SIZE     = 6;
    private static final int STEPS_OUTPUT_STATISTICS   = 5;
    private final static float DEFAULT_MAX_LF_RATIO    = 2.5f;
    private final static float DEFAULT_NORMAL_LF_RATIO = 2.0f;
    private final static float DEFAULT_ACCEPTABLE_DERIVATION_RATIO = 1.5f;

    private final LoadFactorCalculator loadFactorCalculator;
    private final WeakReference<Component> component;
    private final List<Float> loadFactorHistory = new LinkedList<Float>();
    private float maxDetectedLoadFactor = 0f;
    private boolean overload = false;
    private int calls = 0;
    private float maxLFRatio = DEFAULT_MAX_LF_RATIO;
    private float normalLFRatio = DEFAULT_NORMAL_LF_RATIO;

    /**
     * Initializes a new {@link LoadFactorLimitManager}.
     *
     * @param loadFactorCalculator
     * @param component
     */
    public LoadFactorLimitManager(final ServiceLookup services, final LoadFactorCalculator loadFactorCalculator, WeakReference<Component> component) {
        this.loadFactorCalculator = loadFactorCalculator;
        this.component = component;

        ConfigurationService confService = services.getService(com.openexchange.config.ConfigurationService.class);
        if (null != confService) {
            String maxLFRatioProp = confService.getProperty("io.ox/office//module/maxLFRatio", Float.toString(DEFAULT_MAX_LF_RATIO));
            String normalLFRatioProp = confService.getProperty("io.ox/office//module/normalLFRatio", Float.toString(DEFAULT_NORMAL_LF_RATIO));

            try {
                maxLFRatio = Float.parseFloat(maxLFRatioProp);
            } catch (Exception e) { LOG.warn("io.ox/office//module/maxLFRatio cannot be parsed - using default!"); }

            try {
                normalLFRatio = Float.parseFloat(normalLFRatioProp);
            } catch (Exception e) { LOG.warn("io.ox/office//module/normalLFRatio cannot be parsed - using default!"); }
        }

    }

    /**
     * Determines, if we can add new load to the current component type.
     *
     * @return
     *  TRUE, if the system is able to handle an additional connection, otherwise
     *  FALSE.
     */
    public synchronized boolean canCreateNewInstance() {
        boolean canCreate = true;

        if (null != loadFactorCalculator) {
            float loadFactor = getLoadFactor();

            addLoadFactor(loadFactor);
            final Float[] loadFactoryHistory = getLoadFactorHistory();
            boolean currOverload = overload;
            if (currOverload) {
                overload = !detectNormalLoad(loadFactor);
            } else {
                overload = detectHighLoad(loadFactoryHistory);
            }
            canCreate = !overload;

            if (currOverload && !overload) {
                LOG.info("RT connection: System leaves high load state and accepts new join requests. Load factor = " + loadFactor );
            }
            if (!canCreate) {
                LOG.info("RT connection: Connection creation refused, system protects itself against high load. Load factor = " + loadFactor);
            }
        }

        return canCreate;
    }

    /**
     * Returns the load factor of the component.
     *
     * @return
     *  The load factor, where 1.0 means one message is waiting in every run loop
     *  instance.
     */
    public float getLoadFactor() {
        return (null != loadFactorCalculator) ? loadFactorCalculator.getCurrentLoad(component.get()) : 0f;
    }

    /**
     * Retrieves the load factor history as an array of floats.
     *
     * @return
     *  An array of floats representing the history of RT load factor, where
     *  the first entry is the latest value.
     */
    private Float[] getLoadFactorHistory() {
        Float[] history = new Float[loadFactorHistory.size()];
        loadFactorHistory.toArray(history);
        return history;
    }

    /**
     * Adds a new load factor to the history.
     *
     * @param loadFactor the load factor to be added
     */
    private void addLoadFactor(final float loadFactor) {
        calls++;
        if (loadFactorHistory.size() > CHECK_LOADFACTOR_SIZE) {
            loadFactorHistory.remove(loadFactorHistory.size() - 1);
        }
        loadFactorHistory.add(0, loadFactor);
    }

    /**
     * Tries to detect high load on the backend on analyzing the load factors
     * reported by the RT run loops.
     *
     * The algorithm tries to find a steady (with 15% derivation) increase of
     * the load factor (in series of 6 values), where the (max / min) load
     * factor quotient is maxLFRatio or higher.
     *
     * @param loadFactorHistory an array of load factor values
     *
     * @return True, if a high load has been detected, otherwise false.
     */
    private boolean detectHighLoad(final Float[] loadFactorHistory) {
        boolean highLoad = false;

        if ((null != loadFactorHistory) && (loadFactorHistory.length > 0)) {
            boolean minSet = false;
            boolean maxSet = false;
            boolean lastSet = false;
            int loadIncr = 0;
            float minLF = 0;
            float maxLF = 0;
            float lastLF = 0;
            float maxLFSeries = 0;
            float minLFSeries = 0;

            for (int i = 0; i < loadFactorHistory.length; i++) {
                final float f = loadFactorHistory[i];

                if (lastSet) { loadIncr += (lastLF >= (f / DEFAULT_ACCEPTABLE_DERIVATION_RATIO)) ? 1 : 0; }
                if (f <= minLF || !minSet) { minLF = f; minSet = true; }
                if (f >= maxLF || !maxSet) { maxLF = f; maxSet = true; }

                if ((i + 1 == CHECK_LOADFACTOR_SIZE) && (loadIncr == CHECK_LOADFACTOR_SIZE - 1)) {
                    maxLFSeries = maxLF;
                    minLFSeries = minLF;
                }

                lastLF = f;
                lastSet = true;
            }

            final float seriesLoadFactorInc = (minLFSeries > 0) ? (maxLFSeries / minLFSeries) : 0;
            highLoad = (maxLFSeries > minLFSeries) && (minLFSeries > 0) && (seriesLoadFactorInc > maxLFRatio);

            if (highLoad) {
                maxDetectedLoadFactor = maxLF;
            }

            if ((calls >= STEPS_OUTPUT_STATISTICS) || highLoad) {
                final StringBuffer buf = new StringBuffer(200);
                final Component comp = (null != component) ? component.get() : null;
                boolean first = true;

                calls = 0;

                buf.append("RT connection: Load factor statistics (");
                for (float f : loadFactorHistory) {
                    if (!first) { buf.append(", "); }
                    buf.append(f);
                    first = false;
                }
                buf.append(")");

                buf.append(", minLF = ");
                buf.append(minLF);
                buf.append(", maxLF = ");
                buf.append(maxLF);
                buf.append(", minLFSeries = ");
                buf.append(minLFSeries);
                buf.append(", maxLFSeries = ");
                buf.append(maxLFSeries);
                buf.append(", maxLFSeries / minLFSeries = ");
                buf.append(seriesLoadFactorInc);
                buf.append(highLoad ? ", high load detected" : ", system in normal state");
                buf.append(" for ");

                if (null != comp) {
                    // provide more information about the component, run loops, pending requests, etc.
                    buf.append(comp.getId());
                    buf.append(" (");

                    if (comp instanceof IConnectionPool) {
                        final IConnectionPool pool = (IConnectionPool)comp;
                        buf.append(pool.getConnectionCount());
                        buf.append(" documents/");
                    }

                    if (null != loadFactorCalculator) {
                        int runLoopsCount = loadFactorCalculator.getRunLoopCount(component.get());
                        int runLoopFillSum = Math.round(loadFactorHistory[0] * runLoopsCount);
                        buf.append(runLoopFillSum);
                        buf.append(" RunLoopFillSum/");
                        buf.append(runLoopsCount);
                        buf.append(" Loops");
                    }
                    buf.append(")");
                }

                if (highLoad) {
                    LOG.warn(buf.toString());
                } else {
                    LOG.debug(buf.toString());
                }
            }
        }

        return highLoad;
    }

    /**
     * Checks the current load factor of the RT run loops to determine that
     * it's save to switch back to normal mode (accepting new connections).
     *
     * @param loadFactor the current load factor of the RT run loops
     *
     * @return True, if the system is in normal mode again, otherwise false.
     */
    private boolean detectNormalLoad(final float loadFactor) {
        boolean normalLoad = (loadFactor < (maxDetectedLoadFactor / normalLFRatio));
        if (normalLoad) {
            maxDetectedLoadFactor = 0;
        }
        return normalLoad;
    }

}
