/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.pptx4j.pml.Presentation.SldMasterIdLst;
import org.pptx4j.pml.Presentation.SldMasterIdLst.SldMasterId;
import org.pptx4j.pml.SlideLayoutIdList;
import org.pptx4j.pml.SlideLayoutIdList.SldLayoutId;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.pptx.components.RootComponent;
import com.openexchange.office.ooxml.pptx.operations.PptxApplyOperationHelper;
import com.openexchange.office.ooxml.pptx.operations.PptxCreateOperationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class PptxOperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    private PresentationMLPackage opcPackage;
    private ThemePart themePart = null;

    private String target = "";
    private final HashMap<String, Part> idToPart = new HashMap<String, Part>();
    private final HashMap<String, String> relTargetToId = new HashMap<String, String>();

    public PptxOperationDocument(Session session, ServiceLookup _services, InputStream _inputDocumentStream, IResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(session, _services, _inputDocumentStream, _resourceManager, _documentProperties);
        try {
            opcPackage = (PresentationMLPackage)PresentationMLPackage.load(_inputDocumentStream);
            themePart = null;
            initIdTables();
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    // the constructor without inputStream creates an empty document
    public PptxOperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(null, _services, null, null, _documentProperties);
        try {
            opcPackage = PresentationMLPackage.createPackage();
            themePart = null;
            initIdTables();
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final JSONObject aOperations = new JSONObject();
        final JSONArray operationsArray = new JSONArray();

        final PptxCreateOperationHelper createOperationHelper = new PptxCreateOperationHelper(this, operationsArray);
        createOperationHelper.createDocumentDefaults(getUserLanguage());
        createOperationHelper.createMasterSlides();
        createOperationHelper.createSlides();
        aOperations.put("operations", operationsArray);
        return aOperations;
    }

	public Map<String, Object> getMetaData()
		throws Exception {

		final JSONArray operationsArray = new JSONArray();
		final PptxCreateOperationHelper createOperationHelper = new PptxCreateOperationHelper(this, operationsArray);
        createOperationHelper.createDocumentDefaults(getUserLanguage());
		final Map<String, Object> operations = new HashMap<String, Object>(1);
        operations.put("operations", operationsArray);
        return operations;
	}

	public Map<String, Object> getActivePart()
		throws Exception {

		final JSONArray operationsArray = new JSONArray();
		final PptxCreateOperationHelper createOperationHelper = new PptxCreateOperationHelper(this, operationsArray);
        createOperationHelper.createMasterSlides();
        createOperationHelper.createSlides();
		final Map<String, Object> operations = new HashMap<String, Object>(1);
        operations.put("operations", operationsArray);
        return operations;
	}

	public Map<String, Object> getNextPart()
		throws Exception {

		return null;
	}

    private void initIdTables() {
		final MainPresentationPart mainPresentationPart = opcPackage.getMainPresentationPart();
		final SldMasterIdLst slideMasterIds = mainPresentationPart.getJaxbElement().getSldMasterIdLst();
		if(slideMasterIds!=null) {
			final List<SldMasterId> slideMasterIdList = slideMasterIds.getSldMasterId();
			for(int i=0; i<slideMasterIdList.size(); i++) {
				final SldMasterId slideMasterId = slideMasterIdList.get(i);
				final SlideMasterPart slideMasterPart = (SlideMasterPart)mainPresentationPart.getRelationshipsPart().getPart(slideMasterId.getRid());
				final String slideMasterIdString = slideMasterId.getId().toString();
				idToPart.put(slideMasterIdString, slideMasterPart);
				relTargetToId.put("../" + mainPresentationPart.getRelationshipsPart().getRelationshipByID(slideMasterId.getRid()).getTarget(), slideMasterIdString);

				// retrieving layouts
				final SlideLayoutIdList slideLayoutIds = slideMasterPart.getJaxbElement().getSldLayoutIdLst(false);
				if(slideLayoutIds!=null) {
					final List<SldLayoutId> slideLayoutIdList = slideLayoutIds.getSldLayoutId();
					for(int j=0; j<slideLayoutIdList.size(); j++) {
						final SldLayoutId slideLayoutId = slideLayoutIdList.get(j);
						final String slideLayoutIdString = slideLayoutId.getId().toString();
						idToPart.put(slideLayoutIdString, slideMasterPart.getRelationshipsPart().getPart(slideLayoutId.getRid()));
						relTargetToId.put(slideMasterPart.getRelationshipsPart().getRelationshipByID(slideLayoutId.getRid()).getTarget(), slideLayoutIdString);
					}
				}
			}
		}
    }

    public HashMap<String, Part> getIdToPartMap() {
    	return idToPart;
    }

    public HashMap<String, String> getRelTargetToIdMap() {
    	return relTargetToId;
    }

    // - the target can be "" then a normal slide is addressed, otherwise it points to a layout or masterslide
    public Part setContextPartByTarget(String target) {
    	this.target = target;
    	setContextPart(getPartByTarget(target));
    	return contextPart;
    }

    @Override 
    public void setContextPart(Part part) {
    	if(contextPart!=part) {
    		contextPart = part;
    		themePart = null;
    		themeFonts = null;
    	}
    }

    @Override
    public Part getContextPart() {
    	return contextPart;
    }

    public Part getPartByTarget(String target) {
    	Part newContext = null;
    	if(target!=null&&!target.isEmpty()) {
    		newContext = idToPart.get(target);
        }
    	return newContext == null ? getPackage().getMainPresentationPart() : newContext;
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {

        if (applyOperations != null) {
            int i = 0;
            JSONObject op = null;
            String opName = null;
            try {
                final PptxApplyOperationHelper applyOperationHelper = new PptxApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                	successfulAppliedOperations = i;
                    op = (JSONObject) aOperations.get(i);
                    final String target = op.optString("target", "");
                    opName = op.getString("name");
                    switch(opName) {
                    	case "insertParagraph" : {
	                        applyOperationHelper.insertParagraph(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
	                        break;
                    	}
                    	case "splitParagraph": {
                    		applyOperationHelper.splitParagraph(op.getJSONArray("start"), target);
                    		break;
                    	}
                    	case "mergeParagraph": {
                            applyOperationHelper.mergeParagraph(op.getJSONArray("start"), target);
                            break;
                    	}
                    	case "insertText": {
                            applyOperationHelper.insertText(op.getJSONArray("start"), target, op.getString("text").replaceAll("\\p{C}", " "), op.optJSONObject("attrs"));
                            break;
                    	}
                    	case "delete": {
                            applyOperationHelper.delete(op.getJSONArray("start"), target, op.optJSONArray("end"));
                            break;
                    	}
                    	case "setAttributes": {
                            applyOperationHelper.setAttributes(op.getJSONArray("start"), target, op.optJSONArray("end"), op.getJSONObject("attrs"));
                            break;
                    	}
                    	case "insertSlide": {
                    		applyOperationHelper.insertSlide(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertMasterSlide": {
                    		applyOperationHelper.insertMasterSlide(op.getString("id"), op.optInt("start"));
                    		break;
                    	}
                    	case "insertLayoutSlide": {
                    		applyOperationHelper.insertLayoutSlide(op.getString("id"), op.getString("target"), op.optInt("start"));
                    		break;
                    	}
                    	case "insertDrawing": {
                    		applyOperationHelper.insertDrawing(op.getJSONArray("start"), target, op.getString("type").toUpperCase(), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "setDocumentAttributes": {
                    		applyOperationHelper.setDocumentAttributes(op.getJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertHardBreak": {
                            applyOperationHelper.insertHardBreak(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                            break;
                    	}
                    	case "changeLayout": {
                    		applyOperationHelper.changeLayout(op.getJSONArray("start"), target);
                    		break;
                    	}
                    	case "moveSlide": {
                    		applyOperationHelper.moveSlide(op.getJSONArray("start"), op.getJSONArray("end"));
                    		break;
                    	}
                    	case "moveLayoutSlide": {
                    		applyOperationHelper.moveLayoutSlide(op.getString("id"), op.getInt("start"), target);
                    		break;
                    	}
                    	case "insertTab": {
                            applyOperationHelper.insertTab(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                            break;
                    	}
                    	case "insertField": {
                    		applyOperationHelper.insertField(op.getJSONArray("start"), target, op.getString("type"), op.getString("type"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "updateField": {
                    		applyOperationHelper.updateField(op.getJSONArray("start"), target, op.optString("type"), op.optString("type"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "group": {
                    		applyOperationHelper.group(op.getJSONArray("start"), target, op.getJSONArray("drawings"), op.getJSONObject("attrs"), op.optJSONArray("childAttrs"));
                    		break;
                    	}
                    	case "ungroup": {
                    		applyOperationHelper.ungroup(op.getJSONArray("start"), target, op.optJSONArray("drawings"));
                    		break;
                    	}
                    	case "move": {
                            applyOperationHelper.move(op.getJSONArray("start"), target, op.optJSONArray("end"), op.getJSONArray("to"));
                            break;
                    	}
                    }
                    
/*
                    else if (opName.equals("move"))
                        applyOperationHelper.move(op.getJSONArray("start"), op.optJSONArray("end"), op.getJSONArray("to"));
                    else if (opName.equals("insertTab"))
                        applyOperationHelper.insertTab(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertHardBreak"))
                        applyOperationHelper.insertHardBreak(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertTable"))
                        applyOperationHelper.insertTable(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertRows"))
                        applyOperationHelper.insertRows(op.getJSONArray("start"), op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertCells"))
                        applyOperationHelper.insertCells(op.getJSONArray("start"), op.optInt("count", 1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertColumn"))
                        applyOperationHelper.insertColumn(op.getJSONArray("start"), op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
                    else if (opName.equals("deleteColumns"))
                        applyOperationHelper.deleteColumns(op.getJSONArray("start"), op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
                    else if (opName.equals("setAttributes"))
                        applyOperationHelper.setAttributes(op.getJSONObject("attrs"), op.getJSONArray("start"), op.optJSONArray("end"));
                    else if (opName.equals("insertField"))
                        applyOperationHelper.insertField(op.getJSONArray("start"), op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertDrawing"))
                        applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type").toUpperCase(), op.optJSONObject("attrs"));
                    else if (opName.equals("insertStyleSheet"))
                        applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent"), op.optBoolean("hidden"), op.getInt("uiPriority"), op.optBoolean("default"));
                    else if (opName.equals("deleteStylesheet"))
                        applyOperationHelper.deleteStylesheet(op.getString("type"), op.getString("styleId"));
                    else if (opName.equals("insertListStyle"))
                        applyOperationHelper.insertListStyle(op.getString("listStyleId"), op.getJSONObject("listDefinition"));
*/
                }
                successfulAppliedOperations=aOperations.length();
            }
            catch(Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    @Override
    public PresentationMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public void setPackage(OpcPackage p) {
       opcPackage = (PresentationMLPackage)p;
    }

    @Override
	public Component getRootComponent(String target) {
    	return new RootComponent(this, target);
	}

    @Override
    public ThemePart getThemePart(boolean createIfMissing)
    	throws FilterException {

    	try {
	    	if(themePart==null) {
	    		Part context = contextPart;
	    		if(context instanceof SlidePart) {
	    			context = ((SlidePart)context).getSlideLayoutPart();
	    		}
	    		if(context instanceof SlideLayoutPart) {
	    			context = ((SlideLayoutPart)context).getSlideMasterPart();
	    		}
	    		final RelationshipsPart masterRelations = context.getRelationshipsPart();
	    		final Relationship rel = masterRelations.getRelationshipByType(Namespaces.THEME);
	    		if(rel!=null) {
	    			themePart = (ThemePart)masterRelations.getPart(rel);
	    		}
	    		if(themePart==null&&createIfMissing) {
	        		final PartName themePartName = new PartName("/ppt/theme/theme1.xml");
		    		themePart = getDefaultThemePart(themePartName);
		    		context.addTargetPart(themePart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
	    		}
	    	}
	    	return themePart;
    	}
    	catch(InvalidFormatException e) {
    		throw new FilterException("xlsx filter, could not create default theme", ErrorCode.CRITICAL_ERROR, e);
    	}
    }
}
