/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom.components;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.style.StyleStyleElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DLNode;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.Styles;
import com.openexchange.office.odt.dom.Cell;
import com.openexchange.office.odt.dom.Column;
import com.openexchange.office.odt.dom.Row;
import com.openexchange.office.odt.dom.Table;

public class TableComponent extends Component {

	final Table table;

	public TableComponent(ComponentContext parentContext, DLNode<Object> tableNode, int componentNumber) {
		super(parentContext, tableNode, componentNumber);
		this.table = (Table)getObject();
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Table)getObject()).getContent().getFirstNode();
		while(nextNode!=null) {
			if(nextNode.getObject() instanceof Row) {
				return new RowComponent(this, nextNode, nextComponentNumber);
			}
			nextNode = nextNode.next;
		}
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		final String styleId = attrs.optString("styleId");
		if(!styleId.isEmpty()) {
    		table.setStyleName(styleId);
		}
		final JSONObject tableProperties = attrs.optJSONObject("table");
		if(tableProperties!=null) {

			// TODO: big fake ... to be able to use the old "addStyle" functionality
    		// we temporarily create a TextSpanElement to provide the actual style
    		final OdfFileDom ownerDocument = operationDocument.getOwnerDocument();
    		final OdfStylableElement e = new TableTableElement(ownerDocument);
    		e.setAttributeNS(Namespaces.DRAW, "table:style-name", table.getStyleName());
    		Styles.addStyle(attrs, e, ownerDocument);
    		table.setStyleName(e.getAttributeNS(Namespaces.TABLE, "style-name"));

			final JSONArray tableGrid = tableProperties.optJSONArray("tableGrid");
			if(tableGrid!=null) {
				applyTableGrid(operationDocument, tableGrid);
			}
		}
	}

	private void applyTableGrid(OdfOperationDocument operationDocument, JSONArray tableGrid)
		throws SAXException, JSONException {

		final List<Column> columns = table.getColumns();
		columns.clear();
		for(int i = 0; i < tableGrid.length(); i++) {
			final Column column = new Column();
			columns.add(column);
	
			// TODO: big fake ... to be able to use the old "addStyle" functionality
			// we temporarily create a TableColumnElement to provide the actual style
			final OdfFileDom ownerDocument = operationDocument.getOwnerDocument();
			final TableTableColumnElement e = new TableTableColumnElement(ownerDocument);
			final String absoluteColumnWidth = (Double.valueOf(tableGrid.getDouble(i) / 100).toString() + "mm");
			applyWidth(e, absoluteColumnWidth, absoluteColumnWidth);
			column.setStyleName(e.getAttributeNS(Namespaces.TABLE, "style-name"));
		}
	}

    private static void applyWidth(TableTableColumnElement columnElement, String absoluteWidth, String relativeWidth) {
        StyleStyleElement columnStyleElement = columnElement.getOrCreateUnqiueAutomaticStyle();
        StyleTableColumnPropertiesElement columnPropsElement = (StyleTableColumnPropertiesElement) columnStyleElement.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableColumnProperties);
        if (columnPropsElement == null) {
            columnPropsElement = columnStyleElement.newStyleTableColumnPropertiesElement();
        }
        // There is a problem with relative width in some ODF applications, therefore we need to set the absolute width as well
        if (absoluteWidth != null) {
            columnPropsElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:column-width", absoluteWidth);
            // remove the relative with, otherwise LO/AO will ignore the absolute, but unfortunately is buggy with the relative
            columnPropsElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "rel-column-width");
        } else if (relativeWidth != null) {
            // if there is no absolute tableElement width, the relative widths have to be used (or "auto" being resolved)
            columnPropsElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:rel-column-width", relativeWidth);
        }
    }
	
	@Override
	public JSONObject createJSONAttrs(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		final String styleId = table.getStyleName();
		JSONObject tableProperties = null;
		if(styleId!=null) {
			attrs.put("styleId", styleId);
			final Map<String, Object> hardFormatting =
				Styles.getAutomaticStyleHierarchyProps(styleId, "table", ((Styles)operationDocument.getDocument().getStylesDom()).getStyles(), attrs.asMap());
			if(hardFormatting.containsKey("table")) {
				tableProperties = (JSONObject)hardFormatting.get("table");
			}
		}
		if(tableProperties==null) {
			tableProperties = new JSONObject();
		}
		final List<Column> columns = table.getColumns();
		if(!columns.isEmpty()) {
			int calculatedTableWidth = 0;
			final List<Integer> tableGrid = new ArrayList<Integer>(columns.size());
			for(Column column:columns) {
				Integer width = null;
		        final String styleName = column.getStyleName();
		        if(styleName!=null) {
			        final Map<String, Object> hardFormatting = 
			        	Styles.getAutomaticStyleHierarchyProps(styleName, "table-column", ((Styles)operationDocument.getDocument().getStylesDom()).getStyles(), attrs.asMap());

			        final JSONObject drawingProps = (JSONObject)hardFormatting.get("column");
			        if(drawingProps!=null) {
			        	final Object o = drawingProps.get("width");
			        	if(o instanceof Number) {
			        		width = Integer.valueOf(((Number)o).intValue());
					        tableGrid.add(width);
					        calculatedTableWidth += width;
			        	}
			        }
		        }
			}
			// ensure that we have collected proper values for the grid,
			// otherwise it is best to provide better defaults
			if(tableGrid.size()!=columns.size()) {
				tableGrid.clear();
				for(int i=0; i < columns.size(); i++) {
					tableGrid.add(100);
				}
				if(!tableProperties.has("width")) {
					tableProperties.put("width", "auto");
				}
			}
			else {
				if(!tableProperties.has("width")) {
					tableProperties.put("width", calculatedTableWidth);
				}
			}
			tableProperties.put("tableGrid", tableGrid);
		}
		if(!tableProperties.isEmpty()) {
			attrs.put("table", tableProperties);
		}
		return attrs;
	}

	public void insertRows(OdfOperationDocument operationDocument, int rowPosition, int count, boolean insertDefaultCells, int referenceRowNumber, JSONObject attrs)
		throws JSONException, SAXException {

		final RowComponent referenceRowComponent = referenceRowNumber!=-1 ? (RowComponent)getChildComponent(referenceRowNumber) : null;
        final DLList<Object> tblContent = table.getContent();
        final Component oldTr = getChildComponent(rowPosition);
        DLNode<Object> referenceNode = oldTr!=null?oldTr.getNode():null;

        for(int i = 0; i<count; i++) {
        	final Row row = !insertDefaultCells && referenceRowComponent != null ? ((Row)referenceRowComponent.getObject()).clone() : new Row();
            final DLNode<Object> rowNode = new DLNode<Object>(row);
            tblContent.addNode(referenceNode, rowNode, true);
            referenceNode = rowNode;
            if(insertDefaultCells) {
                for(int j = 0; j < table.getColumns().size(); j++) {
                	row.getContent().add(new Cell());
                }
            }
            else if(referenceRowComponent!=null) {       // we try to create the new row from the referenceRow
                CellComponent referenceCellComponent = (CellComponent)referenceRowComponent.getNextChildComponent(null, null);
                while(referenceCellComponent!=null) {
                    row.getContent().add(((Cell)referenceCellComponent.getObject()).clone());
                    referenceCellComponent = (CellComponent)referenceCellComponent.getNextComponent();
                }
            }
        }
        if(attrs!=null) {
        	Component c = getNextChildComponent(null, null).getComponent(rowPosition);
	        for(int i=0; i<count; i++) {
	        	c.applyAttrsFromJSON(operationDocument, attrs);
	        	c = c.getNextComponent();
	        }
        }
	}

	public void insertColumn(OdfOperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
		throws JSONException, SAXException {

		boolean before = insertMode.equals("before");
		RowComponent trComponent = (RowComponent)getNextChildComponent(null, null);
        while(trComponent!=null) {
        	CellComponent tcReference = (CellComponent)trComponent.getNextChildComponent(null, null);
        	CellComponent destination = null;
            while(tcReference!=null) {
                if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                    destination = tcReference;
                    break;
                }
                tcReference = (CellComponent)tcReference.getNextComponent();
            }
            final Cell tc = tcReference!=null ? tcReference.getCell().clone() : new Cell();
            tc.setColumnSpan(1);
            final DLList<Object> rowContent = trComponent.getRow().getContent();
            if(destination==null) {
            	rowContent.add(tc);
            }
            else {
            	final ComponentContext contextChild = destination.getContextChild();
            	rowContent.addNode(contextChild.getNode(), new DLNode<Object>(tc), before);
            }
            trComponent = (RowComponent)trComponent.getNextComponent();
        }
        applyTableGrid(operationDocument, tableGrid);
	}

    public void deleteColumns(OdfOperationDocument operationDocument, int gridStart, int gridEnd)
    	throws JSONException {

    	Component trComponent = getNextChildComponent(null, null);
        while(trComponent!=null) {
        	DLNode<Object> startNode = null;
        	DLNode<Object> endNode = null;
        	CellComponent tcComponent = (CellComponent)trComponent.getNextChildComponent(null, null);
            while(tcComponent!=null) {
            	final int x1 = tcComponent.getGridPosition();
            	final int x2 = tcComponent.getNextGridPosition();
            	if(Math.max(x2, gridEnd + 1) - Math.min(x1, gridStart) < (x2 - x1) + ((gridEnd+1)-gridStart)) {
               		final ComponentContext contextChild = tcComponent.getContextChild();
                	if (startNode==null) {
                        startNode = contextChild.getNode();
                	}
                    endNode = contextChild.getNode();
            	}
                if(tcComponent.getNextGridPosition()>gridEnd)
                    break;
                tcComponent = (CellComponent)tcComponent.getNextComponent();
            }
            if(startNode!=null) {
                ((Row)trComponent.getObject()).getContent().removeNodes(startNode, endNode);
            }
            trComponent = trComponent.getNextComponent();
        }
        for(int i = gridEnd; i > gridStart; i--) {
        	table.getColumns().remove(i);
        }
	}
}
