/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.IDrawingType;
import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DrawingType;
import com.openexchange.office.odf.Settings;
import com.openexchange.office.odt.dom.components.AnnotationComponent;
import com.openexchange.office.odt.dom.components.AnnotationEndComponent;
import com.openexchange.office.odt.dom.components.CellComponent;
import com.openexchange.office.odt.dom.components.Component;
import com.openexchange.office.odt.dom.components.FrameComponent;
import com.openexchange.office.odt.dom.components.ParagraphComponent;
import com.openexchange.office.odt.dom.components.RootComponent;
import com.openexchange.office.odt.dom.components.RowComponent;
import com.openexchange.office.odt.dom.components.ShapeComponent;
import com.openexchange.office.odt.dom.components.ShapeRootComponent;
import com.openexchange.office.odt.dom.components.TableComponent;
import com.openexchange.office.odt.dom.components.TextComponent;
import com.openexchange.office.odt.dom.components.TextFieldComponent;
import com.openexchange.office.odt.dom.components.TextLineBreakComponent;
import com.openexchange.office.odt.dom.components.TextTabComponent;

public class JsonOperationProducer {

    final JSONArray operationQueue;

    private final OdfOperationDocument opsDoc;
    private final OdfTextDocument doc;
    private final ODTStyles styles;
    private final Content content;
    private final Settings settings;

    public JsonOperationProducer(OdfOperationDocument opsDoc)
    	throws SAXException {

    	operationQueue = new JSONArray();
    	this.opsDoc = opsDoc;
        doc = (OdfTextDocument)opsDoc.getDocument();
        styles = (ODTStyles)doc.getStylesDom();
        content = (Content)doc.getContentDom();
        settings = (Settings)doc.getSettingsDom();
    }

    public JSONObject getDocumentOperations()
    	throws JSONException, SAXException {

    	styles.createInsertStyleOperations(operationQueue, styles.getOfficeStyles(), false, true);
    	styles.createInsertStyleOperations(operationQueue, content.getAutomaticStyles(), false, true);
    	createHeaderFooterOperations();
    	createTextOperations(new RootComponent(content.getBody()), new ArrayList<Integer>(), null);

    	final JSONObject operations = new JSONObject(1);
        operations.put("operations", operationQueue);
        return operations;
    }

    public void createHeaderFooterOperations()
    	throws JSONException, SAXException {

    	final MasterStyles masterStyles = styles.getMasterStyles(false);
    	if(masterStyles!=null) {
    		for(StyleMasterPage styleMasterPage:masterStyles.getStyleMasterPages()) {
    			for(StyleHeader styleHeader:styleMasterPage.getStyleHeaders()) {
    				addInsertHeaderFooterOperation(styleHeader.getId(), "header_default");
    				createTextOperations(new RootComponent(styleHeader), new ArrayList<Integer>(), styleHeader.getId());
    			}
    			for(StyleFooter styleFooter:styleMasterPage.getStyleFooters()) {
    				addInsertHeaderFooterOperation(styleFooter.getId(), "footer_default");
    				createTextOperations(new RootComponent(styleFooter), new ArrayList<Integer>(), styleFooter.getId());
    			}
    		}
    	}
    }

    public void createTextOperations(Component parentComponent, List<Integer> parentPosition, String target)
    	throws JSONException, SAXException {

        Component component = parentComponent.getNextChildComponent(null, null);
        while (component!=null) {
        	if(component instanceof ParagraphComponent) {
        		createParagraphOperations((ParagraphComponent)component, parentPosition, target);
        	}
        	else if(component instanceof TableComponent) {
        		createTableOperations((TableComponent)component, parentPosition, target);
        	}
            component = component.getNextComponent();
        }
    }

    public void createParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> parentPosition, String target)
    	throws JSONException, SAXException {

    	// insert paragraph and apply paragraph attributes
        final List<Integer> paragraphPosition = new ArrayList<Integer>(parentPosition);
        paragraphPosition.add(paragraphComponent.getComponentNumber());
        JSONObject attrs = paragraphComponent.createJSONAttrs(opsDoc, new JSONObject(4));
        JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");
        if(paragraphAttrs!=null&&paragraphAttrs.has("listStyleId")) {
        	styles.addListStyle(operationQueue, attrs.getJSONObject("paragraph").getString("listStyleId"));
        }
        addInsertParagraphOperation(paragraphPosition, target, attrs);

        // insert components (text and or other drawing objects)
        Component component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
        	final List<Integer> textPosition = new ArrayList<Integer>(paragraphPosition);
        	textPosition.add(component.getComponentNumber());
        	if(component instanceof TextComponent) {
        		addInsertTextOperation(textPosition, ((Text)(component.getObject())).getText(), target);
        	}
        	else if(component instanceof TextTabComponent) {
        		addInsertTabOperation(textPosition, target);
        	}
        	else if(component instanceof TextLineBreakComponent) {
        		addInsertHardBreakOperation(textPosition, target);
        	}
        	else if(component instanceof TextFieldComponent) {
        		addInsertFieldOperation(textPosition, ((TextField)component.getObject()), target);
        	}
        	else if(component instanceof FrameComponent) {
        		createFrameOperations((FrameComponent)component, textPosition, target);
        	}
        	else if(component instanceof ShapeRootComponent) {
        		createShapeOperations(component, textPosition, target);
        	}
        	else if(component instanceof AnnotationComponent) {
        		final Annotation annotation = (Annotation)component.getObject();
        		addInsertCommentOperation(textPosition, target, annotation);
        		createTextOperations(new RootComponent(annotation), new ArrayList<Integer>(), annotation.getId());
        	}
        	else if(component instanceof AnnotationEndComponent) {
        		addInsertRangeOperation(textPosition, target, (AnnotationEnd)component.getObject());
        	}
        	component = component.getNextComponent();
        }

        component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
            attrs = component.createJSONAttrs(opsDoc, new JSONObject());
            if (attrs!=null&&!attrs.isEmpty()) {
	            int startComponent = component.getComponentNumber();
	            int endComponent   = component.getNextComponentNumber()-1;
	            final List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
	            startPosition.add(startComponent);
	            if(startComponent==endComponent) {
	                addSetAttributesOperation(attrs, startPosition, null, target);
	            }
	            else {
	            	final List<Integer> endPosition = new ArrayList<Integer>(paragraphPosition);
	                endPosition.add(endComponent);
	                addSetAttributesOperation(attrs, startPosition, endPosition, target);
	            }
            }
        	component = component.getNextComponent();
        }
    }

    public void createTableOperations(TableComponent tableComponent, List<Integer> parentPosition, String target)
        	throws JSONException, SAXException {

        final List<Integer> tablePosition = new ArrayList<Integer>(parentPosition);
        tablePosition.add(tableComponent.getComponentNumber());
        final JSONObject attrs = tableComponent.createJSONAttrs(opsDoc, new JSONObject(4));
    	addInsertTableOperation(tablePosition, target, attrs);
    	Component component= tableComponent.getNextChildComponent(null, null);
    	while(component!=null) {
    		if(component instanceof RowComponent) {
    	        final List<Integer> rowPosition = new ArrayList<Integer>(tablePosition);
    	        rowPosition.add(component.getComponentNumber());
    			createRowOperations((RowComponent)component, rowPosition, target);
    		}
    		component = component.getNextComponent();
    	}
    }

    public void createRowOperations(RowComponent rowComponent, List<Integer> rowPosition, String target)
        	throws JSONException, SAXException {

        final JSONObject attrs = rowComponent.createJSONAttrs(opsDoc, new JSONObject(4));
    	addInsertRowsOperation(rowPosition, 1, target, attrs);
    	Component component= rowComponent.getNextChildComponent(null, null);
    	while(component!=null) {
    		if(component instanceof CellComponent) {
    			final List<Integer> cellPosition = new ArrayList<Integer>(rowPosition);
    	        cellPosition.add(component.getComponentNumber());
    	        addInsertCellsOperation(cellPosition, target, component.createJSONAttrs(opsDoc, new JSONObject(4)));
    	   		createTextOperations(component, cellPosition, target);
    		}
    		component = component.getNextComponent();
    	}
    }

    public void createFrameOperations(FrameComponent frameComponent, List<Integer> position, String target)
    	throws JSONException, SAXException {

    	addInsertDrawingOperation(position, frameComponent.getType(), frameComponent.createJSONAttrs(opsDoc, new JSONObject()), target);
    	// a frame component (textBox, image may have childs)...
        Component component = frameComponent.getNextChildComponent(null, null);
        while(component!=null) {
        	if(component instanceof ParagraphComponent) {
        		createParagraphOperations((ParagraphComponent)component, position, target);
        	}
        	else if(component instanceof TableComponent) {
        		createTableOperations((TableComponent)component, position, target);
        	}
        	component = component.getNextComponent();
        }
    }

    public void createShapeOperations(Component shapeComponent, List<Integer> position, String target)
    	throws JSONException, SAXException {

    	addInsertDrawingOperation(position, ((IDrawingType)shapeComponent).getType(), shapeComponent.createJSONAttrs(opsDoc,  new JSONObject()), target);
    	Component component = shapeComponent.getNextChildComponent(null, null);
    	while(component!=null) {
    		if(component instanceof ParagraphComponent) {
    			createParagraphOperations((ParagraphComponent)component, position, target);
    		}
    		else if(component instanceof ShapeComponent) {
    	        final List<Integer> childPosition = new ArrayList<Integer>(position);
    	        childPosition.add(component.getComponentNumber());
    			createShapeOperations(component, childPosition, target);
    		}
    		component = component.getNextComponent();
    	}
    }

    public void addInsertParagraphOperation(final List<Integer> start, String target, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertParagraphObject = new JSONObject(4);
        insertParagraphObject.put("name", "insertParagraph");
        insertParagraphObject.put("start", start);
        if(target!=null) {
        	insertParagraphObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertParagraphObject.put("attrs", attrs);
        }
        operationQueue.put(insertParagraphObject);
    }

    public void addInsertTableOperation(final List<Integer> start, String target, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertTableObject = new JSONObject(4);
        insertTableObject.put("name", "insertTable");
        insertTableObject.put("start", start);
        if(target!=null) {
        	insertTableObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertTableObject.put("attrs", attrs);
        }
        operationQueue.put(insertTableObject);
    }

    public void addInsertRowsOperation(final List<Integer> start, int count, String target, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertRowObject = new JSONObject(4);
        insertRowObject.put("name", "insertRows");
        insertRowObject.put("start", start);
        if(count!=1) {
        	insertRowObject.put("count", count);
        }
        if(target!=null) {
        	insertRowObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertRowObject.put("attrs", attrs);
        }
        operationQueue.put(insertRowObject);
    }

    public void addInsertCellsOperation(final List<Integer> start, String target, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject(3);
        insertCellsObject.put("name", "insertCells");
        insertCellsObject.put("start", start);
        if(target!=null) {
        	insertCellsObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertCellsObject.put("attrs", attrs);
        }
        operationQueue.put(insertCellsObject);
    }
    
    public void addInsertTextOperation(final List<Integer> start, String text, String target)
        throws JSONException {

        final JSONObject insertTextObject = new JSONObject(4);
        insertTextObject.put("name", "insertText");
        insertTextObject.put("start", start);
        insertTextObject.put("text", text);
        if(target!=null) {
        	insertTextObject.put("target", target);
        }
        operationQueue.put(insertTextObject);
    }

    public void addInsertTabOperation(final List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertTabObject = new JSONObject(3);
        insertTabObject.put("name", "insertTab");
        insertTabObject.put("start", start);
        if(target!=null) {
        	insertTabObject.put("target", target);
        }
        operationQueue.put(insertTabObject);
    }

    public void addInsertHardBreakOperation(List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertHardBreakObject = new JSONObject(3);
        insertHardBreakObject.put("name", "insertHardBreak");
        insertHardBreakObject.put("start", start);
        if(target!=null) {
        	insertHardBreakObject.put("target", target);
        }
        operationQueue.put(insertHardBreakObject);
    }

    public void addInsertFieldOperation(List<Integer> start, TextField field, String target)
        throws JSONException {
        final JSONObject insertFieldObject = new JSONObject(5);
        insertFieldObject.put("name", "insertField");
        insertFieldObject.put("start", start);
        insertFieldObject.put("type", field.getType());
        insertFieldObject.put("representation", field.getRepresentation());
        if(target!=null) {
        	insertFieldObject.put("target", target);
        }
        operationQueue.put(insertFieldObject);
    }

    public void addInsertDrawingOperation(List<Integer> start, DrawingType type, JSONObject attrs, String target)
        throws JSONException {

        final JSONObject insertDrawingObject = new JSONObject(5);
        insertDrawingObject.put("name", "insertDrawing");
        insertDrawingObject.put("start", start);
        insertDrawingObject.put("type", type.toString());
        if(!attrs.isEmpty()) {
        	insertDrawingObject.put("attrs", attrs);
        }
        if(target!=null) {
        	insertDrawingObject.put("target", target);
        }
        operationQueue.put(insertDrawingObject);
    }

    public void addSetAttributesOperation(final JSONObject attrs, final List<Integer> startPosition, final List<Integer> endPosition, String target)
        throws JSONException {

    	if(!attrs.isEmpty()) {
	        JSONObject setAttributeObject = new JSONObject(5);
	        setAttributeObject.put("name", "setAttributes");
	        setAttributeObject.put("attrs", attrs );
	        setAttributeObject.put("start", startPosition);
	        if(target!=null) {
	        	setAttributeObject.put("target", target);
	        }
	        if(endPosition!=null) {
	            setAttributeObject.put("end", endPosition);
	        }
	        operationQueue.put(setAttributeObject);
    	}
    }

    public void addInsertCommentOperation(final List<Integer> start,  String target, Annotation annotation)
        throws JSONException {

    	final JSONObject setCommentObject = new JSONObject(6);
    	setCommentObject.put("name", "insertComment");
    	setCommentObject.put("start", start);
        if(target!=null) {
        	setCommentObject.put("target", target);
        }
        setCommentObject.put("id", annotation.getId());
        if(annotation.getCreator()!=null) {
        	setCommentObject.put("author", annotation.getCreator());
        }
        if(annotation.getDate()!=null) {
        	setCommentObject.put("date", annotation.getDate());
        }
        operationQueue.put(setCommentObject);
    }

    public void addInsertRangeOperation(final List<Integer> start,  String target, AnnotationEnd annotationEnd)
        throws JSONException {

    	final JSONObject insertRangeObject = new JSONObject(6);
    	insertRangeObject.put("name", "insertRange");
    	insertRangeObject.put("start", start);
    	insertRangeObject.put("type", "comment");
        if(target!=null) {
        	insertRangeObject.put("target", target);
        }
    	insertRangeObject.put("position", "end");
    	insertRangeObject.put("id", annotationEnd.getId());
        operationQueue.put(insertRangeObject);
    }

    public void addInsertHeaderFooterOperation(String id, String type)
        throws JSONException {

    	final JSONObject insertHeaderFooterObject = new JSONObject(3);
    	insertHeaderFooterObject.put("name", "insertHeaderFooter");
    	insertHeaderFooterObject.put("id", id);
    	insertHeaderFooterObject.put("type", type);
        operationQueue.put(insertHeaderFooterObject);
    }
}
