/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.NavigableSet;
import java.util.TreeSet;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.table.TableNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableStyleNameAttribute;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.OfficeStyles;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.ods.dom.SmlUtils.CellRefRange;

@SuppressWarnings("serial")
public class Sheet extends ElementNSImpl implements IElementWriter, IStyleAccess {

	private final Rows rows;
	private final Columns columns;

	// each cell that is merged
	private List<MergeCell> mergeCells;

	// hyperlink ranges
	private List<Hyperlink> hyperlinks;
	
	// office forms if available ...
	// take care, these forms must be written before table:shapes, otherwise
	// LO won't display form controls (this detail would be good to be part of
	// the odf spec or LO should fix the import filter to not assume a specific
	// element order)
	private ElementNSImpl officeForms = null;

	// each drawing of a page
	private final Drawings drawings = new Drawings();

	// named expressions
	private NamedExpressions namedExpressions = null;

	// conditional formats
	private ConditionalFormats conditionalFormats = null;

	private List<Object> childs;

	boolean columnPlaceholderAvailable = false;
	boolean rowPlaceholderAvailable = false;
	boolean isLinkedTable = false;

	public Sheet(OdfFileDom ownerDocument, String StyleName)
		throws DOMException, SAXException {

		super(ownerDocument, TableTableElement.ELEMENT_NAME.getUri(), TableTableElement.ELEMENT_NAME.getQName());

		final OfficeStyles officeStyles = ((Content)getOwnerDocument()).getStyles();
		columns = new Columns(officeStyles);
		rows = new Rows(officeStyles);
	}

	public String getSheetName() {
		final Attr name = this.getAttributeNodeNS(OdfDocumentNamespace.TABLE.getUri(), "name");
		return name!=null ? name.getValue() : null;
	}

	public void setSheetName(String name) {
		final TableNameAttribute tableNameAttr = new TableNameAttribute((OdfFileDom)getOwnerDocument());
		tableNameAttr.setNodeValue(name);
		setAttributeNodeNS(tableNameAttr);
	}

	public String getStyleName() {
		final Attr name = this.getAttributeNodeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name");
		return name!=null ? name.getValue() : null;
	}

	private void setStyleName(String name) {
		final TableStyleNameAttribute tableStyleNameAttr = new TableStyleNameAttribute((OdfFileDom)getOwnerDocument());
		tableStyleNameAttr.setNodeValue(name);
		setAttributeNodeNS(tableStyleNameAttr);
	}

    public void replaceStyle(OfficeStyles officeStyles, String newStyle) {
    	if(getStyleName()!=null) {
    		officeStyles.decStyleUsage(getStyleFamily(), getStyleName());
    	}
    	if(newStyle!=null) {
    		officeStyles.incStyleUsage(getStyleFamily(), newStyle);
    	}
    	setStyleName(newStyle);
    }

    public String getStyleFamily() {
    	return "table";
    }

    public OdfStyleFamily getOdfStyleFamily() {
    	return OdfStyleFamily.Table;
    }

    public boolean isLinkedTable() {
		return isLinkedTable;
	}

	public void setIsLinkedTable(boolean isLinkedTable) {
		this.isLinkedTable = isLinkedTable;
	}

	public static int getMaxRowCount() {
		return 1048576;
	}

	public int getMaxColCount() {
		return ((Content)getOwnerDocument()).getMaxColumnCount();
	}

	/* returns a live list to the columns */
	public Columns getColumns() {
		return columns;
	}

	/* returns a live list to the rows */
    public TreeSet<Row> getRows() {
        return rows;
    }

    public Drawings getDrawings() {
    	return drawings;
    }

    public void setOfficeForms(ElementNSImpl officeForms) {
    	this.officeForms = officeForms;
    }

    public Row getRow(int r, boolean forceCreate, boolean cutFloor, boolean cutCeiling) {
    	if(rows.isEmpty()) {
    		if(!forceCreate) {
    			return null;
    		}
    		// creating new rows up to r
    		final Row newRow = new Row(null, 0, 0, false);
    		newRow.setRepeated(r+1);
    		newRow.setDefaultCellStyle("Default");
    		rows.add(newRow);
    	}
    	Row retValue = rows.floor(new Row(null, r));
    	if((retValue==null)||(retValue.getRow()+retValue.getRepeated()<=r)) {
    		if(!forceCreate) {
    			return null;
    		}
    		// we have to create new rows from end of floorValue up to r
        	final Row newRow = new Row(null, retValue!=null?retValue.getRow()+retValue.getRepeated():0, 0, false);
        	newRow.setRepeated((r-newRow.getRow())+1);
        	newRow.setDefaultCellStyle("Default");
        	rows.add(newRow);
        	retValue = newRow;
    	}
    	// rows up to r are available
    	if(cutFloor&&retValue.getRow()!=r) {
			final Row newRow = new Row(retValue.getStyleName(), r, 0, false);
			duplicateCells(retValue, newRow, true);
			newRow.setRepeated((retValue.getRow()+retValue.getRepeated())-r);
			newRow.setDefaultCellStyle(retValue.getDefaultCellStyle());
	    	rows.add(newRow);
	    	retValue.setRepeated(r-retValue.getRow());
	    	retValue = newRow;
    	}
    	if(cutCeiling&&((retValue.getRow()+retValue.getRepeated())-1)>r) {
			final Row newRow = new Row(retValue.getStyleName(), r+1, 0, false);
			duplicateCells(retValue, newRow, true);
			newRow.setRepeated(((retValue.getRow()+retValue.getRepeated())-1)-r);
			newRow.setDefaultCellStyle(retValue.getDefaultCellStyle());
	    	rows.add(newRow);
	    	retValue.setRepeated(retValue.getRepeated()-newRow.getRepeated());
    	}
    	return retValue;
    }

    public Column getColumn(int c, boolean forceCreate, boolean cutFloor, boolean cutCeiling) {
    	if(columns.isEmpty()) {
    		if(!forceCreate) {
    			return null;
    		}
    		// creating new columns up to c
    		final Column newColumn = new Column(null, 0, c);
    		columns.add(newColumn);
    	}
    	Column retValue = columns.floor(new Column(null, c));
    	if((retValue==null)||(retValue.getMax()<c)) {
    		if(!forceCreate) {
    			return null;
    		}
    		// we have to create new columns from end of floorValue up to c
        	final Column newColumn = new Column(null, retValue!=null?retValue.getMax()+1:0, c);
        	columns.add(newColumn);
        	retValue = newColumn;
    	}
    	// rows up to c are available
    	if(cutFloor&&retValue.getMin()!=c) {
			final Column newColumn = new Column(retValue.getStyleName(), c, retValue.getMax());
			newColumn.setDefaultCellStyle(retValue.getDefaultCellStyle());
	    	columns.add(newColumn);
	    	retValue.setMax(c-1);
	    	retValue = newColumn;
    	}
    	if(cutCeiling&&(retValue.getMax()>c)) {
			final Column newColumn = new Column(retValue.getStyleName(), c+1, retValue.getMax());
			newColumn.setDefaultCellStyle(retValue.getDefaultCellStyle());
	    	columns.add(newColumn);
	    	retValue.setMax(c);
    	}
    	return retValue;
    }

    private void duplicateCells(Row source, Row dest, boolean cloneCellContent) {

    	final Iterator<Cell> iter = source.getCells().iterator();
    	while(iter.hasNext()) {
    		dest.getCells().add(iter.next().clone(cloneCellContent));
    	}
    }

    public void insertRows(int sheetIndex, int start, int count)
    	throws SAXException {

    	// assuring that the first row is available
    	getRow(start, true, true, false);

    	final Object[] rs = rows.tailSet(new Row(null, start), true).toArray();
    	for(int i=rs.length-1;i>=0;i--) {
    		final Row ro = (Row)rs[i];
    		rows.remove(ro);
    		int newRowNumber = ro.getRow()+count;
    		if(newRowNumber<getMaxRowCount()) {
	    		ro.setRow(ro.getRow()+count);
	    		rows.add(ro);
	    		if(newRowNumber+ro.getRepeated()>getMaxRowCount()) {
	    			ro.setRepeated(getMaxRowCount()-newRowNumber);
	    		}
    		}
    	}
    	Row previousRow = null;
    	if(start>0) {
    		previousRow = getRow(start-1, false, false, false);
    	}
    	final Row newRow = new Row(previousRow!=null?previousRow.getStyleName():null, start);
    	if(previousRow!=null) {
	    	newRow.setDefaultCellStyle(previousRow.getDefaultCellStyle());
	    	duplicateCells(previousRow, newRow, false);
    	}
    	newRow.setRepeated(count);
    	rows.add(newRow);

    	// taking care of hyperlinks
    	if(hyperlinks!=null) {
    		SmlUtils.insertRows(hyperlinks, start, count, false);
    	}

    	// taking care of conditional formats
    	if(conditionalFormats!=null) {
    		final Iterator<Condition> conditionalFormatIter = conditionalFormats.getConditionalFormatList().values().iterator();
    		while(conditionalFormatIter.hasNext()) {
				SmlUtils.insertRows(conditionalFormatIter.next().getRanges(), start, count, true);
    		}
    	}

        // taking care of merged cells
    	if(mergeCells!=null) {
    		SmlUtils.insertRows(mergeCells, start, count, false);
    	}

    	// taking care of tables
    	final DatabaseRanges databaseRanges = ((Content)getOwnerDocument()).getDatabaseRanges(false);
    	if(databaseRanges!=null) {
	    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
	    	while(databaseRangeIter.hasNext()) {
	    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
	    		if(databaseRange.getName().startsWith("__Anonymous_Sheet_DB__")) {
	    			final Range sheetRange = databaseRange.getRange();
	    			if(sheetRange.getSheetIndex()==sheetIndex) {
	    				final CellRefRange cellRefRange = sheetRange.getCellRefRange();
	    				if(cellRefRange!=null) {
	                        final SmlUtils.CellRefRange newCellRefRange = SmlUtils.CellRefRange.insertRowRange(cellRefRange, start, count, false);
	                        if(newCellRefRange!=null&&!newCellRefRange.equals(cellRefRange)) {
	                        	databaseRange.setRange(new Range(sheetIndex, newCellRefRange));
	                        	// TODO: take care of changed table columns
	                        }
                        }
	    			}
	    		}
	    	}
	    }
    }

    public void deleteRows(int sheetIndex, int start, int count)
    	throws SAXException {

    	getRow(start, true, true, true);
    	if(count>1) {
    		getRow((start+count)-1, true, false, true);
    	}
    	final Object[] rs = rows.tailSet(new Row(null, start), true).toArray();
    	for(Object obj:rs) {
			final Row ro = (Row)obj;
	   		rows.remove(ro);
    		if(ro.getRow()>=start+count) {
    			ro.setRow(ro.getRow()-count);
    			rows.add(ro);
    		}
    	}

    	// taking care of hyperlinks
    	if(hyperlinks!=null) {
    		SmlUtils.deleteRows(hyperlinks, start, count, false);
    	}

    	// taking care of conditional formats
    	if(conditionalFormats!=null) {
    		final Iterator<Condition> conditionalFormatIter = conditionalFormats.getConditionalFormatList().values().iterator();
    		while(conditionalFormatIter.hasNext()) {
    			final Ranges ranges = conditionalFormatIter.next().getRanges();
				SmlUtils.deleteRows(ranges, start, count, false);
				if(ranges.isEmpty()) {
					conditionalFormatIter.remove();
				}
    		}
    	}

        // taking care of merged cells
    	if(mergeCells!=null) {
    		SmlUtils.deleteRows(mergeCells, start, count, true);
    	}
    	
    	final DatabaseRanges databaseRanges = ((Content)getOwnerDocument()).getDatabaseRanges(false);
    	if(databaseRanges!=null) {
	    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
	    	while(databaseRangeIter.hasNext()) {
	    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
	    		if(databaseRange.getName().startsWith("__Anonymous_Sheet_DB__")) {
	    			final Range sheetRange = databaseRange.getRange();
	    			if(sheetRange.getSheetIndex()==sheetIndex) {
	    				final CellRefRange cellRefRange = sheetRange.getCellRefRange();
	    				if(cellRefRange!=null) {
	                        final SmlUtils.CellRefRange newCellRefRange = SmlUtils.CellRefRange.deleteRowRange(cellRefRange, start, count);
	                        if(newCellRefRange==null) {
	                        	databaseRanges.deleteTable(((Content)getOwnerDocument()), sheetIndex, databaseRange.getName());
	                        }
	                        else if(!newCellRefRange.equals(cellRefRange)) {
	                        	databaseRange.setRange(new Range(sheetIndex, newCellRefRange));
	                        	// TODO: take care of changed table columns
	                        }
                        }
	    			}
	    		}
	    	}
	    }
    }

    public void insertColumns(int sheetIndex, int start, int count)
    	throws SAXException {

    	for(Row row:getRows()) {
    		row.insertCells(start, count);
    	}

    	// assuring that the first column is available
    	getColumn(start, true, true, false);

    	final Object[] rs = columns.tailSet(new Column(null, start), true).toArray();
    	for(int i=rs.length-1;i>=0;i--) {
    		final Column co = (Column)rs[i];
    		columns.remove(co);
    		co.setMin(co.getMin()+count);
    		co.setMax(co.getMax()+count);
    		if(co.getMin()<getMaxColCount()) {
	    		columns.add(co);
	    		if(co.getMax()>=getMaxColCount()) {
	    			co.setMax(getMaxColCount()-1);
	    		}
    		}
    	}

    	// taking care of hyperlinks
    	if(hyperlinks!=null) {
    		SmlUtils.insertColumns(hyperlinks, start, count, false);
    	}

    	// taking care of conditional formats
    	if(conditionalFormats!=null) {
    		final Iterator<Condition> conditionalFormatIter = conditionalFormats.getConditionalFormatList().values().iterator();
    		while(conditionalFormatIter.hasNext()) {
				SmlUtils.insertColumns(conditionalFormatIter.next().getRanges(), start, count, true);
    		}
    	}

        // taking care of merged cells
    	if(mergeCells!=null) {
    		SmlUtils.insertColumns(mergeCells, start, count, false);
    	}

    	// taking care of tables
    	final DatabaseRanges databaseRanges = ((Content)getOwnerDocument()).getDatabaseRanges(false);
    	if(databaseRanges!=null) {
	    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
	    	while(databaseRangeIter.hasNext()) {
	    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
    			final Range sheetRange = databaseRange.getRange();
    			if(sheetRange.getSheetIndex()==sheetIndex) {
    				final CellRefRange cellRefRange = sheetRange.getCellRefRange();
    				if(cellRefRange!=null) {
                        final SmlUtils.CellRefRange newCellRefRange = SmlUtils.CellRefRange.insertColumnRange(cellRefRange, start, count, false);
                        if(newCellRefRange!=null&&!newCellRefRange.equals(cellRefRange)) {
                        	databaseRange.setRange(new Range(sheetIndex, newCellRefRange));
                        	databaseRange.changeColumn(cellRefRange, start, count);
                        }
                    }
    			}
	    	}
	    }
    }

    public void deleteColumns(int sheetIndex, int start, int count)
    	throws SAXException {

    	for(Row row:getRows()) {
    		row.deleteCells(start, count);
    	}

    	getColumn(start, true, true, true);
    	if(count>1) {
    		getColumn((start+count)-1, true, false, true);
    	}
    	final Object[] rs = columns.tailSet(new Column(null, start), true).toArray();
    	for(Object obj:rs) {
			final Column co = (Column)obj;
	   		columns.remove(co);
    		if(co.getMin()>=start+count) {
    			co.setMin(co.getMin()-count);
    			co.setMax(co.getMax()-count);
    			columns.add(co);
    		}
    	}

    	// taking care of hyperlinks
    	if(hyperlinks!=null) {
    		SmlUtils.deleteColumns(hyperlinks, start, count, false);
    	}

    	// taking care of conditional formats
    	if(conditionalFormats!=null) {
    		final Iterator<Condition> conditionalFormatIter = conditionalFormats.getConditionalFormatList().values().iterator();
    		while(conditionalFormatIter.hasNext()) {
    			final Ranges ranges = conditionalFormatIter.next().getRanges();
				SmlUtils.deleteColumns(ranges, start, count, false);
				if(ranges.isEmpty()) {
					conditionalFormatIter.remove();
				}
    		}
    	}

        // taking care of merged cells
    	if(mergeCells!=null) {
    		SmlUtils.deleteColumns(mergeCells, start, count, true);
    	}

    	// taking care of tables
    	final DatabaseRanges databaseRanges = ((Content)getOwnerDocument()).getDatabaseRanges(false);
    	if(databaseRanges!=null) {
	    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
	    	while(databaseRangeIter.hasNext()) {
	    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
    			final Range sheetRange = databaseRange.getRange();
    			if(sheetRange.getSheetIndex()==sheetIndex) {
    				final CellRefRange cellRefRange = sheetRange.getCellRefRange();
    				if(cellRefRange!=null) {
                        final SmlUtils.CellRefRange newCellRefRange = SmlUtils.CellRefRange.deleteColumnRange(cellRefRange, start, count);
                        if(newCellRefRange==null) {
                        	databaseRanges.deleteTable(((Content)getOwnerDocument()), sheetIndex, databaseRange.getName());
                        }
                        else if(!newCellRefRange.equals(cellRefRange)) {
                        	databaseRange.setRange(new Range(sheetIndex, newCellRefRange));
                        	databaseRange.changeColumn(cellRefRange, start, -count);
                        }
                    }
    			}
	    	}
	    }
    }

    public void clearCellRange(int r1, int r2, int c1, int c2) {

    	getRow(r1, false, true, false);
    	getRow(r2, false, false, true);

    	final NavigableSet<Row> rowSelection = getRows().subSet(new Row(null, r1), true, new Row(null, r2), true);
        final Iterator<Row> rowIter = rowSelection.iterator();
        while(rowIter.hasNext()) {
        	final Row row = rowIter.next();

        	// creating cell entries....
        	row.getCell(c1, false, true, false);
        	row.getCell(c2, false, false, true);

        	final Iterator<Cell> cellSubSetIter = row.getCells().subSet(new Cell(c1), true, new Cell(c2), true).iterator();
        	while(cellSubSetIter.hasNext()) {
        		cellSubSetIter.next();
        		cellSubSetIter.remove();
        	}

        	// check if we need to create empty repeated cells, only necessary if a ceilling cell is available
        	final Cell ceiling = row.getCell(c2+1, false, false, false);
        	if(ceiling!=null) {
        		final Cell newCell = new Cell(c1);
        		newCell.setRepeated((c2-c1)+1);
        		row.getCells().add(newCell);
        	}
        }
    }

    public NamedExpressions getNamedExpressions(boolean forceCreate) {
		if(namedExpressions==null&&forceCreate) {
			namedExpressions = new NamedExpressions((Content)getOwnerDocument());
		}
		return namedExpressions;
	}

	public ConditionalFormats getConditionalFormats(boolean forceCreate) {
		if(conditionalFormats==null&&forceCreate) {
			conditionalFormats = new ConditionalFormats((Content)getOwnerDocument());
		}
		return conditionalFormats;
	}

	public List<MergeCell> getMergeCells() {
    	if(mergeCells==null) {
    		mergeCells = new ArrayList<MergeCell>();
    	}
    	return mergeCells;
    }

    public List<Hyperlink> getHyperlinks() {
    	if(hyperlinks==null) {
    		hyperlinks = new ArrayList<Hyperlink>();
    	}
    	return hyperlinks;
    }

    public Hyperlink getHyperlink(int column, int row) {
    	if(hyperlinks!=null) {
    		for(Hyperlink hyperlink:hyperlinks) {
    			if(hyperlink.getCellRefRange().isInside(column, row)) {
    				return hyperlink;
    			}
    		}
    	}
    	return null;
    }

    /* returns a live list to the each child */
	public List<Object> getChilds() {
		if(childs==null) {
			childs = new ArrayList<Object>();
		}
		return childs;
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		if(!getColumns().isEmpty()&&columnPlaceholderAvailable==false) {
			getChilds().add(getColumns().first());
			columnPlaceholderAvailable = true;
		}
		if(!getRows().isEmpty()&&rowPlaceholderAvailable==false) {
			getChilds().add(getRows().first());
			rowPlaceholderAvailable = true;
		}

		output.startElement(getNamespaceURI(), getLocalName(), getNodeName());
		SaxContextHandler.addAttributes(output, getAttributes(), null);

		// office forms needs to be written before shapes
		if(officeForms!=null) {
			SaxContextHandler.serializeElement(output, officeForms);
		}
		drawings.writeSheetDrawings(output);
		for(Object child:getChilds()) {
			if(child instanceof Column) {
				// taking care of column-group-elements
				int currentColumnGroupLevel = 0;
				for(Column column:columns) {
					currentColumnGroupLevel += Column.writeGroupDifference(output, currentColumnGroupLevel, column.getGroupLevel(), column.isStartingGroup());
					column.writeObject(output);
				}
				Column.writeGroupDifference(output, currentColumnGroupLevel, 0, false);
			}
			else if(child instanceof Row) {
				int currentRowGroupLevel = 0;
				for(Row row:rows) {
					currentRowGroupLevel += Row.writeGroupDifference(output, currentRowGroupLevel, row.getGroupLevel(), row.isStartingGroup());
					row.writeObject(output, this);
				}
				Column.writeGroupDifference(output, currentRowGroupLevel, 0, false);
			}
			else if(child instanceof IElementWriter) {
				((IElementWriter)child).writeObject(output);
			}
			else if(child instanceof ElementNSImpl) {
				SaxContextHandler.serializeElement(output, (ElementNSImpl)child);
			}
		}
		if(conditionalFormats!=null) {
			conditionalFormats.writeObject(output);
		}
		if(namedExpressions!=null) {
			namedExpressions.writeObject(output);
		}
		output.endElement(getNamespaceURI(), getLocalName(), getNodeName());
	}
}
