/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.table.TableContainsHeaderAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableDisplayFilterButtonsAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableFieldNumberAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableHasPersistentDataAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableIsSelectionAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableOnUpdateKeepSizeAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableOnUpdateKeepStylesAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableOrientationAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableRefreshDelayAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableTargetRangeAddressAttribute;
import org.odftoolkit.odfdom.dom.element.table.TableDatabaseRangeElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterConditionElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterOrElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterSetItemElement;
import org.odftoolkit.odfdom.pkg.OdfName;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;

import com.openexchange.office.odf.SaxContextHandler;

public class ContentValidation {

	private String name;
	private Ranges ranges;

	private String containsHeader;
	private String displayFilterButtons;
	private String hasPersistentData;
	private String isSelection;
	private String onUpdateKeepSize;
	private String onUpdateKeepStyles;
	private String orientation;
	private String refreshDelay;
	private List<ElementNSImpl> childs;

	public ContentValidation(Content content, Attributes attributes) {
		name = attributes.getValue(TableNameAttribute.ATTRIBUTE_NAME.getQName());
		if(name==null) {
			name = "__Anonymous_Sheet_DB__0";
		}
		containsHeader = attributes.getValue(TableContainsHeaderAttribute.ATTRIBUTE_NAME.getQName());
		displayFilterButtons = attributes.getValue(TableDisplayFilterButtonsAttribute.ATTRIBUTE_NAME.getQName());
		hasPersistentData = attributes.getValue(TableHasPersistentDataAttribute.ATTRIBUTE_NAME.getQName());
		isSelection = attributes.getValue(TableIsSelectionAttribute.ATTRIBUTE_NAME.getQName());
		onUpdateKeepSize = attributes.getValue(TableOnUpdateKeepSizeAttribute.ATTRIBUTE_NAME.getQName());
		onUpdateKeepStyles = attributes.getValue(TableOnUpdateKeepStylesAttribute.ATTRIBUTE_NAME.getQName());
		orientation = attributes.getValue(TableOrientationAttribute.ATTRIBUTE_NAME.getQName());
		refreshDelay = attributes.getValue(TableRefreshDelayAttribute.ATTRIBUTE_NAME.getQName());
		ranges = new Ranges(content, attributes.getValue(TableTargetRangeAddressAttribute.ATTRIBUTE_NAME.getQName()));
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public boolean getContainsHeader() {
		return containsHeader==null?Boolean.TRUE:containsHeader.equals("true");
	}

	public void setContainsHeader(boolean containsHeader) {
		this.containsHeader = containsHeader ? "true" : "false";
	}

	public List<ElementNSImpl> getChilds() {
		if(childs==null) {
			childs = new ArrayList<ElementNSImpl>();
		}
		return childs;
	}

	public JSONObject createTableAttrs()
		throws JSONException {

		final JSONObject tableAttrs = new JSONObject(2);
		tableAttrs.put("headerRow", getContainsHeader());
		tableAttrs.put("filtered", Boolean.TRUE);
		return tableAttrs;
	}

	public void changeTableColumn(Content content, int col, JSONObject attrs) {
		ElementNSImpl tableFilter = findAndRemoveColumn(col);
		List<String> entries = null;
        JSONObject filter = attrs.optJSONObject("filter");
        if(filter != null) {
            String type = filter.optString("type");
            if( type != null && type.equals("discrete")) {
                JSONArray entriesArray = filter.optJSONArray("entries");
                if(entriesArray != null && entriesArray.length() > 0) {
                    entries = new ArrayList<String>();
                    Iterator<Object> entryIt = entriesArray.iterator();
                    while(entryIt.hasNext()) {
                        entries.add((String)entryIt.next());
                    }
                }
            }
        }
        if(entries!=null) {
            if(tableFilter==null) {
                //insert new filter
                tableFilter = new TableFilterElement(content);
                getChilds().add(0, tableFilter);
            }
            //insert new FilterOr with FilterCondition
            TableFilterOrElement tableFilterOr = new TableFilterOrElement(content);
            tableFilter.insertBefore(tableFilterOr, null);
            TableFilterConditionElement tableFilterCondition = new TableFilterConditionElement(content);
            tableFilterOr.insertBefore(tableFilterCondition, null);
            tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:field-number", Integer.toString(col) );
            tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:value", entries.get(0) );
            tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:operator", "=" );
            ListIterator<String> entriesIt = entries.listIterator();
            while(entriesIt.hasNext()) {
                TableFilterSetItemElement filterSetItem = new TableFilterSetItemElement(content); 
                tableFilterCondition.insertBefore(filterSetItem, null);
                filterSetItem.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:value", entriesIt.next() );
            }
        }
	}

    private ElementNSImpl findAndRemoveColumn(int col) {
        final Node tableFilter = getFilterElementChild();
        if(tableFilter!=null) {
            // if it contains a TableFilterAnd -> remove it, it breaks our AutoFilter support ATM
            Node filterChild = tableFilter.getFirstChild();
            while(filterChild!=null) {
                Node nextChild = filterChild.getNextSibling();
                if(filterChild.getLocalName().equals("filter-and")) {
                    tableFilter.removeChild(filterChild);
                } else if(filterChild.getLocalName().equals("filter-or")) {
                    Node orChild = ((ElementNSImpl)filterChild).getFirstChild();
                    if(orChild.getLocalName().equals("filter-and")) {
                        tableFilter.removeChild(filterChild);
                    } else if(orChild.getLocalName().equals("filter-condition")) {
                        String column = ((ElementNSImpl)orChild).getAttributeNS(TableFieldNumberAttribute.ATTRIBUTE_NAME.getUri(), "field-number");
                        if(Integer.parseInt(column)==col) {
                            tableFilter.removeChild(filterChild);
                        }
                    }
                }
                filterChild = nextChild;
            }
        }
        return (ElementNSImpl)tableFilter;
    }

    private Node getFilterElementChild() {
       	for(ElementNSImpl e:getChilds()) {
       		if(e.getLocalName().equals("filter")) {
       			return e;
       		}
       	}
       	return null;
    }

	public void writeObject(SerializationHandler output, Content content) {

		final OdfName elementName = TableDatabaseRangeElement.ELEMENT_NAME;
    	SaxContextHandler.startElement(output, elementName);
    	SaxContextHandler.addOdfAttribute(output, TableNameAttribute.ATTRIBUTE_NAME, null, name);
    	if(containsHeader!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableContainsHeaderAttribute.ATTRIBUTE_NAME, null, containsHeader);
    	}
    	SaxContextHandler.addOdfAttribute(output, TableDisplayFilterButtonsAttribute.ATTRIBUTE_NAME, null, displayFilterButtons!=null?displayFilterButtons:"true");
    	if(hasPersistentData!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableHasPersistentDataAttribute.ATTRIBUTE_NAME, null, hasPersistentData);
    	}
    	if(isSelection!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableIsSelectionAttribute.ATTRIBUTE_NAME, null, isSelection);
    	}
    	if(onUpdateKeepSize!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableOnUpdateKeepSizeAttribute.ATTRIBUTE_NAME, null, onUpdateKeepSize);
    	}
    	if(onUpdateKeepStyles!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableOnUpdateKeepStylesAttribute.ATTRIBUTE_NAME, null, onUpdateKeepStyles);
    	}
    	SaxContextHandler.addOdfAttribute(output, TableOrientationAttribute.ATTRIBUTE_NAME, null, orientation!=null?orientation:"column");
    	if(refreshDelay!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableRefreshDelayAttribute.ATTRIBUTE_NAME, null, refreshDelay);
    	}
    	if(ranges!=null) {
        	SaxContextHandler.addOdfAttribute(output, TableTargetRangeAddressAttribute.ATTRIBUTE_NAME, null, ranges.convertToString(content, true));
    	}
    	if(childs!=null) {
    		for(ElementNSImpl child:childs) {
    			SaxContextHandler.serializeElement(output, child);
    		}
    	}
    	SaxContextHandler.endElement(output, elementName);
	}
}
