/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.List;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeBooleanValueAttribute;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeCurrencyAttribute;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeStringValueAttribute;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeValueAttribute;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeValueTypeAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableFormulaAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNumberColumnsRepeatedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNumberColumnsSpannedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNumberRowsSpannedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.xlink.XlinkHrefAttribute;
import org.odftoolkit.odfdom.dom.attribute.xlink.XlinkTypeAttribute;
import org.odftoolkit.odfdom.dom.element.number.DataStyleElement;
import org.odftoolkit.odfdom.dom.element.table.TableCoveredTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.number.OdfNumberCurrencyStyle;
import org.odftoolkit.odfdom.incubator.doc.number.OdfNumberDateStyle;
import org.odftoolkit.odfdom.incubator.doc.number.OdfNumberPercentageStyle;
import org.odftoolkit.odfdom.incubator.doc.number.OdfNumberTimeStyle;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfName;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.ElementNS;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.ods.dom.SmlUtils.CellRef;

public class Cell implements Comparable<Cell> {

	public static class ErrorCode {

		final String errorCode;

		public ErrorCode(String errorCode) {
			this.errorCode = errorCode;
		}
		public String getError() {
			return errorCode;
		}
	}

	int column;
	private String cellStyle = null;
	private int repeated = 1;

	// the cell content is stored in one of the two following variables
	private Object content = null;				// immutable objects only, otherwise the clone needs to be changed
	private String formula = null;
	private ElementNS element = null;
	private CellAttributesEnhanced cellAttributesEnhanced = null;

	Cell(int c) {
		column = c;
	}

	public int getColumn() {
		return column;
	}

	public void setColumn(int c) {
		column = c;
	}

	public String getCellStyle() {
    	return cellStyle;
    }

    public void setCellStyle(String val) {
    	cellStyle = val;
    }

    public int getRepeated() {
    	return repeated;
    }

    public void setRepeated(int val) {
    	repeated = val;
    }

    public CellAttributesEnhanced getCellAttributesEnhanced(boolean createIfMissing) {
    	if(createIfMissing&&cellAttributesEnhanced==null) {
    		cellAttributesEnhanced = new CellAttributesEnhanced();
    	}
    	return cellAttributesEnhanced;
    }

    public ElementNSImpl createNode(OdfFileDom fileDom) {

    	if(element==null) {
			element = new ElementNS(fileDom, TableTableCellElement.ELEMENT_NAME.getUri(), TableTableCellElement.ELEMENT_NAME.getQName());
    	}
    	return element;
    }

    public ElementNSImpl getNode() {
    	return element;
    }

    // mergeCell can be null if cell is not merged / covered
	public void writeObject(SerializationHandler output, Sheet sheet, Row row, MergeCell mergeCell)
		throws SAXException {

		boolean covered = false;
		boolean spannedAttribute = false;

		if(mergeCell!=null) {
			final CellRef mergeStart = mergeCell.getCellRefRange().getStart();
			if(mergeStart.getColumn()==getColumn()&&row.getRow()==mergeStart.getRow()) {
				// only the top left cell of the merge range gets the spanned row/column attributes
				spannedAttribute = true;
			}
			else {
				covered = true;
			}
		}
		final OdfName elementCellName = covered ? TableCoveredTableCellElement.ELEMENT_NAME :  TableTableCellElement.ELEMENT_NAME;
		SaxContextHandler.startElement(output, elementCellName);
		if(getRepeated()>1) {
			SaxContextHandler.addOdfAttribute(output, TableNumberColumnsRepeatedAttribute.ATTRIBUTE_NAME, null, Integer.toString(getRepeated()));
		}
		if(getCellStyle()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableStyleNameAttribute.ATTRIBUTE_NAME, null, getCellStyle());
		}
		if(spannedAttribute) {
			final int columns = mergeCell.getCellRefRange().getEnd().getColumn() - mergeCell.getCellRefRange().getStart().getColumn();
			final int rows = mergeCell.getCellRefRange().getEnd().getRow() - mergeCell.getCellRefRange().getStart().getRow();
			if(columns>0||rows>0) {
				SaxContextHandler.addOdfAttribute(output, TableNumberColumnsSpannedAttribute.ATTRIBUTE_NAME, null, Integer.toString(columns+1));
				SaxContextHandler.addOdfAttribute(output, TableNumberRowsSpannedAttribute.ATTRIBUTE_NAME, null, Integer.toString(rows+1));
			}
		}

		// writing cell content
		if(cellAttributesEnhanced!=null) {
			cellAttributesEnhanced.writeAttributes(output);
		}
		if(content!=null) {
			if(formula!=null) {
            	SaxContextHandler.addOdfAttribute(output, TableFormulaAttribute.ATTRIBUTE_NAME, null, "of:=".concat(formula));
			}
			if(content instanceof String) {
				final String contentString = (String)content;
				if(!contentString.isEmpty()) {
					SaxContextHandler.addOdfAttribute(output, OfficeValueTypeAttribute.ATTRIBUTE_NAME, null, "string");
	            	SaxContextHandler.addOdfAttribute(output, CalcExtValueTypeAttributeName, null, "string");
	            	writeContentString(output, sheet, row, contentString);
				}
			}
			else if(content instanceof Boolean) {
            	SaxContextHandler.addOdfAttribute(output, OfficeValueTypeAttribute.ATTRIBUTE_NAME, null, "boolean");
            	SaxContextHandler.addOdfAttribute(output, CalcExtValueTypeAttributeName, null, "boolean");
            	SaxContextHandler.addOdfAttribute(output, OfficeBooleanValueAttribute.ATTRIBUTE_NAME, null, ((Boolean)content)?"true":"false");
			}
			else if(content instanceof ErrorCode) {
            	SaxContextHandler.addOdfAttribute(output, OfficeValueTypeAttribute.ATTRIBUTE_NAME, null, "string");
            	SaxContextHandler.addOdfAttribute(output, OfficeStringValueAttribute.ATTRIBUTE_NAME, null, "");
            	SaxContextHandler.addOdfAttribute(output, CalcExtValueTypeAttributeName, null, "error");
            	final String errorCode = ((ErrorCode)content).getError();
            	if(errorCode!=null) {
            		writeContentString(output, sheet, row, errorCode);
            	}
			}
			else if(content instanceof Number) {
				String valueType = "float";
				String styleName = getCellStyle();
				if(styleName==null) {
					styleName = row.getDefaultCellStyle();
					if(styleName==null) {
						final Column col = sheet.getColumn(getColumn(), false, false, false);
						if(col!=null) {
							styleName = col.getDefaultCellStyle();
						}
					}
				}
				final OdfFileDom xDoc = (OdfFileDom)sheet.getOwnerDocument();
	            final OdfDocument odfDoc = (OdfDocument)xDoc.getDocument();
	            final OdfOfficeStyles officeStyles = odfDoc.getStylesDom().getOfficeStyles();
	            OdfStyle ownStyle = officeStyles.getStyle(styleName, OdfStyleFamily.TableCell);
	            boolean officeValueAttributeRequired = true;

	            if(ownStyle == null) {
	                ownStyle = odfDoc.getContentDom().getAutomaticStyles().getStyle(styleName, OdfStyleFamily.TableCell);
	            }
	            if(ownStyle!=null) {
		            final String dataStyleName = ownStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "data-style-name");
		            if(dataStyleName != null) {
		                DataStyleElement dataStyle = officeStyles.getAllDataStyles().get(dataStyleName);
		                if(dataStyle == null) {
		                    dataStyle = odfDoc.getContentDom().getAutomaticStyles().getAllDataStyles().get(dataStyleName);
		                }
		                if(dataStyle!=null) {
		                	if(dataStyle instanceof OdfNumberCurrencyStyle) {
		                        String currencyCode = ((OdfNumberCurrencyStyle) dataStyle).getCurrencyCode();
		                        if(currencyCode!=null) {
		                        	SaxContextHandler.addOdfAttribute(output, OfficeCurrencyAttribute.ATTRIBUTE_NAME, null, currencyCode);
		                        }
		                        valueType = "currency";
		                    }
		                	else if(dataStyle instanceof OdfNumberPercentageStyle) {
		                		valueType = "percentage";
		                	}
		                	else if(dataStyle instanceof OdfNumberDateStyle) {
		                		// TODO: date-value instead of office-value should be created
		                		valueType = "date";
		                	}
		                	else if(dataStyle instanceof OdfNumberTimeStyle) {
//		    	            	SaxContextHandler.addOdfAttribute(output, OfficeTimeValueAttribute.ATTRIBUTE_NAME, null, MapHelper.doubleToTime(((Number)content).doubleValue()));
//		    	            	officeValueAttributeRequired = false;
		                		valueType = "time";
		                	}
		                }
		            }
	            }
	            if(officeValueAttributeRequired) {
	            	SaxContextHandler.addOdfAttribute(output, OfficeValueAttribute.ATTRIBUTE_NAME, null, ((Number)content).toString());
	            }
	            SaxContextHandler.addOdfAttribute(output, OfficeValueTypeAttribute.ATTRIBUTE_NAME, null, valueType);
            	SaxContextHandler.addOdfAttribute(output, CalcExtValueTypeAttributeName, null, valueType);
			}
		}
		final List<Drawing> anchoredDrawings = sheet.getDrawings().getAnchoredDrawings(new DrawingAnchor(this.getColumn(), row.getRow()));
		if(anchoredDrawings!=null) {
			for(Drawing drawing:anchoredDrawings) {
				drawing.writeDrawing(output);
			}
		}
		if(element!=null) {
			SaxContextHandler.serializeChildElements(output, element);
		}
		SaxContextHandler.endElement(output, elementCellName);
	}

	private void writeContentString(SerializationHandler output, Sheet sheet, Row row, String contentString)
		throws SAXException {

		final String[] paragraphs = contentString.split("\r\n|\n|\r");
    	for(int i=0; i<paragraphs.length; i++) {
        	// simply writing text ... what to do if the content is a error an starts with '#'
        	SaxContextHandler.startElement(output, TextPElement.ELEMENT_NAME);
        	final Hyperlink hyperlink = i==0 ? sheet.getHyperlink(column, row.getRow()) : null;
        	if(hyperlink!=null) {
        		SaxContextHandler.startElement(output, TextAElement.ELEMENT_NAME);
        		SaxContextHandler.addOdfAttribute(output, XlinkTypeAttribute.ATTRIBUTE_NAME, null, "simple");
        		SaxContextHandler.addOdfAttribute(output, XlinkHrefAttribute.ATTRIBUTE_NAME, null, hyperlink.getUrl());
        	}
        	output.characters(paragraphs[i]);
        	if(hyperlink!=null) {
        		SaxContextHandler.endElement(output, TextAElement.ELEMENT_NAME);
        	}
        	SaxContextHandler.endElement(output, TextPElement.ELEMENT_NAME);
    	}
	}

	public Object getCellContent() {
		return content;
    }

    public void setCellContent(Object value) {
    	content = value;
    }

    public String getCellFormula() {
    	return formula;
    }

    public void setCellFormula(String formula) {
    	this.formula = formula;
    }

    public Cell clone(boolean cloneContent) {
    	final Cell clone = new Cell(column);
    	clone.setRepeated(getRepeated());
    	clone.setCellStyle(getCellStyle());
    	if(cloneContent) {
	    	if(element!=null) {
	    		clone.element = element.cloneNode(true);
	    	}
	    	else {
	    		if(content!=null) {
	    			clone.content = content;
	    		}
	    		clone.formula = formula;
	    	}
    	}
    	return clone;
    }

	@Override
	public int compareTo(Cell o) {
		return column - o.getColumn();
	}

	private static final OdfName CalcExtValueTypeAttributeName = OdfName.newName(OdfDocumentNamespace.CALCEXT, "value-type");
}
