/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.tools.error;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.FilterException;

public class FilterExceptionToErrorCode {

    public enum LogLevel {
        LOGLEVEL_DEBUG,
        LOGLEVEL_INFO,
        LOGLEVEL_WARN,
        LOGLEVEL_ERROR
    }

    static public ErrorCode map(final FilterException e, final ErrorCode defErrorCode) {
        ErrorCode errorCode = defErrorCode;
        final String subErrorType = e.getSubType();

        // special handling for file storage exceptions
        switch (e.getErrorcode()) {
            case CRITICAL_ERROR: errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR; break;
            case WRONG_PASSWORD: errorCode = ErrorCode.LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR; break;
            case UNSUPPORTED_ENCRYPTION_USED: errorCode = ErrorCode.LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR; break;
            case COMPLEXITY_TOO_HIGH:
                errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR;
                if (FilterException.ST_MAX_SHEET_COUNT_EXCEEDED.equals(subErrorType)) {
                    errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_SHEET_COUNT_ERROR;
                }
                break;
            case MEMORY_USAGE_TOO_HIGH: errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR; break;
            case MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED: errorCode = ErrorCode.GENERAL_SYSTEM_BUSY_ERROR; break;
            case STRICT_OOXML_NOT_SUPPORTED: errorCode = ErrorCode.LOADDOCUMENT_STRICT_OOXML_NOT_SUPPORTED_ERROR; break;
            case FEATURE_NOT_SUPPORTED: {
                errorCode = ErrorCode.LOADDOCUMENT_FEATURE_NOT_SUPPORTED_ERROR;
                if (FilterException.ST_FRAME_ATTACHED_TO_FRAME_NOT_SUPPORTED.equals(subErrorType)) {
                    errorCode = ErrorCode.LOADDOCUMENT_FEATURE_NOT_SUPPORTED_FRAME_ATTACHED_TO_FRAME_ERROR;
                }
                break;
            }
            default: errorCode = defErrorCode; break;
        }

       return errorCode;
    }

    /**
     * Determines the LogLevel that should be used to output the
     * log message.
     *
     * @param e the filter exception caught and where the needed log-level
     *          should be determined
     * @return the log-level that should be used for logging this exception
     */
    static public LogLevel determineLogLevel(final FilterException e) {
        LogLevel logLevel;

        // filter exception should at most be logged on WARN level
        switch (e.getErrorcode()) {
            case CRITICAL_ERROR: logLevel = LogLevel.LOGLEVEL_WARN; break;
            case WRONG_PASSWORD: logLevel = LogLevel.LOGLEVEL_INFO; break;
            case UNSUPPORTED_ENCRYPTION_USED: logLevel = LogLevel.LOGLEVEL_INFO; break;
            case COMPLEXITY_TOO_HIGH: logLevel = LogLevel.LOGLEVEL_INFO; break;
            case MEMORY_USAGE_TOO_HIGH: logLevel = LogLevel.LOGLEVEL_WARN; break;
            case MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED: logLevel = LogLevel.LOGLEVEL_WARN; break;
            case STRICT_OOXML_NOT_SUPPORTED: logLevel = LogLevel.LOGLEVEL_INFO; break;
            case FEATURE_NOT_SUPPORTED: logLevel = LogLevel.LOGLEVEL_INFO; break;
            default: logLevel = LogLevel.LOGLEVEL_WARN; break;
        }

        return logLevel;
    }

    /**
     * Logs a filter exception message using the provided log-level.
     *
     * @param logger the logger to be used for the logging message.
     * @param level the log-level to be used for the logging message.
     * @param message the message to be logged.
     * @param e the filter exception
     */
    static public void log(final Log logger, LogLevel level, final String logText, final String context, final FilterException e) {
        Validate.notNull(logger);

        // add filter exception type
        String message = logText;
        final StringBuffer out = new StringBuffer(logText);
        out.append(" Type: " + e.getErrorcode().toString());

        // add optional document context information with the appeal to attach a document to reproduce the issue
        if (StringUtils.isNotEmpty(context)) {
            out.append("\nDocument context: " + context + ". If you write an issue for this problem, please add the document to enable development to fix it.");
        }
        message = out.toString();

        switch (level) {
            case LOGLEVEL_DEBUG: logger.debug(message, e); break;
            case LOGLEVEL_INFO: logger.info(message, e); break;
            case LOGLEVEL_WARN: logger.warn(message, e); break;
            case LOGLEVEL_ERROR: logger.error(message, e); break;
            default: logger.error(message, e); break;
        }
    }
}
