/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl.calc;

import java.util.concurrent.atomic.AtomicBoolean;

import org.json.JSONArray;
import org.json.JSONObject;

import com.openexchange.office.tools.doc.DocumentMetaData;
import com.openexchange.office.tools.doc.OXDocument;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.realtime.packet.ID;

/**
 * Encapsulates an asynchronous load request to be processed by a request processor. It
 * provides a isFinished method which can be used to determine, if the request has been
 * processed.
 *
 * @author Carsten Driesner
 * @since 7.6.2
 */
public class Request {

    private Type type = Type.REQUEST_LOAD;

    private final ID fromId;

    private final IRequestProcessor processor;

    private final JSONObject requestOps;

    private final OXDocument docLoader;

    private int documentOSN = 0;

    private String rtResourceID = null;

    private DocumentMetaData docMetaData = null;

    private int previewUIOpsCount = 0;

    private final UserData userData;

    private final AtomicBoolean finished = new AtomicBoolean(false);

    public enum Type {
        REQUEST_LOAD, REQUEST_LOAD_REMAINING, REQUEST_LOAD_SYNC
    }

    /**
     * Creates a sync request which is used to sync clients with a load request which is still in progress.
     *
     * @param folderId
     * @param fileId
     * @param fromId
     * @param processor
     * @param documentOSN
     */
    public Request(final String rtResourceID, final ID fromId, IRequestProcessor processor, int documentOSN, final DocumentMetaData docMetaData, final UserData userData) {
        this.type = Type.REQUEST_LOAD_SYNC;
        this.rtResourceID = rtResourceID;
        this.fromId = fromId;
        this.processor = processor;
        this.documentOSN = documentOSN;
        this.docLoader = null;
        this.requestOps = null;
        this.docMetaData = docMetaData;
        this.userData = userData;
    }

    /**
     * Creates a asynchronous load request to be processed by a request processor.
     *
     * @param folderId
     * @param fileId
     * @param fromId
     * @param operations
     * @param processor
     * @param documentOSN
     * @throws Exception
     */
    public Request(final String rtResourceID, final ID fromId, final JSONArray operations, IRequestProcessor processor, int documentOSN, final DocumentMetaData docMetaData, final UserData userData) throws Exception {
        this.type = Type.REQUEST_LOAD;
        this.rtResourceID = rtResourceID;
        this.fromId = fromId;
        this.processor = processor;
        this.documentOSN = documentOSN;
        this.docLoader = null;
        this.docMetaData = docMetaData;
        this.userData = userData;
        requestOps = new JSONObject();
        requestOps.put("operations", operations);
    }

    /**
     * Creates an asynchronous load request which should process the remaining parts of a request by a request processor.
     *
     * @param folderId
     * @param fileId
     * @param fromId
     * @param previewImporter
     * @param processor
     * @param documentOSN
     * @throws Exception
     */
    public Request(final String rtResourceID, final ID fromId, final OXDocument documentLoader, IRequestProcessor processor, int documentOSN, int previewUIOps, final UserData userData) throws Exception {
        this.type = Type.REQUEST_LOAD_REMAINING;
        this.rtResourceID = rtResourceID;
        this.fromId = fromId;
        this.processor = processor;
        this.documentOSN = documentOSN;
        this.docLoader = documentLoader;
        this.requestOps = null;
        this.previewUIOpsCount = previewUIOps;
        this.docMetaData = documentLoader.getDocumentMetaData();
        this.userData = userData;
    }

    /**
     * Retrieves the type of a request.
     *
     * @return
     */
    public Type getType() {
        return type;
    }

    /**
     * Retrieves the operations of the request.
     *
     * @return
     */
    public JSONObject getOperations() {
        return requestOps;
    }

    /**
     * Retrieves the ID which is the context of the request.
     *
     * @return
     */
    public final ID getId() {
        return fromId;
    }

    /**
     * Retrieves the request processor which is responsible to process the request.
     *
     * @return
     */
    public final IRequestProcessor getProcessor() {
        return processor;
    }

    /**
     * Retrieves the responsible document loader instance for this request.
     *
     * @return
     */
    public final OXDocument getDocumentLoader() {
        return docLoader;
    }

    /**
     * Retrieves the document osn.
     *
     * @return
     */
    public int getDocumentOSN() {
        return documentOSN;
    }

    /**
     * Retrieves the document id.
     *
     * @return
     */
    public String getDocumentId() {
        return rtResourceID;
    }

    /**
     * Retrieve the number of preview operations.
     *
     * @return The number of preview UI operations.
     */
    public int getPreviewUIOpsCount() {
        return this.previewUIOpsCount;
    }

    /**
     * Retrieves the meta data valid by this request. This data may be
     * out dated and should not by used for current operations.
     *
     * @return
     *  The document meta data used by this request.
     */
    public DocumentMetaData getDocumentMetaData() {
        return this.docMetaData;
    }

    /**
     * Retrieves the UserData instance connected with this request.
     *
     * @return
     *  The UserData instance connected with this request.
     */
    public final UserData getUserData() {
        return this.userData;
    }

    /**
     * Determines if the request has been processed or not.
     *
     * @return
     */
    public boolean isFinished() {
        return finished.get();
    }

    /**
     * Sets the request to the finished state.
     */
    public void finished() {
        finished.set(true);
    }

    /**
     * Creates the document id.
     *
     * @param resourceId the resource id from the connection id
     * @return
     */
    static public String buildDocumentId(final String resourceId) {
        return new StringBuilder("document:" + resourceId).toString();
    }
}
