/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import java.util.concurrent.atomic.AtomicBoolean;

import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.realtime.packet.ID;
import com.openexchange.session.Session;

/**
 * A class to be executed by a thread to flush the document and afterwards
 * starts to rename the document. ATTENTION: This is necessary for some types
 * of storages which don't support persistent file ids. It that case the unique
 * real-time ID also changes, which in the end leads to reference problems for
 * the clients.
 *
 * @author Carsten Driesner
 */
public class FlushDocumentRenameRunnable implements Runnable {

    // - Members ----------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(FlushDocumentRenameRunnable.class);
    private final DocumentConnection connection;
    private final ID editorId;
    private final Session session;
    private final AtomicBoolean completed = new AtomicBoolean(false);
    private ErrorCode errorCode;
    private final UserData userData;
    private final String newFileName;

    /**
     * Initializes a new instance of a CompleteEditRightsTimeoutRunnable.
     *
     * @param connection The reference to the Connection instance that needs the time-out mechanism.
     * @param session The session of the client that loses the edit rights.
     * @param newEditorId The ID of the client that receives the edit rights.
     */
    public FlushDocumentRenameRunnable(final DocumentConnection connection, final Session session, final ID editorId, final UserData userData, final String newFileName) {
        super();
        this.connection = connection;
        this.editorId = editorId;
        this.session = session;
        this.userData = userData;
        this.newFileName = newFileName;
    }

    public boolean completed() {
        return completed.get();
    }

    public ErrorCode getErrorCode() {
        return errorCode;
    }

    /**
     * This method must be called by a thread.
     */
    @Override
    public void run() {
        if ((null != connection) && (!connection.isDisposed())) {
            errorCode = connection.flushDocument(session, editorId, false, true, false, true);
            completed.set(true);

            try {
                // a HACK to make sure that complete async rename is executed
                // after the rename
                Thread.sleep(2000);
            } catch (InterruptedException e) {
                // do nothing
            }

            connection.completeAsyncRename(session, editorId, userData, newFileName, errorCode);
        }
    }
}
