/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.components;

import java.lang.annotation.Annotation;
import java.text.ParseException;
import java.util.HashSet;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlRootElement;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.ContentAccessor;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;

public abstract class Component extends ComponentContext {

    private static HashSet<Class<?>> rootElements = new HashSet<Class<?>>();

    public enum Type {
    	PARAGRAPH,
    	TABLE,
    	TR,
    	TC,
    	TAB,
    	COMMENT_REFERENCE,
    	COMMENT_RANGE_START,
    	COMMENT_RANGE_END,
    	COMPLEX_FIELD_START,
    	COMPLEX_FIELD_SEPARATE,
    	COMPLEX_FIELD_END,
    	HARDBREAK_DEFAULT,
    	HARDBREAK_PAGE,
    	HARDBREAK_COLUMN,
    	AC_SHAPE,
    	AC_GROUP,
    	AC_IMAGE
    }

    public enum SplitMode {
        ATTRIBUTE,
        DELETE
    }

    /*
     * returns the corresponding component for the given oxo position. if positionCount is zero a
     * RootComponent is returned. The contextPart is usually the MainDocumentPart and changed only
     * for Header & Footer.
     *
     */
    public static Component getComponent(Component rootComponent, JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return rootComponent;
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            Component c = rootComponent.getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    /*
     * simply returns o, but in case o is a JAXBElement and a RootElement annotation is available
     * then the parent content list is updated and the contentModel is returned.
     * */
    public static Object getContentModel(IndexedNode<Object> node, Object parent) {

        if (node.getData() instanceof javax.xml.bind.JAXBElement) {
            boolean hasRootElement = rootElements.contains(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
            if(!hasRootElement) {
                final Annotation[] annotations = ((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType().getAnnotations();
                for(Annotation annotation : annotations) {
                    if(annotation instanceof XmlRootElement) {
                        rootElements.add(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
                        hasRootElement = true;
                        break;
                    }
                }
            }
            if(hasRootElement) {
                node.setData(((JAXBElement<?>)node.getData()).getValue());
            }
        }
        if(node.getData() instanceof Child) {
        	((Child)node.getData()).setParent(parent);
        }
        return node.getData();
    }

    protected int componentNumber;

    public Component(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node);
        componentNumber = _componentNumber;
    }
    public Object getObject() {
        return getNode().getData();
    }
    public Component getNextComponent() {
    	if(getParentContext()==null) {
    		return null;
    	}
        ComponentContext parentContext = this;
        Component nextComponent = null;
        do {
            ComponentContext previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }

    public ComponentContext getContext(Class<?> contextClass) {
        ComponentContext parentContext = getParentContext();
    	while(!(parentContext instanceof Component)&&(parentContext!=null)) {
    		if(parentContext.getClass()==contextClass) {
    			return parentContext;
    		}
    		parentContext = parentContext.getParentContext();
    	}
    	return null;
    }

    // the ComponentContext c is inserted behind one of the classes in the
    // parentContextList, it is inserted at least behind the parent Component,
    // so the parentContextList is allowed to be null
    //
    // the parentContext of c is inserted, so c does not need to have a parentContext
    public void insertContext(ComponentContext c, HashSet<Class<?>> parentContextList) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }

        final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
        parentContent.setNode(childContext.getNode(), c.getNode());
        final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)c.getNode().getData()).getContent();
        content.addNode(childContext.getNode());
        c.setParentContext(parentContext);
        childContext.setParentContext(c);
        ((Child)c.getNode().getData()).setParent(parentContext.getNode().getData());
        ((Child)childContext.getNode().getData()).setParent(c.getNode().getData());
    }

    public void removeContext(Class<?> contextClass) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            if(parentContext.getClass()==contextClass) {
            	final ComponentContext newParentContext = parentContext.getParentContext();
            	((IndexedNodeList<Object>)((ContentAccessor)newParentContext.getNode().getData()).getContent()).setNodes(parentContext.getNode(), (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent());
            	childContext.setParentContext(newParentContext);
            	((Child)childContext.getNode().getData()).setParent(newParentContext.getNode().getData());
                break;
            }
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
    }

    public ComponentContext getContextChild(HashSet<Class<?>> parentContextList) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    public Component getParentComponent() {
    	ComponentContext parentContext = getParentContext();
    	while(!(parentContext instanceof Component)) {
    		parentContext = parentContext.getParentContext();
    	}
    	return (Component)parentContext;
    }

    public abstract Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent);

    public Component getChildComponent(int childComponentNumber) {
    	Component c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(childComponentNumber);
    	}
    	return c;
    }

    public abstract Component insertChildComponent(ComponentContext parentContext, IndexedNode<Object> contextNode, int number, Component child, Type type);

    public Component insertChildComponent(OperationDocument operationDocument, int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type);
        if(attrs!=null) {
            c.applyAttrsFromJSON(operationDocument, attrs);
        }
        return c;
    }
    public Component getComponent(int number) {
    	Component c = this;
    	while(c!=null&&c.getNextComponentNumber()<=number) {
    		c = c.getNextComponent();
    	}
    	return c;
    }
    public int getComponentNumber() {
        return componentNumber;
    }
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }
    public void delete(OperationDocument operationDocument, int count)
    	throws InvalidFormatException {

    	final ComponentContext contextChild = getContextChild(null);

    	final int endComponent = (getComponentNumber()+count)-1;
        Component component = this;
        while(true) {
        	if(component.getNextComponentNumber()>endComponent) {
        		break;
        	}
        	component = component.getNextComponent();
        }
        component.splitEnd(endComponent, SplitMode.DELETE);
        final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)contextChild.getParentContext().getNode().getData()).getContent();
        content.removeNodes(contextChild.getNode(), component.getContextChild(null).getNode());
    }
    public void splitStart(int n, SplitMode splitMode) {
    }
    public void splitEnd(int n, SplitMode splitMode) {
    }
    public abstract void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

    public abstract JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
    	throws JSONException, ParseException, FilterException;
}
