/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;
import java.util.List;

import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.BookViews;
import org.xlsx4j.sml.CTBookView;
import org.xlsx4j.sml.CTColor;
import org.xlsx4j.sml.CTPane;
import org.xlsx4j.sml.CTSelection;
import org.xlsx4j.sml.CTSheetFormatPr;
import org.xlsx4j.sml.CTSheetProtection;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STPane;
import org.xlsx4j.sml.STPaneState;
import org.xlsx4j.sml.STSheetState;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.SheetView;
import org.xlsx4j.sml.SheetViews;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.XlsxOperationDocument;

public class SheetUtils {

    public static Sheet getSheet(SpreadsheetMLPackage spreadsheetMLPackage, int index) {
        WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        Workbook workbook = workbookPart.getJaxbElement();
        return workbook.getSheets().getSheet().get(index);
    }

    public static void applySheetProperties(Workbook workbook, int sheetIndex, Worksheet worksheet, JSONObject sheetProperties)
    	throws JSONException {

        if(sheetProperties==null)
            return;

        final List<Sheet> sheetList = workbook.getSheets().getSheet();
        final Sheet sheet = sheetList.get(sheetIndex);
        boolean visible = sheetProperties.optBoolean("visible", true);
        sheet.setState(visible ? STSheetState.VISIBLE : STSheetState.HIDDEN);

        final Object locked = sheetProperties.opt("locked");
        if(locked!=null) {
            CTSheetProtection sheetProtection = worksheet.getSheetProtection();
            if(locked instanceof Boolean) {
                if(sheetProtection==null) {
                    sheetProtection = Context.getsmlObjectFactory().createCTSheetProtection();
                    worksheet.setSheetProtection(sheetProtection);
                }
                sheetProtection.setSheet((Boolean)locked);
            }
            else {
                if(sheetProtection!=null) {
                    sheetProtection.setSheet(null);
                }
            }
        }
    	final SheetView sheetView = getOrCreateSheetView(worksheet);
        final Object zoom = sheetProperties.opt("zoom");
        if(zoom!=null) {
        	sheetView.setZoomScale(zoom instanceof Number
        			? (long)((((Number)zoom).doubleValue()*100.0)+0.5)
        			: null);
        }
        final Object showGrid = sheetProperties.opt("showGrid");
        if(showGrid!=null) {
        	sheetView.setShowGridLines(showGrid instanceof Boolean
        			? (Boolean)showGrid
        			: null);
        }
        final Object scrollLeft = sheetProperties.opt("scrollLeft");
        final Object scrollTop  = sheetProperties.opt("scrollTop");
        if(scrollLeft!=null||scrollTop!=null) {
    		SmlUtils.CellRef topLeft = SmlUtils.createCellRef(sheetView.getTopLeftCell());
    		if(topLeft==null) {
    			topLeft = new SmlUtils.CellRef(0, 0);
    		}
       		if(scrollLeft!=null) {
       			topLeft.setColumn(scrollLeft instanceof Number
       				? ((Number)scrollLeft).intValue()
       				: 0);
            }
       		if(scrollTop!=null) {
       			topLeft.setRow(scrollTop instanceof Number
       				? ((Number)scrollTop).intValue()
       				: 0);
            }
       		if(topLeft.getColumn()==0&&topLeft.getRow()==0) {
       			sheetView.setTopLeftCell(null);
       		}
       		else {
       			sheetView.setTopLeftCell(SmlUtils.getCellRef(topLeft));
       		}
        }

        CTPane pane = sheetView.getPane();
        final Object splitMode = sheetProperties.opt("splitMode");
        final Object splitWidth = sheetProperties.opt("splitWidth");
        final Object splitHeight = sheetProperties.opt("splitHeight");
        if(splitMode!=null||splitWidth!=null||splitHeight!=null) {
        	if(pane==null) {
        		pane = getOrCreatePane(worksheet);
        	}
        	if(splitMode!=null) {
	        	if(splitMode instanceof String) {
	        		if(((String)splitMode).equals("split")) {
	        			pane.setState(STPaneState.SPLIT);
	        		} else if(((String)splitMode).equals("frozen")) {
	        			pane.setState(STPaneState.FROZEN);
	        		} else if(((String)splitMode).equals("frozenSplit")) {
	        			pane.setState(STPaneState.FROZEN_SPLIT);
	        		}
	        	}
	        	else {
	        		pane.setState(STPaneState.SPLIT);
	        	}
	        }
        	if(splitWidth!=null) {
        		if(splitWidth instanceof Number) {
        			if(pane.getState()==STPaneState.SPLIT) {
        				pane.setXSplit((((Number)splitWidth).doubleValue()*1440.0)/2540);	// 100thmm to TWIP
        			}
        			else {
        				pane.setXSplit(((Number)splitWidth).doubleValue());
        			}
        		}
        		else {
        			pane.setXSplit(null);
        		}
        	}
        	if(splitHeight!=null) {
        		if(splitHeight instanceof Number) {
        			if(pane.getState()==STPaneState.SPLIT) {
        				pane.setYSplit((((Number)splitHeight).doubleValue()*1440.0)/2540);	// 100thmm to TWIP
        			}
        			else {
        				pane.setYSplit(((Number)splitHeight).doubleValue());
        			}
        		}
        		else {
        			pane.setYSplit(null);
        		}
        	}
        }


        boolean hasPane = pane!=null&&(pane.getXSplit()!=0.0D||pane.getYSplit()!=0.0D);
        if(!hasPane) {
        	sheetView.setPane(null);
        } else {

        	STPane activePane = pane.getActivePane();
        	final Object aPane = sheetProperties.opt("activePane");
	        if(aPane!=null) {
	        	activePane = STPane.BOTTOM_RIGHT;
	        	if(aPane instanceof String) {
	        		if(((String)aPane).equals("topLeft")) {
	        			activePane = STPane.TOP_LEFT;
	        		} else if(((String)aPane).equals("topRight")) {
	            		activePane = STPane.TOP_RIGHT;
	        		} else if(((String)aPane).equals("bottomRight")) {
	        			activePane = STPane.TOP_RIGHT;
	        		} else if(((String)aPane).equals("bottomLeftLeft")) {
	        			activePane = STPane.BOTTOM_LEFT;
	        		}
	        	}
	        	else {
	        		pane.setActivePane(activePane);
	        	}
	        }
	        final Object scrollRight = sheetProperties.opt("scrollRight");
	        final Object scrollBottom  = sheetProperties.opt("scrollBottom");
	        if(scrollRight!=null||scrollBottom!=null) {
	    		SmlUtils.CellRef topLeft = SmlUtils.createCellRef(pane.getTopLeftCell());
	    		if(topLeft==null) {
	    			topLeft = new SmlUtils.CellRef(0, 0);
	    		}
	       		if(scrollRight!=null) {
	       			topLeft.setColumn(scrollRight instanceof Number
	       				? ((Number)scrollRight).intValue()
	       				: 0);
	            }
	       		if(scrollBottom!=null) {
	       			topLeft.setRow(scrollBottom instanceof Number
	       				? ((Number)scrollBottom).intValue()
	       				: 0);
	            }
	       		if(topLeft.getColumn()==0&&topLeft.getRow()==0) {
	       			pane.setTopLeftCell(null);
	       		}
	       		else {
	       			pane.setTopLeftCell(SmlUtils.getCellRef(topLeft));
	       		}
	        }
        }

        // excel needs an activeCell and also a sqRef... the active cell must be placed within the sqref range...
        // otherwise excel will report an error in the viewSettings
        final Object selectedRanges = sheetProperties.opt("selectedRanges");
        final Object activeIndex = sheetProperties.opt("activeIndex");
        final Object activeCell = sheetProperties.opt("activeCell");
        if(selectedRanges!=null||activeIndex!=null||activeCell!=null) {
			final CTSelection selection=getSelection(sheetView, true);
			if(activeCell!=null) {
    			if(activeCell instanceof JSONArray) {
    				final SmlUtils.CellRef cellRef = new SmlUtils.CellRef(((JSONArray)activeCell).getInt(0), ((JSONArray)activeCell).getInt(1));
    				selection.setActiveCell(SmlUtils.getCellRef(cellRef));
    			}
    			else {
					selection.setActiveCell(null);
    			}
    		}
        	if(selectedRanges!=null) {
    			if(selectedRanges instanceof JSONArray) {
            		final List<String> sqrefs = selection.getSqref();
            		sqrefs.clear();
            		for(int i=0;i<((JSONArray)selectedRanges).length();i++) {
            			final JSONObject range = ((JSONArray)selectedRanges).getJSONObject(i);
            			final JSONArray start = range.getJSONArray("start");
            			final JSONArray end = range.getJSONArray("end");
            			final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(
            					new SmlUtils.CellRef(start.getInt(0), start.getInt(1)),
            					new SmlUtils.CellRef(end.getInt(0), end.getInt(1)));
            			sqrefs.add(SmlUtils.getCellRefRange(cellRefRange));
            		}
    			}
    			else {
					selection.getSqref().clear();
    			}
    		}
    		if(activeIndex!=null) {
				if(activeIndex instanceof Number) {
					selection.setActiveCellId(((Number)activeIndex).longValue());
				}
				else {
					selection.setActiveCellId(null);
				}
    		}
    		// now check if our selection is valid
    		if(selection.getActiveCell()==null) {
    			selection.setActiveCell("A1");
    		}
    		final SmlUtils.CellRef activeCellRef = SmlUtils.createCellRef(selection.getActiveCell());
    		final List<String> sel = selection.getSqref();
    		if(sel.isEmpty()) {
    			sel.add(SmlUtils.getCellRef(activeCellRef));
    			selection.setActiveCellId(null);
    		}
    		else {
    			final long cellId = selection.getActiveCellId();
    			boolean activeCellIsPartOfSelection = false;
    			// now check if the active cell is part of the sqref
    			if(cellId>=0&&cellId<sel.size()) {
    				final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sel.get((int)cellId));
    				activeCellIsPartOfSelection = cellRefRange.isInside(activeCellRef.getColumn(), activeCellRef.getRow());
    			}
    			if(activeCellIsPartOfSelection==false) {
    				selection.setActiveCellId(null);
    				sel.clear();
    				sel.add(SmlUtils.getCellRef(activeCellRef));
    			}
	        }
        }
    }

	private static CTSelection getSelection(SheetView sheetView, boolean forceCreate) {

		final STPane activePane = sheetView.getPane()!=null?sheetView.getPane().getActivePane():STPane.TOP_LEFT;
		final List<CTSelection> selectionList = sheetView.getSelection();
		for(CTSelection selection:selectionList) {
			if(selection.getPane()==activePane) {
				return selection;
			}
		}
		if(forceCreate) {
			CTSelection selection = Context.getsmlObjectFactory().createCTSelection();
			selection.setPane(activePane);
			selectionList.add(selection);
			return selection;
		}
		return null;
	}

    private static SheetView getOrCreateSheetView(Worksheet worksheet) {

    	SheetViews sheetViews = worksheet.getSheetViews();
    	if(sheetViews==null) {
    		sheetViews = Context.getsmlObjectFactory().createSheetViews();
    		worksheet.setSheetViews(sheetViews);
    	}
    	final List<SheetView> sheetViewList = sheetViews.getSheetView();
    	if(sheetViewList.isEmpty()) {
    		sheetViewList.add(Context.getsmlObjectFactory().createSheetView());
    	}
    	return sheetViewList.get(0);
    }

    private static CTPane getOrCreatePane(Worksheet worksheet) {

    	final SheetView sheetView = getOrCreateSheetView(worksheet);
    	CTPane pane = sheetView.getPane();
    	if(pane==null) {
    		pane = Context.getsmlObjectFactory().createCTPane();
    		sheetView.setPane(pane);
    	}
    	return pane;
    }

    // check if the first visible sheet matches the bookView and removes each tabSelected from sheet
    public static void updateBookView(Workbook workbook) {

        final List<Sheet> sheetList = workbook.getSheets().getSheet();
        Long firstVisible = null;
        for(int i=0; i<sheetList.size(); i++) {
            final Sheet sh = sheetList.get(i);
            if(firstVisible==null&&sh.getState()==STSheetState.VISIBLE) {
                firstVisible = Long.valueOf(i);
            }
        }
        if(firstVisible==null) {
            firstVisible = Long.valueOf(0);
        }
        BookViews bookViews = workbook.getBookViews();
        if(bookViews==null) {
            bookViews = Context.getsmlObjectFactory().createBookViews();
            workbook.setBookViews(bookViews);
        }
        final List<CTBookView> workbookViewList = bookViews.getWorkbookView();
        if(workbookViewList.isEmpty()) {
            final CTBookView bookView = Context.getsmlObjectFactory().createCTBookView();
            workbookViewList.add(bookView);
        }
        for(CTBookView bookView:workbookViewList) {
            final int activeTab = (int)bookView.getActiveTab();
            final int firstView = (int)bookView.getFirstSheet();
            if(activeTab<0||activeTab>=sheetList.size()||sheetList.get(activeTab).getState()!=STSheetState.VISIBLE) {
                bookView.setActiveTab(firstVisible);
            }
            if(firstView<0||firstView>=sheetList.size()||sheetList.get(firstView).getState()!=STSheetState.VISIBLE) {
                bookView.setFirstSheet(firstVisible);
            }
        }
    }

    public static JSONObject createSheetProperties(Sheet sheet)
        throws JSONException {

        if(sheet.getState()==STSheetState.VISIBLE) {
        	return null;
        }
        JSONObject jsonSheetProperties = new JSONObject();
        jsonSheetProperties.put("visible", false);
        return jsonSheetProperties;
    }

    public static JSONObject createWorksheetProperties(Sheet sheet, Worksheet worksheet)
        throws JSONException {

        JSONObject attrs = createSheetProperties(sheet);
        if(attrs==null) {
        	attrs = new JSONObject();
        }
        final CTSheetProtection sheetProtection = worksheet.getSheetProtection();
        if(sheetProtection!=null) {
            if(sheetProtection.isSheet()) {
                attrs.put("locked", true);
            }
        }
        final SheetViews sheetViews = worksheet.getSheetViews();
        if(sheetViews!=null) {
        	final List<SheetView> sheetViewList = sheetViews.getSheetView();
        	if(!sheetViewList.isEmpty()) {
        		final SheetView sheetView = sheetViewList.get(0);
        		final CTPane pane = sheetView.getPane();
        		final double zoomScale = sheetView.getZoomScale() / 100.0;
        		attrs.put("zoom", zoomScale);
        		if(sheetView.isShowGridLines()==false) {
        			attrs.put("showGrid", false);
        		}
        		if(sheetView.isDefaultGridColor()==false) {
        			final long colorIndex = sheetView.getColorId();
        			if(colorIndex!=64) {
        				CTColor color = new CTColor();
        				color.setIndexed(new Long(colorIndex));
                        Commons.jsonPut(attrs, "gridColor", Utils.createColor(null, color));
        			}
        		}
        		final SmlUtils.CellRef topLeft = SmlUtils.createCellRef(sheetView.getTopLeftCell());
        		if(topLeft!=null) {
        			final int column = topLeft.getColumn();
        			final int row = topLeft.getRow();
        			if(column>0) {
        				attrs.put("scrollLeft", topLeft.getColumn());
        			}
        			if(row>0) {
        				attrs.put("scrollTop", topLeft.getRow());
        			}
        		}
    			final STPane activePane = pane!=null?pane.getActivePane():STPane.TOP_LEFT;
    			if(activePane!=STPane.BOTTOM_RIGHT) {
    				if(activePane==STPane.BOTTOM_LEFT) {
    					attrs.put("activePane", "bottomLeft");
    				} else if(activePane==STPane.BOTTOM_RIGHT) {
    					attrs.put("activePane", "bottomRight");
    				} else if(activePane==STPane.TOP_RIGHT) {
    					attrs.put("activePane", "topRight");
    				}
    			}
        		if(pane!=null) {
        			final SmlUtils.CellRef topLeftP = SmlUtils.createCellRef(pane.getTopLeftCell());
        			if(topLeftP!=null) {
            			if(topLeftP.getColumn()>0) {
            				attrs.put("scrollRight", topLeftP.getColumn());
            			}
            			if(topLeftP.getRow()>0) {
            				attrs.put("scrollBottom", topLeftP.getRow());
            			}
        			}
        			double xSplit = pane.getXSplit();
        			double ySplit = pane.getYSplit();
        			switch(pane.getState()) {
					case FROZEN:
						attrs.put("splitMode", "frozen");
						if(xSplit!=0.0) {
							attrs.put("splitWidth", (int)xSplit);
						}
						if(ySplit!=0.0) {
							attrs.put("splitHeight", (int)ySplit);
						}
						break;
					case FROZEN_SPLIT:
						attrs.put("splitMode",  "frozenSplit");
						if(xSplit!=0.0) {
							attrs.put("splitWidth", (int)xSplit);
						}
						if(ySplit!=0.0) {
							attrs.put("splitHeight", (int)ySplit);
						}
						break;
					case SPLIT:
						if(xSplit!=0.0) {
							attrs.put("splitWidth", (int)((xSplit * 2540.0) / 1440 + 0.5));		// TWIP to 100thmm
						}
						if(ySplit!=0.0) {
							attrs.put("splitHeight", (int)((ySplit * 2540.0) / 1440 + 0.5));
						}
						break;
					default:
						break;
        			}
        		}
        		final List<CTSelection> selections = sheetView.getSelection();
        		if(!selections.isEmpty()) {
        			for(CTSelection selection:selections) {
        				if(selection.getPane()==activePane) {
                			final JSONArray ranges = new JSONArray();
            				for(String sqref:selection.getSqref()) {
            					final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqref);
            					if(cellRefRange!=null) {
                        			final JSONObject range = new JSONObject(2);
            			            range.put("start", cellRefRange.getStart().getJSONArray());
            						range.put("end", cellRefRange.getEnd().getJSONArray());
            						ranges.put(range);
            					}
            				}
            				if(!ranges.isEmpty()) {
            					attrs.put("selectedRanges", ranges);
            					attrs.put("activeIndex", selection.getActiveCellId());
            					if(selection.getActiveCell()!=null) {
                					final SmlUtils.CellRef activeCell = SmlUtils.createCellRef(selection.getActiveCell());
                					if(activeCell!=null) {
                						attrs.put("activeCell", activeCell.getJSONArray());
                					}
            					}
            				}
                			break;
        				}
        			}
        		}
        	}
        }
        return attrs;
    }

    public static CTSheetFormatPr getOrCreateSheetFormatPr(Worksheet worksheet) {
    	CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
    	if(sheetFormatPr==null) {
    		sheetFormatPr = Context.getsmlObjectFactory().createCTSheetFormatPr();
    		worksheet.setSheetFormatPr(sheetFormatPr);
    	}
    	return sheetFormatPr;
    }

    public static void applyWorksheetColumnProperties(XlsxOperationDocument operationDocument, Worksheet worksheet, JSONObject columnProperties)
            throws JSONException {

    	if(columnProperties==null)
    		return;
    	final CTSheetFormatPr sheetFormatPr = getOrCreateSheetFormatPr(worksheet);
        final Object width = columnProperties.opt("width");
        if(width!=null) {
        	sheetFormatPr.setDefaultColWidth(ColumnUtils.columnWidthToCharacter(operationDocument, width));
        }
/*
        final Object customWidth = columnProperties.opt("customWidth");
        if(customWidth!=null) {
        }
*/
        final Object visible = columnProperties.opt("visible");
        if(visible!=null) {
			final Double defaultColWidth = sheetFormatPr.getDefaultColWidth();
        	if(visible instanceof Boolean) {
        		if(!(Boolean)visible) {
        			// changing to invisible
        			sheetFormatPr.setDefaultColWidth(new Double(0.0d));
        			if(defaultColWidth!=null) {
        				sheetFormatPr.setBaseColWidth(defaultColWidth.longValue());
        			}
        		}
        		else {	// changing to visible -> defaultColWidth of "0" is not allowed
        			if(defaultColWidth!=null&&defaultColWidth==0) {
        				sheetFormatPr.setDefaultColWidth(null);
        			}
        		}
        	}
        	else {	// changing to visible -> defaultColWidth of "0" is not allowed
    			if(defaultColWidth!=null&&defaultColWidth==0) {
    				sheetFormatPr.setDefaultColWidth(null);
    			}
        	}
        }
    }

    public static void applyWorksheetRowProperties(XlsxOperationDocument operationDocument, Worksheet worksheet, JSONObject rowProperties)
            throws JSONException {

    	if(rowProperties==null)
    		return;

    	final CTSheetFormatPr sheetFormatPr = getOrCreateSheetFormatPr(worksheet);
    	final Object height = rowProperties.opt("height");
    	if(height!=null) {
            if(height instanceof Number) {
                sheetFormatPr.setDefaultRowHeight(RowUtils.map100thmmToPoint(((Number)height).doubleValue()));
            }
            else {
            	sheetFormatPr.setDefaultRowHeight(15);
            }
    	}
    	final Object customHeight = rowProperties.opt("customHeight");
    	if(customHeight!=null) {
    		if(customHeight instanceof Boolean) {
    			sheetFormatPr.setCustomHeight((Boolean)customHeight);
    		}
    		else {
    			sheetFormatPr.setCustomHeight(null);
    		}
    	}
    	final Object visible = rowProperties.opt("visible");
    	if(visible!=null) {
    		if(visible instanceof Boolean) {
    			sheetFormatPr.setZeroHeight(!(Boolean)visible);
    		}
    		else {
    			sheetFormatPr.setZeroHeight(null);
    		}
    	}
    }

    public static JSONObject createWorksheetColumnProperties(XlsxOperationDocument operationDocument, Worksheet worksheet)
        throws JSONException {

        final JSONObject columnProperties = new JSONObject();
        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        columnProperties.put("width", ColumnUtils.excelColumnWidthTo100thmm(operationDocument, operationDocument.getDefaultColumnWidth(worksheet)));
        if(sheetFormatPr!=null) {
        	final Double defaultColWidth = sheetFormatPr.getDefaultColWidth();
        	if(defaultColWidth!=null) {
        		columnProperties.put("customWidth", true);
        		if(defaultColWidth.doubleValue()==0) {
        			columnProperties.put("visible", true);
        		}
        	}
        }
        return columnProperties;
    }

    public static JSONObject createWorksheetRowProperties(Worksheet worksheet)
        throws JSONException {

        final JSONObject rowProperties = new JSONObject();
        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();

        final boolean customHeight = sheetFormatPr!=null&&sheetFormatPr.isCustomHeight();
        if(customHeight) {
        	rowProperties.put("customHeight", true);
        }
        final double defaultRowHeight = customHeight?sheetFormatPr.getDefaultRowHeight():15;
    	rowProperties.put("height", RowUtils.mapPointTo100thmm(defaultRowHeight));

        if(sheetFormatPr!=null&&sheetFormatPr.isZeroHeight()) {
        	rowProperties.put("visible", false);
        }
        return rowProperties;
    }

    /*
     * applying attrs to each physical available cell within the row
     *
     */
    public static void applyRowProperties(XlsxOperationDocument operationDocument, SheetData sheetData, int start, JSONObject attrs)
        throws Exception {

        Row row = sheetData.getRow(start, false);
        if(row==null) {
            return;
        }
        final Iterator<Cell> cellIterator = row.createCellIterator();
        while(cellIterator.hasNext()) {
            CellUtils.applyCellPropertiesToCell(operationDocument, null, sheetData, attrs, row, cellIterator.next());
        }
    }

    /*
     * applying attrs to each physical available cell within the column
     *
     */
    public static void applyColumnProperties(XlsxOperationDocument operationDocument, SheetData sheetData, int start, int end, JSONObject attrs)
        throws Exception {

        final Iterator<Row> rowIterator = sheetData.createRowIterator();
        while(rowIterator.hasNext()) {
            final Row row = rowIterator.next();
            for(int i=start; i<=end; i++) {
                final Cell cell = row.getCell(i, false);
                if(cell!=null) {
                    CellUtils.applyCellPropertiesToCell(operationDocument, null, sheetData, attrs, row, cell);
                }
            }
        }
    }
}
