/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.operations;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.docx4j.dml.CTColorMappingOverride;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PresentationML.JaxbPmlPart;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.openpackaging.parts.PresentationML.ViewPropertiesPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.CTNormalViewPortion;
import org.pptx4j.pml.CTNormalViewProperties;
import org.pptx4j.pml.CTSlideMasterTextStyles;
import org.pptx4j.pml.Presentation;
import org.pptx4j.pml.Presentation.SldMasterIdLst;
import org.pptx4j.pml.Presentation.SldMasterIdLst.SldMasterId;
import org.pptx4j.pml.Presentation.SldSz;
import org.pptx4j.pml.Sld;
import org.pptx4j.pml.SldLayout;
import org.pptx4j.pml.SldMaster;
import org.pptx4j.pml.SlideLayoutIdList;
import org.pptx4j.pml.SlideLayoutIdList.SldLayoutId;
import org.pptx4j.pml.ViewPr;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.IShapeType;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.PptxOperationDocument;
import com.openexchange.office.ooxml.pptx.components.HardBreakComponent;
import com.openexchange.office.ooxml.pptx.components.ParagraphComponent;
import com.openexchange.office.ooxml.pptx.components.RootComponent;
import com.openexchange.office.ooxml.pptx.components.TextComponent;

public class PptxCreateOperationHelper extends CreateOperationHelper {

    private final PresentationMLPackage presentationMLPackage;
    private final HashMap<String, String> relTargetToId = new HashMap<String, String>();

    public PptxCreateOperationHelper(PptxOperationDocument _operationDocument, JSONArray mOperationsArray)
    	throws FilterException {

    	super(_operationDocument, mOperationsArray);
        presentationMLPackage = _operationDocument.getPackage();
    }

	@Override
	public PptxOperationDocument getOperationDocument() {
		return (PptxOperationDocument)operationDocument;
	}

	@Override
	public void createDocumentDefaults(String userLanguage) throws Exception {
		final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
		final Presentation presentation = mainPresentationPart.getJaxbElement();
        final JSONObject documentAttributes = new JSONObject();

        // page settings
        final JSONObject jsonPageSettings = new JSONObject(3);
		final SldSz slideSz = presentation.getSldSz();
        jsonPageSettings.put("width", (int)((slideSz.getCx()/360)+0.5));
        jsonPageSettings.put("height", (int)((slideSz.getCy()/360)+0.5));
        jsonPageSettings.put("orientation", "landscape");
        documentAttributes.put("page", jsonPageSettings);

        // text default styles
        final CTTextListStyle defaultTextStyle = presentation.getDefaultTextStyle();
        if(defaultTextStyle!=null) {
        	final JSONObject textDefaults = DMLHelper.createJsonFromTextListStyle(defaultTextStyle, mainPresentationPart);
        	if(textDefaults!=null&&!textDefaults.isEmpty()) {
        		documentAttributes.put("defaultTextListStyles", textDefaults);
        	}
        }

        // layout settings
        final RelationshipsPart mainPresentationRelations = mainPresentationPart.getRelationshipsPart();
        final Relationship viewRel = mainPresentationRelations.getRelationshipByType(Namespaces.PRESENTATIONML_VIEW_PROPS);
        if(viewRel!=null) {
			final JSONObject layoutSettings = new JSONObject(1);
			final ViewPropertiesPart viewPropertiesPart = (ViewPropertiesPart)mainPresentationRelations.getPart(viewRel);
        	final ViewPr viewPr = viewPropertiesPart.getJaxbElement();
        	final CTNormalViewProperties normalViewProperties = viewPr.getNormalViewPr();
        	if(normalViewProperties!=null) {
        		final CTNormalViewPortion leftViewPortion = normalViewProperties.getRestoredLeft();
        		if(leftViewPortion!=null) {
        			final double leftViewPortionSz = ((double)leftViewPortion.getSz()) / 1000;
        			layoutSettings.put("slidePaneWidth", leftViewPortionSz);
        		}
        	}
        	if(!layoutSettings.isEmpty()) {
        		documentAttributes.put("layout", layoutSettings);
        	}
        }

        final JSONObject setDocumentAttributesObject = new JSONObject(2);
        setDocumentAttributesObject.put("name", "setDocumentAttributes");
        setDocumentAttributesObject.put("attrs", documentAttributes);
        operationsArray.put(setDocumentAttributesObject);
	}

	public void createMasterSlides()
		throws JSONException, ParseException {

		final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
		final SldMasterIdLst slideMasterIds = mainPresentationPart.getJaxbElement().getSldMasterIdLst();
		if(slideMasterIds!=null) {
			final List<SldMasterId> slideMasterIdList = slideMasterIds.getSldMasterId();
			for(int i=0; i<slideMasterIdList.size(); i++) {
				final SldMasterId slideMasterId = slideMasterIdList.get(i);
				final SlideMasterPart slideMasterPart = (SlideMasterPart)mainPresentationPart.getRelationshipsPart().getPart(slideMasterId.getRid());
				getOperationDocument().setContextPart(slideMasterPart);
				final String masterTarget = slideMasterId.getId().toString();
				relTargetToId.put(mainPresentationPart.getRelationshipsPart().getRelationshipByID(slideMasterId.getRid()).getTarget(), masterTarget);
				createMasterSlideOperations(masterTarget);

				// retrieving layouts
				final SlideLayoutIdList slideLayoutIds = slideMasterPart.getJaxbElement().getSldLayoutIdLst();
				if(slideLayoutIds!=null) {
					final List<SldLayoutId> slideLayoutIdList = slideLayoutIds.getSldLayoutId();
					for(int j=0; j<slideLayoutIdList.size(); j++) {
						final SldLayoutId slideLayoutId = slideLayoutIdList.get(j);
						final String layoutTarget = masterTarget + "." + slideLayoutId.getId().toString();
						relTargetToId.put(slideMasterPart.getRelationshipsPart().getRelationshipByID(slideLayoutId.getRid()).getTarget(), layoutTarget);
						createLayoutSlideOperations(layoutTarget, masterTarget);
					}
				}
			}
		}
	}

	public void createSlides()
		throws JSONException, ParseException {

		getOperationDocument().setContext(null);

		final RootComponent rootComponent = new RootComponent(getOperationDocument());
        Component slideComponent = rootComponent.getNextChildComponent(null, null);
        while (slideComponent!=null) {

        	getOperationDocument().setContextPart((Part)slideComponent.getObject());

        	final JSONObject jsonInsertSlideOperation = new JSONObject(4);
    		jsonInsertSlideOperation.put("name", "insertSlide");
    		jsonInsertSlideOperation.put("start", slideComponent.getComponentNumber());
    		final SlidePart slidePart = (SlidePart)slideComponent.getObject();

    		// retrieving the layout target that has to be used for this slide
    		final String slideLayoutTarget = slidePart.getRelationshipsPart().getRelationshipByType(Namespaces.PRESENTATIONML_SLIDE_LAYOUT).getTarget();
    		if(slideLayoutTarget!=null) {
    			final String slideLayoutTargetId = relTargetToId.get(slideLayoutTarget);
    			if(slideLayoutTargetId!=null) {
    				jsonInsertSlideOperation.put("target", slideLayoutTargetId);
    			}
    		}
			createCommonSlideDataOperations(slideComponent, slideComponent.getComponentNumber(), null, jsonInsertSlideOperation, new JSONObject());
        	slideComponent = slideComponent.getNextComponent();
        }
	}

	public void createMasterSlideOperations(String target)
		throws JSONException, ParseException {

		getOperationDocument().setContext(target);

		final RootComponent rootComponent = new RootComponent(getOperationDocument());
		final Component slideComponent = rootComponent.getNextChildComponent(null, null);
		if(slideComponent!=null) {
			final JSONObject jsonInsertMasterSlideOperation = new JSONObject(3);
			jsonInsertMasterSlideOperation.put("name", "insertMasterSlide");
			jsonInsertMasterSlideOperation.put("id", target);
			final SldMaster slideMaster = (SldMaster)(((JaxbPmlPart<?>)slideComponent.getObject()).getJaxbElement());
			final CTSlideMasterTextStyles txStyles = slideMaster.getTxStyles();
			final JSONObject attrs = new JSONObject();
			if(txStyles!=null) {
				final JSONObject styles = new JSONObject();
				final JSONObject titleStyle = DMLHelper.createJsonFromTextListStyle(txStyles.getTitleStyle(), (Part)slideComponent.getObject());
				if(titleStyle!=null&&!titleStyle.isEmpty()) {
					styles.put("title", titleStyle);
				}
				final JSONObject bodyStyle = DMLHelper.createJsonFromTextListStyle(txStyles.getBodyStyle(), (Part)slideComponent.getObject());
				if(bodyStyle!=null&&!bodyStyle.isEmpty()) {
					styles.put("body", bodyStyle);
				}
				final JSONObject otherStyle = DMLHelper.createJsonFromTextListStyle(txStyles.getOtherStyle(), (Part)slideComponent.getObject());
				if(otherStyle!=null&&!otherStyle.isEmpty()) {
					styles.put("other", otherStyle);
				}
				if(!styles.isEmpty()) {
					attrs.put("listStyles", styles);
				}
			}
			createCommonSlideDataOperations(slideComponent, 0, target, jsonInsertMasterSlideOperation, attrs);
		}
	}

	public void createLayoutSlideOperations(String target, String referencedMaster)
		throws JSONException, ParseException {

		getOperationDocument().setContext(target);

		final RootComponent rootComponent = new RootComponent(getOperationDocument());
		final Component slideComponent = rootComponent.getNextChildComponent(null, null);
		if(slideComponent!=null) {
			final JSONObject jsonInsertLayoutSlideOperation = new JSONObject(4);
			jsonInsertLayoutSlideOperation.put("name", "insertLayoutSlide");
			jsonInsertLayoutSlideOperation.put("id", target);
			jsonInsertLayoutSlideOperation.put("target", referencedMaster);
			createCommonSlideDataOperations(slideComponent, 0, target, jsonInsertLayoutSlideOperation, new JSONObject());
		}
	}

	public void createCommonSlideDataOperations(Component slideComponent, int slide, String target, JSONObject slideInsertOperation, JSONObject attrs)
		throws JSONException, ParseException {

		slideComponent.createJSONAttrs(this, attrs);

		final JSONObject pageAttrs = new JSONObject();
		final Object sldElement = ((JaxbPmlPart<?>)slideComponent.getObject()).getJaxbElement();
		if(sldElement instanceof Sld) {
			if(!((Sld)sldElement).isShowMasterSp()) {
				pageAttrs.put("followMasterShapes", false);
			}
			if(!((Sld)sldElement).isShow()) {
				pageAttrs.put("hidden", true);
			}
		}
		else if(sldElement instanceof SldLayout) {
			if(!((SldLayout)sldElement).isShowMasterSp()) {
				pageAttrs.put("followMasterShapes", false);
			}
		}
		if(!pageAttrs.isEmpty()) {
			attrs.put("slide", pageAttrs);
		}

		if(!attrs.isEmpty()) {
			slideInsertOperation.put("attrs", attrs);
		}
		operationsArray.put(slideInsertOperation);

		if(sldElement instanceof SldMaster) {
			createThemeOperations(target, null);
		}
		else if(sldElement instanceof SldLayout) {
			createClrMapOvrThemeOperation(target, ((SldLayout)sldElement).getClrMapOvr());
		}
		else if(sldElement instanceof Sld) {
			createClrMapOvrThemeOperation(target, ((Sld)sldElement).getClrMapOvr());
		}

		final ArrayList<Integer> position = new ArrayList<Integer>();
		position.add(slide);
		Component shapeComponent = slideComponent.getNextChildComponent(null, null);
		while(shapeComponent!=null) {
			createShapeOperations(shapeComponent, position, target);
			shapeComponent = shapeComponent.getNextComponent();
		}
	}

	public void createClrMapOvrThemeOperation(String target, CTColorMappingOverride colorOverride)
		throws JSONException {

		if(colorOverride!=null&&colorOverride.getOverrideClrMapping()!=null) {
			createThemeOperations(target, colorOverride.getOverrideClrMapping());
		}
	}

	public void createShapeOperations(Component shapeComponent, List<Integer> parentPosition, String target)
		throws JSONException, ParseException {

		final List<Integer> position = new ArrayList<Integer>(parentPosition);
		position.add(shapeComponent.getComponentNumber());
		final JSONObject jsonInsertDrawingOperation = new JSONObject(5);
		jsonInsertDrawingOperation.put("name", "insertDrawing");
		jsonInsertDrawingOperation.put("start", position);
		if(target!=null) {
			jsonInsertDrawingOperation.put("target", target);
		}
		jsonInsertDrawingOperation.put("type", ((IShapeType)shapeComponent).getType().toString());
		final JSONObject attrs = shapeComponent.createJSONAttrs(this, new JSONObject());
		if(!attrs.isEmpty()) {
			jsonInsertDrawingOperation.put("attrs", attrs);
		}
		operationsArray.put(jsonInsertDrawingOperation);
		Component shapeChildComponent = shapeComponent.getNextChildComponent(null, null);
		while(shapeChildComponent!=null) {
			if(shapeChildComponent instanceof ParagraphComponent) {
				createParagraphOperations((ParagraphComponent)shapeChildComponent, position, target);
			}
			else {
				createShapeOperations(shapeChildComponent, position, target);
			}
			shapeChildComponent = shapeChildComponent.getNextComponent();
		}
	}

	public void createParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> parentPosition, String target)
		throws JSONException, ParseException {

		final List<Integer> position = new ArrayList<Integer>(parentPosition);
		position.add(paragraphComponent.getComponentNumber());
		final JSONObject jsonInsertParagraphOperation = new JSONObject(4);
		jsonInsertParagraphOperation.put("name", "insertParagraph");
		jsonInsertParagraphOperation.put("start", position);
		if(target!=null) {
			jsonInsertParagraphOperation.put("target", target);
		}
		final JSONObject attrs = paragraphComponent.createJSONAttrs(this, new JSONObject());
		if(!attrs.isEmpty()) {
			jsonInsertParagraphOperation.put("attrs", attrs);
		}
		operationsArray.put(jsonInsertParagraphOperation);
		Component paragraphChild = paragraphComponent.getNextChildComponent(null, null);
		while(paragraphChild!=null) {
			if(paragraphChild instanceof TextComponent) {
				createTextInsertOperation((TextComponent)paragraphChild, position, target);
			}
			else if(paragraphChild instanceof HardBreakComponent) {
				createHardBreakOperation((HardBreakComponent)paragraphChild, position, target);
			}
			paragraphChild = paragraphChild.getNextComponent();
		}
		paragraphChild = paragraphComponent.getNextChildComponent(null, null);
		while(paragraphChild!=null) {
			if(paragraphChild instanceof TextComponent) {
				createTextAttributesOperation((TextComponent)paragraphChild, position, target);
			}
			paragraphChild = paragraphChild.getNextComponent();
		}
	}

	public void createTextInsertOperation(TextComponent textComponent, List<Integer> parentPosition, String target)
		throws JSONException, ParseException {

		final List<Integer> position = new ArrayList<Integer>(parentPosition);
		position.add(textComponent.getComponentNumber());

		final CTRegularTextRun textRun = (CTRegularTextRun)textComponent.getObject();
		final JSONObject jsonInsertTextOperation = new JSONObject(4);
		jsonInsertTextOperation.put("name", "insertText");
		jsonInsertTextOperation.put("start", position);
		if(target!=null) {
			jsonInsertTextOperation.put("target", target);
		}
		jsonInsertTextOperation.put("text", textRun.getT());
		operationsArray.put(jsonInsertTextOperation);
	}

	public void createTextAttributesOperation(TextComponent textComponent, List<Integer> parentPosition, String target)
		throws JSONException, ParseException {

		final JSONObject attrs = textComponent.createJSONAttrs(this, new JSONObject());
		if(!attrs.isEmpty()) {

			final JSONObject jsonSetAttributesOperation = new JSONObject(5);
			jsonSetAttributesOperation.put("name", "setAttributes");

			final List<Integer> start = new ArrayList<Integer>(parentPosition);
			start.add(textComponent.getComponentNumber());
			jsonSetAttributesOperation.put("start", start);

			final CTRegularTextRun textRun = (CTRegularTextRun)textComponent.getObject();
			if(textRun.getT().length()>1) {
				start.set(start.size()-1, (textComponent.getComponentNumber()+textRun.getT().length())-1);
				jsonSetAttributesOperation.put("end", start);
			}

			if(target!=null) {
				jsonSetAttributesOperation.put("target", target);
			}
			jsonSetAttributesOperation.put("attrs", attrs);
			operationsArray.put(jsonSetAttributesOperation);
		}
	}

	public void createHardBreakOperation(HardBreakComponent hardBreakComponent, List<Integer> parentPosition, String target)
		throws JSONException, ParseException {

		final List<Integer> position = new ArrayList<Integer>(parentPosition);
		position.add(hardBreakComponent.getComponentNumber());

		final JSONObject jsonInsertHardBreakOperation = new JSONObject(4);
		jsonInsertHardBreakOperation.put("name", "insertHardBreak");
		jsonInsertHardBreakOperation.put("start", position);
		if(target!=null) {
			jsonInsertHardBreakOperation.put("target", target);
		}
		final JSONObject attrs = hardBreakComponent.createJSONAttrs(this, new JSONObject());
		if(!attrs.isEmpty()) {
			jsonInsertHardBreakOperation.put("attrs", attrs);
		}
		operationsArray.put(jsonInsertHardBreakOperation);
	}
}
