/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx;

import java.io.InputStream;
import java.util.List;
import java.util.StringTokenizer;

import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.pptx4j.pml.Presentation;
import org.pptx4j.pml.Presentation.SldMasterIdLst;
import org.pptx4j.pml.Presentation.SldMasterIdLst.SldMasterId;
import org.pptx4j.pml.SldMaster;
import org.pptx4j.pml.SlideLayoutIdList;
import org.pptx4j.pml.SlideLayoutIdList.SldLayoutId;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.pptx.components.RootComponent;
import com.openexchange.office.ooxml.pptx.operations.PptxApplyOperationHelper;
import com.openexchange.office.ooxml.pptx.operations.PptxCreateOperationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class PptxOperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    private PresentationMLPackage opcPackage;
    private Theme theme = null;

    public PptxOperationDocument(Session session, ServiceLookup _services, InputStream _inputDocumentStream, IResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(session, _services, _inputDocumentStream, _resourceManager, _documentProperties);
        try {
            opcPackage = (PresentationMLPackage)PresentationMLPackage.load(_inputDocumentStream);
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    // the constructor without inputStream creates an empty document
    public PptxOperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(null, _services, null, null, _documentProperties);
        try {
            opcPackage = PresentationMLPackage.createPackage();
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    // - the target can be "" then a normal slide is addressed
    // or target contains tokens separated by "."
    // - if only one token is available, then a master page is addressed
    // - if two tokens are available, then then first represents the master page
    //   and the second the layout page
    public void setContext(String target) {
    	Part newContext = null;
    	if(target!=null&&!target.isEmpty()) {
        	final MainPresentationPart mainPresentationPart = getPackage().getMainPresentationPart();
    		final Presentation mainPresentation = mainPresentationPart.getJaxbElement();
    		StringTokenizer tokenizer = new StringTokenizer(target,".");
    		while(tokenizer.hasMoreTokens()) {
    			final String masterToken = tokenizer.nextToken();
	    		final SldMasterIdLst slideMasterIds = mainPresentation.getSldMasterIdLst();
				final List<SldMasterId> slideMasterIdList = slideMasterIds.getSldMasterId();
				for(int i=0; i<slideMasterIdList.size(); i++) {
	    			final SldMasterId masterId = slideMasterIdList.get(i);
	    			if(masterToken.equals(masterId.getId().toString())) {
	    				newContext = mainPresentationPart.getRelationshipsPart().getPart(masterId.getRid());
	    				break;
	    			}
	    		}
				if(tokenizer.hasMoreTokens()) {
					final String layoutToken = tokenizer.nextToken();
					// we have to set the layout context
					final SldMaster slideMaster = ((SlideMasterPart)newContext).getJaxbElement();
					final SlideLayoutIdList slideLayoutIds = slideMaster.getSldLayoutIdLst();
					final List<SldLayoutId> slideLayoutIdList = slideLayoutIds.getSldLayoutId();
					for(int i=0; i<slideLayoutIdList.size(); i++) {
						final SldLayoutId slideLayoutId = slideLayoutIdList.get(i);
						if(layoutToken.equals(slideLayoutId.getId().toString())) {
							newContext = newContext.getRelationshipsPart().getPart(slideLayoutId.getRid());
							break;
						}
					}
				}
				break;
    		}
        }
    	setContextPart(newContext);
    }

    @Override 
    public void setContextPart(Part part) {
    	if(contextPart!=part) {
    		contextPart = part;
    		theme = null;
    		themeFonts = null;
    	}
    }

    @Override
    public Part getContextPart() {
    	return contextPart!=null ? contextPart : getPackage().getMainPresentationPart();
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {

        if (applyOperations != null) {
            int i = 0;
            JSONObject op = null;
            String opName = null;
            try {
                final PptxApplyOperationHelper applyOperationHelper = new PptxApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                	successfulAppliedOperations = i;
                    op = (JSONObject) aOperations.get(i);
                    setContext(op.optString("target", ""));
                    opName = op.getString("name");
                    switch(opName) {
                    	case "insertParagraph" : {
	                        applyOperationHelper.insertParagraph(op.getJSONArray("start"), op.optJSONObject("attrs"));
	                        break;
                    	}
                    	case "splitParagraph": {
                    		applyOperationHelper.splitParagraph(op.getJSONArray("start"));
                    		break;
                    	}
                    	case "mergeParagraph": {
                            applyOperationHelper.mergeParagraph(op.getJSONArray("start"));
                            break;
                    	}
                    	case "insertText": {
                            applyOperationHelper.insertText(op.getJSONArray("start"), op.getString("text").replaceAll("\\p{C}", " "), op.optJSONObject("attrs"));
                            break;
                    	}
                    	case "delete": {
                            applyOperationHelper.delete(op.getJSONArray("start"), op.optJSONArray("end"));
                            break;
                    	}
                    	case "setAttributes": {
                            applyOperationHelper.setAttributes(op.getJSONObject("attrs"), op.getJSONArray("start"), op.optJSONArray("end"));
                            break;
                    	}
                    	case "insertSlide": {
                    		applyOperationHelper.insertSlide(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertDrawing": {
                    		applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type").toUpperCase(), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "setDocumentAttributes": {
                    		applyOperationHelper.setDocumentAttributes(op.getJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertHardBreak": {
                            applyOperationHelper.insertHardBreak(op.getJSONArray("start"), op.optJSONObject("attrs"));
                            break;
                    	}
                    	case "changeLayout": {
                    		applyOperationHelper.changeLayout(op.getJSONArray("start"));
                    		break;
                    	}
                    }
                    
/*
                    else if (opName.equals("move"))
                        applyOperationHelper.move(op.getJSONArray("start"), op.optJSONArray("end"), op.getJSONArray("to"));
                    else if (opName.equals("insertTab"))
                        applyOperationHelper.insertTab(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertHardBreak"))
                        applyOperationHelper.insertHardBreak(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertTable"))
                        applyOperationHelper.insertTable(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertRows"))
                        applyOperationHelper.insertRows(op.getJSONArray("start"), op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertCells"))
                        applyOperationHelper.insertCells(op.getJSONArray("start"), op.optInt("count", 1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertColumn"))
                        applyOperationHelper.insertColumn(op.getJSONArray("start"), op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
                    else if (opName.equals("deleteColumns"))
                        applyOperationHelper.deleteColumns(op.getJSONArray("start"), op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
                    else if (opName.equals("setAttributes"))
                        applyOperationHelper.setAttributes(op.getJSONObject("attrs"), op.getJSONArray("start"), op.optJSONArray("end"));
                    else if (opName.equals("insertField"))
                        applyOperationHelper.insertField(op.getJSONArray("start"), op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertDrawing"))
                        applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type").toUpperCase(), op.optJSONObject("attrs"));
                    else if (opName.equals("insertStyleSheet"))
                        applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent"), op.optBoolean("hidden"), op.getInt("uiPriority"), op.optBoolean("default"));
                    else if (opName.equals("deleteStylesheet"))
                        applyOperationHelper.deleteStylesheet(op.getString("type"), op.getString("styleId"));
                    else if (opName.equals("insertListStyle"))
                        applyOperationHelper.insertListStyle(op.getString("listStyleId"), op.getJSONObject("listDefinition"));
*/
                }
                successfulAppliedOperations=aOperations.length();
            }
            catch(Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final JSONObject aOperations = new JSONObject();
        final JSONArray operationsArray = new JSONArray();

        final PptxCreateOperationHelper createOperationHelper = new PptxCreateOperationHelper(this, operationsArray);
        createOperationHelper.createDocumentDefaults(getUserLanguage());
        createOperationHelper.createMasterSlides();
        createOperationHelper.createSlides();
        aOperations.put("operations", operationsArray);
        return aOperations;
    }

    @Override
    public PresentationMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public void setPackage(OpcPackage p) {
       opcPackage = (PresentationMLPackage)p;
    }

    @Override
	public Component getRootComponent() {
    	return new RootComponent(this);
	}

    @Override
    public Theme getTheme(boolean createIfMissing)
    	throws FilterException {

    	try {
	    	if(theme==null) {
	    		Part context = contextPart;
	    		if(context instanceof SlidePart) {
	    			context = ((SlidePart)context).getSlideLayoutPart();
	    		}
	    		if(context instanceof SlideLayoutPart) {
	    			context = ((SlideLayoutPart)context).getSlideMasterPart();
	    		}
	    		final RelationshipsPart masterRelations = context.getRelationshipsPart();
	    		final Relationship rel = masterRelations.getRelationshipByType(Namespaces.THEME);
	    		ThemePart themePart = null;
	    		if(rel!=null) {
	    			themePart = (ThemePart)masterRelations.getPart(rel);
	    		}
	    		if(themePart==null&&createIfMissing) {
	        		final PartName themePartName = new PartName("/ppt/theme/theme1.xml");
		    		themePart = getDefaultThemePart(themePartName);
		    		context.addTargetPart(themePart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
	    		}
	    		theme = themePart!=null?themePart.getJaxbElement():null;
	    	}
	    	return theme;
    	}
    	catch(InvalidFormatException e) {
    		throw new FilterException("xlsx filter, could not create default theme", ErrorCode.CRITICAL_ERROR, e);
    	}
    }
}
