/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.components;

import java.math.BigInteger;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.CTTblGridChange;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblPrChange;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridBase;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TblWidth;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.components.ITable;
import com.openexchange.office.ooxml.docx.DocxOperationDocument;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.TextUtils;
import com.openexchange.office.ooxml.docx.tools.Utils;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class TableComponent extends DocxComponent implements ITable {

    public TableComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
    }
    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

        Component nextComponent = null;
        final IndexedNode<Object> tblNode = getNode();
        final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)tblNode.getData()).getContent();
        for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
            final IndexedNode<Object> childNode = nodeList.getNode(i);
            final Object o = getContentModel(childNode, tblNode.getData());
            if(o instanceof Tr) {
                nextComponent = new TrComponent(this, childNode, nextComponentNumber);
            }
        }
        return nextComponent;
    }
	@Override
	public void applyAttrsFromJSON(com.openexchange.office.ooxml.OperationDocument operationDocument, JSONObject attrs) {
		if(attrs==null) {
			return;
		}
		try {
	        final Tbl tbl = (Tbl)getObject();
    		TblPr tblPr = tbl.getTblPr();
        	if(tblPr==null) {
        		tblPr = Context.getWmlObjectFactory().createTblPr();
        		tblPr.setParent(tbl);
        		tbl.setTblPr(tblPr);
        	}
    		if(attrs.has("styleId")) {
	            Table.applyTableStyle(attrs.getString("styleId"), tblPr);
	        }
    		TblGrid tblGrid = tbl.getTblGrid();
    		if(tblGrid==null) {
    			tblGrid = Context.getWmlObjectFactory().createTblGrid();
    			tblGrid.setParent(tbl);
    			tbl.setTblGrid(tblGrid);
    		}
	        Table.applyTableProperties((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, attrs.optJSONObject("table"), this, tblGrid, tblPr);
        	final Object changes = attrs.opt("changes");
        	if(changes!=null) {
        		boolean removeTableChangeTrack = false;
        		if(changes instanceof JSONObject) {
        			final Object modified = ((JSONObject)changes).opt("modified");
        			if(modified!=null) {
        				if(modified instanceof JSONObject) {
        					CTTblPrChange tblPrChange = tblPr.getTblPrChange();
        					if(tblPrChange==null) {
        						tblPrChange = Context.getWmlObjectFactory().createCTTblPrChange();
        						tblPrChange.setParent(tblPr);
        						tblPr.setTblPrChange(tblPrChange);
        					}
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)modified, tblPrChange);
        					final Object attrsModified = ((JSONObject)modified).opt("attrs");
        					if(attrsModified!=null) {
        						if(attrsModified instanceof JSONObject) {
        							CTTblPrBase tblPrChangePr = tblPrChange.getTblPr();
        							if(tblPrChangePr==null) {
        								tblPrChangePr = Context.getWmlObjectFactory().createCTTblPrBase();
        								tblPrChangePr.setParent(tblPrChange);
        								tblPrChange.setTblPr(tblPrChangePr);
        							}
        							if(((JSONObject)attrsModified).has("styleId")) {
        								Table.applyTableStyle(((JSONObject) attrsModified).getString("styleId"), tblPrChangePr);
        							}
        							final JSONObject tableAttrChanges = ((JSONObject)attrsModified).optJSONObject("table");
        							if(tableAttrChanges!=null) {
        								TblGridBase tblGridBase = null;
        								Object tblGridObj = tableAttrChanges.opt("tableGrid");
        								if(tblGridObj!=null) {
        									if(tblGridObj instanceof JSONArray) {
		        								CTTblGridChange tblGridChange = tblGrid.getTblGridChange();
		        								if(tblGridChange==null&&tableAttrChanges.has("tableGrid")) {
		        									tblGridChange=Context.getWmlObjectFactory().createCTTblGridChange();
		        									tblGridChange.setParent(tblGrid);
		        									tblGrid.setTblGridChange(tblGridChange);
		        								}
		        								tblGridBase = tblGridChange.getTblGrid();
		        								if(tblGridBase==null) {
		        									tblGridBase = Context.getWmlObjectFactory().createTblGridBase();
		        									tblGridBase.setParent(tblGridChange);
		        									tblGridChange.setTblGrid(tblGridBase);
		        								}
        									}
        									else {
        										tblGrid.setTblGridChange(null);
        									}
        								}
        								Table.applyTableProperties((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, tableAttrChanges, null, tblGridBase, tblPrChangePr);
        							}
        						}
        						else {
        							tblPrChange.setTblPr(null);
        						}
        					}
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)attrsModified, tblPrChange);
        				}
        				else {
        					removeTableChangeTrack = true;
        				}
        			}
        			// removed / inserted changetracking has to be applied to each row
        			if(((JSONObject)changes).opt("inserted")!=null||((JSONObject)changes).opt("removed")!=null) {
        				Component trComponent = getNextChildComponent(null, null);
        				while(trComponent!=null) {
        					trComponent.applyAttrsFromJSON(operationDocument, attrs);
        					trComponent = trComponent.getNextComponent();
        				}
        			}
        		}
        		else {
        			removeTableChangeTrack = true;
        		}
        		if(removeTableChangeTrack) {
        			if(tblPr!=null) {
        				tblPr.setTblPrChange(null);
        			}
        			if(tblGrid!=null) {
        				tblGrid.setTblGridChange(null);
        			}
        		}
        	}
		}
		catch(Exception e) {
		}
	}

	@Override
	public JSONObject createJSONAttrs(com.openexchange.office.ooxml.operations.CreateOperationHelper createOperationHelper, JSONObject attrs)
		throws JSONException {

		final Tbl tbl = (Tbl)getObject();
		final DocxOperationDocument operationDocument = (com.openexchange.office.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument();
        final ServiceLookup services = operationDocument.getServiceLookup();
        final Session session = operationDocument.getSession();
        final int maxTableColumns = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableColumns", 15);
        final int maxTableRows = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableRows", 1500);
        final int maxTableCells = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableCells", 1500);

        final TblGrid tableGrid = tbl.getTblGrid();
        final JSONArray tableGridArray = Table.getTableGrid(tableGrid);
        int gridCount = tableGridArray!=null?tableGridArray.length() : 0;
        int columns = gridCount;
        int rows = 0;
        int rowWidth = 0;
        boolean useCellWidth = true;

        Component rowComponent = getNextChildComponent(null, null);
        while(rowComponent!=null) {
            rows++;
            // check, if table cells in the first row have a width defined (in "dxa")
            if (rows == 1) {
                Component cellComponent = rowComponent.getNextChildComponent(null, null);
                while(cellComponent!=null) {
                    TcPr cellProperties = ((Tc)cellComponent.getObject()).getTcPr();
                    if (cellProperties != null) {
                        int cellWidth = 0;
                        TblWidth cellWidthObject = cellProperties.getTcW();
                        if (cellWidthObject != null) {
                            String cellWidthType = cellWidthObject.getType();
                            BigInteger bigInt = cellWidthObject.getW();
                            if (bigInt != null) {
                                cellWidth = bigInt.intValue();
                                if ((cellWidthType != null) && (cellWidthType.equals("dxa"))) {
                                    rowWidth = rowWidth + cellWidth * 2540 / 72 / 20;
                                } else {
                                    useCellWidth = false;
                                }
                            } else {
                                useCellWidth = false;
                            }
                        } else {
                            useCellWidth = false;
                        }
                    } else {
                        useCellWidth = false;
                    }
                    cellComponent = cellComponent.getNextComponent();
                }
            }
            rowComponent = rowComponent.getNextComponent();
        }

        if (useCellWidth == false) {
            rowWidth = 0;  // <- do not use the value
        }

        boolean maxTableSizeExceeded = (columns>maxTableColumns)||(rows>maxTableRows)||((columns*rows)>maxTableCells);

        JSONObject tableChanges = null;
        TblPr tblPr = tbl.getTblPr();
        if(tblPr!=null) {
            // setting the table style that is used...
            TblStyle tblStyle = tblPr.getTblStyle();
            if(tblStyle!=null) {
                String style = tblStyle.getVal();
                if(style!=null&&style.length()>0) {
                    attrs.put("styleId", style);
                }
            }
        }
        final boolean isRootTable = !(tbl.getParent() instanceof Tc);
        Table.createTableProperties(operationDocument.getPackage(), tbl.getTblPr(), tableGridArray, isRootTable, rowWidth, attrs);
        final CTTblPrChange tblPrChange = tbl.getTblPr().getTblPrChange();
        if(tblPrChange!=null) {
        	tableChanges = new JSONObject(2);
        	final JSONObject tableModified = Utils.createJSONFromTrackInfo(operationDocument, tblPrChange);
        	JSONArray tableGridChangeArray = null;
        	if(tableGrid!=null) {
        		final CTTblGridChange tblGridChange = tableGrid.getTblGridChange();
        		if(tblGridChange!=null) {
        			tableGridChangeArray = Table.getTableGrid(tblGridChange.getTblGrid());
        		}
        	}
        	final CTTblPrBase tblChangePr = tblPrChange.getTblPr();
        	if(tblChangePr!=null) {
            	final JSONObject tableChangedAttrs = new JSONObject();
            	final TblStyle tblStyle = tblChangePr.getTblStyle();
            	if(tblStyle!=null) {
            		String style = tblStyle.getVal();
            		if(style!=null&&!style.isEmpty()) {
            			tableChangedAttrs.put("styleId", style);
            		}
            	}
            	Table.createTableProperties(operationDocument.getPackage(), tblPrChange.getTblPr(), tableGridChangeArray, isRootTable, rowWidth, tableChangedAttrs);
        		if(!tableChangedAttrs.isEmpty()) {
        			tableModified.put("attrs", tableChangedAttrs);
        		}
        	}
        	if(!tableModified.isEmpty()) {
        		tableChanges.put("modified", tableModified);
        	}
        	if(!tableChanges.isEmpty()) {
        		attrs.put("changes", tableChanges);
        	}
        }
        JSONObject sizeExceeded = null;
        if(maxTableSizeExceeded) {
            sizeExceeded = new JSONObject();
            sizeExceeded.put("rows", rows);
            sizeExceeded.put("columns", columns);
            attrs.put("sizeExceeded", sizeExceeded);
        }
		return attrs;
	}

	// ITable interfaces
	@Override
	public void insertRows(com.openexchange.office.ooxml.OperationDocument operationDocument, int destinationRowComponent, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
	    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	final TrComponent trReferenceRow = referenceRow!=-1 ? (TrComponent)getChildComponent(referenceRow) : null;
    	final Tbl tbl = (Tbl)getObject();
        final IndexedNodeList<Object> tblContent = tbl.getContent();
        final Component oldTr = getChildComponent(destinationRowComponent);
        IndexedNode<Object> referenceNode = oldTr!=null?oldTr.getNode():null;

        for(int i = 0; i<count; i++) {
            final Tr tr = Context.getWmlObjectFactory().createTr();
            tr.setParent(tbl);
            final IndexedNode<Object> rowNode = new IndexedNode<Object>(tr);
            tblContent.addNode(referenceNode, rowNode, true);
            referenceNode = rowNode;
            if(insertDefaultCells) {
                JSONArray tableGrid = Table.getTableGrid(tbl.getTblGrid());
                if(tableGrid!=null) {
                    for(int j = 0; j<tableGrid.length(); j++) {
                        Tc tc = Context.getWmlObjectFactory().createTc();
                        tc.setParent(tr);
                        tr.getContent().add(tc);
                    }
                }
            }
            else if(trReferenceRow!=null) {       // we try to create the new row from the referenceRow
                final Tr trReference = (Tr)trReferenceRow.getObject();
                final TrPr sourceTrPr = trReference.getTrPr();
                if(sourceTrPr!=null) {
                    final TrPr newTrPr = XmlUtils.deepCopy(sourceTrPr);
                    TextUtils.resetMarkupId(newTrPr);
                    newTrPr.setParent(tr);
                    tr.setTrPr(newTrPr);
                }
                TcComponent tcReferenceComponent = (TcComponent)trReferenceRow.getNextChildComponent(null, null);
                while(tcReferenceComponent!=null) {
                    final Tc tcReference = (Tc)tcReferenceComponent.getObject();
                    final Tc newTc = Context.getWmlObjectFactory().createTc();
                    newTc.setParent(tr);
                    tr.getContent().add(newTc);
                    TcPr tcPrReference = tcReference.getTcPr();
                    if(tcPrReference!=null) {
                        final TcPr newTcPr = XmlUtils.deepCopy(tcPrReference);
                        TextUtils.resetMarkupId(newTcPr);
                        newTcPr.setParent(newTc);
                        newTc.setTcPr(newTcPr);
                    }
                    tcReferenceComponent = (TcComponent)tcReferenceComponent.getNextComponent();
                }
            }
        }
        if(attrs!=null) {
        	Component c = getNextChildComponent(null, null).getComponent(destinationRowComponent);
	        for(int i=0; i<count; i++) {
	        	c.applyAttrsFromJSON(operationDocument, attrs);
	        	c = c.getNextComponent();
	        }
        }
	}

	@Override
	public void splitTable(int rowPosition) {
    	final Tbl sourceTbl = (Tbl)getObject();
    	// to clone the table we temporarily remove the content
    	final IndexedNodeList<Object> sourceTblContent = sourceTbl.getContent();
    	sourceTbl.setContent(null);
    	final Tbl destTbl = XmlUtils.deepCopy(sourceTbl);
    	TextUtils.resetMarkupId(destTbl);
    	sourceTbl.setContent(sourceTblContent);
    	final ContentAccessor sourceTblParent = (ContentAccessor)getParentComponent().getNode().getData();
    	destTbl.setParent(sourceTblParent);
    	((IndexedNodeList<Object>)sourceTblParent.getContent()).addNode(getNode(), new IndexedNode<Object>(destTbl), false);
    	IndexedNodeList.moveNodes(sourceTblContent, destTbl.getContent(), rowPosition, 0, sourceTblContent.size()-rowPosition, destTbl);
	}

	@Override
	public void mergeTable() {
    	final TableComponent sourceTable = (TableComponent)getNextComponent();
    	final IndexedNodeList<Object> sourceContent = ((Tbl)sourceTable.getObject()).getContent();
    	final IndexedNodeList<Object> destContent = ((Tbl)getObject()).getContent();
    	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), getObject());
    	((IndexedNodeList<Object>)((ContentAccessor)((Child)sourceTable.getObject()).getParent()).getContent()).removeNodes(sourceTable.getNode(), null);
	}

	@Override
	public void insertColumn(com.openexchange.office.ooxml.OperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
		throws JSONException {

		boolean before = insertMode.equals("before");
        TrComponent trComponent = (TrComponent)getNextChildComponent(null, null);
        while(trComponent!=null) {
            TcComponent tcReference = (TcComponent)trComponent.getNextChildComponent(null, null);
            TcComponent destination = null;
            while(tcReference!=null) {
                if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                    destination = tcReference;
                    break;
                }
                tcReference = (TcComponent)tcReference.getNextComponent();
            }
            final Tc tc = Context.getWmlObjectFactory().createTc();
            if(tcReference!=null) {
                TcPr referenceTcPr = ((Tc)tcReference.getObject()).getTcPr();
                tc.setParent(trComponent.getObject());
                if(referenceTcPr!=null) {
                    TcPr newTcPr = XmlUtils.deepCopy(referenceTcPr);
                    TextUtils.resetMarkupId(newTcPr);
                    newTcPr.setParent(tc);
                    tc.setTcPr(newTcPr);
                }
            }
            final IndexedNodeList<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
            if(destination==null) {
            	rowContent.add(tc);
            }
            else {
            	// context child can be a SdtRowContext or TcComponent
            	final ComponentContext contextChild = destination.getContextChild(null);
            	rowContent.addNode(contextChild.getNode(), new IndexedNode<Object>(tc), before);
            }
            trComponent = (TrComponent)trComponent.getNextComponent();
        }
        final Tbl tbl = (Tbl)getObject();
        TblGrid tblGrid = tbl.getTblGrid();
        if(tblGrid==null) {
        	tblGrid = Context.getWmlObjectFactory().createTblGrid();
        	tblGrid.setParent(tbl);
        	tbl.setTblGrid(tblGrid);
        }
        Table.setTableGrid((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, this, tblGrid, tableGrid);
	}

	@Override
    public void deleteColumns(com.openexchange.office.ooxml.OperationDocument operationDocument, int gridStart, int gridEnd)
    	throws JSONException {

    	Component trComponent = getNextChildComponent(null, null);
        while(trComponent!=null) {
        	IndexedNode<Object> startNode = null;
        	IndexedNode<Object> endNode = null;
            TcComponent tcComponent = (TcComponent)trComponent.getNextChildComponent(null, null);
            while(tcComponent!=null) {
                if((tcComponent.getGridPosition()>=gridStart&&tcComponent.getGridPosition()<=gridEnd)||
                    (tcComponent.getNextGridPosition()-1>=gridStart&&tcComponent.getNextGridPosition()-1<=gridEnd)){
                	final ComponentContext contextChild = tcComponent.getContextChild(null);
                	if (startNode==null) {
                        startNode = contextChild.getNode();
                	}
                    endNode = contextChild.getNode();
                }
                if(tcComponent.getNextGridPosition()>gridEnd)
                    break;
                tcComponent = (TcComponent)tcComponent.getNextComponent();
            }
            if(startNode!=null) {
                ((Tr)trComponent.getObject()).getContent().removeNodes(startNode, endNode);
            }
            trComponent = trComponent.getNextComponent();
        }
        JSONArray tableGrid = Table.getTableGrid(((Tbl)getObject()).getTblGrid());
        JSONArray newTableGrid = new JSONArray();
        for(int i=0;i<tableGrid.length();i++) {
            if(i<gridStart||i>gridEnd) {
                newTableGrid.put(tableGrid.get(i));
            }
        }
        final Tbl tbl = (Tbl)getObject();
        TblGrid tblGrid = tbl.getTblGrid();
        if(tblGrid==null) {
        	tblGrid = Context.getWmlObjectFactory().createTblGrid();
        	tblGrid.setParent(tbl);
        	tbl.setTblGrid(tblGrid);
        }
        Table.setTableGrid((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, this, tblGrid, newTableGrid);
	}
}
