/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.hazelcast.doc;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.openexchange.exception.OXException;
import com.openexchange.management.ManagementAware;
import com.openexchange.management.ManagementObject;
import com.openexchange.office.hazelcast.access.HazelcastAccess;
import com.openexchange.office.hazelcast.management.HazelcastDocumentDirectoryMBean;
import com.openexchange.office.hazelcast.management.HazelcastDocumentDirectoryManagement;
import com.openexchange.office.hazelcast.serialization.document.PortableDocumentState;
import com.openexchange.office.hazelcast.serialization.document.PortableMemberPredicate;
import com.openexchange.office.hazelcast.serialization.document.PortableRestoreID;
import com.openexchange.office.hazelcast.utils.DocRestoreIDMap;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentState;
import com.openexchange.osgi.ExceptionUtils;



public class HazelcastDocumentDirectory implements DocumentDirectory, ManagementAware<HazelcastDocumentDirectoryMBean> {
    private final Log LOG = com.openexchange.log.Log.loggerFor(HazelcastDocumentDirectory.class);

    private final String mapIdentifier;
    private final HazelcastDocumentDirectoryManagement managementObject;

    public HazelcastDocumentDirectory(final String mapIdentifier) {
        this.mapIdentifier = mapIdentifier;
        this.managementObject = new HazelcastDocumentDirectoryManagement(this);
    }

    @Override
    public DocumentState get(DocRestoreID id) throws OXException {
        DocumentState foundDoc = null;
        PortableRestoreID currPortableRestoreId = new PortableRestoreID(id);

        IMap<PortableRestoreID, PortableDocumentState> allDocStates = getDocStateMapping();
        PortableDocumentState portableDoc = allDocStates.get(currPortableRestoreId);
        if (portableDoc != null) {
            foundDoc = PortableDocumentState.createFrom(portableDoc);
        }

        return foundDoc;
    }

    @Override
    public Map<DocRestoreID, DocumentState> get(Collection<DocRestoreID> ids) throws OXException {
        final Map<DocRestoreID, DocumentState> foundDocs = new HashMap<DocRestoreID, DocumentState>();
        final Set<PortableRestoreID> docResIds = new HashSet<PortableRestoreID>();

        for (DocRestoreID id : ids) {
            docResIds.add(new PortableRestoreID(id));
        }

        if (!docResIds.isEmpty()) {
            IMap<PortableRestoreID, PortableDocumentState> allResources = getDocStateMapping();
            Map<PortableRestoreID, PortableDocumentState> matchingPortableDocs = allResources.getAll(docResIds);
            if (matchingPortableDocs != null) {
                for (Entry<PortableRestoreID, PortableDocumentState> entry : matchingPortableDocs.entrySet()) {
                    final PortableRestoreID foundID = entry.getKey();
                    final PortableDocumentState foundDoc = entry.getValue();
                    final DocRestoreID docResId = PortableRestoreID.createFrom(foundID);
                    foundDocs.put(docResId, PortableDocumentState.createFrom(foundDoc));
                }
            }
        }

        return foundDocs;
    }

    @Override
    public DocumentState set(final DocRestoreID id, final DocumentState docState) throws OXException {
        final PortableDocumentState currentPortableDoc = new PortableDocumentState(docState);
        final PortableRestoreID currentPortableID = new PortableRestoreID(id);

        PortableDocumentState previousPortableDocState = null;
        DocumentState prevDocState = null;
        try {
            final IMap<PortableRestoreID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState != null) {
                // but the previous resource provides a presence
                allDocStates.put(currentPortableID, currentPortableDoc);
            } else {
                previousPortableDocState = allDocStates.putIfAbsent(currentPortableID, currentPortableDoc);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != previousPortableDocState) {
            prevDocState = PortableDocumentState.createFrom(previousPortableDocState);
        }

        return prevDocState;
    }

    @Override
    public void setIfAbsent(DocRestoreID id, DocumentState docState) throws OXException {
        final PortableDocumentState currentPortableDoc = new PortableDocumentState(docState);
        final PortableRestoreID currentPortableID = new PortableRestoreID(id);

        PortableDocumentState previousPortableDocState = null;
        try {
            final IMap<PortableRestoreID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState == null) {
                allDocStates.put(currentPortableID, currentPortableDoc);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }
    }

    @Override
    public DocumentState remove(DocRestoreID id) throws OXException {
        final PortableRestoreID currentPortableID = new PortableRestoreID(id);

        PortableDocumentState previousPortableDocState = null;
        DocumentState prevDocState = null;
        try {
            final IMap<PortableRestoreID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState != null) {
                allDocStates.remove(currentPortableID);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != previousPortableDocState) {
            prevDocState = PortableDocumentState.createFrom(previousPortableDocState);
        }

        return prevDocState;
    }

    @Override
    public Map<DocRestoreID, DocumentState> remove(Collection<DocRestoreID> ids) throws OXException {
        final Map<DocRestoreID, DocumentState> removed = new HashMap<DocRestoreID, DocumentState>();
        final Set<PortableRestoreID> docResIds = new HashSet<PortableRestoreID>();

        for (DocRestoreID id : ids) {
            docResIds.add(new PortableRestoreID(id));
        }

        if (!docResIds.isEmpty()) {
            IMap<PortableRestoreID, PortableDocumentState> allResources = getDocStateMapping();
            Map<PortableRestoreID, PortableDocumentState> matchingPortableDocs = allResources.getAll(docResIds);

            if (matchingPortableDocs != null) {
                for (Entry<PortableRestoreID, PortableDocumentState> entry : matchingPortableDocs.entrySet()) {
                    final PortableRestoreID foundID = entry.getKey();
                    final PortableDocumentState foundDoc = entry.getValue();
                    removed.put(PortableRestoreID.createFrom(foundID), PortableDocumentState.createFrom(foundDoc));

                    allResources.remove(foundID);
                }
            }
        }

        return removed;
    }

    /**
     * Find all Resources in this directory that are located on a given member node.
     *
     * @param member The cluster member
     * @return all Resources in this directory that are located on the given member node.
     * @throws OXException
     */
    public DocRestoreIDMap<DocumentState> getDocumentsOfMember(Member member)throws OXException {
        IMap<PortableRestoreID, PortableDocumentState> allResources = getDocStateMapping();
        PortableMemberPredicate memberPredicate = new PortableMemberPredicate(member);
        Set<Entry<PortableRestoreID, PortableDocumentState>> matchingResources = allResources.entrySet(memberPredicate);

        DocRestoreIDMap<DocumentState> foundIds = new DocRestoreIDMap<DocumentState>();
        Iterator<Entry<PortableRestoreID, PortableDocumentState>> iterator = matchingResources.iterator();
        while(iterator.hasNext()) {
            try {
                Entry<PortableRestoreID, PortableDocumentState> next = iterator.next();

                foundIds.put(PortableRestoreID.createFrom(next.getKey()),
                             PortableDocumentState.createFrom(next.getValue()));
            } catch (Exception e) {
                LOG.error("Couldn't add resource that was found for member " + member, e);
            }
        }
        return foundIds;
    }

    @Override
    public ManagementObject<HazelcastDocumentDirectoryMBean> getManagementObject() {
        return managementObject;
    }

    /**
     * Get the mapping of full IDs to the Resource e.g. ox://marc.arens@premium/random <-> Resource. The resource includes the
     * {@link RoutingInfo} needed to address clients identified by the {@link ID}
     *
     * @return the map used for mapping full IDs to ResourceMaps.
     * @throws OXException if the map couldn't be fetched from hazelcast
     */
    public IMap<PortableRestoreID, PortableDocumentState> getDocStateMapping() throws OXException {
        HazelcastInstance hazelcast = HazelcastAccess.getHazelcastInstance();
        return hazelcast.getMap(mapIdentifier);
    }
}
