/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.message;

import org.apache.commons.lang.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tools.session.ServerSessionAdapter;


public class MessageHelper {

    public final static String KEY_SESSION = "session";
    public final static String KEY_RESOURCE = "resource";


    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(MessageHelper.class);

    /**
     * Retrieves the session object from a stanza. The method
     * tries to use the fromId or JSON request.
     *
     * @param stanza The stanza to be used to extract the session.
     * @param component The component part of the stanza payload to retrieve a session
     * @param path The path part of the stanza payload to retrieve a session
     * @return The session or null, if it cannot be extracted.
     */
    static public ServerSession getServerSession(final Stanza stanza, final String component, final String path) {
        Validate.notEmpty(component, "component must be set!");
        Validate.notEmpty(path, "path must be set!");

        final ID id = stanza.getFrom();
        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(component, path), false);

        return getServerSession(jsonRequest, id);
    }

    /**
     * Retrieves the session object from a JSON request or ID. The method
     * tries to use the JSON request first and if not successful uses the
     * provided ID to retrieve the session. Either of these
     *
     * @param jsonRequest The json object containing the request data or null if not available.
     * @param sessionId The ID of the client requesting a service or null if not available.
     * @return The server session object or null if no session could be
     * retrieved from the jsonRequest nor from the ID.
     */
    static public ServerSession getServerSession(final JSONObject jsonRequest, final ID sessionId) {
        ServerSession serverSession = null;

        // try to get session from json request
        if (null != jsonRequest) {
            try {
                final Object sessionData = jsonRequest.opt(KEY_SESSION);
                if (sessionData instanceof JSONObject) {
                    // normal real-time requests uses a session json object where resource specifies the session-id
                    final JSONObject jsonSession = jsonRequest.getJSONObject(KEY_SESSION);
                    if (null != jsonSession) {
                        final Session session = ServerServiceRegistry.getInstance().getService(SessiondService.class, true).getSession(
                            jsonSession.getString(KEY_RESOURCE));
                        serverSession = ServerSessionAdapter.valueOf(session);
                    }
                } else if (sessionData instanceof String) {
                    // in special cases the session id is directly provided as a string
                    final Session session = ServerServiceRegistry.getInstance().getService(SessiondService.class, true).getSession((String)sessionData);
                    serverSession = ServerSessionAdapter.valueOf(session);
                }
            } catch (Exception e) {
                LOG.error("RT connection: Message handler not able to retrieve server session from json request!", e);
            }
        }

        // try to get session from ID
        if ((null == serverSession)) {
            try {
                serverSession = (null != sessionId) ? sessionId.toSession() : null;
            } catch (Exception e) {
                LOG.error("RT connection: Message handler not able to get server session!", e);
            }
        }

        return serverSession;
    }

    /**
     * Extracts the json request from a stanza using the provided element path.
     *
     * @param stanza The stanza where the json request should be extracted.
     * @param elementPath The path to be use to extract the json request.
     * @param keepActions
     * @return
     *  A json object containing the json request or an empty json object, if
     *  data requested data cannot be found.
     */
    static public JSONObject getJSONRequest(Stanza stanza, ElementPath elementPath, boolean keepActions) {
        final JSONObject jsonRequest = new JSONObject();

        // We're iterating over all messages that are constructed with the office.operations element path
        for (final PayloadTree payload : stanza.getPayloadTrees(elementPath)) {
            final Object data = payload.getRoot().getData();

            if (data instanceof JSONArray) {
                final JSONArray jsonArray = (JSONArray) data;
                final JSONObject jsonObject = new JSONObject();

                try {
                    jsonObject.put(MessagePropertyKey.KEY_ACTIONS, jsonArray);
                } catch (JSONException e) {
                    LOG.error("RT connection: getJSONRequest catches exception trying to to put actions to the JSON object. ", e);
                }

                OperationHelper.appendJSON(jsonRequest, jsonObject, keepActions);
            } else if (data instanceof JSONObject) {
                OperationHelper.appendJSON(jsonRequest, (JSONObject) data, keepActions);
            }
        }

        return jsonRequest;
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     * This function is for special purpose where a method already created
     * a JSON result object with error data. In that case finalize shall not
     * overwrite the first error data.
     *
     * @param jsonRequest
     *  The request to be processed.
     *
     * @param jsonExtra
     *  Extra data to be added to the JSON result object.
     *
     * @return
     *  The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(JSONObject tmpResult, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != tmpResult) {
            try {
                // add unique message id, if given in request
                if ((null != jsonRequest) && jsonRequest.has(MessagePropertyKey.KEY_UNIQUE_ID)) {
                    tmpResult.put(MessagePropertyKey.KEY_UNIQUE_ID, jsonRequest.get(MessagePropertyKey.KEY_UNIQUE_ID));
                }
            } catch (JSONException e) {
                LOG.error("RT connection: getJSfinalizeJSONResult catches exception trying to to put unique id to the JSON object. ", e);
            }

            // add extra data, if set
            if (null != jsonExtra) {
                OperationHelper.appendJSON(tmpResult, jsonExtra);
            }
        }

        return tmpResult;
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param errorCode The result of the request as error code.
     * @param jsonRequest The request to be processed.
     * @param jsonExtra Extra data to be added to the JSON result object.
     *
     * @return The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(ErrorCode errorCode, JSONObject jsonRequest, JSONObject jsonExtra) {
        JSONObject jsonResult = new JSONObject();
        return finalizeJSONResult(jsonResult, errorCode, jsonRequest, jsonExtra);
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param errorCode The result of the request as error code.
     * @return The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(final ErrorCode errorCode) {
        JSONObject jsonResult = new JSONObject();
        return finalizeJSONResult(jsonResult, errorCode, null, null);
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param tmpResult
     *  A json object which is used to merge the following data into it.
     *
     * @param errorCode
     *  The result of the request as error code.
     *
     * @param jsonRequest
     *  The request to be processed.
     *
     * @param jsonExtra
     *  Extra data to be added to the JSON result object.
     *
     * @return
     *  The final JSON result object to be sent to the client-side. The object
     *  is the same as tmpResult extended with the properties provided as
     *  arguments.
     */
    static public JSONObject finalizeJSONResult(JSONObject tmpResult, ErrorCode errorCode, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != tmpResult) {
            try {
                // add unique message id, if given in request
                if ((null != jsonRequest) && jsonRequest.has(MessagePropertyKey.KEY_UNIQUE_ID)) {
                    tmpResult.put(MessagePropertyKey.KEY_UNIQUE_ID, jsonRequest.get(MessagePropertyKey.KEY_UNIQUE_ID));
                }
                tmpResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
            } catch (JSONException e) {
                LOG.error("RT connection: getJSfinalizeJSONResult catches exception trying to to put unique id to the JSON object. ", e);
            }

            // add extra data, if set
            if (null != jsonExtra) {
                OperationHelper.appendJSON(tmpResult, jsonExtra);
            }
        }

        return tmpResult;
    }

    /**
     * Checks the json result object is empty and sets the "hasErrors"
     * property to indicate an errornous state.
     *
     * @param jsonResult
     *  A json object containing the result of a request which should be
     *  checked.
     *
     * @return
     *  The jsonResult which contains a "hasError" property if it doesn't
     *  contain any properties.
     */
    static public JSONObject checkJSONResult(JSONObject jsonResult) {
        if ((null != jsonResult) && jsonResult.isEmpty()) {
            try {
                jsonResult.put(MessagePropertyKey.KEY_HAS_ERRORS, true);
            } catch (JSONException e) {
                LOG.error("RT connection: checkJSONResult catches exception trying to to put hasErrors to the JSON object. ", e);
            }
        }

        return jsonResult;
    }

    /**
     * Retrieves the sender ID from stanza.
     *
     * @param stanza The stanza provided by the Realtime framework that should
     * be used to retrieve the sender ID from.
     *
     * @return
     *  The id of the user who sent the request. If the request was done synchronously this is
     *  the "real" ID and not the synthetic.
     *  ATTENTION: In that case the fromID must be used for the answer message.setTo() method!
     */
    static public ID getIDFromStanza(Stanza stanza) {
        ID id = null;
        if (null != stanza) {
            id = stanza.getOnBehalfOf();
            if (null == id) {
                id = stanza.getFrom();
            }
        }

        return id;
    }

}
