/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.realtime.impl.calc;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.OXException;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineClipBoard;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.realtime.doc.DocumentEventHelper;
import com.openexchange.office.realtime.doc.FilterExceptionToErrorCode;
import com.openexchange.office.realtime.doc.ImExportHelper;
import com.openexchange.office.realtime.doc.OXDocument;
import com.openexchange.office.realtime.impl.ConnectionStatus;
import com.openexchange.office.realtime.impl.DocumentConnection;
import com.openexchange.office.realtime.impl.UserData;
import com.openexchange.office.realtime.log.DocContextLogger;
import com.openexchange.office.realtime.tools.DebugHelper;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.realtime.tools.SystemInfoHelper;
import com.openexchange.office.tools.actions.ActionParameters;
import com.openexchange.office.tools.doc.DocumentMetaData;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.message.MessageChunk;
import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;


/**
 * Spreadsheet-specific implementation of a DocumentConection. This class provides methods
 * to handle the spreadsheet-specific requests, loading/saving spreadsheet documents.
 *
 * @author Carsten Driesner
 * @since 7.6.2
 */
public class CalcConnection extends DocumentConnection implements IRequestProcessor {
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(CalcConnection.class);
    private final static int MAX_OPERATIONS = 1000;
    private final static ActionHandler m_handler = new ActionHandler(CalcConnection.class);

    private boolean m_asyncSpreadsheetLoaded = false;
    private ICalcEngineClient m_calcClient = null;
    private String m_calcDocumentHandle;
    private final AsyncLoadRequestQueue m_loadRequestQueue;
    private JSONObject m_uiOperations = null;

    public CalcConnection(final ServiceLookup serviceLookup, final ID id, final AsyncLoadRequestQueue queue, final String componentID) {
        super(serviceLookup, id, m_handler, componentID);
        m_loadRequestQueue = queue;
    }

    /**
     * Determines if the asynchronous loading has been completed or not.
     *
     * @return TRUE if the asynchronous loading has been completed otherwise FALSE.
     */
    synchronized public boolean isAsyncSpreadsheetLoaded() {
        return m_asyncSpreadsheetLoaded;
    }

    /**
     * Sets the asynchronous loading state.
     *
     * @param set Sets the asynchronous loading state accordingly.
     */
    synchronized void setAsyncSpreadsheetLoaded(boolean set) {
        m_asyncSpreadsheetLoaded = set;
    }

    /**
     * Provides a stanza for the first message sent to the client joining a document (chat-room). The OX Documents client expects to find
     * the document operation within the stanza provided by this method.
     *
     * @param onBehalfOf The ID of the client which wants to join the document.
     * @return The stanza sent to the client which wants to join the document. The stanza includes the document operations and operations
     *         which are currently pending for the next save. The stanza also has the update state including the editor ID.
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(final ID onBehalfOf) {
        final ID fromId = getId();
        final DocFileHelper fileHelper = getFileHelper();
        Message welcomeMessage = null;

        if (isValidID(onBehalfOf) && (null != fileHelper)) {
            final UserData userData = this.getUserData(onBehalfOf);
            final ServerSession serverSession = MessageHelper.getServerSession(null, onBehalfOf);

            LOG.debug("RT connection: [getWelcomeMessge] called for calc document: " + DebugHelper.getDocumentFolderAndFileId(userData) + ", identity: " + getIdentity());

            // Check, if we can retrieve the session from the join request. Include using the session id
            // which cannot be processed correctly in some cases, if the session storage hazelcast bundle
            // is not started!!
            if ((null == serverSession) || (!isSessionIdAccessbile(userData.getSessionId()))) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR, userData.getFolderId(), userData.getFileId());
            }

            // Check global save state - stop and provide error message, if global save is set!
            // This prevents the user to edit a document which is saved by background save which
            // lasts longer. The user wouldn't be able to save it on close
            if (this.getGlobalSaveState()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_BACKGROUNDSAVE_IN_PROGRESS_ERROR, userData.getFolderId(), userData.getFileId());
            }

            // Check for the document type and optimize access in case the document has been loaded successfully
            if ((fileHelper != null) && (m_calcDocumentHandle != null)) {
                DocumentMetaData currentMetaData = null;
                synchronized (getSyncAccess()) {
                    // we need to make a clone to set the write protected state for every client separately
                    currentMetaData = getLastKnownMetaData().clone();
                }

                // Optimizing the spreadsheet access for the second and other upcoming clients
                // No need to access the document and load it more than once.
                boolean writeAccess = FileHelper.canWriteToFile(getServices(), serverSession, serverSession.getUserId(), userData.getFileId());
                currentMetaData.setWriteProtectedState(!writeAccess);

                if (isAsyncSpreadsheetLoaded()) {
                    // In this case we already have finished loading and can therefore handle
                    // this by just sending operations synchronously
                    impl_handleEditRights(onBehalfOf, writeAccess);
                    welcomeMessage = impl_createWelcomeMessage(onBehalfOf, m_uiOperations, -1, false, null, -1, true, currentMetaData);
                    return welcomeMessage;
                } else {
                    // Loading of the document into the CalcEngine is in process, therefore
                    // we have to sync us with the first client that initiated the loading.
                    m_loadRequestQueue.putRequest(new Request(getResourceID(), onBehalfOf, this, -1, currentMetaData));
                    welcomeMessage = impl_createAsyncLoadWelcomeMessage(fromId, onBehalfOf, null, 0);
                    return welcomeMessage;
                }
            }

            final OXDocument oxDocument = new OXDocument(serverSession, getServices(), userData, fileHelper, isNewDocLoaded(), getResourceManager(), getStorageHelper(), isDebugOperations(), null);
            final int documentOSN = oxDocument.getUniqueDocumentId();
            final JSONObject operations = new JSONObject();

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            ErrorCode errorCode = oxDocument.getLastError();

            // check error code after loading the document
            if (errorCode.isError()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
            }

            // Set current meta data for the document
            synchronized (getSyncAccess()) {
                setLastKnownMetaData(oxDocument.getDocumentMetaData());
            }

            // must be called after the document meta data have been set
            boolean writeAccess = !oxDocument.getDocumentMetaData().getWriteProtectedState();
            impl_handleEditRights(onBehalfOf, writeAccess);

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            final JSONObject importerProps = new JSONObject();
            JSONObject previewData = null;

            // check error code before continuing to retrieve operations
            if (errorCode.getCode() == ErrorCode.NO_ERROR.getCode()) {
                // Initialize result object with operations from persistent document first.
                // Check for FilterExceptions that can occur due to problems with the file.
                try {
                    final IImporter importer = ImExportHelper.getImporter(getServices(), getFileHelper());
                    this.impl_setMemoryImporterProperties(importerProps);

                    if (OXDocument.supportsPreview(importer)) {
                        // If preview is supported use it to have a better user experience
                        oxDocument.preparePreviewOperations(importer, importerProps);
                        previewData = oxDocument.getPreviewOperations();
                    } else {
                        // load the whole document as a fall back for missing preview capabilities
                        final JSONObject documentOperations = oxDocument.getOperations(importer, importerProps, null);
                        OperationHelper.appendJSON(operations, documentOperations);
                    }
                } catch (final FilterException e) {
                    final String docUserData = DocContextLogger.getContextStringWithDoc(onBehalfOf.toString(), userData, oxDocument.getDocumentMetaData().getFileName());
                    if (isDebugOperations()) {
                        LOG.error("RT connection: getWelcomeMessage catched filter exception, context: " + docUserData, e);
                    } else {
                        LOG.error("RT connection: getWelcomeMessage catched filter exception", e);
                    }

                    errorCode = FilterExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR);
                    if ((e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_TOO_HIGH) || (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED)) {
                        // try to figure what happened with the memory usage of the filter as
                        // we want to provide two different error codes.
                        errorCode = ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR;
                        if (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED) {
                            final SystemInfoHelper.MemoryInfo memInfo = SystemInfoHelper.getMemoryInfo();
                            final String freeMemString = (memInfo == null) ? "unknown" : ((Long) ((memInfo.maxHeapSize - memInfo.usedHeapSize) / SystemInfoHelper.MBYTES)).toString();
                            LOG.info("Document could not be loaded, because current available heap space is not sufficient. Current free heap space: " + freeMemString + " MB");
                        }
                    }
                } catch (final Exception e) {
                    errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                    LOG.error("RT connection: getWelcomeMessage, Exception catched", e);
                }
            }

            // check error code again as the filter and other parts can throw errors, too
            if (errorCode.isError()) {
                welcomeMessage = impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
                return welcomeMessage;
            }

            if (m_calcDocumentHandle == null && (operations.has("operations") || (null != previewData))) {
                final CalcEngineClientFactory calcFactory = getServices().getService(CalcEngineClientFactory.class);

                try {
                    LOG.debug("RT connection: going to create calc client and document, identity: " + getIdentity());
                    m_calcClient = calcFactory.get();
                    final String calcDocHandle = CalcEngineHandleGenerator.newHandle();
                    m_calcClient.createDocument(calcDocHandle);
                    m_calcDocumentHandle = calcDocHandle;

                    LOG.debug("RT connection: context[file:" + DebugHelper.getDocumentFolderAndFileId(userData) + ",doc-handle:" + m_calcDocumentHandle + "] calc document created, identity: " + getIdentity());

                    // return a asynchronous loading welcome message
                    boolean singleSheet = false;
                    JSONArray uiOperations = null;
                    JSONArray previewOperations = null;
                    JSONArray opArray = null;

                    if (operations.has("operations")) {
                        opArray = operations.getJSONArray("operations");
                    } else if (null != previewData) {
                        previewOperations = previewData.optJSONArray("operations");
                        singleSheet = (oxDocument.getSheetCount() == 1);
                    }

                    if (null != previewOperations) {
                        errorCode = impl_processPreviewRequest(fromId, onBehalfOf, previewOperations, singleSheet);
                        if (errorCode.isError()) {
                            return impl_createErrorMessage4WelcomeMessage(onBehalfOf, errorCode, getConnectionStatusClone());
                        }

                        synchronized (getSyncAccess()) {
                            // Create a copy of the ui operations array, so we can be sure that
                            // we don't see changes by the parallel running thread loading the
                            // remaining sheets.
                            uiOperations = JSONHelper.shallowCopy(m_uiOperations.getJSONArray("operations"));
                        }
                    }

                    // create request and process load request in the calc load request queue asynchronously
                    if ((null != opArray) && (opArray.length() > 0)) {
                        // In case we don't have preview operations we have to process
                        // the whole operations asynchronously. The welcome message just
                        // provide the info that the loading is done asynchronously.
                        final Request request = new Request(this.getResourceID(), onBehalfOf, opArray, this, documentOSN, oxDocument.getDocumentMetaData());
                        m_loadRequestQueue.putRequest(request);
                        welcomeMessage = impl_createAsyncLoadWelcomeMessage(getId(), onBehalfOf, uiOperations, -1);
                    } else if (null != uiOperations) {
                        if (singleSheet) {
                            // In case we just have one sheet and loaded the document synchronously
                            // we are done and return the final ui operations.
                            this.setAsyncSpreadsheetLoaded(true);
                            welcomeMessage = impl_createWelcomeMessage(onBehalfOf, m_uiOperations, -1, false, null, -1, true, oxDocument.getDocumentMetaData());
                        } else {
                            // In case we have preview operations we provide the OXDocument instance
                            // to complete the loading of the remaining sheets in a background thread.
                            final int activeSheetIndex = oxDocument.getActiveSheetIndex();
                            final Request request = new Request(this.getResourceID(), onBehalfOf, oxDocument, this, documentOSN, uiOperations.length());
                            m_loadRequestQueue.putRequest(request);
                            welcomeMessage = impl_createAsyncLoadWelcomeMessage(getId(), onBehalfOf, uiOperations, activeSheetIndex);
                        }
                    } else {
                        LOG.error("RT connection: Impossible case trying to process spreadsheet request");
                        welcomeMessage = impl_createErrorMessage4WelcomeMessage(onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR, getConnectionStatusClone());
                        return welcomeMessage;
                    }
                } catch (Throwable t) {
                    // handles exception from calc init/load
                    return impl_handlExceptionForInitCalc(t, onBehalfOf);
                }
            }
        } else {
            DocumentEventHelper.addOpenErrorEvent(fileHelper);
            LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return welcomeMessage;
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework.
     * This method is called when the last client left the document and the
     * Realtime-Framework wants to dispose the GroupDispatcher instance.
     * Attention: If getSignOffMessage or onLeave is marked with the annotation
     * "Asynchronous" onDispose is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id The ID of the last client that left the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        LOG.debug("RT connection: onDispose called for calc document: " + DebugHelper.getDocumentFolderAndFileId(null, this.getResourceID()) + ", identity: " + getIdentity());

        try {
            final UserData userData = this.getUserData(id);
            // Use a try/catch throwable block to prevent throwing low-level
            // exceptions. Make the most important calls first!
            this.m_loadRequestQueue.purgeDocumentRequests(this.getResourceID());

            if (isSaveOnDispose()) {
                flushDocument(MessageHelper.getServerSession(null, id), userData, true, true);
            }

            impl_resetConnectionStatusOnDispose();
        } catch (Throwable e) {
            LOG.error("RT connection: Exception while cleanup connection instance", e);
        } finally {
            // Special handling destroying the calcengine document. In
            // case of an error this can throw low-level exceptions.
            if (m_calcClient != null) {
                ICalcEngineClient calcClient = m_calcClient;
                String calcDocHandle = m_calcDocumentHandle;
                m_calcClient = null;
                m_calcDocumentHandle = null;

                try {
                    LOG.trace("RT connection: destroyDocment for calc document " + ((null != calcDocHandle) ? calcDocHandle : "null") + ", identity: " + getIdentity());
                    if (calcDocHandle != null)
                        calcClient.destroyDocument(calcDocHandle);
                } catch (Throwable e) {
                    LOG.error("RT connection: Error deleting spreadsheet document " + ((null != calcDocHandle) ? calcDocHandle : "null") + ", identity: " + getIdentity(), e);
                }
                try {
                    LOG.trace("RT connection: dispose on calc client " + ((null != calcDocHandle) ? calcDocHandle : "unknown handle") + ", identity: " + getIdentity());
                    calcClient.dispose();
                } catch (Throwable e) {
                    LOG.error("RT connection: Error disposing spreadsheet client " + ((null != calcDocHandle) ? calcDocHandle : "null") + ", identity: " + getIdentity(), e);
                }
            } else {
                LOG.debug("RT connection: Calc client instance is null, identity: " + getIdentity());
            }

            synchronized (getSyncAccess()) {
                // Reset members which can reference big memory elements as RT-core
                // references our Connection object even after calling onDispose().
                m_uiOperations = null;
            }
            super.onDispose(id);
        }
    }

    /**
     * Handles the request to apply new actions to the document. The method controls that the sender has the edit rights and it makes some
     * basic operation checking to protected the document consistency.
     *
     * @notice Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     *         element: "message", payloads: [{ element: "action", data: "applyactions" }, { namespace: getComponentID(), element: "actions", data:
     *         "[{operations: [...]}, ...]" }] }
     * @param stanza A Stanza which contain the operations to be applied to the current document.
     * @return The message
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleApplyActions");

        // Check that we have a ID for the sender stanza and that the sender has edit rights.
        final ID fromId = stanza.getFrom();
        Message returnMessage = new Message();

        if (this.isValidMemberAndEditor(fromId)) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza,
                new ElementPath(getComponentID(), MessagePropertyKey.KEY_ACTIONS), true);

            returnMessage = impl_applyActions(fromId, MessageHelper.getServerSession(jsonRequest, fromId), jsonRequest);
        }

        return returnMessage;
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "copy" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopy(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleCopy");

        final ID fromId = MessageHelper.getIDFromStanza(stanza);
        if (isValidInternalID(fromId)) {
            LOG.debug("RT connection sending [copy] result to: " + stanza.getFrom().toString());

            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
            final Session session = MessageHelper.getServerSession(jsonRequest, fromId);

            try {
                JSONObject jsonResult = new JSONObject();
                if (m_calcClient != null) {
                    try {
                        CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                        event.sClipboardHandle = CalcEngineClipBoard.newHandle();
                        event.sUserSession = jsonRequest.getJSONObject("session").getString("resource");
                        JSONObject sourceSelection = new JSONObject();
                        sourceSelection.put("sheet", jsonRequest.get("sheet"));
                        sourceSelection.put("start", jsonRequest.get("start"));
                        if( jsonRequest.has("end")) {
                            sourceSelection.put("end", jsonRequest.get("end"));
                        }
                        event.sSourceSelectionJSON = sourceSelection.toString();
                        executeSpreadsheetCopyFailSafe(fromId, event);
                        jsonResult.put("handle", event.sClipboardHandle);
                    } catch (CalcEngineException e) {
                        LOG.error("Exception on handleQuery", e);
                        impl_handleCalcEngineError(session, e.getErrorcode());
                        throw new OXException(e);
                    }
                }

                final Message returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(
                        MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json",
                        getComponentID(),
                        "spreadsheet")).build()));

                // send the message to the one who requested the copy-request
                send(returnMessage);

                // update all Clients with the result, too
                updateClients(new MessageData(null, getConnectionStatusClone(), jsonResult), null);
            } catch (Exception e) {
                LOG.error("Exception on handleCopy()", e);
            }
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "paste" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handlePaste(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handlePaste");

        // We need to get a real ID from the stanza to check the edit rights
        final Message returnMessage = new Message();
        final ID fromId = retrieveRealID(stanza);
        if (this.isValidMemberAndEditor(fromId)) {
            LOG.debug("RT connection: Handling [paste], called from: " + fromId.toString());

            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
            final ServerSession session = MessageHelper.getServerSession(jsonRequest, fromId);
            JSONObject pasteAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject paste = new JSONObject();

            try {
                paste.put("name", "paste");
                paste.put("UserSession", jsonRequest.getJSONObject("session").getString("resource"));
                JSONObject targetSelection = new JSONObject();
                targetSelection.put("sheet", jsonRequest.get("sheet"));
                targetSelection.put("start", jsonRequest.get("start"));
                if( jsonRequest.has("end")) {
                    targetSelection.put("end", jsonRequest.get("end"));
                }
                paste.put("TargetSelection", targetSelection );
                paste.put("ClipboardHandle", jsonRequest.getString("handle").toString());
                paste.put("osn", getConnectionStatusClone().getOperationStateNumber());
                paste.put("opl", 1);
                operations.put(0, paste);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                pasteAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.error("RT connection: handlePaste catches exception while creating JSON actions object. ", e);
            }

            JSONObject jsonResult = new JSONObject();
            final Message applyMessage = impl_applyActions(fromId, session, pasteAction);

            ElementPath path = new ElementPath(getComponentID(), "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by impl_applyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json", getComponentID(), "spreadsheet")).build()));
            send(returnMessage);
        } else {
            // send immediately an answer to the requesting client
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR, null, null),
                    "json", getComponentID(), "spreadsheet")).build()));
            send(returnMessage);

            // send a hang-up as the client sent a paste-request without "edit" rights
            try {
                sendHangUp(fromId,
                    getConnectionStatusClone(),
                    ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR,
                    HangUpReceiver.HANGUP_ID, "Relaying [hangup] to client trying to paste without edit rights");
            } catch (Exception e) {
                LOG.error("RT connection: Exception while sending [hangup] to editor client due to inconsistent OSN", e);
            }
        }
    }

    /**
     * Stanzas layout: UpdateView { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "updateview" }, { namespace: getComponentID(), element: "spreadsheet", data: "{... : ...,}"
     * }] }
     *
     * @param stanza
     * @throws OXException
     */
    public Boolean handleUpdateView(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleUpdateView");

        Message returnMessage = null;
        final ID fromId = stanza.getFrom();
        if (isValidInternalID(fromId)) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getSpreadsheetViewUpdate(fromId, serverSession, jsonRequest);
            if (jsonResult != null && !jsonResult.has("error")) {
                returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(
                        MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json", getComponentID(), "spreadsheet")).build()));

                // send the message to the one who requested to rename the document
                send(returnMessage);

                LOG.debug("RT connection: Relaying [updateView] result to: " + stanza.getFrom().toString());

                // update all Clients with the result, too
                updateClients(new MessageData(null, getConnectionStatusClone(), jsonResult), null);
            }
        }
        return new Boolean(returnMessage != null);
    }

    /**
     * Creates a welcome message in case of an exception while trying to
     * create a new calc engine instance/load operations into a calc engine
     * instance. ATTENTION: This should only be called from getWelcomeMessage().
     *
     * @param t the exception thrown while trying to create a new calc engine instance/load doc
     * @param onBehalfOf the id of the user joining the document
     * @return a welcome message containing the correct error code
     */
    private Message impl_handlExceptionForInitCalc(final Throwable t, final ID onBehalfOf) {
        Message welcomeMessage = null;
        ErrorCode errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;

        LOG.error("RT connection: Error creating calc engine instance/loading document into calc engine, exception catched!", t);
        if ((t instanceof NoClassDefFoundError) ||
            (t instanceof UnsatisfiedLinkError) ||
            (t instanceof IllegalArgumentException)) {
            errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
        }

        welcomeMessage = impl_createErrorMessage4WelcomeMessage(onBehalfOf, errorCode, getConnectionStatusClone());
        return welcomeMessage;
    }

    /**
     * Processes a "applyaction" request from a client. Must be called from
     * handle-methods, when they need applyActions support. Validation
     * checking must be done by the caller!
     *
     * @param fromId
     * @param session
     * @return
     */
    private Message impl_applyActions(final ID fromId, final ServerSession session, final JSONObject jsonRequest) throws OXException {
        final Message returnMessage = new Message();

        MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
        MessageChunk uiActionChunk = null;
        JSONObject jsonResult = new JSONObject();
        boolean emptyActionChunk = false;

        // increase number of received actions
        this.incActionsReceived();

        LOG.debug("RT connection: [applyActions] applied from: " + fromId.toString());

        // Server must check the consistency of the operations sent by the current editor client.
        // If check fails the client must receive a 'hangup' message and be excluded from the
        // document group.
        final ErrorCode checkOpErr = OperationHelper.hasValidOperations(actionChunk, getConnectionStatusClone().getOperationStateNumber());
        if (checkOpErr.isError()) {
            try {
                // send hangup message and ignore operations
                sendHangUp(fromId, getConnectionStatusClone(), checkOpErr, HangUpReceiver.HANGUP_ID,
                           "sent hangup due to incorrect osn/opl in message chunk.");
            } catch (Exception e) {
                LOG.error("RT connection: Exception while sending [hangup] to editor client due to invalid operations", e);
            }
            if (impl_resetEditor(fromId)) {
                // also send a update message with the new/reseted editor to all clients
                updateClients(new MessageData(null, getConnectionStatusClone()), fromId);
            }
            return returnMessage;
        }

        boolean isPaste = false;
        boolean isSort = false;
        // update spreadsheet document
        JSONArray senderOperations = null;
        if (m_calcClient != null && actionChunk.getOperations().has(MessagePropertyKey.KEY_ACTIONS)) {
            try {
                JSONObject firstSourceOp = new JSONObject();
                int sourceOperationLength = 0;
                JSONArray chunkActions = actionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                for (int chunkAction = 0; chunkAction < chunkActions.length(); ++chunkAction) {
                    JSONObject localJsonResult = new JSONObject();
                    JSONObject sheetOperations = chunkActions.getJSONObject(chunkAction);
                    JSONArray operations = sheetOperations.getJSONArray("operations");
                    sourceOperationLength += operations.length();
                    JSONObject firstChunkOperation = operations.getJSONObject(0);
                    // put locale into the first operation of _all_ chunks
                    if (!firstChunkOperation.has("parse")) {
                        String userLanguage = session.getUser().getPreferredLanguage();
                        firstChunkOperation.put("parse", userLanguage);
                    }

                    if (chunkAction == 0) {
                        firstSourceOp = firstChunkOperation;
                    }
                    if(operations.length() == 1) {
                        String opName = operations.getJSONObject(0).getString("name");
                        if(opName.equals("paste")){
                            isPaste = opName.equals("paste");
                        } else if(opName.equals("sort")) {
                            isSort = true;
                        }
                    }
                    if (isPaste) {
                        JSONObject pasteOp = operations.getJSONObject(0);
                        CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                        event.sUserSession = pasteOp.getString("UserSession");
                        event.sTargetSelectionJSON = pasteOp.getString("TargetSelection");
                        event.sClipboardHandle = pasteOp.getString("ClipboardHandle");
                        executeSpreadsheetPasteFailSafe(fromId, event);
                        localJsonResult = new JSONObject(event.sPasteOperationsJSON);
                    } else {
                        localJsonResult = executeSpreadsheetOperationFailsafe(fromId, sheetOperations.toString());
                    }
                    if (chunkAction == 0) {
                        jsonResult = localJsonResult;
                    } else {
                        // add result elements to jsonResult
                        if (localJsonResult.has("undoCount")) {
                            jsonResult.put("undoCount", localJsonResult.get("undoCount"));
                        }
                        if (localJsonResult.has("redoCount")) {
                            jsonResult.put("redoCount", localJsonResult.get("redoCount"));
                        }
                        mergeChanged(jsonResult, localJsonResult);
                    }
                }
                LOG.debug(jsonResult.toString());
                if (jsonResult.has("changedOperations")) {
                    JSONObject actions = new JSONObject();
                    JSONObject changedOperations = jsonResult.getJSONObject("changedOperations");
                    JSONArray changedOpsArray = new JSONArray();
                    JSONArray ops = changedOperations.optJSONArray("operations");

                    emptyActionChunk = (ops == null || ops.length() == 0);
                    changedOpsArray.put(0, changedOperations);
                    actions.put(MessagePropertyKey.KEY_ACTIONS, changedOpsArray);
                    actionChunk = new MessageChunk(actions, actionChunk.getSender());
                    jsonResult.remove("changedOperations");
                }
                if (jsonResult.has("senderOperations")) {
                    senderOperations = jsonResult.getJSONArray("senderOperations");
                }
                JSONObject actions = new JSONObject();
                JSONArray uiActionsArray = new JSONArray();
                JSONObject uiOperations = new JSONObject();
                if (jsonResult.has("uiOperations")) {
                    JSONArray uiOperationsArray = jsonResult.getJSONArray("uiOperations");
                    int osnDiff = sourceOperationLength - uiOperationsArray.length();
                    int opState = getConnectionStatusClone().getOperationStateNumber();
                    for (int op = 0; op < uiOperationsArray.length(); ++op) {
                        uiOperationsArray.getJSONObject(op).put("opl", 1).put("osn", opState + op);
                    }
                    if (osnDiff > 0) {
                        uiOperationsArray.getJSONObject(uiOperationsArray.length() - 1).put("opl", osnDiff + 1);
                    }
                    uiOperations.put("operations", uiOperationsArray);
                } else {
                    // create a dummy operation to align osl/opn
                    int osn = firstSourceOp.getInt("osn");
                    JSONObject dummyOp = new JSONObject();
                    dummyOp.put("name", "noOp");
                    dummyOp.put("osn", osn);
                    dummyOp.put("opl", sourceOperationLength);
                    JSONArray operationsArray = new JSONArray();
                    operationsArray.put(0, dummyOp);
                    uiOperations.put("operations", operationsArray);
                }
                uiActionsArray.put(0, uiOperations);
                actions.put(MessagePropertyKey.KEY_ACTIONS, uiActionsArray);
                uiActionChunk = new MessageChunk(actions, fromId);
            } catch (JSONException e) {
                LOG.debug("Error executing spreadsheet operation", e);
            } catch (CalcEngineException e) {
                LOG.error("RT connection: Error executing spreadsheet operation. Calc engine worker notifies problems", e);
                impl_handleCalcEngineError(session, e.getErrorcode());
                throw new OXException(e);
            } catch (Exception e) {
                LOG.error("RT connection: Error executing spreadsheet operation", e);
                throw new OXException(e);
            }
        }

        // handling operation state number for operations from clients
        int serverOSN = this.getConnectionStatusClone().getOperationStateNumber();
        try {
            MessageChunk osnActionChunk = uiActionChunk != null ? uiActionChunk : actionChunk;
            if (osnActionChunk.getOperations().hasAndNotNull(MessagePropertyKey.KEY_ACTIONS)) {
                JSONArray allActions = osnActionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                if (lastAction.has("operations")) {
                    JSONArray allOps = lastAction.getJSONArray("operations");
                    JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                    // update statistics with incoming operations requests
                    DocumentEventHelper.addIncomingEvent(getFileHelper());

                    if (lastOp.has("osn") && lastOp.has("opl")) {
                        int osn = -1;
                        synchronized (getSyncAccess()) {
                            osn = lastOp.getInt("osn") + lastOp.getInt("opl");
                            getConnectionStatus().setOperationStateNumber(osn);
                            serverOSN = osn;
                        }
                        LOG.debug("RT connection: handleApplyActions, lastOp received = " + OperationHelper.operationToString(lastOp));
                        LOG.debug("RT connection: handleApplyActions, current server osn = " + osn);
                    }
                }
            }
        } catch (JSONException e) {
            LOG.error("RT connection: handleApplyActions: Error setting operation state number to operation", e);
        }

        // update the shared state
        int msgChunkListSize = 0;
        synchronized (getSyncAccess()) {
            if (!emptyActionChunk) {
                addMessageChunkToQueue(actionChunk);
            }
            if (uiActionChunk != null) {
                OperationHelper.appendPackedMessageChunk(m_uiOperations, senderOperations, uiActionChunk, serverOSN);
            }
            msgChunkListSize = getMessageQueueCount();
        }

        LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));

        final MessageData messageData = new MessageData(
            uiActionChunk != null ? uiActionChunk : actionChunk,
                getConnectionStatusClone(),
                MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null));

        // Send the message to all participants (excluding the one who sent it originally)
        returnMessage.setFrom(fromId);
        returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "update")).build()));

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
        }

        if (isPaste||isSort) {
            relayToAll(returnMessage);
        } else {
            relayToAllExceptSender(returnMessage);
        }

        // update statistics with distributed operations count
        int otherClientCount = 0;

        final ConnectionStatus connectionStatus = getConnectionStatus();
        synchronized (connectionStatus) {
            otherClientCount = connectionStatus.getActiveClients() - 1;
        }

        DocumentEventHelper.addDistributedEvent(getFileHelper(), otherClientCount);

        // send update to sender, only
        this.clearActionsReceived();

        final Message senderMessage = new Message();
        final MessageData senderMessageData = new MessageData(null, getConnectionStatusClone(), jsonResult);
        senderMessage.setFrom(fromId);
        senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(senderMessageData, MessageData.class.getName(), getComponentID(), "update")).build()));
        LOG.debug("RT connection: Relaying [update] to editor for keep alive, applied from: " + fromId.toString());
        relayToID(senderMessage, fromId);

        return returnMessage;
    }

    /**
     * Sends a client request to receive updated view data to the calc engine.
     *
     * @param fromId The ID of the client which requested the view update.
     * @param session The session of the client which requested the view update.
     * @param jsonRequest The request as JSON object containing the required data to process the view update.
     * @return The result of the view update request.
     * @throws OXException
     */
    private JSONObject getSpreadsheetViewUpdate(final ID fromId, Session session, JSONObject jsonRequest) throws OXException {
        JSONObject updateViewOperation = createUpdateViewOperation(jsonRequest);
        JSONObject resultObject = null;

        try {
            resultObject = executeSpreadsheetOperationFailsafe(fromId, updateViewOperation.toString());
        } catch (CalcEngineException e) {
            LOG.error("Exception on getSpreadsheetViewUpdate()", e);
            impl_handleCalcEngineError(session, e.getErrorcode());
            throw new OXException(e);
        } catch (Exception e) {
            LOG.debug("Exception on getSpreadsheetViewUpdate()", e);
        }

        return MessageHelper.checkJSONResult(resultObject);
    }

    // /* disabled !
    // * restore not possible by non well designed core ...
    // * refactor the underlying design and reactivate this method
    // */
    // private void restoreSpreadsheetFromOperationCache(final String lastFailedOperation,
    // final StringBuffer resultOfLastOperation) throws Exception {
    //
    // // let exception out !
    // // outside code rely on that .-)
    //
    // if (m_messageChunkList != null) {
    // final int c = m_messageChunkList.size();
    // int i = 0;
    // final String[] restoreOperations = new String[c];
    //
    // for (MessageChunk chunk : m_messageChunkList) {
    // final JSONObject jsonObj = chunk.getOperations();
    // final String jsonStr = jsonObj.toString();
    // restoreOperations[i++] = jsonStr;
    // }
    //
    // LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] start restore of document");
    // final ECalcEngineError error = m_calcClient.restoreDocument (m_calcDocumentHandle, restoreOperations);
    // if (error != ECalcEngineError.E_NONE)
    // throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] restore failed.");
    // }
    //
    // LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] apply last operation (after restore)");
    // resultOfLastOperation.setLength(0); // reset !
    // final ECalcEngineError error = m_calcClient.executeOperation (m_calcDocumentHandle, lastFailedOperation, resultOfLastOperation);
    // if (error != ECalcEngineError.E_NONE)
    // throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] apply of last operation (after restore) - failed.");
    // }

    /**
     * @param requestData
     * @return
     */
    private static JSONObject createUpdateViewOperation(JSONObject requestData) {
        final JSONArray operationsArray = new JSONArray();
        final JSONObject operations = new JSONObject();

        try {
            requestData.put("name", "updateView");
            LOG.debug(requestData.toString());
            operationsArray.put(requestData);
            operations.put("operations", operationsArray);
        } catch (JSONException e) {
            LOG.error("Exception on createUpdateViewOperation", e);
        }
        return operations;
    }


    /**
     * merge the content of the 'contents' element of a local result to the global result
     *
     * @param globalChanged changed cells information of previous operations
     * @param localChanged changed cells information of current operations
     * @return information of all cells, older information is overwritten by new cells
     */
    private JSONArray mergeContents(JSONArray globalChanged, JSONArray localChanged) {
        try {
            Set<Long> localHashes = new HashSet<Long>();

            Iterator<Object> localIt = localChanged.iterator();
            while (localIt.hasNext()) {
                JSONObject cell = (JSONObject) localIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                localHashes.add(new Long(posValue));
            }

            Iterator<Object> globalIt = globalChanged.iterator();
            while (globalIt.hasNext()) {
                JSONObject cell = (JSONObject) globalIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                if (!localHashes.contains(new Long(posValue))) {
                    localChanged.put(cell);
                }
            }
        } catch (JSONException e) {
            LOG.error("Exception on mergeContents", e);
        }
        return localChanged;
    }

    /**
     * merge the content of the 'changed' element of a local result to the global result
     *
     * @param jsonResult global result object
     * @param localJsonResult local result object
     */
    private void mergeChanged(JSONObject jsonResult, JSONObject localJsonResult) {
        try {
            if (localJsonResult.has("changed")) {
                if (!jsonResult.has("changed")) {
                    jsonResult.put("changed", new JSONObject());
                }
                JSONObject globalChanged = jsonResult.getJSONObject("changed");
                if (!globalChanged.has("type") || !globalChanged.getString("type").equals("all")) {
                    JSONObject localChanged = localJsonResult.getJSONObject("changed");
                    globalChanged.put("type", localChanged.getString("type"));
                    if (!globalChanged.getString("type").equals("all")) {
                        // if still not 'all' is changed merge the cell/column/row ranges
                        if (localChanged.has("sheets")) {
                            if (!globalChanged.has("sheets")) {
                                globalChanged.put("sheets", localChanged.getJSONObject("sheets"));
                            } else {
                                JSONObject globalSheets = globalChanged.getJSONObject("sheets");
                                JSONObject localSheets = localChanged.getJSONObject("sheets");
                                Set<String> localSheetSet = localSheets.keySet();

                                Iterator<String> localSheetIt = localSheetSet.iterator();
                                while (localSheetIt.hasNext()) {
                                    String localSheetKey = localSheetIt.next();
                                    JSONObject localSheet = localSheets.getJSONObject(localSheetKey);
                                    if (!globalSheets.has(localSheetKey)) {
                                        globalSheets.put(localSheetKey, localSheet);
                                    } else {
                                        JSONObject globalSheet = globalSheets.getJSONObject(localSheetKey);
                                        //
                                        if (localSheet.has("contents")) {
                                            if (globalSheet.has("contents")) {
                                                globalSheet.put(
                                                    "contents",
                                                    mergeContents(globalSheet.getJSONArray("contents"), localSheet.getJSONArray("contents")));
                                            } else {
                                                globalSheet.put("contents", localSheet.getJSONArray("contents"));
                                            }
                                        }

                                        if (localSheet.has("cells")) {
                                            if (!globalSheet.has("cells")) {
                                                globalSheet.put("cells", localSheet.getJSONArray("cells"));
                                            } else {
                                                // merge cells "cells:[{start:[1,1],end:[3,3]}, {start:[5,7]}]"
                                                JSONArray globalCells = globalSheet.getJSONArray("cells");
                                                JSONArray localCells = localSheet.getJSONArray("cells");
                                                CellsMerger cellsMerger = new CellsMerger(globalCells);
                                                cellsMerger.add(localCells);
                                                globalSheet.put("cells", cellsMerger.getCells());
                                            }
                                        }
                                        if (localSheet.has("rows")) {
                                            if (!globalSheet.has("rows")) {
                                                globalSheet.put("rows", localSheet.getJSONArray("rows"));
                                            } else {
                                                // merge rows "rows:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalRows = globalSheet.getJSONArray("rows");
                                                JSONArray localRows = localSheet.getJSONArray("rows");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalRows);
                                                rowColsMerger.add(localRows);
                                                globalSheet.put("rows", rowColsMerger.getRowCols());
                                            }
                                        }
                                        if (localSheet.has("columns")) {
                                            if (!globalSheet.has("columns")) {
                                                globalSheet.put("columns", localSheet.getJSONArray("columns"));
                                            } else {
                                                // merge columns "columns:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalColumns = globalSheet.getJSONArray("columns");
                                                JSONArray localColumns = localSheet.getJSONArray("columns");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalColumns);
                                                rowColsMerger.add(localColumns);
                                                globalSheet.put("cols", rowColsMerger.getRowCols());
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            if (localJsonResult.has("changedOperations")) {
                JSONArray lclChangedOp = localJsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                JSONArray resultChangedOp = jsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                for (int op = 0; op < lclChangedOp.length(); ++op) {
                    resultChangedOp.put(lclChangedOp.get(op));
                }
            }
            final String subElements[] =
                {
                    "senderOperations",
                    "uiOperations"
                };
            for(int sub = 0; sub < subElements.length; ++sub)
            {
                if (localJsonResult.has(subElements[sub])) {
                    JSONArray lclChangedOp = localJsonResult.getJSONArray(subElements[sub]);
                    JSONArray resultChangedOp = jsonResult.getJSONArray(subElements[sub]);
                    for (int op = 0; op < lclChangedOp.length(); ++op) {
                        resultChangedOp.put(lclChangedOp.get(op));
                    }
                }
            }
        } catch (JSONException e) {
            LOG.debug("Exception in RT connection: mergeChanged" + jsonResult.toString() + " / " + localJsonResult.toString());
        }
    }

    /**
     * Processes a undo/redo request from the client-side.
     * ATTENTION: This method is called by handlers that are called by a 'query'
     * real-time request. This implies several limitations, which must be handled
     * in this method. Especially the following points must be
     * 1.) The ID from the stanza is an internal one, which cannot be mapped
     *     to the client-ids.
     * 2.) The session data is not available for via the ID. Must be additionally
     *     sent through the request (will by extract by retrieveRealID)
     *
     * @param stanza The stanza with the data for the undo/redo request.
     * @param isUndo Specifies, if this is a undo or redo request.
     * @throws OXException
     */
    private void processUndoRedo(Stanza stanza, boolean isUndo) throws OXException {
        final ID fromId = this.retrieveRealID(stanza);

        if (this.isValidMemberAndEditor(fromId)) {
            LOG.trace("RT connection: Handling " + (isUndo ? "[undo]" : "[redo]") + ", called from: " + fromId.toString());

            final ConnectionStatus connectionStatus = getConnectionStatus();
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "document"), false);
            final ServerSession session = MessageHelper.getServerSession(stanza, "office", ActionParameters.KEY_RTDATA);
            JSONObject undoAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject undo = new JSONObject();

            try {
                undo.put("name", isUndo ? "undo" : "redo");
                undo.put("osn", connectionStatus.getOperationStateNumber());
                undo.put("opl", 1);
                operations.put(0, undo);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                undoAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.error("RT connection: Exception while creating undoActions JSON object.", e);
            }

            final Message applyMessage = impl_applyActions(fromId, session, undoAction);
            JSONObject jsonResult = new JSONObject();

            final ElementPath path = new ElementPath(getComponentID(), "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by impl_applyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json", getComponentID(), "undo")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "undo" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleUndo(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleUndo");

        processUndoRedo(stanza, true);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "redo" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRedo(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleRedo");

        processUndoRedo(stanza, false);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "replace" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleReplaceAll(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleReplaceAll");

        final ID fromId = this.retrieveRealID(stanza);
        if (this.isValidMemberAndEditor(fromId)) {
            final Session session = MessageHelper.getServerSession(null, fromId);

            LOG.trace("RT connection: Handling [replaceAll], called from: " + fromId.toString());
            try {
                final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
                final JSONArray operationsArray = new JSONArray();
                final JSONObject operations = new JSONObject();

                try {
                    jsonRequest.put("name", "replace");
                    LOG.debug(jsonRequest.toString());
                    operationsArray.put(jsonRequest);
                    operations.put("operations", operationsArray);
                } catch (JSONException e) {
                    LOG.error("JSONException on handleReplaceAll", e);
                }

                // return operations;
                JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
                jsonResult = MessageHelper.checkJSONResult(jsonResult);
                if (jsonResult.has("changedOperations")) {
                    JSONObject actions = new JSONObject();
                    JSONArray changedOperations = new JSONArray();
                    changedOperations.put(0, jsonResult.getJSONObject("changedOperations"));
                    actions.put(MessagePropertyKey.KEY_ACTIONS, changedOperations);
                    MessageChunk actionChunk = new MessageChunk(actions, fromId);
                    jsonResult.remove("changedOperations");

                    // update the shared state
                    synchronized (getSyncAccess()) {
                        addMessageChunkToQueue(actionChunk);
                    }
                }

                final Message returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(stanza.getFrom());
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json", getComponentID(), "spreadsheet")).build()));

                LOG.debug("RT connection sending [replaceAll] result to: " + stanza.getFrom().toString());

                // send the message to the one who requested to rename the document
                send(returnMessage);

                LOG.debug("RT connection relaying [replaceAll] result, called from: " + stanza.getFrom().toString());

                // update all Clients with the result, too
                updateClients(new MessageData(null, getConnectionStatusClone(), jsonResult), null);
            } catch (JSONException e) {
                LOG.error("Exception on handleReplaceAll", e);
            } catch (CalcEngineException e) {
                LOG.error("Exception on handleReplaceAll", e);
                impl_handleCalcEngineError(session, e.getErrorcode());
                throw new OXException(e);
            } catch (Exception e) {
                LOG.error("Exception on handleReplaceAll", e);
            }
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "sort" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleSort(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleSort");

        final ID fromId = retrieveRealID(stanza);
        if (this.isValidMemberAndEditor(fromId)) {
            LOG.trace("RT connection: Handling [sort], called from: " + fromId.toString());

            final ConnectionStatus connectionStatus = getConnectionStatus();
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
            final JSONObject sortAction = new JSONObject();
            final JSONArray actions = new JSONArray();
            final JSONObject operationObj = new JSONObject();
            final JSONArray operations = new JSONArray();
            final JSONObject sort = new JSONObject();
            final ServerSession session = MessageHelper.getServerSession(jsonRequest, fromId);

            try {
                LOG.debug(jsonRequest.toString());

                sort.put("name", "sort");
                sort.put("osn", connectionStatus.getOperationStateNumber());
                sort.put("opl", 1);
                final Iterator <Map.Entry<String, Object>> reqIt = jsonRequest.entrySet().iterator();
                while(reqIt.hasNext())
                {
                    Map.Entry<String, Object> nextElement = reqIt.next();
                    sort.put(nextElement.getKey(), nextElement.getValue());
                }
                operations.put(0, sort);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                sortAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.error("RT connection: handleSort catches exception while creating JSON actions object. ", e);
            }

            final Message applyMessage = impl_applyActions(fromId, session, sortAction);
            JSONObject jsonResult = new JSONObject();

            final ElementPath path = new ElementPath(getComponentID(), "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by impl_applyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json", getComponentID(), "spreadsheet")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "query" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleQuery(Stanza stanza) throws OXException {
        checkForDisposed();
        checkForThreadId("handleQuery");

        final ID fromId = stanza.getFrom();
        if (this.isValidInternalID(fromId)) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "spreadsheet"), false);
            final Session session = MessageHelper.getServerSession(jsonRequest, fromId);

            LOG.debug("RT connection: Handling [query], called from: " + fromId.toString());
            try {
                final JSONArray operationsArray = new JSONArray();
                JSONObject operations = new JSONObject();
                jsonRequest.put("name", "query");
                LOG.debug(jsonRequest.toString());
                operationsArray.put(jsonRequest);
                operations.put("operations", operationsArray);

                // return operations;
                JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
                jsonResult = MessageHelper.checkJSONResult(jsonResult);
                final Message returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json", getComponentID(), "spreadsheet")).build()));

                LOG.debug("RT connection sending [query] result to: " + stanza.getFrom().toString());

                // send the message to the one who requested to rename the document
                send(returnMessage);

                LOG.debug("RT connection relaying [query] result, called from: " + stanza.getFrom().toString());

                // update all Clients with the result, too
                updateClients(new MessageData(null, getConnectionStatusClone(), jsonResult), null);
            } catch (CalcEngineException e) {
                LOG.error("Exception on handleQuery", e);
                impl_handleCalcEngineError(session, e.getErrorcode());
                throw new OXException(e);
            } catch (Exception e) {
                LOG.error("Exception on handleQuery", e);
            }
        }
    }

    /**
     * Implements the request processing for the asynchronous loading of documents.
     * Currently there are two different requests:
     * 1. LoadRequest - initiates the loading of the document into the CalcEngine
     * with chunks to circumvent problems with the string length.
     * 2. SyncRequest - sync any other clients that requests the same doc as a
     * previous client to guarantee that the CalcEngine is ready to process further
     * requests.
     *
     * @param request
     *  The current request which must be processed by the Connection object.
     */
    @Override
    public void processRequest(final Request request) throws OXException {
        final ConnectionStatus connectionStatus = getConnectionStatus();

        switch (request.getType()) {
        case REQUEST_LOAD_REMAINING:
        case REQUEST_LOAD:
            Message loadMessage = null;
            ErrorCode errorCode = ErrorCode.NO_ERROR;

            try {
                synchronized (getSyncAccess()) {
                    if (m_uiOperations == null) {
                        m_uiOperations = new JSONObject();
                    }
                }

                LOG.trace("RT connection: loading operations in chunks for document: " + request.getDocumentId());

                final ID fromId = request.getId();

                JSONArray uiOperations = new JSONArray();
                JSONArray opArray = null;
                if (request.getOperations() != null) {
                    // for a normal load request we can pull the operations from the
                    // operations JSONObject.
                    opArray = request.getOperations().getJSONArray("operations");
                } else {
                    // in case of a remaining load request, we have to trigger the
                    // import of the remaining sheets and retrieve the operations from
                    // the response.
                    OXDocument docLoader = request.getDocumentLoader();
                    final JSONObject remainingOps = docLoader.getRemainingOperations(null);
                    if (remainingOps != null) {
                        opArray = remainingOps.getJSONArray("operations");
                    }
                }

                if (opArray.length() > MAX_OPERATIONS) {
                    int opCount = opArray.length();
                    int copyPosition = 0;

                    while (copyPosition < opCount) {
                        final JSONObject opPart = new JSONObject();
                        final JSONArray partArray = new JSONArray();
                        for (int op = 0; op < MAX_OPERATIONS && copyPosition < opCount; ++op) {
                            partArray.put(op, opArray.get(copyPosition));
                            ++copyPosition;
                        }

                        opPart.put("operations", partArray);
                        LOG.trace("RT connection: sending operations chunk to CalcEngine for document: " + request.getDocumentId());
                        boolean docDisposed = isDisposed();

                        LOG.trace("RT connection: check document is disposed (before executeSpreadsheetOperationFailsafe) disposed=" + docDisposed);

                        if (!docDisposed) {
                            final JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));
                            docDisposed = isDisposed();

                            if (!docDisposed) {
                                if (resultObject.has("uiOperations")) {
                                    try {
                                        JSONArray newUIOps = resultObject.getJSONArray("uiOperations");
                                        for (int i = 0; i < newUIOps.length(); i++) {
                                            uiOperations.put(newUIOps.get(i));
                                        }
                                    } catch (JSONException e) {
                                        LOG.error("RT connection: Error adding ui operations to JSON object", e);
                                    }
                                }
                            }
                        }

                        LOG.trace("RT connection: check document is disposed (after executeSpreadsheetOperationFailsafe) disposed=" + docDisposed);

                        if (docDisposed) {
                            // remove possible pending requests from clients already left the connection
                            this.m_loadRequestQueue.purgeDocumentRequests(request.getDocumentId());
                            break;
                        } else {
                            Thread.sleep(1000);
                        }
                    }

                    synchronized (getSyncAccess()) {
                        if (!isDisposed()) {
                            LOG.trace("RT connection: put ui operations into m_uiOperations");
                            final JSONArray operations = m_uiOperations.optJSONArray("operations");
                            if (null != operations) {
                                JSONHelper.appendArray(operations, uiOperations);
                            } else {
                                m_uiOperations.put("operations", uiOperations);
                            }
                        }
                    }
                } else if (opArray.length() > 0) {
                    final JSONObject opPart = new JSONObject();
                    opPart.put("operations", opArray);

                    final JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));
                    if (resultObject.has("uiOperations")) {
                        try {
                            synchronized (getSyncAccess()) {
                                if (!isDisposed()) {
                                    LOG.trace("RT connection: put complete ui operations into m_uiOperations");
                                    uiOperations = resultObject.getJSONArray("uiOperations");
                                    final JSONArray operations = m_uiOperations.optJSONArray("operations");
                                    if (null != operations) {
                                        JSONHelper.appendArray(operations, uiOperations);
                                    } else {
                                        m_uiOperations.put("operations", uiOperations);
                                    }
                                }
                            }
                        } catch (JSONException e) {
                            LOG.error("RT connection: Error adding ui operations to JSON object", e);
                        }
                    }
                    LOG.debug(resultObject.toString());
                }

                if (!isDisposed()) {
                    final JSONObject finishedResult = executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"loadFinished\"}]}");

                    if (finishedResult.has("uiOperations")) {
                        synchronized (getSyncAccess()) {
                            if (!isDisposed()) {
                                JSONArray currUIOp = m_uiOperations.getJSONArray("operations");
                                JSONHelper.appendArray(currUIOp, finishedResult.getJSONArray("uiOperations"));
                                m_uiOperations.put("operations", currUIOp);
                            }
                        }
                    }
                    LOG.trace("RT connection: loading operations in chunks finished for document: " + request.getDocumentId());
                }

                JSONObject operations = null;
                synchronized (getSyncAccess()) {
                    operations = m_uiOperations;
                }

                loadMessage = impl_createWelcomeMessage(request.getId(), operations, request.getDocumentOSN(), true, null, request.getPreviewUIOpsCount(), true, request.getDocumentMetaData());
            } catch (NoClassDefFoundError e) {
                LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
                errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
            } catch (UnsatisfiedLinkError e) {
                LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
                errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
            } catch (CalcEngineException e) {
                // do nothing in case if shutdown
                if (isShutdown()) {
                    LOG.debug("RT connection: Shutdown detected during asynchronous spreadsheet operation processing!", e);
                    return;
                }
                LOG.error("RT connection: Error executing spreadsheet operations, calc engine notifies problems!", e);
                errorCode = ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR;
            } catch (InterruptedException e) {
                LOG.error("RT connection: Interrupted while executing spreadsheet operations - may be we are shutting down", e);
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                return;
            } catch (FilterException e) {
                LOG.error("RT connection: Exception while asynchronous spreadsheet loading", e);
                errorCode = FilterExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR);
            } catch (Throwable e) {
                LOG.error("RT connection: Error executing spreadsheet operations", e);
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
            } finally {
                // Set request to finish to enable new requests to be asynchronously
                // processed.
                if (!isDisposed()) {
                    this.setAsyncSpreadsheetLoaded(true);

                    if (errorCode.getCode() == ErrorCode.NO_ERROR.getCode()) {
                        // in case of no error we just sent the finalLoad update
                        try {
                            boolean isMember = false;
                            synchronized (connectionStatus) {
                                isMember = connectionStatus.hasActiveUser(request.getId().toString());
                            }
                            if (isMember) {
                                this.relayToID(loadMessage, request.getId());
                            }
                        } catch (Throwable t) {
                            LOG.debug("RT connection: Not able to send message to the document clients, may be client already left", t);
                        }
                    } else {
                        // in case of an error we have to sent a finalLoad update which
                        // has 'hasErrors' set and 'errorCode' includes more information
                        // about the root cause.
                        ConnectionStatus errorConnectionStatus = getConnectionStatusClone();
                        errorConnectionStatus.setFinalLoad(true);

                        try {
                            final MessageData messageData = new MessageData(null, errorConnectionStatus, MessageHelper.finalizeJSONResult(errorCode, null, null));
                            this.updateClients(messageData, request.getId());
                        } catch (Throwable t) {
                            LOG.warn("RT connection: Not able to send message to the document clients, maybe client already left", t);
                        } finally {
                            // Purge the remaining sync requests from the queue
                            // as we already sent a 'finalLoad' update which
                            // includes an error code describing the root cause.
                            final String documentId = request.getDocumentId();
                            this.m_loadRequestQueue.purgeDocumentRequests(documentId);
                        }
                    }
                }
                LOG.trace("RT connection: request finished for document: " + request.getDocumentId());
                request.finished();
            }
            break;

        case REQUEST_LOAD_SYNC:
            // Sync another client with asynchronous loading process
            // This code should only be reached in case of a successful loading
            request.finished();
            if (!isDisposed()) {
                LOG.trace("RT connection: additional client can now work on document: " + request.getDocumentId());

                JSONObject operations = null;
                synchronized (getSyncAccess()) {
                    operations = m_uiOperations;
                }
                Message welcomeMessage = impl_createWelcomeMessage(request.getId(), operations, request.getDocumentOSN(), true, null, -1, true, request.getDocumentMetaData());
                try {
                    boolean isMember = false;
                    synchronized (connectionStatus) {
                        isMember = connectionStatus.hasActiveUser(request.getId().toString());
                    }
                    if (isMember) {
                        this.relayToID(welcomeMessage, request.getId());
                    }
                } catch (Throwable t) {
                    LOG.debug("RT connection: Not able to send message to the document clients", t);
                }
            }
            break;
        }
    }

    /**
     * Tries to process a request for the calc engine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param operation The request to be processed by the calc engine.
     * @return The result of the calc engine.
     * @throws Exception
     */
    protected JSONObject executeSpreadsheetOperationFailsafe(final ID fromId, final String operation) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute operation"));

        final StringBuffer result = new StringBuffer(256);
        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            final ICalcEngineClient calcClient = m_calcClient;
            error = (null != calcClient) ? calcClient.executeOperation(m_calcDocumentHandle, operation, result) : ECalcEngineError.E_TIMEOUT;
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                try {
                    sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HangUpReceiver.HANGUP_ALL, null);
                } catch (Exception e) {
                    LOG.error("RT connection: Exception catched while trying to send hangup for all clients", e);
                }
            }
            throw new CalcEngineException("error [" + error + "] on execute operation '" + operation + "'.", error);
        }

        final String resultStr = result.toString();
        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_DEBUG).toLog("... result : '" + resultStr + "'"));
        JSONObject resultObject = new JSONObject(resultStr);
        return resultObject;
    }

    /**
     * Tries to process a copy request for the calcengine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param event The copy event to be processed by the calc engine.
     * @throws Exception in case anything went wrong
     */
    void executeSpreadsheetCopyFailSafe(final ID fromId, final CalcEngineClipBoardEvent event) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute copy"));

        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            final ICalcEngineClient calcClient = m_calcClient;
            error = (null != calcClient) ? calcClient.copy(m_calcDocumentHandle, event) : ECalcEngineError.E_TIMEOUT;
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                try {
                    sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HangUpReceiver.HANGUP_ALL, null);
                } catch (Exception e) {
                    LOG.error("RT connection: Exception catched while trying to send hangup for all clients", e);
                }
            }
            throw new CalcEngineException("error [" + error + "] on execute copy.'", error);
        }
    }

    /**
     * Tries to process a copy request for the calcengine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param event The copy event to be processed by the calc engine.
     * @throws Exception in case anything went wrong
     */
    void executeSpreadsheetPasteFailSafe(final ID fromId, final CalcEngineClipBoardEvent event) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute paste"));

        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            final ICalcEngineClient calcClient = m_calcClient;
            error = (null != calcClient) ? calcClient.paste(m_calcDocumentHandle, event) : ECalcEngineError.E_TIMEOUT;
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                try {
                    sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HangUpReceiver.HANGUP_ALL, null);
                } catch (Exception e) {
                    LOG.error("RT connection: Exception catched while trying to send hangup for all clients", e);
                }
            }
            throw new CalcEngineException("error [" + error + "] on execute paste.'", error);
        }
    }

    /**
     * Creates a welcome message for the client where it's notified that an asynchronous load process is used on the backend side. That
     * means that the client must wait until the final load update is sent.
     *
     * @param fromId The id of this real-time connection instance.
     * @param onBehalfOf The id of the client that requested to join the connection instance.
     * @param previewUIOperations Optional operations for the preview mode.
     * @param previewActiveSheetIndex Optional preview active sheet index.
     * @return A welcome message prepared for asynchronous loading by the client.
     */
    protected Message impl_createAsyncLoadWelcomeMessage(ID fromId, ID onBehalfOf, final JSONArray previewUIOperations, final int previewActiveSheetIndex) {
        Message welcomeMessage = null;
        // Provide a welcome message where the syncLoad attribute is set to false
        // which notifies the client that it has to wait for a specific update
        // message which has 'finalLoad' attribute set to true.
        final DocFileHelper fileHelper = getFileHelper();
        final ConnectionStatus connectionStatus = getConnectionStatusClone();
        final JSONObject actionData = new JSONObject();
        connectionStatus.setSyncLoad(false);

        LOG.debug("RT connection: Completed synchronous loading for document " + Request.buildDocumentId(this.getResourceID()) + ", asynchronous loading started.");

        // set preview operations
        JSONObject previewJSONData = null;
        try {
            if (null != previewUIOperations) {
                final JSONObject previewData = new JSONObject();
                previewJSONData = new JSONObject();
                previewData.put("operations", previewUIOperations);
                previewData.put("activeSheet", previewActiveSheetIndex);
                previewJSONData.put("preview", previewData);
            }
        } catch (JSONException e) {
            LOG.warn("Exception while creating preview object for asynchronous loading", e);
        }

        final MessageData messageData = new MessageData(
            new MessageChunk(actionData, getId()), connectionStatus, MessageHelper.finalizeJSONResult(ErrorCode.NO_ERROR, null, previewJSONData));

        // set empty actions as payload for the first welcome message
        welcomeMessage = new Message();
        welcomeMessage.setFrom(fromId);
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(
            new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "getactions")).build()));

        LOG.debug("RT connection: Sending preview [welcome message] to: " + onBehalfOf.toString());

        return welcomeMessage;
    }

    /**
     * Process the preview request and fill the global ui operations object.
     *
     * @param fromId The id of the client which requested the preview.
     * @param onBehalfOf The id of the client in which context this preview should be processed.
     * @param previewOperations The operations for the preview.
     * @param singleSheet Specifies if this is a single sheet document or not.
     */
    private ErrorCode impl_processPreviewRequest(final ID fromId, final ID onBehalfOf, final JSONArray previewOperations, boolean singleSheet) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        try {
            final String documentId = Request.buildDocumentId(this.getResourceID());

            synchronized (getSyncAccess()) {
                m_uiOperations = new JSONObject();
            }

            LOG.debug("RT connection: loading preview operations in chunks for document: " + documentId);

            final JSONArray uiOperations = new JSONArray();
            if (previewOperations.length() > MAX_OPERATIONS) {
                int opCount = previewOperations.length();
                int copyPosition = 0;

                while (copyPosition < opCount) {
                    JSONObject opPart = new JSONObject();
                    JSONArray partArray = new JSONArray();
                    for (int op = 0; op < MAX_OPERATIONS && copyPosition < opCount; ++op) {
                        partArray.put(op, previewOperations.get(copyPosition));
                        ++copyPosition;
                    }

                    opPart.put("operations", partArray);
                    LOG.debug("RT connection: sending preview operations chunk to CalcEngine for document: " + documentId);

                    JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));

                    if (resultObject.has("uiOperations")) {
                        JSONHelper.appendArray(uiOperations, resultObject.getJSONArray("uiOperations"));
                    }
                }

                synchronized (getSyncAccess()) {
                    m_uiOperations.put("operations", uiOperations);
                }
            } else {
                final JSONObject operations = new JSONObject();
                operations.put("operations", previewOperations);

                JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, operations.toString(false));
                if (resultObject.has("uiOperations")) {
                    synchronized (getSyncAccess()) {
                        m_uiOperations.put("operations", resultObject.getJSONArray("uiOperations"));
                    }
                }
                LOG.debug(resultObject.toString());
            }

            if (singleSheet) {
                // single sheet needs a load finished - no more operations will follow
                JSONObject finishedResult = executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"loadFinished\"}]}");
                if (finishedResult.has("uiOperations")) {
                    JSONArray currUIOp = m_uiOperations.getJSONArray("operations");
                    JSONHelper.appendArray(currUIOp, finishedResult.getJSONArray("uiOperations"));
                    m_uiOperations.put("operations", currUIOp);
                }
            } else {
                executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"sheetLoaded\"}]}");
            }
            LOG.debug("RT connection: loading preview operations in chunks finished for document: " + documentId);
        } catch (NoClassDefFoundError e) {
            LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
            errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
        } catch (UnsatisfiedLinkError e) {
            LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
            errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
        } catch (CalcEngineException e) {
            // do nothing in case if shutdown
            LOG.error("RT connection: Error executing spreadsheet operations, calc engine notifies problems!", e);
            errorCode = ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR;
        } catch (JSONException e) {
            LOG.error("RT connection: Error apply JSON operation", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        } catch (Exception e) {
            LOG.error("RT connection: Error executing spreadsheet operations", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        } catch (Throwable e) {
            LOG.error("RT connection: Error executing spreadsheet operations", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        }

        return errorCode;
    }

    /**
     * Tries to handle calc engine errors in a general way.
     *
     * @param session The session of the client which requested the calc engine to process the operations that lead to a calc engine error.
     * @param e The calc engine error we received on operation processing.
     * @return TRUE if the error is fatal and the instance must be disposed, FALSE if not fatal. Attention: Currently EVERY error is treated
     *         as fatal.
     */
    private boolean impl_handleCalcEngineError(Session session, ECalcEngineError e) {
        final boolean result = true;

        // Set this instance to dispose as we encountered a calc engine error.
        // Dispose calls onDispose() which will save the document.
        LOG.error("RT connection: Due to the calc engine error: " + e + ", the connection instance is going to be disposed");
        try {
            // Delay disposing this instance to ensure that the hangup message
            // is sent before the not member message reaches the clients due to
            // dispose. See #37712
            Thread.sleep(1500);
        } catch (InterruptedException ex) { }
        this.dispose();

        return result;
    }

}
