/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.time.StopWatch;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.hazelcast.doc.DocumentDirectory;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.realtime.tools.SystemInfoHelper;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.actions.ActionParameters;
import com.openexchange.office.tools.directory.DocResourceID;
import com.openexchange.office.tools.directory.DocumentState;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.memory.MemoryObserver;
import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.realtime.group.GroupDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;

/**
 * The abstract Connection class implements the necessary specialties of a
 * connection instance which supports persistence. The class doesn't implement
 * document specific parts.
 * {@link Connection}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public @NotThreadSafe
abstract class Connection extends GroupDispatcher {
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(Connection.class);

    private ServiceLookup m_services = null;
    private ContextAwareLogHelp m_logHelp = null;
    private final Object m_syncAccess = new Object();
    private final AtomicBoolean m_disposed = new AtomicBoolean();
    private final AtomicBoolean m_shutdown = new AtomicBoolean();
    private final AtomicBoolean m_saveOnDispose = new AtomicBoolean(true);
    private final AtomicBoolean m_processed = new AtomicBoolean();
    private final AtomicInteger m_saveDocumentInProgress = new AtomicInteger();
    private final AtomicBoolean m_globalBackgroundSave = new AtomicBoolean();
    private final Date m_startupTime;
    private DocumentDirectory m_persistentDocumentDirectory = null;
    private Date m_lastSaveTime = null;
    private ErrorCode m_lastFailSafeSaveError = ErrorCode.NO_ERROR;
    private long m_threadId = 0;
    private final String m_componentID;
    private final static ElementPath rtDataPath = new ElementPath(ActionParameters.KEY_OFFICE, ActionParameters.KEY_RTDATA);
    private final String m_identity = UUID.randomUUID().toString();

    public enum HangUpReceiver {
        HANGUP_ALL,
        HANGUP_ID
    }

    /**
     * An enum to set the reason for the currently triggered failSafeSave call
     */
    public enum FailSafeSaveReason {
        TOO_MANY_OPS,
        MODIFICATION_TIMEOUT
    }

    /**
     * Initializes a new {@link Connection}.
     *
     * @param services A service lookup instance which provides access to other services
     *                 needed by the OX Documents GroupDispatcher.
     * @param id       The ID which references the resource used for this OX Documents
     *                 GroupDispatcher. Normally references a file in the infostore.
     * @throws OXException
     */
    public Connection(final ServiceLookup serviceLookup, final ID id, final ActionHandler handler, final String componentID) {
        super(id, handler);

        LOG.debug("RT connection: Connection created: " + id.toString() + ", identity: " + getIdentity());

        // initialize ServiceLookup and ResourceManager
        m_componentID = componentID;
        m_services = serviceLookup;
        m_disposed.set(false);
        m_startupTime = new Date();
        m_persistentDocumentDirectory = serviceLookup.getService(com.openexchange.office.hazelcast.doc.DocumentDirectory.class);
    }

    /**
     * Provides a context aware logging instance to used for logging messages.
     *
     * @return The context aware logging instance.
     */
    protected ContextAwareLogHelp mem_LogHelp() {
        try {
            if (m_logHelp == null) {
                m_logHelp = new ContextAwareLogHelp(LOG);
            }
            return m_logHelp;
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Provides the create thread id of this connection instance.
     *
     * @return
     */
    public long getCreateThreadId() {
        return this.m_threadId;
    }

    public String getIdentity() {
        return this.m_identity;
    }

    /**
     * Provides the service lookup instance.
     *
     * @return
     */
    protected ServiceLookup getServices() {
        return m_services;
    }

    protected void setGlobalSaveState() {
        m_globalBackgroundSave.compareAndSet(false, true);
    }

    protected boolean getGlobalSaveState() {
        return this.m_globalBackgroundSave.get();
    }

    /**
     * Provides the configuration services.
     *
     * @return
     */
    protected ConfigurationService getConfigurationService() {
        return m_services.getService(com.openexchange.config.ConfigurationService.class);
    }

    protected IDBasedFileAccess getFileAccess(final Session session) {
        IDBasedFileAccessFactory factory = m_services.getService(IDBasedFileAccessFactory.class);
        return factory.createAccess(session);
    }

    protected DocumentDirectory getDocumentDirectory() {
        return this.m_persistentDocumentDirectory;
    }

    /**
     * Checks the global (hazelcast based) save state for this document.
     *
     * @return
     *  TRUE, if the global save state if set, otherwise FALSE.
     */
    protected boolean isGlobalSaveStateSet() {
        boolean isInSaveState = false;
        final DocumentDirectory docDirectory = getDocumentDirectory();

        if (null != docDirectory) {
            final DocResourceID docResId = DocResourceID.createDocResourceID(this.getId().getContext(), this.getId().getResource());

            try {
                final DocumentState docState = docDirectory.get(docResId);

                if (null != docState) {
                    isInSaveState = docState.getSaveInProgress();
                }
            } catch (OXException e) {
                LOG.warn("RT connection: Couldn't retrive global save state for document " + docResId.toString(), e);
            }
        }

        return isInSaveState;
    }

    /**
     * Provides the sync access instance, which is used to synchronize the
     * access to shared members in a multi-thread environment. Some RT handlers
     * are called by RT in a multi-thread way. Accessing members in these methods
     * must by synchronized by the implementation.
     *
     * @return
     *  The synchronization object to be used for multi-thread access to
     *  shared members.
     */
    public Object getSyncAccess() {
        return this.m_syncAccess;
    }

    /**
     * Provides the startup time of this connection instance.
     *
     * @return
     */
    public Date getStartupTime() {
        return this.m_startupTime;
    }

    /**
     * Provides the state of save on dispose.
     *
     * @return
     */
    public boolean isSaveOnDispose() {
        return this.m_saveOnDispose.get();
    }

    /**
     * Changes the save on dispose state.
     *
     * @param value
     */
    protected void setSaveOnDispose(boolean value) {
        this.m_saveOnDispose.set(value);
    }

    /**
     * Returns if the connection instance has been disposed by the RT-framework or not.
     *
     * @return TRUE if RT called the onDispose() method or FALSE if not. The connection
     */
    public boolean isDisposed() {
        return this.m_disposed.get();
    }

    /**
     * Set this instance to dispose state.
     */
    protected void setDisposed() {
        m_disposed.set(true);
    }

    /**
     * Provides the processed state.
     *
     * @return
     */
    public boolean isProcessed() {
        return this.m_processed.get();
    }

    /**
     * Sets the processed state.
     *
     * @param newValue
     */
    public void setProcessedState(boolean newValue) {
        this.m_processed.set(newValue);
    }

    /**
     * Determine whether we are in a shutdown process or not.
     *
     * @return TRUE if the instance was informed by the ConnectionComponent that a shutdown has been requested, FALSE if not.
     */
    public boolean isShutdown() {
        return this.m_shutdown.get();
    }

    /**
     * Sets the shutdown state indicating that we are in a shutdown process where nothing should be done.
     */
    public void setShutdown() {
        this.m_shutdown.getAndSet(true);
    }

    /**
     * Retrieves the component ID of the connection instance.
     * @return
     *  The component ID of the connection instance.
     */
    protected final String getComponentID() {
        return m_componentID;
    }

    /**
     * Provides the time stamp of the latest save operation. If there have been no save
     * operation, the Connection creation time is returned.
     *
     * @return The date of the latest save operation.
     */
    public Date getSaveTimeStamp() {
        return (null != this.m_lastSaveTime) ? m_lastSaveTime : getStartupTime();
    }

    /**
     * Provides the time of modification or null if no modification has been made since loading the document.
     *
     * @return Time of the last modification or null if no modification has been made,
     */
    public Date getLastSaveTime() {
        return m_lastSaveTime;
    }

    /**
     * Sets the last save time to the current time.
     */
    protected void setLastSaveTime() {
        m_lastSaveTime = new Date();
    }

    /**
     * Increments the save in progress counter atomically and provides
     * the previous counter value.
     *
     * @return
     *  The old counter value, before the atomic increment. The caller
     *  should check the value and can continue, if the value is zero. If not,
     *  the caller should decrement the counter and try again/wait.
     */
    protected int incAndCheckSaveInProgress() {
        return m_saveDocumentInProgress.getAndIncrement();
    }

    /**
     * Decrement the save in progress counter atomically.
     *
     * @return
     *  The updated value of the atomic counter.
     */
    protected int decSaveInProgress() {
        return this.m_saveDocumentInProgress.decrementAndGet();
    }

    /**
     * Determines, if saving the document is in progress.
     *
     * @return
     *  TRUE if saving is in progress, otherwise FALSE.
     */
    protected boolean isSaveInProgress() {
        return (this.m_saveDocumentInProgress.get() > 0);
    }

    /**
     * Determines, if saving the document is in progress. Waits the
     * specified amount of time until TRUE is returned.
     *
     * @param timeout
     *  Number of milliseconds to wait for a possible save in progress.
     *
     * @return
     *  TRUE if save is in progress, otherwise FALSE.
     */
    protected boolean isSaveInProgressWithWait(long timeout) {
        boolean saveInProgress = isSaveInProgress();

        if ((timeout > 0) && saveInProgress) {
            final StopWatch stopWatch = new StopWatch();
            final long sleep = Math.max(1, (timeout / 10));

            stopWatch.start();
            do {
                try {
                    Thread.sleep(sleep);
                } catch (InterruptedException e) {
                    break;
                }

                if (stopWatch.getTime() > timeout) {
                    break;
                }
            } while (isSaveInProgress());

            saveInProgress = isSaveInProgress();
        }

        return saveInProgress;
    }

    /**
     * Blocks until the timeout has been reached or the save in progress lock
     * is available.
     *
     * @param timeout
     *  time out value in milliseconds, zero means no timeout.
     * @return
     *  TRUE if the save in progress lock has been successfully locked. FALSE
     *  if the timeout has been reached.
     */
    protected boolean waitForSaveInProgressAvailableWithTimeout(long timeout) {
        final StopWatch stopWatch = new StopWatch();
        final long sleep = Math.max(1, (timeout / 10));
        boolean success = false;

        stopWatch.start();
        while (true) {
            int oldValue = m_saveDocumentInProgress.getAndIncrement();
            if (oldValue == 0) {
                success = true;
                break;
            } else {
                m_saveDocumentInProgress.decrementAndGet();
            }

            if (stopWatch.getTime() < timeout) {
                try {
                    Thread.sleep(sleep);
                } catch (InterruptedException e) {
                    break;
                }
            } else {
                // time is up - giving up
                break;
            }
        }

        return success;
    }

    /**
     * Provides the last fail safe save error code that saving the document encountered.
     *
     * @return The error code of the last fail safe save process (in the background).
     */
    public synchronized ErrorCode getLastFailSafeSaveError() {
        return m_lastFailSafeSaveError;
    }

    /**
     * Sets the last fail safe save error code that saving the document encountered.
     *
     * @param latestErrorCode The error code of the last fail safe save process (in the background).
     */
    protected synchronized void setLastFailSafeSaveError(ErrorCode latestErrorCode) {
        m_lastFailSafeSaveError = latestErrorCode;
    }


    /**
     * Tries to determine the "real" ID of the sender of the provided stanza.
     * There are two sorts of IDs: "internal" and general. An internal ID is not
     * able to reference the source (client). A general ID contains the sender.
     * A "real" ID is sometimes necessary to check the "rights" of a client/sender.
     *
     * @param stanza The stanza received from the real-time framework.
     *
     * @return The real ID or null, if no "real" ID could be retrived from the stanza
     */
    protected ID retrieveRealID(final Stanza stanza) {
        ID realID = null;

        // first retrieve the id using getFrom()/onBehalfOf()
        ID tempId = stanza.getFrom();
        if (null == tempId) {
            tempId = stanza.getOnBehalfOf();
        }

        if (tempId.isInternal()) {
            // Internal IDs are not "real" id, therefore try to extract additional
            // data from the stanza to construct a "real" ID.
            Collection<PayloadElement> payloadElements = stanza.filterPayloadElements(rtDataPath);
            if (null != payloadElements && !payloadElements.isEmpty()) {
                Iterator<PayloadElement> iter = payloadElements.iterator();
                PayloadElement payloadElement = iter.next();
                Object data = payloadElement.getData();
                if (data instanceof JSONObject) {
                    try {
                        // try to create a "general" ID from the real-time data
                        final JSONObject rtData = (JSONObject)data;
                        final ServerSession session = MessageHelper.getServerSession(rtData, tempId);
                        realID = getRealtimeID(rtData.getString(ActionParameters.KEY_RTID), session);
                    } catch (Exception e) {
                        LOG.error("Exception while creating real ID from stanza", e);
                    }
                }
            }
        } else {
            realID = tempId;
        }

        return realID;
    }

    /**
     * Creates a full real-time ID from the uuid string and the session of the user.
     *
     * @param uuid The uuid string of the user of the session.
     * @param session The session of the user that's rt-id should be created.
     *
     * @return The real-time, if possible otherwise an exception is thrown.
     * @throws Exception
     */
    protected ID getRealtimeID(final String uuid, final Session session) throws Exception {
        return new ID("ox", SessionUtils.getUserIdString(session), SessionUtils.getContextIdString(session), uuid);
    }

    /**
     * Checks if a ID is valid in the context of this connection. This means
     * that the ID is not null and member of the connection or the id of the
     * group.
     *
     * @param id The ID to be checked for validity.
     * @return TRUE if valid, otherwise FALSE.
     */
    protected boolean isValidID(final ID id) {
        return (null != id) && (isMember(id) || (this.getId().equals(id)));
    }

    /**
     * Checks if a ID is valid in the context of this connection. It also
     * accepts internal IDs, which are generated by RT, if a http-request
     * is done which wants to synchronize with a connection instance.
     *
     * @param id The ID to be checked for validity.
     * @return TRUE if valid, otherwise FALSE.
     */
    protected boolean isValidInternalID(final ID id) {
        return (null != id) && (id.isInternal() || isValidID(id));
    }

    /**
     * Checks the validity of a session.
     *
     * @param session A session to be checked.
     *
     * @return TRUE, if the session is valid otherwise FALSE.
     */
    protected boolean isValidSession(final Session session) {
        return (null != session);
    }

    /**
     * Checks if the connection instance has been disposed and throws a
     * RuntimeException if true, otherwise nothing happens.
     *
     * @throws RuntimeException In case of a disposed connection.
     */
    protected void checkForDisposed() throws RuntimeException {
        if (isDisposed()) {
            throw new RuntimeException("Connection instance has already been disposed - access not allowed");
        }
    }

    /**
     * Checks for the correct thread id and outputs a log error message, if we
     * detect that a wrong thread called us.
     *
     * @param methodName
     *  The name of the method that should be checked. Must NOT be null.
     */
    protected void checkForThreadId(final String methodName) {
        Validate.notNull(methodName, "Method name must not be null");
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId)) {
            LOG.debug("RT connection: " + methodName + " called from different thread " + String.valueOf(threadId));
        }
    }

    /**
     * Handles the "firstjoined" notification provided by the Realtime-Framework. This method is called
     * when the first client joins/creates the document.
     *
     * @param id The ID of the first client that joins/creates to the document (chat-room). (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#firstJoined(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void firstJoined(ID id) {
        super.firstJoined(id);

        m_threadId = Thread.currentThread().getId();
    }

    /**
     * Make a fail safe revision of the current document. This method is called by a timer
     * thread and therefore must be implemented thread-safe. Attention: There are members
     * which are set to null in onDispose() to return memory to the Java VM as soon as possible.
     * Therefore access to these member must be surrounded by a check for null!
     *
     * @param failSafeSaveReason The reason, why this failSafeSaveDocument method is currently
     *	called. This parameter may be null.
     *
     * @return The error code describing if the fail safe save was successful or not it describes
     * what's the root cause of a possible failure.
     */
    public abstract ErrorCode failSafeSaveDocument(FailSafeSaveReason failSafeSaveReason);

    /**
     * Tries to flush a modified document due to a dispose notification from the Realtime framework.
     *
     * @return TRUE if the saving the document was successful otherwise FALSE. The method also provides TRUE if there was no need to save
     *         the document as there are no pending operations.
     */
    public abstract boolean saveDocumentOnDispose();

    /**
     * Provides the number of pending operations on this connection.
     *
     * @return The number of operations.
     */
    public abstract long getPendingOperationsCount();

    /**
     * Removes all pending operations from the queue. This is for testing
     * purpose only and should never be called in a production system.
     */
    public abstract void resetOperationQueue();

    /**
     * Sends an update message to all clients of the document except the sender. This method can be used if a update should only be sent to
     * all other clients, because the sender already have all information available.
     *
     * @param messageData The message data to be sent to all clients.
     * @param senderId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     *                 This client won't be part of the clients that will receive the update message.
     */
    protected void updateClientsExceptSender(MessageData messageData, ID senderId) {
        impl_updateClients(messageData, senderId, true);
    }

    /**
     * Sends an update message to all clients of the document.
     *
     * @param messageData The message data to be sent to all clients.
     * @param fromId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     */
    protected void updateClients(MessageData messageData, ID fromId) {
        impl_updateClients(messageData, fromId, false);
    }

    /**
     * Sends an update message to a specific client of the document.
     *
     * @param messageData
     * @param clientId
     */
    protected void updateClient(MessageData messageData, final ID clientId) {
        impl_updateClient(messageData, clientId);
    }

    /**
     * Internal method to send an update message to a set of clients of the document.
     *
     * @param messageData The message data to be sent to all clients.
     * @param fromId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     * @param exceptSender If set to TRUE the sender/fromId won't be part of the clients that will received the update message.
     */
    private void impl_updateClients(MessageData messageData, ID fromId, boolean exceptSender) {
        final Message updateMessage = new Message();
        final ID senderId = (null != fromId) ? fromId : getId();

        updateMessage.setFrom(senderId);
        updateMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), m_componentID, "update")).build()));

        LOG.debug("RT connection: Handling [" + (exceptSender ? "updateClientsExceptSender" : "updateClients") + "], originated by: " + senderId.toString());

        try {
            if (exceptSender) {
                relayToAllExceptSender(updateMessage);
            } else {
                relayToAll(updateMessage);
            }
        } catch (OXException e) {
            if (exceptSender) {
                LOG.error("RT connection: [updateClients] catches exception for relayToAllExceptSender.", e);
            } else {
                LOG.error("RT connection: [updateClients] catches exception for relayToAll.", e);
            }
        }
    }

    /**
     * Internal method to send an update message to a specific client.
     *
     * @param messageData
     * @param clientId
     */
    private void impl_updateClient(MessageData messageData, final ID clientId) {
        final Message updateMessage = new Message();
        final ID senderId = (null != clientId) ? clientId : getId();

        if (isValidID(senderId)) {
            updateMessage.setFrom(senderId);
            updateMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), m_componentID, "update")).build()));

            LOG.debug("RT connection: Handling [updateClient], originated by: " + senderId.toString());

            try {
                this.relayToID(updateMessage, senderId);
            } catch (OXException e) {
                LOG.error("RT connection: [updateClients] catches exception for relayToId.", e);
            }
        }
    }

    /**
     * Checks and sends a flushErrorInfo message to all clients if the result JSON object contains an error, which describes problems on
     * flushing a document to OX Drive.
     *
     * @param errorCode The error code of the flushDocument operation, which could be zero or non-zero (that is the error code).
     * @param fromId The ID of the client that triggered the flushDocument. Must not be null, otherwise the flushErrorInfo message cannot be
     *            sent.
     * @param exceptSender Specifies if the client with fromId should get the flushErrorInfo message or not.
     */
    protected void impl_sendFlushInfoOnError(ErrorCode errorCode, ID fromId, boolean exceptSender) {
        final ID senderId = (null != fromId) ? fromId : getId();

        if (isValidID(senderId) && errorCode.isError()) {
            final JSONObject result = new JSONObject();
            try {
                result.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
                impl_sendFlushInfo(result, senderId, exceptSender);
            } catch (JSONException e) {
                LOG.error("Exception while setting up result for flushErrorInfo notification", e);
            }
        }
    }

    /**
     * Sends a flushErrorInfo message to all clients using the provided JSON object that must contain the error code.
     *
     * @param result The JSONObject directly returned by the flushDocument() call containing the error code.
     * @param fromId The ID of the client that triggered the flushDocument. Must not be null, otherwise the flushErrorInfo message cannot be
     *            sent.
     * @param exceptSender Specifies if the client with fromId should get the flushErrorInfo message or not.
     */
    protected void impl_sendFlushInfo(JSONObject result, ID senderId, boolean exceptSender) {
        try {
            final MessageData messageData = new MessageData(null, null, result);
            final Message flushErrorInfoMessage = new Message();

            flushErrorInfoMessage.setFrom(senderId);
            flushErrorInfoMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), m_componentID, "flusherrorinfo")).build()));

            if (exceptSender) {
                relayToAllExceptSender(flushErrorInfoMessage);
            } else {
                relayToAll(flushErrorInfoMessage);
            }
        } catch (OXException e) {
            LOG.error("RT connection: [flushErrorInfo] catches exception for relayToAll.", e);
        }
    }

    /**
     * Sends a "hangup" message to clients to switch to read-only mode and show an error message.
     *
     * @param fromId       The ID of the client that sends the "hangup" message. Must not be null to ensure that the message is send.
     * @param componentID  The component ID used by the group dispatcher.
     * @param statusToSend The connection status to be sent via the hangup message. Must not be null.
     * @param errorCode    The error code to be sent via the "hangup" message. This should specify why the "hangup" has been sent to the client.
     *                     Can be null if no error can be specified which triggers a generic error message on the client.
     * @param receiver     Specifies the receiver(s) of the hang-up message.
     * @param logMessage   The message to be added to the logging entry in debug mode only.
     */
    protected void sendHangUp(ID fromId, final ConnectionStatus statusToSend, ErrorCode errorCode, final HangUpReceiver receiver, final String logMessage) throws Exception {
        Validate.notNull(fromId, "Without fromId no hang up!");
        Validate.notNull(statusToSend, "Connection status to be sent must be set!");

        final MessageData messageData = new MessageData(null, statusToSend, MessageHelper.finalizeJSONResult(errorCode, null, null));
        final Message returnMessage = new Message();

        returnMessage.setFrom(fromId);
        returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), m_componentID, "hangup")).build()));

        LOG.debug("RT connection: send hang up message: " + (logMessage == null ? "unknown reason" : logMessage));

        if (HangUpReceiver.HANGUP_ALL == receiver) {
            relayToAll(returnMessage);
        } else {
            relayToID(returnMessage, fromId);
        }
    }

    /**
     * Logs optional performance data.
     *
     * @param session     The session of the client, that should log the
     *                    performance data. Must not be null
     * @param jsonRequest The json request from the client.
     */
    protected void logPerformanceData(final Session session, final JSONObject jsonRequest) {
        Validate.notNull(session, "Session must be valid!");
        Validate.notNull(jsonRequest, "jsonRequest must be valid!");

        final JSONObject performanceData = jsonRequest.optJSONObject("performanceData");
        final JSONObject recentFile = jsonRequest.optJSONObject("file");

        if (null != performanceData) {
            // if performance data is not null, this means, that performance logging is enabled -> only checking log path
            final String loggingPath = ConfigurationHelper.getStringOfficeConfigurationValue(
                getServices(), session, "//module/performanceDataLogPath", null);
            if (null != loggingPath) {
                OperationHelper.logPerformanceData(performanceData, recentFile, loggingPath);
            }
        }
    }

    /**
     * Sets the importer properties regarding memory information to prevent
     * out-of-memory exceptions, while loading big documents.
     *
     * @param importerProps The json object that is used for importer properties.
     *
     * @return The maximum memory to be used by the import filter.
     */
    protected long impl_setMemoryImporterProperties(final JSONObject importerProps) {
        long maxHeapSizeForFilter = 0;

        final SystemInfoHelper.MemoryInfo memInfo = SystemInfoHelper.getMemoryInfo();
        if (null != memInfo) {
            try {
                // The maximal heap size for the filter to load the document should not exceed
                // half of the maximal heap size of the JVM.
                maxHeapSizeForFilter = memInfo.maxHeapSize / 2;
                importerProps.put("maxMemoryUsage", maxHeapSizeForFilter);
                double percentage = ((double)(memInfo.maxHeapSize - SystemInfoHelper.MINIMAL_FREE_BACKEND_HEAPSPACE)) / (double)memInfo.maxHeapSize;
                MemoryObserver memObserver = MemoryObserver.getMemoryObserver();
                MemoryObserver.setPercentageUsageThreshold(percentage);
                importerProps.put(DocumentProperties.PROP_MEMORYOBSERVER, memObserver);
                importerProps.put(DocumentProperties.PROP_MAX_MEMORY_USAGE, maxHeapSizeForFilter);
            } catch (JSONException e) {
                LOG.error("Exception catched while setting up importer memory properties", e);
            }
        }
        return maxHeapSizeForFilter;
    }
}
