/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.presenter.impl;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * An enumeration to provide specific error information from the server to
 * the client side. The enumeration provides method to convert to JSON object
 * or to be created from a JSON object.
 *
 * {@link ErrorCode}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class ErrorCode implements Cloneable {

	public static final ErrorCode NO_ERROR = new ErrorCode(0, "NO_ERROR", "No error", 0);
	public static final ErrorCode GENERAL_MAX_PARTICIPANTS_FOR_PRESENTATION_REACHED_ERROR = new ErrorCode(100, "GENERAL_MAX_PARTICIPANTS_FOR_PRESENTATION_REACHED_ERROR", "The maximum numer of participants for this presentation has been reached. Joining is not possible.", 2);

	public final static int CODE_NO_ERROR          = 0;
    public final static String CONSTANT_NO_ERROR   = "NO_ERROR";
    public final static String CONSTANT_NO_ERROR_DESCRIPTION = "No error";

    // must be adapted
    private final static int ERRORCLASS_COUNT      = 4;
    public final static int ERRORCLASS_NO_ERROR    = 0;
    public final static int ERRORCLASS_WARNING     = 1;
    public final static int ERRORCLASS_ERROR       = 2;
    public final static int ERRORCLASS_FATAL_ERROR = 3;


    private final int m_code;
    private final String m_codeAsStringConstant;
    private final String m_description;
    // can be changed by client code to change severity dependent on the context
    private int m_errorClass;

    /**
     * Creates a ErrorCode instance with the code, a unique constant string for the
     * code and a description.
     *
     * @param code
     *  The error code.
     *
     * @param codeAsString
     *  The code as a unique constant string.
     *
     * @param description
     *  The description of the error.
     */
    private ErrorCode(int code, final String codeAsString, final String description, int errorClass) {
        this.m_code = code;
        this.m_codeAsStringConstant = codeAsString;
        this.m_description = description;
        this.m_errorClass = errorClass;
    }

    /**
     * Determines if this instance represents an error or not.
     *
     * @return
     *  TRUE if the instance represents an error and FALSE if not.
     */
    public boolean isError() {
        return (m_errorClass >= ERRORCLASS_ERROR);
    }

    /**
     * Determines if this instance represents no error, means
     * it could be a warning or no error at all.
     *
     * @return
     *  TRUE if this instance represents no error or a warning, otherwise
     *  FALSE is returned.
     */
    public boolean isNoError() {
        return (m_errorClass <= ERRORCLASS_WARNING);
    }

    /**
     * Determines if this instance represents a warning or not.
     *
     * @return
     *  TRUE if the instance represents a warning and FALSE if not.
     */
    public boolean isWarning() {
        return (m_errorClass == ERRORCLASS_WARNING);
    }

    /**
     * Provides a description of the error.
     *
     * @return
     *  The description.
     */
    public String getDescription() {
       return m_description;
    }

    /**
     * Provides a unique number which describes the error case.
     *
     * @return
     *  The error code.
     */
    public int getCode() {
       return m_code;
    }

    /**
     * Provides a unique constant string for the error case.
     * @return
     */
    public String getCodeAsStringConstant() {
        return m_codeAsStringConstant;
    }

    /**
     * Provides the error class of the error.
     *
     * @return
     *  The error class of the error, which describes how severe the
     *  error condition must be treated by the code.
     */
    public int getErrorClass() {
        return m_errorClass;
    }

    /**
     * Creates a new error class instance which contains the same values
     * as this instance.
     */
    @Override
    public ErrorCode clone() {
        return new ErrorCode(this.m_code, this.m_codeAsStringConstant, this.m_description, this.m_errorClass);
    }

    /**
     * Sets the error class of the error. Sometimes an error has a
     * different impact and therefore the error class must be adapted.
     * Use this method with care as it can have a deep impact on the
     * error processing.
     *
     * @param errorClass
     */
    public static ErrorCode setErrorClass(final ErrorCode errorCode, int errorClass) {
        ErrorCode clone = errorCode.clone();
        if ((errorClass >= 0) && (errorClass < ERRORCLASS_COUNT)) {
            clone.m_errorClass = errorClass;
        }
        return clone;
    }

    /**
     * Provides a string representation of the error code instance.
     *
     * @return
     *  The string representation of the error code instance.
     */
    @Override
    public String toString() {
      return m_code + ": " + m_description;
    }

    /**
     * Provides a JSONObject which contains the state of the error
     * code instance.
     *
     * @return
     *  A JSONObject which contains the state of the error code
     *  instance or an empty object in case of an error.
     */
    public JSONObject getAsJSON() {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("errorClass", this.getErrorClass());
            jsonObject.put("errorCode", this.getCode());
            jsonObject.put("error", this.getCodeAsStringConstant());
            jsonObject.put("errorDescription", this.getDescription());
        } catch (final JSONException e) {
            //
        }
        return jsonObject;
    }

    /**
     * Creates a JSONObject that can be directly used as a result object
     * for server answers.
     *
     * @return
     *  A JSONObject which can be used as a server answer or an empty
     *  object in case of an error.
     */
    public JSONObject getAsJSONResultObject() {
    	JSONObject resultObject = new JSONObject();
    	JSONObject errorObject = this.getAsJSON();
    	try {
    		resultObject.put("error", errorObject);
    	} catch (final JSONException e) {
    		//
    	}
    	return resultObject;
    }

    /**
     * Extracts the error code from a JSON object that is the full representation
     * of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error code shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error code or the default value if the code couldn't be extracted
     *  from the JSONObject.
     */
    static public int getErrorCodeFromJSON(JSONObject jsonObject, int defaultValue) {
        int result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optInt("errorCode", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error constant string from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error constant string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error constant string or the default value if the code couldn't
     *  be extracted from the JSONObject.
     */
    static public String getErrorConstantFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("error", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error description from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error description string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error description cannot be extracted.
     *
     * @return
     *  The error description or the default value if the string couldn't be
     *  extracted from the JSONObject.
     */
    static public String getErrorDescriptionFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("errorDescription", defaultValue);
        }
        return result;
    }
 }
