/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.JAXBException;

import org.docx4j.XmlUtils;
import org.docx4j.dml.chart.CTAxDataSource;
import org.docx4j.dml.chart.CTBubbleSer;
import org.docx4j.dml.chart.CTChart;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTMultiLvlStrRef;
import org.docx4j.dml.chart.CTNumDataSource;
import org.docx4j.dml.chart.CTNumRef;
import org.docx4j.dml.chart.CTSerTx;
import org.docx4j.dml.chart.CTStrRef;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.chart.IListSer;
import org.docx4j.dml.chart.ISerContent;
import org.docx4j.dml.spreadsheetdrawing.ICellAnchor;
import org.docx4j.openpackaging.contenttype.ContentType;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.DefaultXmlPart;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.DrawingML.Drawing;
import org.docx4j.openpackaging.parts.DrawingML.JaxbDmlPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.docx4j.relationships.Relationships;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.BookViews;
import org.xlsx4j.sml.CTAutoFilter;
import org.xlsx4j.sml.CTBookView;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTConditionalFormatting;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTFilterColumn;
import org.xlsx4j.sml.CTHyperlink;
import org.xlsx4j.sml.CTHyperlinks;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.CTSheetFormatPr;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTTablePart;
import org.xlsx4j.sml.CTTableParts;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.IDataValidation;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.SmlUtils.CellRef;
import org.xlsx4j.sml.SmlUtils.CellRefRange;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.ooxml.drawingml.ChartWrapper;
import com.openexchange.office.ooxml.drawingml.DMLChartSpace;
import com.openexchange.office.ooxml.drawingml.DMLGraphic;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.AutoFilterHelper;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ChartUtils;
import com.openexchange.office.ooxml.xlsx.tools.ChartUtils.ColRow;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.TableHelper;
import com.openexchange.office.ooxml.xlsx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

	private final static Logger log = LoggerFactory.getLogger(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final WorkbookPart workbookPart;
    private final Map<String, CTXf> styleMap;
    private final Map<String, Long> styleIdToIndex = new HashMap<String, Long>();

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        spreadsheetMLPackage = _operationDocument.getPackage();
        workbookPart = spreadsheetMLPackage.getWorkbookPart();
        styleMap = BuiltinIds.getUsedStyles(operationDocument.getStylesheet(false), null, styleIdToIndex);
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    public void setCellContent(Cell cell, Object cellValue, Object cellResult, String cellRef, Long cellSi)
    	throws InvalidFormatException {

    	if(cellValue instanceof String) {
            final String v = (String)cellValue;
            if(v.length()>0&&v.startsWith("=")) {
                final CTCellFormula cellFormula = Context.getsmlObjectFactory().createCTCellFormula();
                cell.setT(null);
                cell.setF(cellFormula);
                if(cellSi==null) {
                    cellFormula.setT(STCellFormulaType.NORMAL);
                    cellFormula.setValue(v.substring(1));
                }
                else {
                    cellFormula.setT(STCellFormulaType.SHARED);
                    if(cellRef!=null) {
                        cellFormula.setValue(v.substring(1));
                        cellFormula.setRef(cellRef);
                    }
                    cellFormula.setSi(cellSi);
                }
            }
            else if(v.length()>0&&v.startsWith("#")) {
                cell.setT(STCellType.E);
                cell.setV(v);
                cell.setF(null);
            }
            else {
                cell.setF(null);
                if(v.startsWith("'")) {
                	cell.setT(STCellType.S);
                	cell.setV(getOperationDocument().applySharedString(v));
                }
                else {
                    cell.setT(STCellType.STR);
                    cell.setV(v);
                }
            }
        }
        else if(cellValue instanceof Number) {
            cell.setT(STCellType.N);
            cell.setV(((Number)cellValue).toString());
            cell.setF(null);
        }
        else if(cellValue instanceof Boolean) {
            cell.setT(STCellType.B);
            cell.setV((Boolean)cellValue ? "1" : "0");
            cell.setF(null);
        }
        else if(cellValue==JSONObject.NULL) {
            // empty cell ->
            cell.setT(null);
            cell.setF(null);
            cell.setIs(null);
            cell.setV(null);
        }
        if(cellResult!=null) {
        	// cellResult only possible if cellType == formula
        	if(cell.getF()!=null) {
        		if(cellResult instanceof Number) {
        			cell.setT(null);			// reseting error type
        			cell.setV(((Number)cellResult).toString());
        		}
        		else if (cellResult instanceof String) {
        			cell.setT(((String)cellResult).length()>0&&((String)cellResult).startsWith("#")?STCellType.E:STCellType.STR);
        			cell.setV((String)cellResult);
        		}
        	}
        }
    }

    public WorksheetPart getWorksheetPart(int sheetIndex) {
        final Sheet sheet = SheetUtils.getSheet(spreadsheetMLPackage, sheetIndex);
        final RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        return (WorksheetPart)relationshipPart.getPart(sheet.getId());
    }

    public Worksheet getWorksheet(int sheetIndex) {
        return getWorksheetPart(sheetIndex).getJaxbElement();
    }

    public void setCellContents(int sheetIndex, JSONArray start, JSONArray rowArray)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::setCellContents: size of start parameter != 2");

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        int y = start.getInt(1);
        for(int i=0; i<rowArray.length(); i++) {
            final JSONArray cellArray = rowArray.getJSONArray(i);
            int x = start.getInt(0);
            for(int j=0; j<cellArray.length(); j++) {

                final Row row = sheetData.getRow(y, true);

                final JSONObject cellObject = cellArray.optJSONObject(j);
                if(cellObject!=null) {
                    final Cell cell = row.getCell(x, true);
	                final Object value = cellObject.opt("value");
	                final Object result = cellObject.opt("result");
	                if(value!=null || result!=null) {
	                	setCellContent(cell, value, result , null, null);
	                }
	                final JSONObject attrs = cellObject.optJSONObject("attrs");
	                if(attrs!=null) {
	                	CellUtils.applyCellPropertiesToCell(operationDocument, worksheetPart, sheetData, attrs, row, cell);
	                }
                }
                x++;
            }
            y++;
        }
    }

    public void fillCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, Object optValue, Object optResult, JSONObject optAttrs, int optShared, JSONArray optRef)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of end parameter != 2");

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }

        final boolean clearContent = optValue==JSONObject.NULL;
        final boolean setContent = ((optValue!=null)||(optResult!=null))&&!clearContent;
        if(setContent||optAttrs!=null) {
            final Long cellSi = optRef!=null ? sheetData.getNextSi() : null;
            for(int y=start.getInt(1); y<=optEnd.getInt(1); y++) {
                for(int x=start.getInt(0); x<=optEnd.getInt(0); x++) {
                    final Row row = sheetData.getRow(y, true);
                    final Cell cell = row.getCell(x, true);
                    if(setContent||clearContent) {
                        if(optRef==null) {
                            setCellContent(cell, optValue, optResult, null, null);
                        }
                        else {
                            final String cellRef = optRef.getInt(0)==x&&optRef.getInt(1)==y ? SmlUtils.getCellRefRange(start.getInt(0), start.getInt(1), optEnd.getInt(0), optEnd.getInt(1)) : null;
                            setCellContent(cell, optValue, optResult, cellRef, cellSi);
                        }
                    }
                    if(optAttrs!=null) {
                    	CellUtils.applyCellPropertiesToCell(operationDocument, worksheetPart, sheetData, optAttrs, row, cell);
                    }
                }
            }
        }
    }

    public void clearCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, boolean remove)
        throws Exception {

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            return;
        }
        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyOperationHelper::clearCellContents: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyOperationHelper::clearCellContents: size of end parameter != 2");

        final SmlUtils.RowRangeIterator rowRangeIterator = new SmlUtils.RowRangeIterator(start.getInt(1), optEnd.getInt(1), sheetData.createRowIterator());
        while(rowRangeIterator.hasNext()) {
            final Row row = rowRangeIterator.next();
            if(remove) {
            	row.clearCellRange(start.getInt(0), optEnd.getInt(0));
            	if(row.isDefault()) {
            		rowRangeIterator.remove();
            	}
            }
            else {
            	final SmlUtils.CellRangeIterator cellRangeIterator = new SmlUtils.CellRangeIterator(start.getInt(0), optEnd.getInt(0), row.createCellIterator());
            	while(cellRangeIterator.hasNext()) {
            		final Cell cell = cellRangeIterator.next();
                    setCellContent(cell, JSONObject.NULL, null, null, null);
                }
            }
        }
    }

    private boolean nullOnlyAttributes(JSONObject attrs) {
        final Iterator<Entry<String, Object>> iter = attrs.entrySet().iterator();
        while(iter.hasNext()) {
            final Entry<String, Object> entry = iter.next();
            if(entry.getValue() instanceof JSONObject) {
                if(nullOnlyAttributes((JSONObject)entry.getValue())==false) {
                    return false;
                }
            }
            else if(entry.getValue()!=JSONObject.NULL) {
                return false;
            }
        }
        return true;
    }
    public void setRowAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        final Worksheet worksheet = getWorksheet(sheetIndex);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        final boolean nullOnly = nullOnlyAttributes(attrs);
        for(int y=start; y<=optEnd; y++) {
            SheetUtils.applyRowProperties(getOperationDocument(), sheetData, y, attrs);
            final Row row = sheetData.getRow(y, !nullOnly);
            if(row!=null) {
                RowUtils.applyRowProperties(getOperationDocument(), attrs, row, getOperationDocument().getStylesheet(true));
                sheetData.applyCachedRowData(row);
            }
        }
    }

    public void setColumnAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        final Worksheet worksheet = getWorksheet(sheetIndex);
        final List<Cols> colsList = worksheet.getCols();
        if(colsList.size()==0) {
            colsList.add(Context.getsmlObjectFactory().createCols());
        }

        // TODO: we only apply our attributes to the first Cols within the cols list, so additional
        // entries have to be merged ...
        start++;
        optEnd++;

        final Cols cols = colsList.get(0);
        Utils.createColumnAttributeRanges(cols, Long.valueOf(start), Long.valueOf((optEnd-start)+1));
        for(final Col col:cols.getCol()) {
            if(col.getMin()>optEnd)
                return;
            if(col.getMax()<start)
                continue;
            SheetUtils.applyColumnProperties(getOperationDocument(), worksheet.getSheetData(), (int)(col.getMin()-1), (int)(col.getMax()-1), attrs);
            ColumnUtils.applyColumnProperties(getOperationDocument(), attrs, col, worksheet, getOperationDocument().getStylesheet(true));
        }
    }

    public void insertRows(int sheetIndex, int start, int optEnd)
        throws Exception {

        if(start<0) {
            throw new RuntimeException("xlsx::insertRows: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::insertRows: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertRows(getOperationDocument().getStylesheet(true), start, count);

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            SmlUtils.insertRows(mergeCells.getMergeCell(), start, count, false);
            if(mergeCells.getMergeCell().isEmpty()) {
            	worksheet.setMergeCells(null);
            }
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            SmlUtils.insertRows(hyperlinks.getHyperlink(), start, count, false);
            if(hyperlinks.getHyperlink().isEmpty()) {
            	worksheet.setHyperlinks(null);
            }
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.insertRows(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, 1);
    }

    public void deleteRows(int sheetIndex, int start, int optEnd) {

        if(start<0) {
            throw new RuntimeException("xlsx::deleteRows: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::deleteRows: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        else {
            optEnd = start;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteRows, no sheetData available");
            return;
        }
        sheetData.deleteRows(start, count);

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            SmlUtils.deleteRows(mergeCells.getMergeCell(), start, count, true);
            if(mergeCells.getMergeCell().isEmpty()) {
            	worksheet.setMergeCells(null);
            }
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            SmlUtils.deleteRows(hyperlinks.getHyperlink(), start, count, false);
            if(hyperlinks.getHyperlink().isEmpty()) {
            	worksheet.setHyperlinks(null);
            }
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.deleteRows(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, -1);
    }

    public void insertColumns(int sheetIndex, int start, int optEnd)
        throws Exception {

        if(start<0) {
            throw new RuntimeException("xlsx::insertColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::insertColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertColumns(getOperationDocument().getStylesheet(true), start, count);

        // taking care of column attributes
        for(final Cols cols:worksheet.getCols()) {
            for(final Col col:cols.getCol()) {
                if(col.getMax()<start)
                    continue;
                else if(col.getMin()>start) {
                    col.setMin(Math.min(col.getMin()+count, 16384));
                    col.setMax(Math.min(col.getMax()+count, 16384));
                }
                else if(col.getMax()>=start) {
                    col.setMax(Math.min(col.getMax()+count, 16384));
                }
            }
        }

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
        	SmlUtils.insertColumns(mergeCells.getMergeCell(), start, count, false);
        	if(mergeCells.getMergeCell().isEmpty()) {
        		worksheet.setMergeCells(null);
        	}
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
        	SmlUtils.insertColumns(hyperlinks.getHyperlink(), start, count, false);
        	if(hyperlinks.getHyperlink().isEmpty()) {
        		worksheet.setHyperlinks(null);
        	}
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.insertColumns(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, 1);

        // taking care that the inserted columns are not hidden
        final JSONObject attrs = new JSONObject();
        final JSONObject column = new JSONObject();
        column.put("visible", JSONObject.NULL);
        attrs.put("column", column);
        setColumnAttributes(sheetIndex, start, optEnd, attrs);
    }

    public void deleteColumns(int sheetIndex, int start, int optEnd) {

        if(start<0) {
            throw new RuntimeException("xlsx::deleteColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::deleteColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        else {
            optEnd = start;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteColumns, no sheetData available");
            return;
        }
        sheetData.deleteColumns(start, count);

        // taking care of column attributes
        final List<Cols> cols = worksheet.getCols();
        for(int ci = cols.size()-1; ci>=0; ci--) {
            final List<Col> colList = cols.get(ci).getCol();
            final int cstart = start + 1;
            final int cend = start + count;
        	for(int i = colList.size()-1; i>=0; i--) {

        		final Col col = colList.get(i);
        		long min = col.getMin();
        		long max = col.getMax();

        		if(max<cstart) {
        		 	break;
        		}
        		if(min>cend) {
        			col.setMin(min-count);
        			col.setMax(max-count);
        		}
        		else if (min>=cstart&&max<=cend) {
        			colList.remove(i);
        			if(colList.isEmpty()) {
        				cols.remove(ci);
        			}
        		}
        		else if (cstart>=min&&cend<=max) {
        			col.setMax(max-count);
        		}
        		else if (cend>max) {
        			col.setMax(cstart-1);
        		}
        		else if (cstart<min) {
        			col.setMin(cstart);
        			col.setMax(max-count);
        		}
        	}
        }

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
        	SmlUtils.deleteColumns(mergeCells.getMergeCell(), start, count, true);
        	if(mergeCells.getMergeCell().isEmpty()) {
        		worksheet.setMergeCells(null);
        	}
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
        	SmlUtils.deleteColumns(hyperlinks.getHyperlink(), start, count, false);
        	if(hyperlinks.getHyperlink().isEmpty()) {
        		worksheet.setHyperlinks(null);
        	}
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.deleteColumns(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, -1);
    }

    private SmlUtils.CellRefRange modifyCellRefRange(SmlUtils.CellRefRange cellRefRange, int start, int count,ColRow colrow, int rel, boolean append) {

        SmlUtils.CellRefRange newCellRefRange = null;
        if(rel>0) {
            if(colrow==ColRow.Column) {
                newCellRefRange = SmlUtils.CellRefRange.insertColumnRange(cellRefRange, start, count, append);
            }
            else {
                newCellRefRange = SmlUtils.CellRefRange.insertRowRange(cellRefRange, start, count, append);
            }
        }
        else {
            if(colrow==ColRow.Column) {
                newCellRefRange = SmlUtils.CellRefRange.deleteColumnRange(cellRefRange, start, count);
            }
            else {
                newCellRefRange = SmlUtils.CellRefRange.deleteRowRange(cellRefRange, start, count);
            }
        }
        return newCellRefRange;
    }

    private void handleInsertDeleteColRow(int sheetIndex, int start, int count, ColRow colrow, int rel){

        final Workbook workbook = workbookPart.getJaxbElement();
        final List<Sheet> allSheets = workbook.getSheets().getSheet();
        final String sheetName = allSheets.get(sheetIndex).getName();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();

        for (int i = 0; i < allSheets.size(); i++) {
            final Sheet sheet = allSheets.get(i);
            final Part part = workbookRelationshipsPart.getPart(sheet.getId());
            if(part instanceof WorksheetPart) {
                final WorksheetPart sheetPart = (WorksheetPart)part;
                final RelationshipsPart sheetRelationshipsPart = sheetPart.getRelationshipsPart();
                if(sheetRelationshipsPart!=null) {
                    final Worksheet worksheet = sheetPart.getJaxbElement();

                    if(sheetIndex==i) {

                        // check dataValidations
                        final List<IDataValidations> validations = OperationDocument.getDataValidations(worksheet);
                        for(final IDataValidations dataValidations:validations) {
                            final List<? extends IDataValidation> dataValidationList = dataValidations.getDataValidation();
                            for(int j = dataValidationList.size()-1;j>=0;j--) {
                                final IDataValidation dataValidation = dataValidationList.get(j);
                                final List<String> sqrefList = dataValidation.getsqref();
                                for(int k=sqrefList.size()-1; k>=0; k--) {
                                    final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqrefList.get(k));
                                    if(cellRefRange!=null) {
                                        final SmlUtils.CellRefRange newCellRefRange = modifyCellRefRange(cellRefRange, start, count, colrow, rel, true);
                                        if(newCellRefRange==null) {
                                            sqrefList.remove(k);
                                        }
                                        else if(!newCellRefRange.equals(cellRefRange)) {
                                            sqrefList.set(k, SmlUtils.getCellRefRange(newCellRefRange));
                                        }
                                    }
                                }
                                if(sqrefList.size()==0) {
                                	// TODO: we need to ensure that also calcengine + frontend is deleting this data validation
                                    dataValidationList.remove(j);
                                }
                            }
                        }

                        // check ConditionalFormattings
                        final List<CTConditionalFormatting> conditionalFormattings = worksheet.getConditionalFormatting();
                        for(int j = conditionalFormattings.size()-1;j>=0;j--) {
                            final CTConditionalFormatting conditionalFormatting = conditionalFormattings.get(j);
                            final List<String> sqrefList = conditionalFormatting.getSqref();
                            for(int k=sqrefList.size()-1; k>=0; k--) {
                                final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqrefList.get(k));
                                if(cellRefRange!=null) {
                                    final SmlUtils.CellRefRange newCellRefRange = modifyCellRefRange(cellRefRange, start, count, colrow, rel, true);
                                    if(newCellRefRange==null) {
                                        sqrefList.remove(k);
                                    }
                                    else if(!newCellRefRange.equals(cellRefRange)) {
                                    	// TODO: we need to ensure that also calcengine + frontend is deleting this conditional formatting
                                        sqrefList.set(k, SmlUtils.getCellRefRange(newCellRefRange));
                                    }
                                }
                            }
                            if(sqrefList.size()==0) {
                            	conditionalFormattings.remove(j);
                            }
                        }

                        // check autoFilter
                        final CTAutoFilter autoFilter = worksheet.getAutoFilter();
                        if(autoFilter!=null) {
                            final SmlUtils.CellRefRange cellRefRange = autoFilter.getCellRefRange(false);
                            if(cellRefRange!=null) {
                                final SmlUtils.CellRefRange newCellRefRange = this.modifyCellRefRange(cellRefRange, start, count, colrow, rel, false);
                                if(newCellRefRange==null) {
                                    AutoFilterHelper.deleteTable(this, sheetIndex, null);
                                }
                                else if(!newCellRefRange.equals(cellRefRange)) {
                                    autoFilter.setCellRefRange(newCellRefRange);
                                    AutoFilterHelper.insertDefinedTableName(this, Long.valueOf(sheetIndex), null, newCellRefRange);

                                    // applying changed colIds
                                    if(colrow==ColRow.Column) {
                                        final List<CTFilterColumn> filterColumnList = autoFilter.getFilterColumn();
                                        for(int j=0; j<filterColumnList.size(); j++) {
                                            final CTFilterColumn filterColumn = filterColumnList.get(j);
                                            final long colId = filterColumn.getColId();
                                            if(rel>0&&colId>=start) {
                                                filterColumn.setColId(colId+count);
                                            }
                                            else if(((colId>=start)&&(colId<(start+count)))) {
                                                filterColumnList.remove(j--);
                                            }
                                            else if(colId>=start+count) {
                                                filterColumn.setColId(colId-count);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // for each worksheet -->

                    // check charts
                    final org.xlsx4j.sml.CTDrawing ctDrawing = worksheet.getDrawing();
                    if(ctDrawing!=null) {
                        final Drawing drawing = (Drawing)sheetRelationshipsPart.getPart(ctDrawing.getId());
                        if(drawing!=null) {
                            if(drawing.getRelationshipsPart().getRelationshipByType("http://schemas.openxmlformats.org/officeDocument/2006/relationships/chart")!=null) {

                                // if we have a drawing and if the drawing is containing a chart... we have to go deeper into detail
                                ChartUtils.handleInsertDeleteColRow(operationDocument, drawing, sheet.getName(), sheetName, start, count, colrow, rel);
                            }
                        }
                    }
                }
            }
        }
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
        throws InvalidFormatException, JAXBException, JSONException {

        // first we have to get a free relationship that can be used for our new sheet
        int sheetNumber = 1;

        Relationship relShip = null;
        PartName partName = null;
        do {
            partName = new PartName("/xl/worksheets/sheet" + Integer.toString(sheetNumber++) + ".xml");
            relShip = workbookPart.getRelationshipsPart().getRel(partName);
        }
        while(relShip!=null);

        // now we have a free partName... and we will look for a free SheetId
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();

        long sheetId = 1;
        do {
            boolean idFound = false;
            for(int i = 0; i < sheetList.size(); i++) {
                if(sheetList.get(i).getSheetId()==sheetId) {
                    idFound = true;
                    break;
                }
            }
            if(idFound==false)
                break;
            sheetId++;
        }
        while(true);

        final WorksheetPart worksheetPart = spreadsheetMLPackage.createWorksheetPart(partName, sheetName, sheetId);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        if(sheetFormatPr==null) {
            sheetFormatPr = Context.getsmlObjectFactory().createCTSheetFormatPr();
            worksheet.setSheetFormatPr(sheetFormatPr);
            sheetFormatPr.setBaseColWidth(new Long(10L));
            sheetFormatPr.setDefaultRowHeight(14.5d);
        }

        // setting the correct index position
        final Sheet sheet = sheetList.remove(sheetList.size() - 1);
        sheetList.add(sheetIndex, sheet);
        if(attrs!=null) {
            SheetUtils.applySheetProperties(workbook, sheetIndex, worksheet, attrs.optJSONObject("sheet"));
            SheetUtils.applyWorksheetColumnProperties(operationDocument, worksheet, attrs.optJSONObject("column"));
            SheetUtils.applyWorksheetRowProperties(operationDocument, worksheet, attrs.optJSONObject("row"));
        }

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            for(final CTDefinedName definedName:definedNames.getDefinedName()) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId>=sheetIndex) {
                        definedName.setLocalSheetId(localSheetId+1);
                    }
                }
            }
        }
    }

    public void moveSheet(int sheetIndex, int to) {
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        sheetList.add(to, sheetList.remove(sheetIndex));

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(final CTDefinedName definedName:definedNameList) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    long newSheetId;
                    if(localSheetId==sheetIndex) {
                        newSheetId = to;
                    }
                    else {
                        newSheetId = localSheetId;
                        if(sheetIndex<newSheetId) {
                            newSheetId--;
                        }
                        if(newSheetId>=to) {
                            newSheetId++;
                        }
                    }
                    definedName.setLocalSheetId(newSheetId);
                }
            }
        }
    }

    public void deleteSheet(int sheetIndex) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        final Sheet sheet = sheetList.remove(sheetIndex);
        final String relId = sheet.getId();
        final RelationshipsPart relationshipsPart = workbookPart.getRelationshipsPart();
        relationshipsPart.removeRelationship(relationshipsPart.getRelationshipByID(relId));

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(int i=0;i<definedNameList.size();i++) {
                final CTDefinedName definedName = definedNameList.get(i);
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId==sheetIndex) {
                        definedNameList.remove(i--);
                    }
                    else if(localSheetId>sheetIndex) {
                        definedName.setLocalSheetId(localSheetId-1);
                    }
                }
            }
        }
    }

    public void copySheet(int sheet, int to, String sheetName)
    	throws Docx4JException, InvalidFormatException, JAXBException, JSONException {

    	insertSheet(to, sheetName, null);
    	if(to<=sheet) {
    		sheet++;
    	}
    	// create and insert a new copy of our worksheet
        final WorksheetPart sourceWorksheetPart = getWorksheetPart(sheet);
        final WorksheetPart destWorksheetPart   = getWorksheetPart(to);
        final Worksheet sourceWorksheet = sourceWorksheetPart.getJaxbElement();
        final Worksheet destWorksheet = XmlUtils.deepCopy(sourceWorksheet, Context.getJcSML());
        destWorksheetPart.setJaxbElement(destWorksheet);

        // now we have to clone the relationships
    	cloneRelationshipsPart(destWorksheetPart, sourceWorksheetPart, "http://schemas.openxmlformats.org/officeDocument/2006/relationships/drawing");

        final Workbook workbook = workbookPart.getJaxbElement();
        final String encodedOldName = encodeSheetName(workbook.getSheets().getSheet().get(sheet).getName());
        final String encodedNewName = encodeSheetName(workbook.getSheets().getSheet().get(to).getName());
        updateWorksheetName(destWorksheetPart, encodedOldName, encodedNewName);
        
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
        	// duplicating definedNames with localSheetId
        	final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
        	for(int i=definedNameList.size()-1;i>=0;i--) {
        		final CTDefinedName sourceName = definedNameList.get(i);
        		if(sourceName.getLocalSheetId()!=null&&sourceName.getLocalSheetId().longValue()==sheet) {
        			final CTDefinedName destName = sourceName.clone();
        			destName.setLocalSheetId(Long.valueOf(to));
                    destName.setValue(updateSheetName(destName.getValue(), encodedOldName, encodedNewName));
        			definedNameList.add(destName);
        		}
        	}
        }
    }

    private void cloneRelationshipsPart(Part destPart, Part sourcePart, String type)
    	throws Docx4JException, InvalidFormatException, JAXBException, PartUnrecognisedException {

   		final RelationshipsPart sourceRelationshipsPart = sourcePart.getRelationshipsPart(false);
        if(sourceRelationshipsPart!=null) {
            final Relationships sourceRelationships = sourceRelationshipsPart.getRelationships();
	    	final List<Relationship> sourceRelationshipList = sourceRelationships.getRelationship();
	        for(int relIndex = 0; relIndex<sourceRelationshipList.size(); relIndex++) {
	        	final Relationship sourceRelationship = sourceRelationshipList.get(relIndex);
	        	final String sourceType = sourceRelationship.getType();
	        	final String targetMode = sourceRelationship.getTargetMode();
	        	if((targetMode==null||targetMode.equalsIgnoreCase("Internal"))&&!sourceType.equals(Namespaces.HYPERLINK)) {
	        		if(type==null||sourceType.equals(type)) {
	        		    try {
    						final Part part = sourceRelationshipsPart.getPart(sourceRelationship.getId());
    						if(part instanceof JaxbDmlPart) {
    							cloneDmlPart(destPart, (JaxbDmlPart<?>)part, sourceRelationship.getId());
    							continue;
    						}
    						else if(part instanceof DefaultXmlPart) {
    							cloneDefaultXmlPart(destPart, (DefaultXmlPart)part, sourceRelationship.getId());
    							continue;
    						}
	        		    }
	        		    catch(InvalidFormatException e) {
	        		        //
	        		    }
	        		}
	        	}
        		destPart.getRelationshipsPart(true).getRelationships().getRelationship().add(XmlUtils.deepCopy(sourceRelationship, org.docx4j.jaxb.Context.getJcRelationships()));
	        }
        }
    }

    public void insertHyperlink(int sheetIndex, JSONArray start, JSONArray optEnd, String url)
    	throws JSONException {

        if(optEnd==null) {
            optEnd = start;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks==null) {
        	hyperlinks = Context.getsmlObjectFactory().createCTHyperlinks();
        	worksheet.setHyperlinks(hyperlinks);
        }
        final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
		final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
        for(int i=hyperlinkList.size()-1;i>=0;i--) {
        	final CTHyperlink hyperlink = hyperlinkList.get(i);
        	if((hyperlink.getCellRefRange(false)==null)||cellRefRange.contains(hyperlink.getCellRefRange(false))) {
        		hyperlinkList.remove(i);
        	}
        }
        final CTHyperlink hyperlink = Context.getsmlObjectFactory().createCTHyperlink();
        hyperlink.setCellRefRange(cellRefRange);
        hyperlink.setId(Commons.setUrl(worksheetPart, hyperlink.getId(), url));
        hyperlinks.getHyperlink().add(hyperlink);
    }

    public void deleteHyperlink(int sheetIndex, JSONArray start, JSONArray optEnd)
    	throws JSONException {

    	if(optEnd==null) {
            optEnd = start;
        }
    	final Worksheet worksheet = getWorksheet(sheetIndex);
    	final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
    	if(hyperlinks!=null) {
    		final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
    		final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
    		for(int i=hyperlinkList.size()-1;i>=0;i--) {
    			final CTHyperlink hyperlink = hyperlinkList.get(i);
    			if((hyperlink.getCellRefRange(false)==null)||cellRefRange.intersects(hyperlink.getCellRefRange(false))) {
    				hyperlinkList.remove(i);
    			}
    		}
    		if(hyperlinkList.isEmpty()) {
    			worksheet.setHyperlinks(null);
    		}
    	}
    }

    private void cloneDmlPart(Part destPart, JaxbDmlPart<?> sourcePart, String relationshipId)
    	throws Docx4JException, InvalidFormatException, JAXBException, PartUnrecognisedException {

		final JaxbDmlPart<?> clonedPart = (JaxbDmlPart<?>)JaxbDmlPart.newPartForContentType(sourcePart.getContentType(), sourcePart.getPartName().getName());
    	final ByteArrayOutputStream baos = new ByteArrayOutputStream();
    	sourcePart.marshal(baos);
		clonedPart.unmarshal(new ByteArrayInputStream(baos.toByteArray()));
        destPart.addTargetPart(clonedPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS, relationshipId);
        cloneRelationshipsPart(clonedPart, sourcePart, null);
    }

    private void cloneDefaultXmlPart(Part destPart, DefaultXmlPart sourcePart, String relationshipId)
    	throws Docx4JException, JAXBException {

		final Document destDoc = (Document) sourcePart.getDocument().cloneNode(true);
		final DefaultXmlPart clonedPart = new DefaultXmlPart(sourcePart.getPartName());
		clonedPart.setContentType(new ContentType(sourcePart.getContentType()));
		clonedPart.setRelationshipType(sourcePart.getRelationshipType());
		destPart.addTargetPart(clonedPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS, relationshipId);
		clonedPart.setDocument(destDoc);
		cloneRelationshipsPart(clonedPart, sourcePart, null);
    }

    public void setSheetName(int sheetIndex, String newSheetName) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheet _sheet = workbook.getSheets().getSheet().get(sheetIndex);

        final String oldSheetName = _sheet.getName();
        final String encodedOldName = encodeSheetName(oldSheetName);
        final String encodedNewName = encodeSheetName(newSheetName);

        if(encodedOldName.length()<=0&&encodedNewName.length()<=0) {
            return;
        }

        _sheet.setName(newSheetName);

        // updating defined names
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            for(final CTDefinedName definedName:definedNames.getDefinedName()) {
                definedName.setValue(updateSheetName(definedName.getValue(), encodedOldName, encodedNewName));
            }
        }
        final List<Sheet> allSheets = workbook.getSheets().getSheet();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();
        for (int i = 0; i < allSheets.size(); i++) {
            final Sheet sheet = allSheets.get(i);
            final Part part = workbookRelationshipsPart.getPart(sheet.getId());
            if(part instanceof WorksheetPart) {
            	updateWorksheetName((WorksheetPart)part, encodedOldName, encodedNewName);
            }
        }
    }

    /* following method changes the absolute sheet addressing of each formula, chart formula
     * and dataValidation from encodedOldName to encodedNewName. This is done if a sheet
     * was renamed or copied */
    private void updateWorksheetName(WorksheetPart sheetPart, String encodedOldName, String encodedNewName) {
	    final RelationshipsPart sheetRelationshipsPart = sheetPart.getRelationshipsPart();
	    if(sheetRelationshipsPart!=null) {
	        final Worksheet worksheet = sheetPart.getJaxbElement();
	        final SheetData sheetData = worksheet.getSheetData();
	        if(sheetData!=null) {
	            // updating cells
	            final Iterator<Row> rowIterator = sheetData.createRowIterator();
	            while(rowIterator.hasNext()) {
	                final Row row = rowIterator.next();
	                final Iterator<Cell> cellIterator = row.createCellIterator();
	                while(cellIterator.hasNext()) {
	                    final Cell cell = cellIterator.next();
	                    final CTCellFormula cellFormula = cell.getF();
	                    if(cellFormula!=null) {
	                        cellFormula.setR1(updateSheetName(cellFormula.getR1(), encodedOldName, encodedNewName));
	                        cellFormula.setR2(updateSheetName(cellFormula.getR2(), encodedOldName, encodedNewName));
	                        cellFormula.setRef(updateSheetName(cellFormula.getRef(), encodedOldName, encodedNewName));
	                        cellFormula.setValue(updateSheetName(cellFormula.getValue(), encodedOldName, encodedNewName));
	                    }
	                }
	            }
	            // updating data validations
	            final List<IDataValidations> validations = OperationDocument.getDataValidations(worksheet);
	            for(final IDataValidations dataValidations:validations) {
	                for(final IDataValidation dataValidation:dataValidations.getDataValidation()) {
	                    dataValidation.setFormula1(updateSheetName(dataValidation.getFormula1(), encodedOldName, encodedNewName));
	                    dataValidation.setFormula2(updateSheetName(dataValidation.getFormula2(), encodedOldName, encodedNewName));
	                }
	            }
	            // updating charts
	            final org.xlsx4j.sml.CTDrawing ctDrawing = worksheet.getDrawing();
	            if(ctDrawing!=null) {
	                final Drawing drawing = (Drawing)sheetRelationshipsPart.getPart(ctDrawing.getId());
	                if(drawing!=null) {
	                    if(drawing.getRelationshipsPart().getRelationshipByType("http://schemas.openxmlformats.org/officeDocument/2006/relationships/chart")!=null) {
	                        final List<ICellAnchor> drawings = drawing.getJaxbElement().getEGAnchor();
	                        for (final ICellAnchor ca : drawings) {
	                            final Map<String, Object> drawingPropertyMap = DMLGraphic.getDrawingType(operationDocument, drawing, ca);
	                            final ChartWrapper chart = Drawings.getChart(drawingPropertyMap);
	                            if (chart != null) {
	                                final List<IListSer> series = chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart();
	                                for (final IListSer serie : series) {
	                                    for (final ISerContent content : serie.getSer()) {
	                                        final CTSerTx tx = content.getTx();
	                                        if(tx!=null) {
	                                            final CTStrRef strRef = tx.getStrRef();
	                                            if(strRef!=null) {
	                                                strRef.setF(updateSheetName(strRef.getF(), encodedOldName, encodedNewName));
	                                            }
	                                        }
	                                        final CTAxDataSource ctAx = content.getCat();
	                                        if(ctAx!=null) {
	                                            final CTMultiLvlStrRef multiLvlStrRef = ctAx.getMultiLvlStrRef();
	                                            if(multiLvlStrRef!=null) {
	                                                multiLvlStrRef.setF(updateSheetName(multiLvlStrRef.getF(), encodedOldName, encodedNewName));
	                                            }
	                                            final CTNumRef numRef = ctAx.getNumRef();
	                                            if(numRef!=null) {
	                                                numRef.setF(updateSheetName(numRef.getF(), encodedOldName, encodedNewName));
	                                            }
	                                            final CTStrRef strRef = ctAx.getStrRef();
	                                            if(strRef!=null) {
	                                                strRef.setF(updateSheetName(strRef.getF(), encodedOldName, encodedNewName));
	                                            }
	                                        }
	                                        final CTNumDataSource numDataSource = content.getVal();
	                                        if(numDataSource!=null) {
	                                           final CTNumRef numRef = numDataSource.getNumRef();
	                                           if(numRef!=null) {
	                                               numRef.setF(updateSheetName(numRef.getF(), encodedOldName, encodedNewName));
	                                           }
	                                        }
	                                        if(content instanceof CTBubbleSer) {
	                                        	final CTNumDataSource bubbleSize = ((CTBubbleSer)content).getBubbleSize();
	                                        	if(bubbleSize!=null) {
	                                        		final CTNumRef numRef = bubbleSize.getNumRef();
	                                        		if(numRef!=null) {
	                                        			numRef.setF(updateSheetName(numRef.getF(), encodedOldName, encodedNewName));
	                                        		}
	                                        	}
	                                        }
	                                    }
	                                }
	                            }
	                        }
	                    }
	                }
	            }
	        }
	    }
    }

    public static int getNameOffset(String content, int startOffset) {

        boolean stringLiteral = false;
        boolean complexString = false;
        for(int i=startOffset; i<content.length();i++) {
            final char n = content.charAt(i);
            if(complexString) {
                if(n=='\'') {
                    if(i+1<content.length()&&content.charAt(i+1)=='\'') {
                        i++;
                    }
                    else {
                        complexString = false;
                    }
                }
            }
            else if(stringLiteral) {
                if(n=='"') {
                    if(i+1<content.length()&&content.charAt(i+1)=='"') {
                        i++;
                    }
                    else {
                        stringLiteral = false;
                    }
                }
            }
            else if(n=='\'') {
                complexString = true;
            }
            else if(n=='"') {
                stringLiteral = true;
            }
            else if(n=='!') {
                return i;
            }
        }
        return -1;
    }

    private static String simpleCharPattern ="^[\\w.\\xa1-\\u2027\\u202a-\\uffff]+$";

    public static String updateSheetName(String content, String oldName, String newName) {
        String newContent = content;
        List<Integer> replaceOffsets = null;
        if(content!=null&&content.length()>oldName.length()) {
            for(int startOffset = 0; startOffset<content.length();) {
                final int nameOffset = getNameOffset(content, startOffset);
                if(nameOffset<0) {
                    break;
                }
                final int possibleLength = nameOffset - startOffset;
                if(possibleLength>=oldName.length()) {
                    boolean replace = false;
                    final boolean isComplex = content.charAt(nameOffset-1)=='\'';
                    final String subString = content.substring(nameOffset-oldName.length(), nameOffset);

                    final int mOffset = nameOffset - oldName.length();
                    if(isComplex&&oldName.charAt(0)=='\'') {
                        if(subString.equals(oldName)) {
                            replace = true;
                        }
                    }
                    else if(oldName.charAt(0)!='\'') {
                        if(subString.equals(oldName)) {
                            if(mOffset>startOffset) {
                                final String prev = content.substring(mOffset-1, mOffset);
                                if(!prev.matches(simpleCharPattern)) {
                                    replace = true;
                                }
                            }
                            else {
                                replace = true;
                            }
                        }
                    }
                    if(replace) {
                        if(replaceOffsets==null) {
                            replaceOffsets = new ArrayList<Integer>();
                        }
                        replaceOffsets.add(mOffset);
                    }
                }
                startOffset = nameOffset + 1;
            }
        }
        if(replaceOffsets!=null) {
            final StringBuffer buffer = new StringBuffer(content);
            for(int i=replaceOffsets.size()-1;i>=0;i--) {
                final int offset = replaceOffsets.get(i);
                buffer.replace(offset, offset + oldName.length(), newName);
            }
            newContent = buffer.toString();
        }
        return newContent;
    }

    public static String encodeSheetName(String sheetName) {
        final StringBuffer encodedName = new StringBuffer(sheetName.length());
        for(int i=0; i<sheetName.length();i++) {
            final char c = sheetName.charAt(i);
            encodedName.append(c);
            if(c=='\'') {
                encodedName.append(c);
            }
        }
        if(!sheetName.matches(simpleCharPattern)) {
            encodedName.insert(0, '\'');
            encodedName.append('\'');
        }
        return encodedName.toString();
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs)
    	throws JSONException {

    	if(attrs!=null) {
            final Workbook workbook = workbookPart.getJaxbElement();
            final Worksheet worksheet = getWorksheet(sheetIndex);
            final JSONObject sheetProperties = attrs.optJSONObject("sheet");
            if(sheetProperties!=null) {
                SheetUtils.applySheetProperties(workbook, sheetIndex, worksheet, attrs.optJSONObject("sheet"));
                final boolean visible = sheetProperties.optBoolean("visible", true);
                if(!visible) {
                    SheetUtils.updateBookView(workbook);
                }
            }
            final JSONObject columnProperties = attrs.optJSONObject("column");
            if(columnProperties!=null) {
                SheetUtils.applyWorksheetColumnProperties(getOperationDocument(), worksheet, columnProperties);
            }
            final JSONObject rowProperties = attrs.optJSONObject("row");
            if(rowProperties!=null) {
                SheetUtils.applyWorksheetRowProperties(getOperationDocument(), worksheet, rowProperties);
            }
        }
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
        throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        final Worksheet worksheet = getWorksheet(sheetIndex);
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells==null) {
            mergeCells = Context.getsmlObjectFactory().createCTMergeCells();
            worksheet.setMergeCells(mergeCells);
        }
        // first we will remove each mergeCell that is covering our new mergeCellRange
        final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(new SmlUtils.CellRef(start.getInt(0), start.getInt(1)), new SmlUtils.CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            final CTMergeCell mergeCell = mergeCellList.get(i);
            final SmlUtils.CellRefRange mergedRefRange = mergeCell.getCellRefRange(false);
            if(mergedRefRange==null) {
                log.warn("xlsx export: mergeRange is invalid");
            }
            else {
                if(cellRefRange.intersects(mergedRefRange)) {
                    mergeCellList.remove(i);
                }
            }
        }
        if(type.equals("merge")) {
            final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
            mergeCell.setCellRefRange(new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1))));
            mergeCellList.add(mergeCell);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setCellRefRange(new CellRefRange(new CellRef(start.getInt(0), row), new CellRef(optEnd.getInt(0), row)));
                mergeCellList.add(mergeCell);
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setCellRefRange(new CellRefRange(new CellRef(column, start.getInt(1)), new CellRef(column, optEnd.getInt(1))));
                mergeCellList.add(mergeCell);
            }
        }
        if(mergeCells.getMergeCell().isEmpty()) {
            worksheet.setMergeCells(null);
        }
    }

    public Long getStyleIndex(String styleId) {
        if(styleId==null||styleId.isEmpty()) {
            return Long.valueOf(0);
        }
        return styleIdToIndex.get(styleId);
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, String parent, boolean hidden, int uiPriority, boolean defaultStyle, boolean autostyle)
        throws Exception {

        final CTStylesheet stylesheet = operationDocument.getStylesheet(true);
    	CTXf cellStyleXf = null;
    	if(autostyle) {
    		final CTCellXfs ctXfs = stylesheet.getCellXfs();
    		if(ctXfs!=null&&styleId.startsWith("a")) {
        		final List<CTXf> ctXfList = ctXfs.getXf();
    			final int autoStyleIndex = Integer.valueOf(styleId.substring(1));
    			if(autoStyleIndex>=0&&autoStyleIndex<=ctXfList.size()) {
    				if(autoStyleIndex<ctXfList.size()) {
    					cellStyleXf = ctXfList.get(autoStyleIndex);
    				}
    				else {
    					cellStyleXf = Context.getsmlObjectFactory().createCTXf();
    		            styleMap.put(styleId, cellStyleXf);
    		            styleIdToIndex.put(styleId, Long.valueOf(ctXfList.size()));
    					ctXfList.add(cellStyleXf);
    				}
    				if(parent!=null&&!parent.isEmpty()) {
    					final CTCellStyles cellStyles = stylesheet.getCellStyles();
    					if(cellStyles!=null) {
    						for(CTCellStyle cellStyle:cellStyles.getCellStyle()) {
    							if(parent.equals(cellStyle.getName())) {
    								cellStyleXf.setXfId(cellStyle.getXfId());
    								break;
    							}
    						}
    					}
    				}
    			}
    		}
    	}
    	else {
	        CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
	        if(cellStyleXfs==null) {
	            cellStyleXfs = Context.getsmlObjectFactory().createCTCellStyleXfs();
	        }
	        cellStyleXf = styleMap.get(styleId);
	        if(cellStyleXf==null) {
	            cellStyleXf = Context.getsmlObjectFactory().createCTXf();
	            final List<CTXf> ctXfList = cellStyleXfs.getXf();
	            final Long ctXfIndex = new Long(ctXfList.size());
	            ctXfList.add(cellStyleXf);

	            CTCellStyles cellStyles = stylesheet.getCellStyles();
	            if(cellStyles==null) {
	                cellStyles = Context.getsmlObjectFactory().createCTCellStyles();
	            }
	            final List<CTCellStyle> cellStyleList = cellStyles.getCellStyle();
	            final CTCellStyle cellStyle = Context.getsmlObjectFactory().createCTCellStyle();
	            cellStyleList.add(cellStyle);
	            cellStyle.setBuiltinId(BuiltinIds.getBuiltinId(styleName));
	            cellStyle.setName(styleName);
	            cellStyle.setHidden(hidden);
	            cellStyle.setXfId(ctXfIndex);
	            styleMap.put(styleId, cellStyleXf);
	            styleIdToIndex.put(styleId, ctXfIndex);
	        }
	        final JSONObject cellApplyProperties = attrs.optJSONObject("apply");
	        if(cellApplyProperties!=null) {
	            final Object applyBorder = cellApplyProperties.opt("border");
	            if(applyBorder!=null) {
	                cellStyleXf.setApplyBorder(applyBorder instanceof Boolean&&!((Boolean)applyBorder).booleanValue()?false:null);
	            }
	            final Object applyAlign = cellApplyProperties.opt("align");
	            if(applyAlign!=null) {
	                cellStyleXf.setApplyAlignment(applyAlign instanceof Boolean&&!((Boolean)applyAlign).booleanValue()?false:null);
	            }
	            final Object applyNumber = cellApplyProperties.opt("number");
	            if(applyNumber!=null) {
	                cellStyleXf.setApplyNumberFormat(applyNumber instanceof Boolean&&!((Boolean)applyNumber).booleanValue()?false:null);
	            }
	            final Object applyProtect = cellApplyProperties.opt("protect");
	            if(applyProtect!=null) {
	                cellStyleXf.setApplyProtection(applyProtect instanceof Boolean&&!((Boolean)applyProtect).booleanValue()?false:null);
	            }
	            final Object applyFont = cellApplyProperties.opt("font");
	            if(applyFont!=null) {
	                cellStyleXf.setApplyProtection(applyFont instanceof Boolean&&!((Boolean)applyFont).booleanValue()?false:null);
	            }
	            final Object applyFill = cellApplyProperties.opt("fill");
	            if(applyFill!=null) {
	                cellStyleXf.setApplyFill(applyFill instanceof Boolean&&!((Boolean)applyFill).booleanValue()?false:null);
	            }
	        }
    	}
    	if(cellStyleXf!=null) {
	        final JSONObject cellProperties = attrs.optJSONObject("cell");
	        if(cellProperties!=null) {
	            CellUtils.applyCellProperties(operationDocument, cellProperties, cellStyleXf, stylesheet, autostyle);
	        }
	        final JSONObject characterProperties = attrs.optJSONObject("character");
	        if(characterProperties!=null) {
	            CellUtils.applyCharacterProperties(operationDocument, characterProperties, cellStyleXf, stylesheet, autostyle);
	        }
    	}
    }

    private Drawing getDrawing(int sheetIndex)
        throws Exception {

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        final org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final String drawingsId = drawing.getId();
        if(drawingsId==null||drawingsId.isEmpty()) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
        return (Drawing)worksheetRelationships.getPart(drawingsId);
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws Exception {

        Drawing drawingPart = null;
        final int sheetNumber = start.getInt(0);
        final WorksheetPart worksheetPart = getWorksheetPart(sheetNumber);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) { // -> drawing part needs to be created
            drawing = Context.getsmlObjectFactory().createCTDrawing();
            worksheet.setDrawing(drawing);

            drawingPart = new Drawing(new PartName("/xl/drawings/drawing"+ Integer.toString(sheetNumber+1) + ".xml"));
            drawingPart.setJaxbElement(org.docx4j.jaxb.Context.getDmlSpreadsheetDrawingObjectFactory().createCTDrawing());
            final Relationship dmlRelationship = worksheetPart.addTargetPart(drawingPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
            drawing.setId(dmlRelationship.getId());
        }
        else {
            drawingPart = (Drawing)worksheetPart.getRelationshipsPart().getPart(drawing.getId());
        }
        Drawings.insertDrawing(getOperationDocument(), start.getInt(1), drawingPart, type, attrs);

        applyDrawingAttributes(start, attrs);
    }

    public void applyDrawingAttributes(JSONArray start, JSONObject attrs)
        throws Exception {

        final org.docx4j.openpackaging.parts.DrawingML.Drawing drawingsPart = getDrawing(start.getInt(0));
        Drawings.applyDrawingAttributes(Drawings.applyAnchorAttributes(start.getInt(1), drawingsPart, attrs), attrs);

        final org.docx4j.dml.spreadsheetdrawing.CTDrawing drawings = drawingsPart.getJaxbElement();
        final ICellAnchor cellAnchor = drawings.getEGAnchor().get(start.getInt(1));
        final HashMap<String, Object> drawingPropertyMap = DMLGraphic.getDrawingType(operationDocument, drawingsPart, cellAnchor);

        final JSONObject chartProperties = attrs.optJSONObject("chart");
        if(chartProperties!=null) {
            DMLChartSpace.applyChartSpaceProperties(chartProperties, Drawings.getChart(drawingPropertyMap), attrs.optJSONObject("fill"));
        }
    }

    public void deleteDrawing(JSONArray start)
        throws Exception {

        final Drawing drawing = getDrawing(start.getInt(0));
        final List<ICellAnchor> egAnchor = drawing.getJaxbElement().getEGAnchor();
        egAnchor.remove(start.getInt(1));
        if(egAnchor.isEmpty()) {
        	// removing the drawing relation, so the drawing will not be saved
        	final Worksheet worksheet = getWorksheet(start.getInt(0));
        	worksheet.setDrawing(null);
        }
    }

    public void insertChartDataSeries(JSONArray start, int series, JSONObject attrs)
        throws Exception {

        final Workbook workbook = workbookPart.getJaxbElement();
        final String sheetName = workbook.getSheets().getSheet().get(start.getInt(0)).getName();
        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.insertChartSpace(Drawings.getChart(drawingPropertyMap), sheetName, series, attrs);
    }

    public void setChartDataSeriesAttributes(JSONArray start, int series, JSONObject attrs)
            throws Exception {

            final Workbook workbook = workbookPart.getJaxbElement();
            final String sheetName = workbook.getSheets().getSheet().get(start.getInt(0)).getName();
            final Drawing drawing = getDrawing(start.getInt(0));
            final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
            DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
            DMLChartSpace.setDataSeriesAttributes(Drawings.getChart(drawingPropertyMap), sheetName, series, attrs);
        }

    public void deleteChartDataSeries(JSONArray start, int series)
        throws Exception {

        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.deleteChartSpace(Drawings.getChart(drawingPropertyMap), series);
    }

    public void insertName(Long sheet, String name, String formula, Boolean hidden) {

    	if(sheet<0) {
    		sheet = null;
    	}

        final Workbook workbook = workbookPart.getJaxbElement();
        DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            definedNames = Context.getsmlObjectFactory().createDefinedNames();
            workbook.setDefinedNames(definedNames);
        }
        final HashMap<String, CTDefinedName> definedNameMap = definedNames.getDefinedNameMap();
        final CTDefinedName definedName = Context.getsmlObjectFactory().createCTDefinedName();
        definedName.setName(name);
        definedName.setLocalSheetId(sheet);
        definedName.setValue(formula);
        if(hidden!=null) {
            definedName.setHidden(hidden.booleanValue()?true:null);
        }
        definedNameMap.put(DefinedNames.getDefinedNameKey(sheet, name), definedName);
    }

    public void changeName(Long sheet, String name, String formula, String newName) {

    	if(sheet<0) {
    		sheet = null;
    	}
    	final Workbook workbook = workbookPart.getJaxbElement();
        final DefinedNames definedNames = workbook.getDefinedNames();
        final CTDefinedName oldName = definedNames.getDefinedNameMap().get(DefinedNames.getDefinedNameKey(sheet, name));
        definedNames.getDefinedNameMap().remove(DefinedNames.getDefinedNameKey(sheet, name));
        if(newName!=null) {
        	oldName.setName(newName);
        }
        if(formula!=null) {
        	oldName.setValue(formula);
        }
        definedNames.getDefinedNameMap().put(DefinedNames.getDefinedNameKey(sheet, newName!=null?newName:name), oldName);
    }

    public void deleteName(Long sheet, String name) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            definedNames.getDefinedNameMap().remove(DefinedNames.getDefinedNameKey(sheet, name));
        }
        log.debug("xlsx:deleteName ... could not find name within definedNameList");
    }

	public void setChartAxisAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
		DMLChartSpace.setAxisAttributes(getChart(start), axis, attrs);
	}

	public void setChartGridlineAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
		DMLChartSpace.setChartGridlineAttributes(getChart(start), axis, attrs);
	}

	public void setChartTitleAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
	    final ChartWrapper chart = getChart(start);

		if (axis.equals("main")) {
		    final CTChartSpace chartSpace = chart.getChartSpace();
			final CTChart ctChart = chartSpace.getChart();
			CTTitle title = ctChart.getTitle();
			if (title == null) {
				title = new CTTitle();
				ctChart.setTitle(title);
			}
			DMLChartSpace.setTitleFromAttrs(title, attrs);
		} else {
			DMLChartSpace.setChartTitleAttributes(chart, axis, attrs);
		}
	}

	public void setChartLegendAttributes(JSONArray start, JSONObject attrs) throws Exception {
		DMLChartSpace.setLegendFromAttrs(getChart(start), attrs);
	}

	private ChartWrapper getChart(JSONArray start) throws Exception {
	    final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
	    return Drawings.getChart(drawingPropertyMap);
	}

	public void setDocumentAttributes(JSONObject attrs) {
		final JSONObject documentAttrs = attrs.optJSONObject("document");
		if(documentAttrs!=null) {
			final int activeSheet = documentAttrs.optInt("activeSheet", 0);
	        final Workbook workbook = workbookPart.getJaxbElement();
			BookViews bookViews = workbook.getBookViews();
			if(bookViews==null) {
				bookViews = Context.getsmlObjectFactory().createBookViews();
				workbook.setBookViews(bookViews);
			}
    		final List<CTBookView> bookViewList = bookViews.getWorkbookView();
    		if(bookViewList.isEmpty()) {
    			final CTBookView newBookView = Context.getsmlObjectFactory().createCTBookView();
    			bookViewList.add(newBookView);
    		}
			final CTBookView bookView = bookViewList.get(0);
			bookView.setActiveTab(new Long(activeSheet));
		}
	}
}
