/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.odf;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.ods.dom.Content;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.doc.DocumentComplexity;
import com.openexchange.office.tools.memory.MemoryObserver;
import com.openexchange.office.tools.memory.MemoryObserver.MemoryListener;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * The access point to the ODF Toolkit, obfuscating all OX Office dependencies
 * from the Toolkit.
 *
 */
public class OdfOperationDoc {

	private OdfOperationDocument mDocument;
	private static final Logger LOG = LoggerFactory.getLogger(OdfOperationDoc.class);
    private static final String STATUS_ID_FILE = "debug/statusid.txt";
    private MemoryListener memoryListener = null;

    final private long FACTOR_FOR_REQUIRED_HEAPSPACE = 80;
    private static final String MODULE = "//module/";
    private static final String SPREADSHEET = "//spreadsheet/";
    private static final String DEBUG_OPERATIONS = "debugoperations";
    private static final String MAX_TABLE_COLUMNS = "maxTableColumns";
    private static final String MAX_TABLE_ROWS = "maxTableRows";
    private static final String MAX_TABLE_CELLS = "maxTableCells";
    private static final String MAX_SHEETS = "maxSheets";
    private final DocumentProperties documentProperties;
	private static boolean lowMemoryAbort = false;
	private static int successfulAppliedOperations = 0;

    public OdfOperationDoc(Session session, ServiceLookup _services, InputStream _inputDocumentStream, IResourceManager resourceManager, DocumentProperties configuration) throws Exception {
    	try {
    		documentProperties = configuration;

    		registerMemoryListener();

	    	boolean saveDebugOperations = Boolean.parseBoolean(ConfigurationHelper.getOfficeConfigurationValue(_services, session, MODULE + DEBUG_OPERATIONS, "false"));
	        configuration.put(DEBUG_OPERATIONS, saveDebugOperations);
	        int maxTableColumns = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_COLUMNS, 15);
	        configuration.put(MAX_TABLE_COLUMNS, maxTableColumns);
	        int maxTableRows = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_ROWS, 1500);
	        configuration.put(MAX_TABLE_ROWS, maxTableRows);
	        int maxTableCells = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_CELLS, 1500);
	        configuration.put(MAX_TABLE_CELLS, maxTableCells);
	        int maxSheets = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, SPREADSHEET + MAX_SHEETS, 256);
	        configuration.put(MAX_SHEETS, maxSheets);

	        // complexity check... TODO: distinguish between odt and odf
	        final long maxAllowedXmlSizeMatchingComplexity = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, "//spreadsheet/maxCells", 500000) * 100;
        	final long maxAllowedXmlSizeMatchingMemory = configuration.optLong(DocumentProperties.PROP_MAX_MEMORY_USAGE, -1) / FACTOR_FOR_REQUIRED_HEAPSPACE;
        	final InputStream inputDocumentStream = DocumentComplexity.checkDocumentComplexity(_inputDocumentStream, maxAllowedXmlSizeMatchingComplexity, maxAllowedXmlSizeMatchingMemory);

	        mDocument = new OdfOperationDocument(inputDocumentStream, resourceManager, configuration);
    	}
    	catch(Throwable e) {
    		removeMemoryListener();
    		throw getRethrowException(e);
    	}
    }

	public static boolean isLowMemoryAbort() {
	    return lowMemoryAbort;
	}
	public static void setLowMemoryAbort(boolean l) {
	    lowMemoryAbort = l;
	}
	public static void abortOnLowMemory() {
	    if(lowMemoryAbort) {
	        throw new RuntimeException();
	    }
	}
	public static void setSuccessfulAppliedOperations(int opCount) {
		successfulAppliedOperations = opCount;
	}
	public static int getSuccessfulAppliedOperations() {
		return successfulAppliedOperations;
	}
    public void registerMemoryListener() {
        MemoryObserver memoryObserver = (MemoryObserver)documentProperties.get(DocumentProperties.PROP_MEMORYOBSERVER);
        if(memoryObserver!=null) {
            if(memoryListener==null) {
                memoryListener = new MemoryListener() {
                    @Override
                    public void memoryTresholdExceeded(long usedMemory, long maxMemory) {
                        notifyLowMemory();
                    }
                };
                memoryObserver.addListener(memoryListener);
            }
            if(MemoryObserver.isUsageThresholdExceeded()) {
                notifyLowMemory();
            }
        }
    }

    public void removeMemoryListener() {
        if(memoryListener!=null) {
            MemoryObserver memoryObserver = (MemoryObserver)documentProperties.get(DocumentProperties.PROP_MEMORYOBSERVER);
            if(memoryObserver!=null) {
                memoryObserver.removeListener(memoryListener);
                memoryListener = null;
            }
        }
    }

    // take care... this function is called from the low-memory
    // thread, so breakpoints will not work... It is called before
    // gc is started and or an OutOfMemory Exception is thrown, so we
    // try to free up as much as possible ... the filter will then
    // most probably exit with an exception
    private void notifyLowMemory() {
        LOG.info("ODF Filter: Low memory notification received");
        setLowMemoryAbort(true);
        if(mDocument!=null) {
	        // trying to free memory ...
	        final OdfPackage odfPackage = mDocument.getPackage();
	        final OdfDocument odfDocument = mDocument.getDocument();
	        if(odfDocument!=null) {
	        	try {
	        		final OdfStylesDom stylesDom = odfDocument.getStylesDom();
	        		stylesDom.abort();
	        	}
	        	catch(SAXException e) {
	        	}
	        	if(odfDocument instanceof OdfSpreadsheetDocument) {
	        		try {
	        			final Content content = (Content)((OdfSpreadsheetDocument)odfDocument).getContentDom();
	        			content.getSheets().clear();
	        		}
	        		catch(SAXException e) {
	        		}
	        	}
	        }
	        if(odfPackage!=null) {
	        	odfPackage.freeMemory();
	        }
        }
    }

    public static FilterException getRethrowException(Throwable e) {
    	FilterException ret;
        if(isLowMemoryAbort()) {
            ret = new FilterException("ODS FilterException...", ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
        }
        else if(e instanceof FilterException) {
    		final FilterException filterException = (FilterException)e;
    		if(filterException.getErrorcode()==ErrorCode.COMPLEXITY_TOO_HIGH) {
    			LOG.warn("could not load/save document, the document complexity is too high");
    		}
    		else if(filterException.getErrorcode()==ErrorCode.CRITICAL_ERROR) {
    			LOG.error(e.getMessage(), e);
    		}
    		ret = (FilterException)e;
    	}
    	else if (e instanceof java.util.zip.ZipException && e.getMessage().contains("only DEFLATED entries can have EXT descriptor")) {
            ret = new FilterException(e, ErrorCode.UNSUPPORTED_ENCRYPTION_USED);
        }
    	else if (e instanceof OutOfMemoryError) {
    		ret = new FilterException(e, ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
    	}
    	else {
    		LOG.error(e.getMessage(), e);
    		ret = new FilterException(e, ErrorCode.CRITICAL_ERROR);
    	}
    	ret.setOperationCount(OdfOperationDoc.getSuccessfulAppliedOperations());
    	return ret;
    }

    /**
     * Receives the (known) operations of the ODF text document
     *
     * @return the operations as JSON
     */
    // ToDo OX - JSONObject is to be considered..
    public JSONObject getOperations()
    	throws SAXException, JSONException, FilterException {

    	JSONObject ops = mDocument.getOperations();
        if (ops != null) {
            LOG.info("\n\n*** ALL OPERATIONS:\n{0}", ops.toString());
        } else {
            LOG.info("\n\n*** ALL OPERATIONS:\n" + null);
        }
        return ops;
    }

    /**
     * Applies the (known) operations to upon the latest state of the ODF text
     * document
     *
     * @param operationString ODF operations as String
     * @return the number of operations being accepted
     * @throws FilterException
     */
    public int applyOperations(String operationString) throws FilterException {
        LOG.info("\n*** EDIT OPERATIONS:\n{0}", operationString);
        int operationsCount = 0;
        try {
            operationsCount = mDocument.applyOperations(operationString);
        } catch (Exception ex) {
            LOG.info("Error simluated");
            if (null != ex.getMessage() && ex.getMessage().equals("ERROR_SIMULATED")) {
                throw new FilterException("Simulated error triggered by operation: createError", ErrorCode.UNSUPPORTED_OPERATION_USED);
            } else {
                throw new RuntimeException(ex);
            }
        }
        return operationsCount;
    }

    /**
     * writes the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    void writeStatusID(String statusId) {
        OdfPackage pkg = mDocument.getPackage();
        LOG.info("Overwriting ODT document status ID " + STATUS_ID_FILE);
        pkg.insert(statusId.getBytes(), STATUS_ID_FILE, "text/plain");
    }

    /**
     * Read the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    String readStatusID() {
        OdfPackage pkg = mDocument.getPackage();
        String statusId = null;
        if (pkg != null && pkg.contains(STATUS_ID_FILE)) {
            // ..from the ODF ZIP
            byte[] revisionByteArray = pkg.getBytes(STATUS_ID_FILE);
            if (revisionByteArray != null && revisionByteArray.length != 0) {
                BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(revisionByteArray)));
                // read the first line of the file, only containing one number
                try {
                    statusId = reader.readLine();
                } catch (IOException e) {
                    LOG.info("No existing status ID file found!");
                }
                LOG.info("Found an existing status ID:{0}", statusId);
            }
        }
        return statusId;
    }

    public long getContentSize() {
        return mDocument.getContentSize();
    }

    /**
     * Returns the TextOperationDocument encapsulating the DOM view
     *
     * @return ODF text document
     */
    public OdfDocument getDocument() {
        return mDocument.getDocument();
    }

    /**
     * Close the OdfPackage and release all temporary created data. After
     * execution of this method, this class is no longer usable. Do this as the
     * last action to free resources. Closing an already closed document has no
     * effect.
     */
    public void close() {
        mDocument.close();
    }
}
