/*
 *
 *  Copyright (C) 2016 OX Software GmbH All rights reserved.
 *
 *  This program is proprietary software and licensed to you under Open-Xchange
 *  Inc's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks.
 *
 */

package com.openexchange.office.monitoring;

import javax.management.NotCompliantMBeanException;
import javax.management.StandardMBean;

import com.openexchange.annotation.NonNull;
import com.openexchange.office.IMonitoring;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.tools.monitoring.CalcEngineEventType;
import com.openexchange.office.tools.monitoring.CloseType;
import com.openexchange.office.tools.monitoring.OpenType;
import com.openexchange.office.tools.monitoring.OperationsType;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.monitoring.Statistics;

/**
 * {@link OfficeMonitoring}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class OfficeMonitoring extends StandardMBean implements IMonitoring {

    /**
     * Domain
     */
    public static final String DOMAIN = "com.openexchange.office";

    /**
     * Initializes a new {@link OfficeMonitoring}.
     * @param statistics
     * @throws NotCompliantMBeanException
     */
    public OfficeMonitoring(@NonNull Statistics statistics) throws NotCompliantMBeanException {
        super(IMonitoring.class);

        m_statistics = statistics;
    }

    // - IMonitoringCommon

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Total()
     */
    @Override
    public long getDocumentsCreated_Total() {
        return m_statistics.getDocumentsCreated(null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsCreated_Text()
     */
    @Override
    public long getDocumentsCreated_Text() {
        return m_statistics.getDocumentsCreated(DocumentType.TEXT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Spreadsheet()
     */
    @Override
    public long getDocumentsCreated_Spreadsheet() {
        return m_statistics.getDocumentsCreated(DocumentType.SPREADSHEET);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Presentation()
     */
    // TODO (KA): enable if needed: @Override
    public long getDocumentsCreated_Presentation() {
        return m_statistics.getDocumentsCreated(DocumentType.PRESENTATION);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IMonitoring#getMedianDocumentsSize_Total()
     */
    @Override
    public long getDocumentsSizeMedian_Total() {
        return m_statistics.getDocumentsSizeMedian(null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getMedianDocumentsSize_Text_KB()
     */
    @Override
    public long getDocumentsSizeMedian_Text() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.TEXT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getMedianDocumentsSize_Spreadsheet()
     */
    @Override
    public long getDocumentsSizeMedian_Spreadsheet() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.SPREADSHEET);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getMedianDocumentsSize_Presentation()
     */
    // TODO (KA): @Override
    public long getDocumentsSizeMedian_Presentation() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.PRESENTATION);
    }

    // - IMonitoringText

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Total()
     */
    @Override
    public long getDocumentsOpened_Text_Total() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Msx()
     */
    @Override
    public long getDocumentsOpened_Text_OOXML() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.MSX);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Binary()
     */
    @Override
    public long getDocumentsOpened_Text_Binary() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.BINARY);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Odf()
     */
    @Override
    public long getDocumentsOpened_Text_ODF() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.ODF);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsClosed_Text_Total()
     */
    @Override
    public long getDocumentsClosed_Text_Total() {
        return m_statistics.getDocumentsClosed(DocumentType.TEXT, CloseType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsClosed_Text_Timeout()
     */
    @Override
    public long getDocumentsClosed_Text_Timeout() {
        return m_statistics.getDocumentsClosed(DocumentType.TEXT, CloseType.TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsSaved_Text_Total()
     */
    @Override
    public long getDocumentsSaved_Text_Total() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_Close()
     */
    @Override
    public long getDocumentsSaved_Text_Close() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_100ops()
     */
    @Override
    public long getDocumentsSaved_Text_100ops() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.OPS_100);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_15min()
     */
    @Override
    public long getDocumentsSaved_Text_15mins() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.OPS_15MINS);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentOperations_Text_Incoming()
     */
    @Override
    public long getDocumentsOperations_Text_Incoming() {
        return m_statistics.getDocumentsOperations(DocumentType.TEXT, OperationsType.INCOMING);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentOperations_Text_Outgoing_Distributeted()
     */
    @Override
    public long getDocumentsOperations_Text_Distributed() {
        return m_statistics.getDocumentsOperations(DocumentType.TEXT, OperationsType.DISTRIBUTED);
    }

    // - IMonitoringSpreadsheet

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_Total() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Msx()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_OOXML() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.MSX);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Binary()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_Binary() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.BINARY);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Odf()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_ODF() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.ODF);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsClosed_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsClosed_Spreadsheet_Total() {
        return m_statistics.getDocumentsClosed(DocumentType.SPREADSHEET, CloseType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsClosed_Spreadsheet_Timeout()
     */
    @Override
    public long getDocumentsClosed_Spreadsheet_Timeout() {
        return m_statistics.getDocumentsClosed(DocumentType.SPREADSHEET, CloseType.TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsSaved_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_Total() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_Close()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_Close() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_100ops()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_100ops() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.OPS_100);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_15min()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_15mins() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.OPS_15MINS);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentOperations_Spreadsheet_Incoming()
     */
    @Override
    public long getDocumentsOperations_Spreadsheet_Incoming() {
        return m_statistics.getDocumentsOperations(DocumentType.SPREADSHEET, OperationsType.INCOMING);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentOperations_Spreadsheet_Outgoing_Distributeted()
     */
    @Override
    public long getDocumentsOperations_Spreadsheet_Distributed() {
        return m_statistics.getDocumentsOperations(DocumentType.SPREADSHEET, OperationsType.DISTRIBUTED);
    }

    // - IMonitoringCalcEngine

    @Override
    public long getCalcEngine_Documents_Reloaded() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.DOC_RELOADED);
    }

    @Override
    public long getCalcEngine_Jobs_Processed() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.JOB_PROCESSED);
    }

    @Override
    public long getCalcEngine_Jobs_Error() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.JOB_ERROR);
    }

    @Override
    public long getCalcEngine_Median_JobTime() {
        return m_statistics.getCalcEngineMedianJobTime();
    }

    @Override
    public long getCalcEngine_Peak_JobTime() {
        return m_statistics.getCalcEnginePeakJobTime();
    }

    @Override
    public long getCalcEngine_Documents_Init() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.DOC_INIT);
    }

    @Override
    public long getCalcEngine_Documents_Exit() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.DOC_EXIT);
    }


    @Override
    public long getCalcEngine_Documents_Error() {
        return m_statistics.getCalcEngineData(CalcEngineEventType.DOC_ERROR);
    }

    // - Implementation

    /**
     * @return The statistics object, all office runtime information is collected at
     */
    public @NonNull Statistics getStatistics() {
        return m_statistics;
    }

    // - Members ---------------------------------------------------------------

    private Statistics m_statistics = null;
}

