/*
 *
  *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.json.tools;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.groupware.ldap.User;
import com.openexchange.log.Log;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.tools.MimeTypeHelper;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.doc.DocumentFormatHelper.OfficeDocumentType;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.office.tools.resource.Resource;
import com.openexchange.office.tools.resource.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * {@link OXDocument}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class OXDocumentHelper {

	public static class OfficeDocumentTypeProperties {
	    public String mimeType;
	    public String module;
	    public String extension;
	    public OfficeDocumentType docType;

	    public OfficeDocumentTypeProperties(String mimeType, String module, String extension, OfficeDocumentType docType) {
	        this.mimeType = mimeType;
	        this.module = module;
	        this.extension = extension;
	        this.docType = docType;
	    }
	}

    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     *  The session of the client that requested a service which needs documents
     *  access.
     *
     * @param fileHelper
     *  The FileHelper instance that is responsible for the file access regarding
     *  the document.
     *
     * @param resourceManager
     *  The ResourceManager instance which stores resources addressed by the document
     *  to be available via REST-API and other documents.
     */
    public OXDocumentHelper(ServiceLookup services, Session session, final InputStream documentStream, ResourceManager resourceManager) {
        super();

        m_services = services;
        m_resourceManager = resourceManager;
        m_session = session;

        if (null != documentStream) {
            try {
                m_documentBuffer = IOUtils.toByteArray(documentStream);
            } catch (IOException e) {
                LOG.error("RT connection: unable to retrieve document stream, exception catched", e);
            } finally {
                IOUtils.closeQuietly(documentStream);
            }
        }

        if (null != m_documentBuffer) {
            final ZipInputStream zipInputStm = new ZipInputStream(new ByteArrayInputStream(m_documentBuffer));
            ZipEntry entry = null;
            byte[] documentBuffer = null;
            byte[] operationsBuffer = null;
            byte[] extendedDataBuffer = null;

            try {
                // create a dummy resource manager, if no one is given
                if (null == m_resourceManager) {
                    m_resourceManager = new ResourceManager(m_services);
                }

                while ((entry = zipInputStm.getNextEntry()) != null) {
                    final String entryName = entry.getName();

                    if (entryName.equals(DocFileHelper.OX_DOCUMENT_EXTENDEDDATA)) {
                        try {
                            // read document identifier
                            extendedDataBuffer = IOUtils.toByteArray(zipInputStm);
                        } catch (IOException e) {
                            LOG.error("RT connection: unable to retrieve extended data stream, exception catched", e);
                        }
                    } else if (entryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                        try {
                            // read original document
                            documentBuffer = IOUtils.toByteArray(zipInputStm);
                        } catch (IOException e) {
                            LOG.error("RT connection: unable to retrieve original document stream, exception catched", e);
                        }
                    } else if (entryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME)) {
                        try {
                            // read original document
                            final String op = IOUtils.toString(zipInputStm);
                            // non continuous osn numbers lead to a load error in the frontend, so we have to remove them as some operations are filtered by the calcengine
                            operationsBuffer = op.replaceAll("\\\"osn\\\":[0-9]+,?", "").getBytes();
                        } catch (IOException e) {
                            LOG.error("RT connection: unable to retrieve operations stream, exception catched", e);
                        }
                    } else if (entryName.startsWith(DocFileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX)) {
                        // read resource and put into the resource manager, if not already contained
                        final String resourceName = entryName.substring(DocFileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX.length());

                        if (!m_resourceManager.containsResource(Resource.getUIDFromResourceName(resourceName))) {
                            byte[] resourceBuffer = null;

                            try {
                                resourceBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve resource document stream, exception catched", e);
                            }

                            m_resourceManager.addResource(resourceBuffer);
                        }
                    }

                    zipInputStm.closeEntry();
                }
            } catch (IOException e) {
                LOG.error("RT connection: Exception catched while reading the document stream", e);
            } finally {
                IOUtils.closeQuietly(zipInputStm);

                // set content of original document
                if (null != documentBuffer) {
                    m_documentBuffer = documentBuffer;
                }
                if (null != extendedDataBuffer) {
                    try {
                        m_uniqueDocumentId = new JSONObject(new String(extendedDataBuffer));
                    } catch (JSONException e) {
                        LOG.error("RT connection: unable to setup json object for unique document identifier", e);
                    }
                }

                // set content of already available operations
                if (null != operationsBuffer) {
                    try {
                        m_documentOperations = new JSONObject(new String(operationsBuffer));
                    } catch (JSONException e) {
                        LOG.error("RT connection: unable to setup json object for document operations", e);
                    }
                }
            }
        }
    }

    /**
     * Retrieves the operations from the document and optional additional operations.
     *
     * @param session
     *  The session of the client requesting the document operations.
     *
     * @param importer
     *  The importer instance to be used to read the document stream and provide the
     *  operations.
     *
     * @param appendOperations
     *  Optional operations which should be appended to the document operations. Normally
     *  used if there are queued operations which are not part of the last document stream.
     *
     * @return
     *  The JSONObject containing all operations from the documents.
     *
     * @throws
     *  FilterException in case the filter implementation is not able to provide the
     *  operations.
     */
    public JSONObject getOperations(Session session, IImporter importer) throws FilterException {
        JSONObject jsonResult = null;

        if ((null != session) && (null != m_documentBuffer) && (null != importer)) {
            final InputStream documentStm = new ByteArrayInputStream(m_documentBuffer);
            try {
                String userLangCode = null;
                try {
	                User user = SessionUtils.getUserInfo(session, m_services);
                    if (null != user) {
                        userLangCode = DocFileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                    }
                } catch (Exception e) {
                	// nothing to do.
                }
                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                jsonResult = importer.createOperations(session, documentStm, docProps);

                String uniqueDocumentId = docProps.optString(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, null);
                if (null != uniqueDocumentId) {
                    try {
                        this.setUniqueDocumentId(Integer.parseInt(uniqueDocumentId));
                    } catch (NumberFormatException e) {
                        // nothing to do as the unique document identifier
                        // is only an optional value
                    }
                }

                LOG.debug("TIME getOperations: " + (System.currentTimeMillis() - startTime));
            } finally {
                IOUtils.closeQuietly(documentStm);
            }
        }

        // getting operations from the original document is mandatory
        if (null != jsonResult) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                OperationHelper.appendJSON(jsonResult, m_documentOperations);
            }
        }

        return jsonResult;
    }

    /**
     * Provides a buffered input stream which is a copy of the input stream
     * provided in the ctor.
     *
     * @return
     *  A buffered input stream or null if the source stream couldn't be read.
     */
    public InputStream getInputStream() {
    	return new ByteArrayInputStream(m_documentBuffer);
    }

    /**
     * Provides the byte buffer used by the OXDocumentHelper instance.
     *
     * @return
     *  The byte buffer of the document or null if the buffer couldn't be
     *  created.
     */
    public final byte[] getDocumentBuffer() {
    	return m_documentBuffer;
    }

    /**
     * Retrieves the last error code set by the latest method.
     *
     * @return
     *  The last error code set by the last method called.
     */
    public ErrorCode getLastError() {
        return m_lastErrorCode;
    }

    /**
     * Retrieves the resource manager used by this document.
     *
     * @return the ResourceManager used to read the document
     */
    ResourceManager getResourceManager() {
        return m_resourceManager;
    }

    /**
     * Provides the unique document id stored in the
     * document.
     *
     * @return
     *  Returns the unique document id stored in the
     *  document or -1 if no id is available.
     */
    public int getUniqueDocumentId() {
        int id = -1;

        if (null != m_uniqueDocumentId) {
            id = m_uniqueDocumentId.optInt("uniqueDocumentIdentifier", -1);
        }

        return id;
    }

    /**
     * Sets a new unique document id to be stored
     * in the document on the next successful
     * flushDocument.
     *
     * @param uniqueId
     *  The new unique document id to be stored.
     */
    public void setUniqueDocumentId(int uniqueId) {
        if (null == m_uniqueDocumentId) {
            m_uniqueDocumentId = new JSONObject();
        }

        try {
            m_uniqueDocumentId.put("uniqueDocumentIdentifier", uniqueId);
        } catch (JSONException e) {
            // this is not a fatal error as we can work without this unique
            // id. just the speed up of the local cache won't work
            LOG.error("RT connection: Couldn't set unique document id to JSONObject", e);
        }
    }

    static public OfficeDocumentTypeProperties determineDefaultOfficeProperties(String docTypeName) {
        String mimeType;
        String module;
        String extensionType;
        OfficeDocumentType officeDocumentType;
        OfficeDocumentTypeProperties docProps = null;

        if (docTypeName.equals("spreadsheet")) {
            module = "spreadsheet";
            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_SPREADSHEET;
            extensionType = "xlsx";
            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET;
        } else if (docTypeName.equals("presentation")) {
            module = "presentation";
            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_PRESENTATION;
            extensionType = "pptx";
            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION;
        } else {
            module = "text";
            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_WORD;
            extensionType = "docx";
            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING;
        }

        return new OfficeDocumentTypeProperties(mimeType, module, extensionType, officeDocumentType);
    }

    /**
     * Provides the import implementation responsible for a specific document
     * type.
     *
     * @param docType
     *  The document type where we want to get the responsible importer.
     *
     * @return
     *  The responsible importer or null if no importer is known for the
     *  provided document type.
     */
    static public IImporter getImporter(ServiceLookup services, OfficeDocumentType docType) {
        // let the backend create the default document, if no default resource is available
        IImporter importer = null;

        switch (docType) {
            case MS_DOCUMENT_WORD: {
                importer = services.getService(com.openexchange.office.ooxml.docx.Importer.class); break;
            }
            case MS_DOCUMENT_SPREADSHEET: {
                importer = services.getService(com.openexchange.office.ooxml.xlsx.Importer.class); break;
            }
            case MS_DOCUMENT_PRESENTATION: {
                importer = services.getService(com.openexchange.office.ooxml.pptx.Importer.class); break;
            }
            default: break;
        }

        return importer;
    }

    // - Members ---------------------------------------------------------------

    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(OXDocumentHelper.class);

    private final ErrorCode m_lastErrorCode = ErrorCode.NO_ERROR;

    private byte[] m_documentBuffer = null;

    private JSONObject m_documentOperations = null;

    private ResourceManager m_resourceManager = null;

    private JSONObject m_uniqueDocumentId = null;

    private final Session m_session;

    private final ServiceLookup m_services;
}
