/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.calcengine.client.impl;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.Validate;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ELogLevel;

//=============================================================================
public class CalcEngineRecorder implements ICalcEngineClient
{
	//-------------------------------------------------------------------------
    private CalcEngineRecorder()
    {}

	//-------------------------------------------------------------------------
    public static synchronized CalcEngineRecorder create(final ICalcEngineClient iWrapped      ,
    													 final String            sRecordingPath)
    	throws Exception
    {
        final CalcEngineRecorder aInst = new CalcEngineRecorder();
        aInst.m_iWrapped       = iWrapped;
        aInst.m_aRecordingPath = new File (sRecordingPath);
        
        aInst.m_aRecordingPath.mkdirs ();
		Validate.isTrue(aInst.m_aRecordingPath.isDirectory (), "Could not create folder for recording CE client data at '"+aInst.m_aRecordingPath+"'.");
        
        return aInst;
    }

	//-------------------------------------------------------------------------
    @Override
    public void createDocument(final String sDocHandle)
    	throws Exception
    {
        m_iWrapped.createDocument    (sDocHandle);
        impl_createDocFile4Recording (sDocHandle);
    };

	//-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sDocHandle)
    	throws Exception
    {
        m_iWrapped.destroyDocument  (sDocHandle);
        impl_closeDocFile4Recording (sDocHandle);
    };
    
    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sDocHandle    ,
    									     final String       sOperationJSON,
    									     final StringBuffer sResultJSON   )
    	throws Exception
    {
        ECalcEngineError eState = ECalcEngineError.E_NONE;

        try
        {
        	impl_recordRequest  (sDocHandle, CalcEngineRecordedFile.OP_EXECUTE, sOperationJSON);
        	eState = m_iWrapped.executeOperation(sDocHandle, sOperationJSON, sResultJSON);
        	impl_recordResponse (sDocHandle, CalcEngineRecordedFile.OP_EXECUTE, sResultJSON.toString ());
        }
        catch (final Throwable ex)
        {
        	eState = ECalcEngineError.E_EXCEPTION;
        	impl_recordError (sDocHandle, CalcEngineRecordedFile.OP_EXECUTE, ex);
        }

        return eState;
    };

	//-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument(final String    sDocHandle          ,
    										final String... lPagedOperationJSONs)
    	throws Exception
    {
    	return m_iWrapped.restoreDocument(sDocHandle, lPagedOperationJSONs);
    }

	//-------------------------------------------------------------------------
    @Override
    public ECalcEngineError copy(final String                   sSourceDocHandle,
    							 final CalcEngineClipBoardEvent aEvent          )
    	throws Exception
    {
        ECalcEngineError eState = ECalcEngineError.E_NONE;

        try
        {
        	final String sRequest = CalcEngineRecordedFile.constructCopyRequest(aEvent);
        	impl_recordRequest  (sSourceDocHandle, CalcEngineRecordedFile.OP_COPY, sRequest);

        	eState = m_iWrapped.copy(sSourceDocHandle, aEvent);
        	
        	final String sResponse = aEvent.sCopyOperationsJSON;
        	impl_recordResponse (sSourceDocHandle, CalcEngineRecordedFile.OP_COPY, sResponse);
        }
        catch (final Throwable ex)
        {
        	eState = ECalcEngineError.E_EXCEPTION;
        	impl_recordError (sSourceDocHandle, CalcEngineRecordedFile.OP_COPY, ex);
        }

        return eState;
    }

	//-------------------------------------------------------------------------
    @Override
    public ECalcEngineError paste(final String                   sTargetDocHandle,
    							  final CalcEngineClipBoardEvent aEvent          )
    	throws Exception
    {
        ECalcEngineError eState = ECalcEngineError.E_NONE;

        try
        {
        	final String sRequest  = CalcEngineRecordedFile.constructPasteRequest(aEvent);
        	impl_recordRequest  (sTargetDocHandle, CalcEngineRecordedFile.OP_PASTE, sRequest);

        	eState = m_iWrapped.paste(sTargetDocHandle, aEvent);
        	
        	final String sResponse = aEvent.sPasteOperationsJSON;
        	impl_recordResponse (sTargetDocHandle, CalcEngineRecordedFile.OP_PASTE, sResponse);
        }
        catch (final Throwable ex)
        {
        	eState = ECalcEngineError.E_EXCEPTION;
        	impl_recordError (sTargetDocHandle, CalcEngineRecordedFile.OP_PASTE, ex);
        }

        return eState;
    }

	//-------------------------------------------------------------------------
    @Override
    public String getVersion()
    	throws Exception
    {
        return m_iWrapped.getVersion();
    }

	//-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel)
    	throws Exception
    {
        m_iWrapped.setLogLevel(eLevel);
    }

    //-------------------------------------------------------------------------
    @Override
    public void dispose ()
    	throws Exception
    {
    	if (m_iWrapped != null)
    		m_iWrapped.dispose ();
    	m_iWrapped = null;
    }

	//-------------------------------------------------------------------------
    private void impl_recordRequest (final String sDocHandle,
    							     final String sOperation,
    							     final String sData     )
        throws Exception
    {
    	final CalcEngineRecordedFile aFile = impl_accessDocFile4Recording (sDocHandle);
    	aFile.recordRequest(sOperation, sData);
    }
    
	//-------------------------------------------------------------------------
    private void impl_recordResponse (final String sDocHandle,
    							      final String sOperation,
    							      final String sData     )
        throws Exception
    {
    	final CalcEngineRecordedFile aFile = impl_accessDocFile4Recording (sDocHandle);
    	aFile.recordResponse(sOperation, sData);
    }

	//-------------------------------------------------------------------------
    private void impl_recordError (final String    sDocHandle,
    							   final String    sOperation,
    							   final Throwable aError    )
        throws Exception
    {
    	final CalcEngineRecordedFile aFile = impl_accessDocFile4Recording (sDocHandle);
    	aFile.recordError(sOperation, aError);
    }
    
	//-------------------------------------------------------------------------
    private CalcEngineRecordedFile impl_createDocFile4Recording (final String sDocHandle)
    	throws Exception
    {
    	final Map< String, CalcEngineRecordedFile > lRecordingFiles = mem_RecordingFiles ();
    	final File                                  aFileOnDisc     = new File (m_aRecordingPath, sDocHandle+CalcEngineRecordedFile.EXTENSION);
    	final CalcEngineRecordedFile                aDocFile        = new CalcEngineRecordedFile ();

        aDocFile.setFile(aFileOnDisc);
        aDocFile.create ();

        lRecordingFiles.put (sDocHandle, aDocFile);
    	
        return aDocFile;
    }

	//-------------------------------------------------------------------------
    private CalcEngineRecordedFile impl_accessDocFile4Recording (final String sDocHandle)
    	throws Exception
    {
    	final Map< String, CalcEngineRecordedFile > lRecordingFiles = mem_RecordingFiles ();
    	final CalcEngineRecordedFile                aDocFile        = lRecordingFiles.get(sDocHandle);
        return aDocFile;
    }

	//-------------------------------------------------------------------------
    private void impl_closeDocFile4Recording (final String sDocHandle)
    	throws Exception
    {
    	final Map< String, CalcEngineRecordedFile > lRecordingFiles = mem_RecordingFiles ();
		final CalcEngineRecordedFile                aFile           = lRecordingFiles.get(sDocHandle);
    	if (aFile != null)
    	{
    		aFile.close ();
    		lRecordingFiles.remove(sDocHandle);
    	}
    }

    //-------------------------------------------------------------------------
    private Map< String, CalcEngineRecordedFile > mem_RecordingFiles ()
        throws Exception
    {
    	if (m_lOpenFiles == null)
    		m_lOpenFiles = new HashMap< String, CalcEngineRecordedFile > ();
    	return m_lOpenFiles;
    }

    //-------------------------------------------------------------------------
    private ICalcEngineClient m_iWrapped = null;

    //-------------------------------------------------------------------------
    private File m_aRecordingPath = null; 

    //-------------------------------------------------------------------------
    private Map< String, CalcEngineRecordedFile > m_lOpenFiles = null;
}
