/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.io.File;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ECalcEngineMode;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientHTTP.EAliveState;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobBase;
import com.openexchange.office.tools.jobpool.JobEnvironment;
import com.openexchange.office.tools.jobpool.JobResults;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;
import com.openxchange.office_communication.tools.exec.Executable;
import com.openxchange.office_communication.tools.exec.SimpleExecutableWatcher;

//=============================================================================
public class CalcEngineJob extends JobBase
{
	//-------------------------------------------------------------------------
	private static final Log LOG = LogFactory.getJclLog(CalcEngineJob.class);

	//-------------------------------------------------------------------------
	public static boolean SIMULATION_MODE = false;

	static
	{
		// handle reading those value from config or environment "gracefully".
		// It's an optional debug variable only. 
		// Dont disturb normal production mode with any error which can occur here !
		try
		{
			// a) because it's not final (!) it could be set by e.g. an unit test directly !
			//    ...
			
			// b) try to read suitable configuration entry
			if (SIMULATION_MODE == false)
			{
				final CalcEngineConfig aCfg = CalcEngineConfig.get();
				SIMULATION_MODE = aCfg.isSimulatorOn();
			}
			
			// c) try to read it from the environment
			if (SIMULATION_MODE == false)
			{
				String sSimulatorMode = System.getProperty(CalcEngineConfig.PROP_SIMULATOR);
				if (StringUtils.isEmpty(sSimulatorMode))
					sSimulatorMode = System.getenv(CalcEngineConfig.PROP_SIMULATOR);
				
				if ( ! StringUtils.isEmpty(sSimulatorMode))
					SIMULATION_MODE = Boolean.parseBoolean(sSimulatorMode);
			}
		}
		catch (Throwable ex)
		{
			SIMULATION_MODE = false;
		}
	}
	
	//-------------------------------------------------------------------------
	public static final String ENV_WORKERBIN          = "env.worker-bin";
	public static final String ENV_PORT               = "env.port";
	public static final String ENV_MAXMEM             = "env.worker-maxmem";
	public static final String ENV_CACHE_PATH         = "env.cache-path";

	//-------------------------------------------------------------------------
	public static final String ARG_OPERATION          = "arg.operation"      ;
	public static final String ARG_DOCUMENT_HANDLE    = "arg.document.handle";
	public static final String ARG_OPERATIONLIST      = "arg.operationlist"  ;
	public static final String ARG_CLIPBOARD_EVENT    = "arg.clipboard.event";
	
	//-------------------------------------------------------------------------
	public static final String OP_CREATE_DOCUMENT     = "createDocument"  ;
	public static final String OP_DESTROY_DOCUMENT    = "destroyDocument" ;
	public static final String OP_RESTORE_DOCUMENT    = "restoreDocument" ;
	public static final String OP_EXECUTE_OPERATION   = "executeOperation";
	public static final String OP_COPY                = "copy"            ;
	public static final String OP_PASTE               = "paste"           ;
    public static final String OP_GET_VERSION         = "getVersion"      ;

	//-------------------------------------------------------------------------
	public static final String RESULT_DOCUMENT_HANDLE = "result.document.handle";
	public static final String RESULT_ERROR_STATE     = "result.error-state"    ;
	public static final String RESULT_4_EXECUTION     = "result.4.execution"    ;
	public static final String RESULT_CLIPBOARD_EVENT = "result.clipboard.event";
	
	//-------------------------------------------------------------------------
	public CalcEngineJob ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	@Override
	protected void activateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.info (aLog.forLevel(ELogLevel.E_INFO).toLog("activate ..."));
		
		// read configuration first to know the connection details
		
		final String  sWorkerBin = aEnv.get (ENV_WORKERBIN , null);
		final Integer nPort      = aEnv.get (ENV_PORT      , 0   );
		final Long    nMaxMemKB  = aEnv.get (ENV_MAXMEM    , 0L  ); // optional !
		final String  sCachePath = aEnv.get (ENV_CACHE_PATH, null); // optional !
		
		Validate.notEmpty (sWorkerBin , "Miss environment var '"+ENV_WORKERBIN+"'.");
		Validate.isTrue   (nPort>0    , "Miss environment var '"+ENV_PORT     +"'.");

		aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_PORT, Integer.toString(nPort));
		
		impl_initInternalAPI (nPort);
		impl_shutdownMightRunningWorker ();
		
		LOG.info (aLog.forLevel(ELogLevel.E_INFO                     )
				      .toLog   ("worker executable:'"+sWorkerBin+"'"));
		
		// start worker process (running in background)
		
		final Executable aExe = new Executable ();

		File aWorkerBin  = new File (sWorkerBin);
		File aWorkerHome = aWorkerBin.getParentFile();
		if (aWorkerHome.isDirectory())
			aExe.setWorkingPath(aWorkerHome.getAbsolutePath());
		
		aExe.setForward4StdOut (CalcEngineWorkerLogSink.create(ELogLevel.E_INFO ));
		aExe.setForward4StdErr (CalcEngineWorkerLogSink.create(ELogLevel.E_ERROR));
		
		if (StringUtils.endsWithIgnoreCase(sWorkerBin, ".jar"))
		{		
			aExe.setExecutable ("java"    );
			aExe.addArgument   ("-jar"    );
			aExe.addArgument   (sWorkerBin);
		}
		else
		{
			aExe.setExecutable (sWorkerBin);
		}
		
		aExe.addArgument   ("-p"                    );
		aExe.addArgument   (Integer.toString(nPort) );
		aExe.addArgument   ("-i"                    );
		aExe.addArgument   ("localhost"             );
		
		if ( ! StringUtils.isEmpty(sCachePath))
		{
			aExe.addArgument ("-arp"    );
			aExe.addArgument (sCachePath);
		}
		
		if (nMaxMemKB > 0) // optional
		{
			aExe.addArgument ("-m"                    );
			aExe.addArgument (Long.toString(nMaxMemKB));
		}
		
		if (SIMULATION_MODE) // for testing purposes only !
		{
        	LOG.warn(aLog.forLevel(ELogLevel.E_WARNING                                       )
        			     .toLog   ("SIMULATION MODE ON - PLEASE CHECK IF THIS WAS EXPECTED."));
			aExe.addArgument ("-s");
		}
		
		m_aProcess = aExe;

		//"Started SelectChannelConnector@localhost:"+nPort;
		//"worker started successfully ..."
		//"worker starts ..."

		final String         sWatchPoint4PortOpenA = "SelectChannelConnector@localhost:"+nPort;
		final String         sWatchPoint4PortOpenB = "worker started successfully ...";
		final CountDownLatch aWatchPointSync       = new CountDownLatch (1);
		aExe.registerWatcher(SimpleExecutableWatcher.create(aWatchPointSync, sWatchPoint4PortOpenA, sWatchPoint4PortOpenB));

		LOG.info (aLog.forLevel (ELogLevel.E_INFO).toLog("start worker ..."));
		m_aProcessSync = new CountDownLatch (1);
		m_aProcess.runAsync (m_aProcessSync);
		
		LOG.info (aLog.forLevel (ELogLevel.E_INFO).toLog("... wait for worker bootstrap"));
		final int     nTimeout = CalcEngineConfig.get().getWorkerInitTimeout();
		final boolean bStarted = aWatchPointSync.await(nTimeout, TimeUnit.MILLISECONDS);
		if ( ! bStarted)
			throw new Exception ("Worker not started in time.");
		LOG.info (aLog.forLevel (ELogLevel.E_INFO).toLog("... worker started"));
		
		// connect to that worker
		
		LOG.info (aLog.forLevel (ELogLevel.E_INFO).toLog("connect to worker ..."));
		CalcEngineDescriptor aDesc         = new CalcEngineDescriptor ();
        					 aDesc.m_eMode = ECalcEngineMode.E_HTTP_CLIENT_SIMPLE;
        					 aDesc.m_sHost = "localhost";
        					 aDesc.m_nPort = nPort;
        ICalcEngineClient    iAPI          = CalcEngineClientFactory.getDirect(aDesc);
		m_iAPI = iAPI;
		LOG.info (aLog.forLevel (ELogLevel.E_INFO).toLog("... worker connection established"));
		
		if (LOG.isDebugEnabled())
			impl_setLogLevel (ELogLevel.E_DEBUG);
		else
		if (LOG.isInfoEnabled())
			impl_setLogLevel (ELogLevel.E_INFO);
		else
		if (LOG.isWarnEnabled())
			impl_setLogLevel (ELogLevel.E_WARNING);
		else
		if (LOG.isErrorEnabled())
			impl_setLogLevel (ELogLevel.E_ERROR);
	}

	//-------------------------------------------------------------------------
	@Override
	protected void validateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG).toLog("validate ..."));

		final boolean bIsAlive = impl_isWorkerAlive ();
		if ( ! bIsAlive)
			throw new Exception ("Worker is not alive any longer ...");
	}

	//-------------------------------------------------------------------------
	@Override
	protected void deactivateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		if (m_aProcess == null)
			return;

		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.info (aLog.forLevel(ELogLevel.E_INFO).toLog("deactivate worker ..."));
		impl_shutdownMightRunningWorker ();
		LOG.info (aLog.forLevel(ELogLevel.E_INFO).toLog("... OK"));

		final Executable aProcess = m_aProcess;
		m_aProcess = null;
		
		if (aProcess.isAlive())
			aProcess.kill ();
	}

	//-------------------------------------------------------------------------
	@Override
	protected void executeImpl(final JobEnvironment aEnv      ,
							   final JobArguments   lArguments,
							   final JobResults     lResults  )
        throws Exception
    {
		Validate.notNull(m_iAPI, "No API / no process ? Did you defined an environment for this job ?");
		
		String sOp = lArguments.get (ARG_OPERATION);
		Validate.notEmpty(sOp, "Invalid argument '"+ARG_OPERATION+"'.");
		
		if (StringUtils.equalsIgnoreCase(sOp, OP_CREATE_DOCUMENT))
			impl_createDocument (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_DESTROY_DOCUMENT))
			impl_destroyDocument (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_RESTORE_DOCUMENT))
			impl_restoreDocument (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_EXECUTE_OPERATION))
			impl_executeOperation (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_COPY))
			impl_copy (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_PASTE))
			impl_paste (lArguments, lResults);
		else
			throw new UnsupportedOperationException ("No support for operation '"+sOp+"' yet. Please implement.");
	}

	//-------------------------------------------------------------------------
	private void impl_createDocument(final JobArguments lArguments,
			   						 final JobResults   lResults  )
        throws Exception
    {
		final String sDocHandle = lArguments.get(ARG_DOCUMENT_HANDLE);

		LOG.info (mem_Log ().forLevel(ELogLevel.E_INFO                )
							.toLog   ("execute : createDocument ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));

		m_iAPI.createDocument(sDocHandle);
    }
	
	//-------------------------------------------------------------------------
	private void impl_destroyDocument(final JobArguments lArguments,
			   						  final JobResults   lResults  )
        throws Exception
    {
		final String sDocHandle = lArguments.get(ARG_DOCUMENT_HANDLE);

		LOG.trace (mem_Log ().forLevel(ELogLevel.E_INFO)
				             .toLog   ("execute : destroyDocument ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));

		m_iAPI.destroyDocument(sDocHandle);
    }

	//-------------------------------------------------------------------------
	private void impl_restoreDocument(final JobArguments lArguments,
			   						  final JobResults   lResults  )
        throws Exception
    {
		final String   sDocHandle  = lArguments.get(ARG_DOCUMENT_HANDLE);
		final String[] lOperations = lArguments.get(ARG_OPERATIONLIST  );

		LOG.info (mem_Log ().forLevel(ELogLevel.E_INFO)
	                        .toLog   ("execute : restoreDocument ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));
		
		final ECalcEngineError eError = m_iAPI.restoreDocument(sDocHandle, lOperations);
		lResults.set (RESULT_ERROR_STATE, eError);
    }

	//-------------------------------------------------------------------------
	private void impl_executeOperation(final JobArguments lArguments,
			   					       final JobResults   lResults  )
        throws Exception
    {
		final String       sDocHandle  = lArguments.get(ARG_DOCUMENT_HANDLE);
		final String       lOperations = lArguments.get(ARG_OPERATIONLIST  );
		final StringBuffer sResult     = new StringBuffer (256);

		LOG.info (mem_Log ().forLevel(ELogLevel.E_INFO)
                            .toLog   ("execute : executeOperation ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));
		
		final ECalcEngineError eError = m_iAPI.executeOperation(sDocHandle, lOperations, sResult);
		lResults.set (RESULT_ERROR_STATE, eError            );
		lResults.set (RESULT_4_EXECUTION, sResult.toString());
    }

	//-------------------------------------------------------------------------
	private void impl_copy(final JobArguments lArguments,
			   			   final JobResults   lResults  )
        throws Exception
    {
		final String                   sDocHandle = lArguments.get(ARG_DOCUMENT_HANDLE);
		final CalcEngineClipBoardEvent aEvent     = lArguments.get(ARG_CLIPBOARD_EVENT);

		LOG.info (mem_Log ().forLevel(ELogLevel.E_INFO)
                            .toLog   ("execute : copy ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));
		
		final ECalcEngineError eError = m_iAPI.copy(sDocHandle, aEvent);
		lResults.set (RESULT_ERROR_STATE    , eError);
		lResults.set (RESULT_CLIPBOARD_EVENT, aEvent); // in/out param !
    }

	//-------------------------------------------------------------------------
	private void impl_paste(final JobArguments lArguments,
			   			    final JobResults   lResults  )
        throws Exception
    {
		final String                   sDocHandle = lArguments.get(ARG_DOCUMENT_HANDLE);
		final CalcEngineClipBoardEvent aEvent     = lArguments.get(ARG_CLIPBOARD_EVENT);

		LOG.info (mem_Log ().forLevel(ELogLevel.E_INFO)
                            .toLog   ("execute : paste ["+CalcEngineLogContextConst.CONTEXT_DOC_HANDLE+":"+sDocHandle+"] ...."));
		
		final ECalcEngineError eError = m_iAPI.paste(sDocHandle, aEvent);
		lResults.set (RESULT_ERROR_STATE    , eError);
		lResults.set (RESULT_CLIPBOARD_EVENT, aEvent); // in/out param !
    }

	//-------------------------------------------------------------------------
	private void impl_shutdownMightRunningWorker ()
	    throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		try
		{
			LOG.info (aLog.forLevel(ELogLevel.E_INFO               )
					      .toLog   ("check if worker is alive ..."));

			if (m_aInternalAPI.isAlive() == EAliveState.E_NO)
			{
				LOG.info (aLog.forLevel(ELogLevel.E_INFO                   )
						      .toLog   ("... no worker alive on that port"));
				return;
			}

			LOG.info (aLog.forLevel(ELogLevel.E_INFO                                 )
				          .toLog   ("... found active worker - try to shutdown them"));

			final boolean bDown = m_aInternalAPI.shutdown();
			if (bDown)
			{
				LOG.info (aLog.forLevel(ELogLevel.E_INFO                          )
				              .toLog   ("... shutdown was triggered successfully"));
				return;
			}

			LOG.warn (aLog.forLevel(ELogLevel.E_WARNING                             )
		                  .toLog   ("... shutdown was not successfully- TRY KILL !"));

			m_aInternalAPI.kill ();
			if (m_aInternalAPI.isAlive() == EAliveState.E_NO)
			{
				LOG.info (aLog.forLevel(ELogLevel.E_INFO                      )
						      .toLog   ("... kill of worker was successfully"));
				return;
			}

			LOG.error (aLog.forLevel(ELogLevel.E_ERROR           )
				           .toLog   ("... kill of worker failed"));
		}
		catch (final com.sun.jersey.api.client.ClientHandlerException exConnection)
		{
			final String sReason = exConnection.getMessage();
			if ( ! StringUtils.containsIgnoreCase(sReason, "connection refused"))
				throw exConnection;
		}
	}

	//-------------------------------------------------------------------------
	private boolean impl_isWorkerAlive ()
	    throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG).toLog("check if worker is alive ..."));

		final boolean bIsAlive = (m_aInternalAPI.isAlive() == EAliveState.E_YES);

		if (bIsAlive)
			LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG).toLog("... YES"));
		else
			LOG.error (aLog.forLevel(ELogLevel.E_ERROR).toLog("... WORKER NO LONGER AVAILABLE."));

		return bIsAlive;
	}

	//-------------------------------------------------------------------------
	private void impl_setLogLevel (final ELogLevel eLevel)
	    throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.info(aLog.forLevel (ELogLevel.E_INFO).toLog("set log level of worker to : "+eLevel));
		m_aInternalAPI.setLogLevel(eLevel);
	}

	//-------------------------------------------------------------------------
	private void impl_initInternalAPI (final int nPort)
	    throws Exception
	{
		if (m_aInternalAPI != null)
			return;

		final CalcEngineClientHTTP aInternalAPI  = CalcEngineClientHTTP.create ();
		final CalcEngineDescriptor aDesc         = new CalcEngineDescriptor ();
							 	   aDesc.m_sHost = "localhost";
							 	   aDesc.m_nPort = nPort;
		aInternalAPI.describeEnvironment(aDesc);
		m_aInternalAPI = aInternalAPI;
	}
	
	//-------------------------------------------------------------------------
	private ContextAwareLogHelp mem_Log ()
	    throws Exception
	{
		if (m_aLog == null)
		{
			m_aLog = new ContextAwareLogHelp (LOG);
			final String sIdentity = ObjectUtils.identityToString  (this          );
			final String sPointer  = StringUtils.substringAfterLast(sIdentity, "@");
			m_aLog.enterContext(CalcEngineLogContextConst.CONTEXT_JOB_ID, sPointer);
		}
		return m_aLog;
	}
	
	//-------------------------------------------------------------------------
	private Executable m_aProcess = null;
	
	//-------------------------------------------------------------------------
	private CountDownLatch m_aProcessSync = null;
	
	//-------------------------------------------------------------------------
	private ICalcEngineClient m_iAPI = null;

	//-------------------------------------------------------------------------
	private CalcEngineClientHTTP m_aInternalAPI = null;

	//-------------------------------------------------------------------------
	private ContextAwareLogHelp m_aLog = null;
}
