/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.List;

import org.apache.commons.lang.Validate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;
import com.openxchange.office_communication.office.jms.core.doc.DocumentCache;
import com.openxchange.office_communication.tools.logging.v2.LogConst;
import com.openxchange.office_communication.tools.persistence.ISimplePersistenceTransacted;
import com.openxchange.office_communication.tools.persistence.SimplePersistenceFactory;
import com.openxchange.office_communication.tools.persistence.impl.DiscPersistence;

//=============================================================================
/** Call the native (JNI) bridge directly within the current process.
 */
public class CalcEngineClientNative implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientNative.class);

    //-------------------------------------------------------------------------
//	private static final boolean IGNORE_ERRORS = false;
	
    //-------------------------------------------------------------------------
    private CalcEngineClientNative ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientNative create()
        throws Exception
    {
    	final CalcEngineClientNative aClient = new CalcEngineClientNative ();
    	return aClient;
    }

    //-------------------------------------------------------------------------
    public synchronized void enableAutoRestore (final String sPath)
    	throws Exception
    {
    	m_sAutoRestorePath      = sPath;
    	m_bIsAutoRestoreEnabled = ! StringUtils.isEmpty(m_sAutoRestorePath);
    	if (m_bIsAutoRestoreEnabled)
    		LOG.info("enable auto restore at '"+sPath+"' ...");
    }
    
    //-------------------------------------------------------------------------
    public synchronized void setWorkerId (final String sId)
        throws Exception
    {
    	m_sWorkerId = sId;
    }

    //-------------------------------------------------------------------------
    @Override
    public synchronized void createDocument(final String sHandle)
        throws Exception
    {
        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final boolean                 bOK     = aEngine.createDocument(sHandle);
        
        if (! bOK)
        	impl_logAndThrowError ("Creating new calc engine document with handle '"+sHandle+"' failed.");
    };
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized void destroyDocument(final String sHandle)
        throws Exception
    {
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sHandle))
        {
        	LOG.warn(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			           .forLevel    (ELogLevel.E_WARNING                                  )
        			           .toLog       ("Handle is unknown."                                 ));
        	return;
        }
        
        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final boolean                 bOK     = aEngine.destroyDocument(sHandle);
        
        if (! bOK)
        	impl_logAndThrowError ("Destroy of calc engine document with handle '"+sHandle+"' failed.");
        
    	impl_cleanCacheForDoc(sHandle);
    };

    //-------------------------------------------------------------------------
    @Override
    public synchronized ECalcEngineError executeOperation(final String       sHandle       ,
                                   			              final String       sOperationJSON,
                                   			              final StringBuffer sResultJSON   )
        throws Exception
    {
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sHandle))
        {
			final boolean bRestored = impl_restoreDocument (sHandle);
			if ( ! bRestored)
			{
	        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
				                    .forLevel    (ELogLevel.E_ERROR                                    )
				                    .toLog       ("document lost"                                      ));
	        	return ECalcEngineError.E_DOCUMENT_LOST;
			}
        }

        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final String                  sResult = aEngine.executeOperation(sHandle, sOperationJSON);
        
        if (sResult.isEmpty())
        	impl_logAndThrowError ("Execute operation on calc engine document '"+sHandle+"' failed.");

		sResultJSON.setLength(0      );
        sResultJSON.append   (sResult);

		impl_cacheDocument (sHandle, sOperationJSON);

		return ECalcEngineError.E_NONE;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                              final String... lPagedOperationJSONs)
        throws Exception
    {
    	throw new UnsupportedOperationException ("Restore from outside not supported yet - we do it by ourself !");
    	
//        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");
//
//        final CalcEngineNativeWrapper aEngine = mem_Engine ();
//    	      boolean                 bOK     = false;
//
//    	// a) create the document with given handle new
//      	LOG.info(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
//      					   .forLevel    (ELogLevel.E_INFO                                     )
//      					   .toLog       ("restore document - create document new"             ));
//
//      	bOK = aEngine.createDocument(sHandle);
//        if (! bOK)
//        	impl_logAndThrowError ("Recreating calc engine document with handle '"+sHandle+"' failed.");
//
//        // b) put all known operations to that new document for restore ...
//      	LOG.info(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
//				   		   .forLevel    (ELogLevel.E_INFO                                     )
//				   		   .toLog       ("restore document - apply operations"                ));
//        
//        for (String sOperationJSON : lPagedOperationJSONs)
//        {
//        	LOG.debug(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
//                    			.forLevel    (ELogLevel.E_DEBUG                                    )
//                    			.toLog       ("apply operation ["+sOperationJSON+"]"               ));
//
//            final String sResult = aEngine.executeOperation(sHandle, sOperationJSON);
//            bOK = !sResult.isEmpty();
//
//			if (! bOK)
//				impl_logAndThrowError ("Restore of operation set failed on document '"+sHandle+"'.");
//        }
//        
//        return ECalcEngineError.E_NONE;
    }
    
    //-------------------------------------------------------------------------
	@Override
	public synchronized ECalcEngineError copy(final String                   sDocHandle,
								              final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sDocHandle), "'"+sDocHandle+"' is an invalid handle.");

        // lost document and automatic restore is handled by called method 'executeOperation' already !

        final String           sOperationJSON = CalcEngineClipBoardEvent.toJSON4CopyEvent (aEvent);
        final StringBuffer     sResultJSON    = new StringBuffer (256);
        final ECalcEngineError eError         = executeOperation (sDocHandle, sOperationJSON, sResultJSON);

        if (eError != ECalcEngineError.E_NONE)
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
                    			.forLevel    (ELogLevel.E_ERROR                                       )
                    			.toLog       ("copy operation failed with error code : "+eError       ));
        	return eError;
        }
        
    	aEvent.sCopyOperationsJSON = sResultJSON.toString();
        return ECalcEngineError.E_NONE;
	}

    //-------------------------------------------------------------------------
	@Override
	public synchronized ECalcEngineError paste(final String                   sDocHandle,
								               final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sDocHandle), "'"+sDocHandle+"' is an invalid handle.");

        // lost document and automatic restore is handled by called method 'executeOperation' already !

        final String           sOperationJSON = CalcEngineClipBoardEvent.toJSON4PasteEvent (aEvent);
        final StringBuffer     sResultJSON    = new StringBuffer (256);
        final ECalcEngineError eError         = executeOperation (sDocHandle, sOperationJSON, sResultJSON);
        
        if (eError != ECalcEngineError.E_NONE)
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
        						.forLevel    (ELogLevel.E_ERROR                                       )
        						.toLog       ("paset operation failed with error code : "+eError      ));
        	return eError;
        }

    	aEvent.sPasteOperationsJSON = sResultJSON.toString();
        return ECalcEngineError.E_NONE;
	}

    //-------------------------------------------------------------------------
    @Override
    public synchronized String getVersion() 
        throws Exception
    {
        final CalcEngineNativeWrapper aEngine  = mem_Engine ();
        final String                  sVersion = aEngine.getVersion();
        return sVersion;
    }

    //-------------------------------------------------------------------------
    @Override
    public synchronized void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
        final CalcEngineNativeWrapper aEngine  = mem_Engine ();
        aEngine.setLogLevel(eLevel);
    }

    //-------------------------------------------------------------------------
    @Override
    public synchronized void dispose ()
    	throws Exception
    {
//		final DocumentCache aCache = mem_DocCache ();
//		if (aCache != null)
//			aCache.clear ();
    }

	//-------------------------------------------------------------------------
    private boolean impl_isHandleKnown (final String sHandle)
       throws Exception
    {
    	// TODO implement me better ... more performance - please
    	
    	final CalcEngineNativeWrapper aEngine  = mem_Engine ();
    	final List< String >          lHandles = aEngine.getHandleList ();
    	final boolean                 bIsKnown = lHandles.contains (sHandle);
    	return bIsKnown;
    }
    
	//-------------------------------------------------------------------------
	private void impl_cacheDocument (final String sHandle    ,
									 final String sOperations)
	    throws Exception
	{
		if ( ! m_bIsAutoRestoreEnabled)
		{
			LOG.debug ("No document cache available. Caching document not possible.");
			return;
		}
		
		if (LOG.isDebugEnabled())
			LOG.debug ("cache operation ["+LogConst.DOC_HANDLE+"="+sHandle+", "+LogConst.OPERATION+"="+sOperations+"] ...");
		else
			LOG.info ("cache operation ["+LogConst.DOC_HANDLE+"="+sHandle+"] ...");
		mem_DocCache ().cacheDocOperation(sHandle, sOperations);
	}
	
	//-------------------------------------------------------------------------
	private boolean impl_restoreDocument (final String sHandle)
	    throws Exception
	{
		try
		{
			if ( ! m_bIsAutoRestoreEnabled)
				return false;
			
			LOG.info ("restore document ["+LogConst.DOC_HANDLE+"="+sHandle+"] ...");
	
			final CalcEngineNativeWrapper aImpl  = mem_Engine ();
	
			LOG.info ("... create document new ["+LogConst.DOC_HANDLE+"="+sHandle+"]");
			if ( ! aImpl.createDocument(sHandle))
			{
				LOG.error(LogConst.FAILED);
				return false;
			}
	
			LOG.info("... apply operations ["+LogConst.DOC_HANDLE+"="+sHandle+"]");
			final List< String > lOperations = mem_DocCache ().getCacheDocOperations(sHandle);
			      int            nOp         = 0;
			for (final String sOperation : lOperations)
			{
				if (StringUtils.isEmpty(sOperation))
				{
					LOG.error("... can not apply <empty> operation ["+LogConst.DOC_HANDLE+"="+sHandle+", op-nr="+nOp+"] ...");
					return false;
				}
				
				LOG.debug("... apply operation ["+LogConst.DOC_HANDLE+"="+sHandle+", op-nr="+nOp+", "+LogConst.OPERATION+"="+sOperation+"] ...");
				nOp++;

				final String lResults = aImpl.executeOperation(sHandle, sOperation);
				if (StringUtils.isEmpty(lResults))
				{
					LOG.error("... apply operation failed ["+LogConst.DOC_HANDLE+"="+sHandle+", op-nr="+nOp+", "+LogConst.OPERATION+"="+sOperation+"] ...");
					LOG.info ("... destroy document ["+LogConst.DOC_HANDLE+"="+sHandle+"] ...");
					aImpl.destroyDocument(sHandle);
	
					// TODO should we clean our cache in case restore from cache failed ?! (error in error in ... ?)
					return false;
				}
			}
			
			LOG.info ("... restore document OK ["+LogConst.DOC_HANDLE+"="+sHandle+"]");
			return true;
		}
		catch (final Throwable ex)
		{
			LOG.error(ex);
			return false;
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_cleanCacheForDoc (final String sHandle)
		throws Exception
	{
		if ( ! m_bIsAutoRestoreEnabled)
			return;

		LOG.info ("clean cache for document ["+LogConst.DOC_HANDLE+"="+sHandle+"] ...");
		mem_DocCache ().removeDocFromCache(sHandle);
	}
	
	//-------------------------------------------------------------------------
    private void impl_logAndThrowError (final String sMessage)
        throws Exception
    {
    	LOG.error (sMessage);
    	throw new Exception (sMessage);
    }
    
    //-------------------------------------------------------------------------
    private CalcEngineNativeWrapper mem_Engine ()
       throws Exception
    {
        if (m_aEngine == null)
            m_aEngine = new CalcEngineNativeWrapper ();
        return m_aEngine;
    }
    
    //-------------------------------------------------------------------------
    private DocumentCache mem_DocCache ()
        throws Exception
    {
    	if (m_aDocCache == null)
    	{
    		Validate.isTrue( ! StringUtils.isEmpty(m_sAutoRestorePath), "Auto restore not enabled - who calls this function ?");
    		
			final DocumentCache aCache  = new DocumentCache ();
			final String[]      lConfig = new String[4];
			
			lConfig[0] = ISimplePersistenceTransacted.CFG_PERSISTENCE_SCOPE;
			lConfig[1] = m_sWorkerId;
			lConfig[2] = DiscPersistence.CFG_DATA_PATH;
			lConfig[3] = m_sAutoRestorePath;
			
			final ISimplePersistenceTransacted iPersistence = SimplePersistenceFactory.get(DiscPersistence.class.getName(), lConfig);
			aCache.setPersistenceLayer(iPersistence);
			
			m_aDocCache = aCache;
    	}
    	return m_aDocCache;
    }

    //-------------------------------------------------------------------------
    private synchronized ContextAwareLogHelp mem_Log ()
        throws Exception
    {
    	if (m_aContextLog == null)
    		m_aContextLog = new ContextAwareLogHelp (LOG);
    	return m_aContextLog;
    }
    
    //-------------------------------------------------------------------------
    private CalcEngineNativeWrapper m_aEngine = null;

    //-------------------------------------------------------------------------
    private String m_sWorkerId = null;
    
    //-------------------------------------------------------------------------
    private boolean m_bIsAutoRestoreEnabled = false;
    
    //-------------------------------------------------------------------------
    private String m_sAutoRestorePath = null;
    
    //-------------------------------------------------------------------------
    private DocumentCache m_aDocCache = null;

    //-------------------------------------------------------------------------
    private ContextAwareLogHelp m_aContextLog = null;
}