/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.Map;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openxchange.office_communication.jms.core.plain.EErrorCode;
import com.openxchange.office_communication.jms.core.plain.JMSClientBase;
import com.openxchange.office_communication.jms.core.plain.JMSRuntimeStats;
import com.openxchange.office_communication.office.jms.messages.CalcEngineMessageBean;
import com.openxchange.office_communication.office.jms.messages.ECalcEngineMethod;
import com.openxchange.office_communication.office.jms.messages.ECalcEngineResult;
import com.openxchange.office_communication.tools.logging.ELogLevel;
import com.openxchange.office_communication.tools.logging.v2.LogConst;
import com.openxchange.office_communication.tools.logging.v2.LogContext;
import com.openxchange.office_communication.tools.logging.v2.LogVars;
import com.openxchange.office_communication.tools.logging.v2.Logger;
import com.openxchange.office_communication.tools.logging.v2.Slf4JLogger;

//=============================================================================
/** Run the calc engine in 'client JMS' mode.
 */
public class CalcEngineClientJMS extends    JMSClientBase
								 implements ICalcEngineClient
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = Slf4JLogger.create(CalcEngineClientJMS.class);
	
	//-------------------------------------------------------------------------
	private CalcEngineClientJMS ()
	    throws Exception
	{}

	//-------------------------------------------------------------------------
	public synchronized static CalcEngineClientJMS create(final String sOutQueue)
	    throws Exception
	{
		System.err.println ("#### DBG : new instance : CalcEngineClientJMS");
		final CalcEngineConfig    aConfig  = CalcEngineConfig.get();
		final int                 nTimeout = aConfig.getRequestTimeout();
		final CalcEngineClientJMS aInst    = new CalcEngineClientJMS  ();
		final long                nTTL     = 10 * 60 * 1000; // 10min
		
	  	aInst.setFullQualifiedRequestQueue(sOutQueue);
//	  	aInst.setTimeout                  (nTimeout );
//	  	aInst.setMessageTTL               (nTTL     );

	  	if (aConfig.isJmsStatisticsEnabled())
	  	{
			System.err.println ("#### DBG : enable JMSRuntimeStats ...");
	  		JMSRuntimeStats.setEnabled             (true );
			JMSRuntimeStats.setAutoDumpIntervalInMS(10000);
	  	}

	  	return aInst;
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ void start ()
		throws Exception
	{
		final LogContext aLog = impl_newLogContext ("start client");
		
		aLog.forLevel    (ELogLevel.E_INFO)
		    .withMessage ("initiate start")
		    .log         ();
		
		super.start ();
	
		aLog.forLevel    (ELogLevel.E_INFO)
		    .withMessage ("started"       )
		    .withVars    (new LogVars ()
			{
				public void defineVars(final Map<String, Object> lVars) throws Exception
				{
					lVars.put ("response-queue", getResponseQueueName());
				}
			})
		   .log();
	}
	
	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ void stop ()
		throws Exception
	{
		final LogContext aLog = impl_newLogContext ("stop client");

		aLog.forLevel    (ELogLevel.E_INFO)
		    .withMessage ("initiate stop" )
		    .withVars    (new LogVars ()
			{
				public void defineVars(final Map<String, Object> lVars) throws Exception
				{
					lVars.put ("response-queue", getResponseQueueName());
				}
			})
		    .log ();

		super.stop ();

		aLog.forLevel    (ELogLevel.E_INFO)
		    .withMessage ("stopped"       )
		    .log         ();
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ void createDocument(final String sDocHandle)
		throws Exception
	{
		final LogContext aLog = impl_newLogContext4Method (ECalcEngineMethod.E_CREATE_DOCUMENT);

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage(LogConst.START  )
		    .log();
		
    	//-----
    	
		final CalcEngineMessageBean aRequest   = new CalcEngineMessageBean();
    	aRequest.setMethod   (ECalcEngineMethod.E_CREATE_DOCUMENT);
    	aRequest.setDocHandle(sDocHandle);

		//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("start request" )
		    .withVars   (new LogVars()
			{
				public void defineVars(final Map<String, Object> lVars) throws Exception
				{
					lVars.put (LogConst.DOC_HANDLE, sDocHandle);
				}
			})
		    .log();

    	//-----

    	EErrorCode            eResult   = null;
    	CalcEngineMessageBean aResponse = null;
    	try
    	{
	    	aResponse = super.requestSynchronous(aRequest);
	    	eResult   = aResponse.getErrorCode();
    	}
    	catch (final TimeoutException exTimeout)
    	{
    		eResult = EErrorCode.E_TIMEOUT;
    	}

    	//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("got response"  )
		    .setVar     (LogConst.DOC_HANDLE, sDocHandle)
		    .setVar     (LogConst.ERROR_CODE, eResult   )
		    .log();
    	
    	//-----

    	if (eResult != EErrorCode.E_OK)
    	{
        	final String sError = aLog.forLevel   (ELogLevel.E_ERROR)
        							  .withMessage(LogConst.END     )
        							  .log();

    		throw new Exception (sError);
    	}

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage(LogConst.END    )
		    .log();
	}
	
	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ void destroyDocument(final String sDocHandle)
		throws Exception
	{
		final LogContext aLog = impl_newLogContext4Method (ECalcEngineMethod.E_DESTROY_DOCUMENT);

		aLog.forLevel   (ELogLevel.E_INFO)
			.withMessage(LogConst.START  )
			.log();
		
		//-----
    	
    	final CalcEngineMessageBean aRequest = new CalcEngineMessageBean();
    	aRequest.setMethod   (ECalcEngineMethod.E_DESTROY_DOCUMENT);
    	aRequest.setDocHandle(sDocHandle                          );
    	
		//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("start request" )
		    .withVars   (new LogVars()
			{
				public void defineVars(final Map<String, Object> lVars) throws Exception
				{
					lVars.put (LogConst.DOC_HANDLE, sDocHandle);
				}
			})
		    .log();
	
    	//-----

    	EErrorCode            eResult   = null;
    	CalcEngineMessageBean aResponse = null;
    	try
    	{
	    	aResponse = super.requestSynchronous(aRequest);
	    	eResult   = aResponse.getErrorCode();
    	}
    	catch (final TimeoutException exTimeout)
    	{
    		eResult = EErrorCode.E_TIMEOUT;
    	}

    	//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("got response"  )
		    .setVar     (LogConst.DOC_HANDLE, sDocHandle)
		    .setVar     (LogConst.ERROR_CODE, eResult   )
		    .log();
		
		//-----
    	
    	if (eResult != EErrorCode.E_OK)
    	{
        	final String sError = aLog.forLevel   (ELogLevel.E_ERROR)
									  .withMessage(LogConst.END     )
									  .log();

        	throw new Exception (sError);
    	}
    	
    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage(LogConst.END    )
		    .log();
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ ECalcEngineError executeOperation(final String       sDocHandle    ,
											                    final String       sOperationJSON,
											                    final StringBuffer sResultJSON   )
		throws Exception
	{
		final LogContext aLog = impl_newLogContext4Method (ECalcEngineMethod.E_EXECUTE_OPERATION)
									.setVar(LogConst.DOC_HANDLE, sDocHandle);

		aLog.forLevel   (ELogLevel.E_INFO)
			.withMessage(LogConst.START  )
			.log        ();
		
		//-----
		
    	final CalcEngineMessageBean aRequest = new CalcEngineMessageBean();
    	aRequest.setMethod   (ECalcEngineMethod.E_EXECUTE_OPERATION);
    	aRequest.setDocHandle(sDocHandle                           );
    	aRequest.setBody     (sOperationJSON                       );

    	//-----
    	
    	aLog.forLevel   (ELogLevel.E_INFO )
		    .withMessage("start request"  )
			.forLevel   (ELogLevel.E_TRACE)
		    .setVar     (LogConst.CONTENT , sOperationJSON)
			.log        ();

    	//-----

    	EErrorCode            eResult   = null;
    	CalcEngineMessageBean aResponse = null;
    	try
    	{
	    	aResponse = super.requestSynchronous(aRequest);
	    	eResult   = aResponse.getErrorCode();
    	}
    	catch (final TimeoutException exTimeout)
    	{
    		eResult = EErrorCode.E_TIMEOUT;
    	}

    	//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("got response"  )
		    .setVar     (LogConst.DOC_HANDLE, sDocHandle)
		    .setVar     (LogConst.ERROR_CODE, eResult   )
		    .log();

    	//-----
    	
    	if (eResult == EErrorCode.E_CUSTOM)
    	{
    		final ECalcEngineResult eDetailedResult = aResponse.getBody();

    		if (eDetailedResult == ECalcEngineResult.E_UNKNOWN_HANDLE)
    		{
    	    	aLog.forLevel   (ELogLevel.E_WARNING)
	    		    .withMessage(LogConst.END       )
    				.setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_DOCUMENT_LOST)
    				.log        ();

    			return ECalcEngineError.E_DOCUMENT_LOST;
    		}
    		
    		throw new UnsupportedOperationException ("No support for custom error '"+eDetailedResult+"'.");
    	}

    	//-----

    	if (eResult == EErrorCode.E_TIMEOUT)
    	{
	    	aLog.forLevel   (ELogLevel.E_WARNING)
			    .withMessage(LogConst.END       )
			    .setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_TIMEOUT)
			    .log        ();

    		return ECalcEngineError.E_TIMEOUT;
    	}

    	//-----

    	if (eResult != EErrorCode.E_OK)
    	{
        	final String sError = aLog.forLevel   (ELogLevel.E_ERROR)
									  .withMessage(LogConst.END     )
									  .log();
				
			throw new Exception (sError);
    	}
    	
    	//-----

    	final String sResult = aResponse.getBody();
    	if (StringUtils.isEmpty(sResult))
    	{	
        	final String sError = aLog.forLevel   (ELogLevel.E_ERROR             )
					  				  .withMessage("no JSON result "+LogConst.END)
					  				  .log();

        	throw new Exception (sError);
    	}

    	aLog.forLevel   (ELogLevel.E_INFO         )
		    .withMessage(LogConst.END             )
		    .forLevel   (ELogLevel.E_DEBUG        )
	        .setVar     (LogConst.CONTENT, sResult)
		    .log        ();

    	sResultJSON.append (sResult);
    	return ECalcEngineError.E_NONE;
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ECalcEngineError restoreDocument(final String    sDocHandle          ,
											                  final String... lPagedOperationJSONs)
		throws Exception
	{
		throw new UnsupportedOperationException("not implemented yet");
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ ECalcEngineError copy(final String                   sDocHandle,
								                    final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
		final LogContext aLog = impl_newLogContext4Method (ECalcEngineMethod.E_COPY)
									.setVar(LogConst.DOC_HANDLE, sDocHandle);
		
		aLog.forLevel   (ELogLevel.E_INFO)
			.withMessage(LogConst.START  )
			.log        ();
		
		//-----
		
        final String                sCopyJSON = CalcEngineClipBoardEvent.toJSON4CopyEvent (aEvent);
		final CalcEngineMessageBean aRequest  = new CalcEngineMessageBean();
		aRequest.setMethod   (ECalcEngineMethod.E_EXECUTE_OPERATION);
		aRequest.setDocHandle(sDocHandle                           );
		aRequest.setBody     (sCopyJSON                            );
		
		//-----
		
		aLog.forLevel   (ELogLevel.E_INFO )
			.withMessage("start request"  )
			.forLevel   (ELogLevel.E_TRACE)
			.setVar     (LogConst.CONTENT , sCopyJSON)
			.log        ();
		
    	//-----

    	EErrorCode            eResult   = null;
    	CalcEngineMessageBean aResponse = null;
    	try
    	{
	    	aResponse = super.requestSynchronous(aRequest);
	    	eResult   = aResponse.getErrorCode();
    	}
    	catch (final TimeoutException exTimeout)
    	{
    		eResult = EErrorCode.E_TIMEOUT;
    	}

    	//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("got response"  )
		    .setVar     (LogConst.DOC_HANDLE, sDocHandle)
		    .setVar     (LogConst.ERROR_CODE, eResult   )
		    .log();
		
		//-----
		
		if (eResult == EErrorCode.E_CUSTOM)
		{
			final ECalcEngineResult eDetailedResult = aResponse.getBody();
			
			if (eDetailedResult == ECalcEngineResult.E_UNKNOWN_HANDLE)
			{
				aLog.forLevel   (ELogLevel.E_WARNING)
					.withMessage(LogConst.END       )
					.setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_DOCUMENT_LOST)
					.log        ();
				
				return ECalcEngineError.E_DOCUMENT_LOST;
			}
			
			throw new UnsupportedOperationException ("No support for custom error '"+eDetailedResult+"'.");
		}
		
		//-----
		
		if (eResult == EErrorCode.E_TIMEOUT)
		{
			aLog.forLevel   (ELogLevel.E_WARNING)
				.withMessage(LogConst.END       )
				.setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_TIMEOUT)
				.log        ();
			
			return ECalcEngineError.E_TIMEOUT;
		}
		
		//-----
		
		if (eResult != EErrorCode.E_OK)
		{
			final String sError = aLog.forLevel   (ELogLevel.E_ERROR)
									  .withMessage(LogConst.END     )
									  .log();
			
			throw new Exception (sError);
		}
		
		//-----
		
		final String sResult = aResponse.getBody();
		if (StringUtils.isEmpty(sResult))
		{	
			final String sError = aLog.forLevel   (ELogLevel.E_ERROR             )
					  				  .withMessage("no JSON result "+LogConst.END)
					  				  .log();
			
			throw new Exception (sError);
		}
		
		aLog.forLevel   (ELogLevel.E_INFO         )
			.withMessage(LogConst.END             )
			.forLevel   (ELogLevel.E_DEBUG        )
			.setVar     (LogConst.CONTENT, sResult)
			.log        ();
		
		aEvent.sCopyOperationsJSON = sResult;
		return ECalcEngineError.E_NONE;
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ ECalcEngineError paste(final String                   sDocHandle,
								                     final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
		final LogContext aLog = impl_newLogContext4Method (ECalcEngineMethod.E_PASTE)
									.setVar(LogConst.DOC_HANDLE, sDocHandle);

		aLog.forLevel   (ELogLevel.E_INFO)
			.withMessage(LogConst.START  )
			.log        ();

		//-----
		
        final String                sPasteJSON = CalcEngineClipBoardEvent.toJSON4PasteEvent (aEvent);
		final CalcEngineMessageBean aRequest   = new CalcEngineMessageBean();
		aRequest.setMethod   (ECalcEngineMethod.E_EXECUTE_OPERATION);
		aRequest.setDocHandle(sDocHandle                           );
		aRequest.setBody     (sPasteJSON                           );

		//-----
		
		aLog.forLevel   (ELogLevel.E_INFO )
			.withMessage("start request"  )
			.forLevel   (ELogLevel.E_TRACE)
			.setVar     (LogConst.CONTENT , sPasteJSON)
			.log        ();

    	//-----

    	EErrorCode            eResult   = null;
    	CalcEngineMessageBean aResponse = null;
    	try
    	{
	    	aResponse = super.requestSynchronous(aRequest);
	    	eResult   = aResponse.getErrorCode();
    	}
    	catch (final TimeoutException exTimeout)
    	{
    		eResult = EErrorCode.E_TIMEOUT;
    	}

    	//-----

    	aLog.forLevel   (ELogLevel.E_INFO)
		    .withMessage("got response"  )
		    .setVar     (LogConst.DOC_HANDLE, sDocHandle)
		    .setVar     (LogConst.ERROR_CODE, eResult   )
		    .log();

		//-----
		
		if (eResult == EErrorCode.E_CUSTOM)
		{
			final ECalcEngineResult eDetailedResult = aResponse.getBody();
			
			if (eDetailedResult == ECalcEngineResult.E_UNKNOWN_HANDLE)
			{
				aLog.forLevel   (ELogLevel.E_WARNING)
					.withMessage(LogConst.END       )
					.setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_DOCUMENT_LOST)
					.log        ();
				
				return ECalcEngineError.E_DOCUMENT_LOST;
			}
			
			throw new UnsupportedOperationException ("No support for custom error '"+eDetailedResult+"'.");
		}

		//-----
		
		if (eResult == EErrorCode.E_TIMEOUT)
		{
			aLog.forLevel   (ELogLevel.E_WARNING)
				.withMessage(LogConst.END       )
				.setVar     (LogConst.ERROR_CODE, ECalcEngineError.E_TIMEOUT)
				.log        ();
			
			return ECalcEngineError.E_TIMEOUT;
		}

		//-----
		
		if (eResult != EErrorCode.E_OK)
		{
			final String sError = aLog.forLevel   (ELogLevel.E_ERROR)
									  .withMessage(LogConst.END     )
									  .log();
			
			throw new Exception (sError);
		}

		//-----
		
		final String sResult = aResponse.getBody();
		if (StringUtils.isEmpty(sResult))
		{	
			final String sError = aLog.forLevel   (ELogLevel.E_ERROR             )
					  				  .withMessage("no JSON result "+LogConst.END)
					  				  .log();
			
			throw new Exception (sError);
		}

		aLog.forLevel   (ELogLevel.E_INFO         )
			.withMessage(LogConst.END             )
			.forLevel   (ELogLevel.E_DEBUG        )
			.setVar     (LogConst.CONTENT, sResult)
			.log        ();

		aEvent.sPasteOperationsJSON = sResult;
		return ECalcEngineError.E_NONE;
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ String getVersion()
		throws Exception
	{
		//throw new UnsupportedOperationException("not implemented yet");
		return "JMS Client 1.0.0";
	}

	//-------------------------------------------------------------------------
	@Override
	public /*no synchronize*/ void setLogLevel(final com.openexchange.office.tools.logging.ELogLevel eLevel)
		throws Exception
	{
		// not supported in JMS anv any longer !
		// dedicated processes needs it's own logging ...
	}

	//-------------------------------------------------------------------------
    @Override
	public /*no synchronize*/ void dispose ()
    	throws Exception
    {
    	stop ();
    }
	
	//-------------------------------------------------------------------------
	private LogContext impl_newLogContext (final String sMessage)
		throws Exception
	{
		final LogContext aLogContext = LOG.inCategory (LogConst.SPREADSHEET                               ,
										        	   LogConst.CLIENT                                    )
										  .setVar     (LogConst.THIS  , ObjectUtils.identityToString(this));
		
		if ( ! StringUtils.isEmpty(sMessage))
			aLogContext.withMessage(sMessage);
			
		return aLogContext;
	}

	//-------------------------------------------------------------------------
	private LogContext impl_newLogContext4Method (final ECalcEngineMethod eMethod)
		throws Exception
	{
		String sMessage = null;
		
		if (eMethod == ECalcEngineMethod.E_CREATE_DOCUMENT)
			sMessage = "create-document";
		else
		if (eMethod == ECalcEngineMethod.E_DESTROY_DOCUMENT)
			sMessage = "destroy-document";
		else
		if (eMethod == ECalcEngineMethod.E_EXECUTE_OPERATION)
			sMessage = "execute-operation";
		else
		if (eMethod == ECalcEngineMethod.E_COPY)
			sMessage = "copy";
		else
		if (eMethod == ECalcEngineMethod.E_PASTE)
			sMessage = "paste";
		else
			throw new UnsupportedOperationException ("No support for '"+eMethod+"'.");
		
		final LogContext aLogContext = impl_newLogContext (sMessage                )
										   .setVar        (LogConst.METHOD, eMethod);
		return aLogContext;
	}
}
