package com.openexchange.office.tools;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageCapability;
import com.openexchange.file.storage.composition.FolderID;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class StorageHelper
{
    // capabilities cached locally
    boolean efficientRetrieval = false;
    boolean fileVersions = false;
    boolean setCapabilitiesCalled = false;
    boolean supportsPersistenIDs = false;
    boolean ignorableVersion = false;

    final String folderId;

    public StorageHelper(final IDBasedFileAccess fileAccess, final String folderId) {
        this.folderId = folderId;
        impl_setCapabilities(fileAccess, folderId);
    }

    public StorageHelper(final ServiceLookup services, final Session session, final String folderId) {
        this.folderId = folderId;

        final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        impl_setCapabilities(fileAccess, folderId);
    }

    public String getFolderId() {
        return folderId;
    }

    public boolean supportsEfficientRetrieval() {
        return efficientRetrieval;
    }

    public boolean supportsFileVersions() {
        return fileVersions;
    }

    public boolean supportsPersistentIDs() {
        return supportsPersistenIDs;
    }

    public boolean supportsIgnorableVersion() {
        return ignorableVersion;
    }

    /**
     * Determines, using the folder id and file access, if the storage supports
     * file versions or not.
     *
     * @param fileAccess
     *  A valid file access .
     *
     * @param folderId
     *  A valid folder id to be checked for rename support or not.
     *
     * @return
     *  TRUE if the storage supports versions, otherwise not.
     */
    private boolean impl_setCapabilities(final IDBasedFileAccess fileAccess, final String folderId) {
        boolean result = false;

        try {
            final FolderID folderIdentifier = new FolderID(folderId);

            efficientRetrieval = fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.EFFICIENT_RETRIEVAL);
            fileVersions = fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.FILE_VERSIONS);
            supportsPersistenIDs = fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.PERSISTENT_IDS);
            ignorableVersion = fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.IGNORABLE_VERSION);
        } catch (NumberFormatException e) {
            // nothing to do
        } catch (OXException e) {
            // nothing to do
        }

        return result;
    }

}
