/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import static com.openexchange.realtime.util.EncodeUtility.decodeResourcePart;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.base.Optional;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.capabilities.CapabilitySet;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.groupware.ldap.User;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineClipBoard;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.realtime.impl.OXDocument.SaveResult;
import com.openexchange.office.realtime.impl.calc.AsyncLoadRequestQueue;
import com.openexchange.office.realtime.impl.calc.CalcEngineException;
import com.openexchange.office.realtime.impl.calc.CellsMerger;
import com.openexchange.office.realtime.impl.calc.IRequestProcessor;
import com.openexchange.office.realtime.impl.calc.Request;
import com.openexchange.office.realtime.impl.calc.RowColsMerger;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.realtime.tools.SystemInfoHelper;
import com.openexchange.office.tools.ApplicationType;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.DocFileHelper;
import com.openexchange.office.tools.DocFileHelper.RenameResult;
import com.openexchange.office.tools.DocumentFormat;
import com.openexchange.office.tools.DocumentFormatHelper;
import com.openexchange.office.tools.DocumentMetaData;
import com.openexchange.office.tools.DocumentType;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.ExceptionToErrorCode;
import com.openexchange.office.tools.FileDescriptor;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.FolderHelper;
import com.openexchange.office.tools.IDUtils;
import com.openexchange.office.tools.RecentFileListManager;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.UserConfigurationHelper;
import com.openexchange.office.tools.UserConfigurationHelper.Mode;
import com.openexchange.office.tools.htmldoc.ConfHelper;
import com.openexchange.office.tools.htmldoc.GenericHtmlDocumentBuilder;
import com.openexchange.office.tools.htmldoc.TextTableLimits;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.memory.MemoryObserver;
import com.openexchange.office.tools.message.MessageChunk;
import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.office.tools.monitoring.CloseEvent;
import com.openexchange.office.tools.monitoring.CloseType;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.OpenEvent;
import com.openexchange.office.tools.monitoring.OpenType;
import com.openexchange.office.tools.monitoring.OperationsEvent;
import com.openexchange.office.tools.monitoring.OperationsType;
import com.openexchange.office.tools.monitoring.SaveEvent;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.realtime.Asynchronous;
import com.openexchange.realtime.group.GroupDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.Duration;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

/**
 * The Connection class implements the necessary specialties of a OX Document instance. This includes loading/saving documents, apply
 * changes, switching the edit rights between clients and send update messages to the connected clients. {@link Connection}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
final public @NotThreadSafe
class Connection extends GroupDispatcher implements IRequestProcessor {

    private final int MAX_OPERATIONS = 1000;

    /**
     * Initializes a new {@link Connection}.
     *
     * @param services A service lookup instance which provides access to other services needed by the OX Documents GroupDispatcher.
     * @param id The ID which references the resource used for this OX Documents GroupDispatcher. Normally references a file in the
     *            Infostore.
     * @throws OXException
     */
    public Connection(ServiceLookup serviceLookup, ID id, AsyncLoadRequestQueue queue) {
        super(id, m_handler);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Connection created: " + id.toString());
        }

        // initialize ServiceLookup and ResourceManager
        m_services = serviceLookup;
        m_resourceManager = new ResourceManager(m_services);
        m_loadRequestQueue = queue;
        m_disposed.set(false);
        m_saveOnDispose.set(true);
        m_startupTime = new Date();
        // No need to support a user-based max time for sending last actions.
        m_nMaxTimeForSendingLastActions = getConfigurationService().getIntProperty(
            "io.ox/office//module/switchingEditRightsTimeout",
            DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS);

        if (m_nMaxTimeForSendingLastActions != 0) {
            // set time in milli seconds
            m_nMaxTimeForSendingLastActions *= 1000;
        }
        setCreateVersion(true);

        // resource layout is: folderId.fileId
        final String resource = id.getResource();

        if (null != resource) {
            final int fileIdDelimiter = resource.indexOf('.');
            String folderId = null;
            String fileId = null;

            if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
                folderId = decodeResourcePart(resource.substring(0, fileIdDelimiter));
                fileId = decodeResourcePart(resource.substring(fileIdDelimiter + 1));
            }

            if ((null != folderId) && (folderId.length() > 0) && (null != fileId) && (fileId.length() > 0)) {
                m_fileHelper = new DocFileHelper(m_services, folderId, fileId);
            }

            m_syncStatus = new Sync(folderId, fileId);
        }

        if (LOG.isDebugEnabled()) {
            if (null != m_fileHelper) {
                LOG.debug("RT connection: Established for resource: " + id.toString());
            } else {
                LOG.debug("RT connection: NOT established for resource: " + id.toString());
            }
        }
    }

    /**
     * Provides the configuration service.
     *
     * @return A configuration service instance.
     */
    private ConfigurationService getConfigurationService() {
        return m_services.getService(com.openexchange.config.ConfigurationService.class);
    }

    /**
     * Handles the "firstjoined" notification provided by the Realtime-Framework. This method is called when the first client joins/creates
     * the document.
     *
     * @param id The ID of the first client that joins/creates to the document (chat-room). (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#firstJoined(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void firstJoined(ID id) {
        m_threadId = Thread.currentThread().getId();
        setCreateVersion(true);

        if (null != id) {
            synchronized (m_firstJoinedID) {
                this.m_firstJoinedID = id.toString();
            }
        }

        if (null != m_fileHelper) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: First client joined RT connection: " + ((null != id) ? id.toString() : "n/a"));
                LOG.debug("RT connection: Thread Id = " + String.valueOf(m_threadId));
            }
        } else {
            LOG.error("RT connection: Import of operations not possible due to invalid file parameters: " + ((null != id) ? id.toString() : ""));
        }

        m_resourceManager.lockResources(true);
    }

    /**
     * Handles the "join" notification provided by the Realtime-Framework. This method is called when a client joins the document.
     *
     * @param id The ID of the client that want to join the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onJoin(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onJoin(ID id) {
        super.onJoin(id);

        checkForDisposed();
        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onJoin called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Client joined: " + ((null != id) ? id.toString() : ""));
        }

        int curClientCount = 0;
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: onJoin called from different thread " + String.valueOf(threadId));
        }

        ServerSession session = null;
        try {
            session = id.toSession();
            if ((null != session) && (null == m_storageHelper)) {
                m_storageHelper = new StorageHelper(m_fileHelper.getFileAccess(session), m_fileHelper.getFolderId());
            }
        }
        catch (OXException e) {
            LOG.error("RT connection: Exception catched while trying to access session", e);
            throw new RuntimeException("Cannot add user without valid session");
        }

        synchronized (m_connectionStatus) {
            if (id != null) {
                try {
                    final int userId = SessionUtils.getUserId(session);

                    m_connectionStatus.addActiveUser(id.toString(), session.getUser().getDisplayName(), userId, new JSONObject());
                } catch (Exception e) {
                    LOG.error("RT connection: onJoin catches exception while retrieving session from joining user ID. ", e);
                }
            }
            curClientCount = m_connectionStatus.getActiveClients();
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                debugPrintActiveClients(m_connectionStatus);
            }
        }

        boolean joiningClientReceivesEditRights = false;
        boolean forceSwitch = false;

        synchronized (m_connectionStatus) {
            String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            String currentEditUser = m_connectionStatus.getCurrentEditingUserId();

            if ((1 == curClientCount) && (null != id)) {
                // first client always receives the edit rights
                joiningClientReceivesEditRights = true;
            } else if (((null != currentEditUser) && (currentEditUser.length() == 0)) && ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() == 0))) {
                // Fix for 30583 & better edit rights handling
                // Currently no one has the edit rights and no one wants to
                // receive the edit rights => next joining client receives the edit rights
                joiningClientReceivesEditRights = true;
            } else {
                // Special case using inactivity heuristic to determine to provide
                // edit rights to the new joining user.
                try {
                    if ((null != currentEditUser) && (currentEditUser.length() > 0)) {
                        int userId1 = m_connectionStatus.getUserId(currentEditUser);
                        int userId2 = IDUtils.getUserIdFromRealTimeId(id);

                        if (userId1 == userId2) {
                            // set new joining client as new editor without sending
                            // preparelosing etc. messages
                            joiningClientReceivesEditRights = true;
                            forceSwitch = true;
                        }
                    }
                } catch (Exception e) {
                    // nothing to do -> this is just a heuristic and can be ignored
                }
            }
        }

        if (joiningClientReceivesEditRights) {
            try {
                // No need that acquireEdit sends any update message
                // therefore set silent to true. As stated above
                // the current state is transferred via getWelcomeMessage.
                acquireEdit(null, id, true, forceSwitch);
            } catch (OXException e) {
                LOG.error("RT connection: onJoin calling acquireEdit failed with exception", e);
            }
        }

        if (curClientCount > 1) {
            // send update to other clients to update their active clients count etc.
            updateClients(new MessageData(null, syncGetConnectionStatus(), null), id);
        }
    }

    /**
     * Handles the "join" notification from the Realtime-Framework. This method is called when a client joins and provides additional
     * information via the stanza.
     *
     * @param id The ID of the client that want to join the document (chat-room).
     * @param stanza
     */
    @Override
    protected void onJoin(ID id, Stanza stanza) {
        super.onJoin(id, stanza);

        checkForDisposed();
        Session session = MessageHelper.getServerSession(null, id);

        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "connectData"), true);
        if ((null != session) && (null != jsonRequest)) {

            // store application type to be used later
            final ApplicationType app = ApplicationType.stringToEnum(jsonRequest.optString("app", null));
            if (ApplicationType.APP_NONE == m_appType) {
                m_appType = app;
            }
            // store if the document was newly created or not
            m_newDocLoaded = jsonRequest.optBoolean("newDocument");

            if (LOG.isDebugEnabled()) {
                Boolean useLocalStorage = null;
                try {
                    useLocalStorage = jsonRequest.getBoolean("useLocalStorage");
                } catch (Exception e) {
                    // Can be ignored, this is just an optional value
                }

                StringBuilder strBuf = new StringBuilder("RT connection: onJoin called with connect data for document: ");
                strBuf.append(getDocumentFolderAndFileId());
                strBuf.append(" UseLocalStorage=");
                strBuf.append((null == useLocalStorage) ? "unknown" : useLocalStorage);
                strBuf.append(" app=");
                strBuf.append(app);
                strBuf.append(" newDocument=");
                strBuf.append(m_newDocLoaded);
                LOG.debug(strBuf.toString());
            }
        }
    }

    /**
     * Handles the "leave" notification provided by the Realtime-Framework. This method is called when a client leaves the document group.
     * Attention: If getSignOffMessage is marked with the annotation "Asynchronous" onLeave is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id The ID of the last client that wants to leave the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id) {
        super.onLeave(id);

        checkForDisposed();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onLeave called from different thread " + String.valueOf(threadId));
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onLeave called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Client " + id.toString() + " leaving => Flushing operations");
        }

        // Update statistics with CLOSE or TIMEOUT type
        if ((null != m_fileHelper) && (null != id)) {
            final String uid = id.toString();
            boolean timeout = false;
            synchronized(m_connectionStatus) {
                timeout = (m_connectionStatus.getDurationOfInactivity(uid) >= ConnectionStatus.INACTIVITY_TRESHOLD_TIME);
            }

            if (timeout) {
                Statistics.handleDocumentEvent(new CloseEvent(m_fileHelper.getDocumentType(), CloseType.TIMEOUT));
            } else {
                Statistics.handleDocumentEvent(new CloseEvent(m_fileHelper.getDocumentType(), CloseType.CLOSE));
            }
        }

        // handles the leaving of a client from the document connection
        impl_handleClientLeaves(id);
    }

    /**
     * Handles the "leave" notification provided by the Realtime-Framework. This method is called when a client leaves the document group.
     * Attention: If getSignOffMessage is marked with the annotation "Asynchronous" onLeave is also called from the same background thread.
     * Therefore this method must be thread-safe, too. ATTENTION: The onLeave(ID id) method is also called by the RT-framework therefore one
     * has to make sure that nothing is done twice.
     *
     * @param id The ID of the last client that wants to leave the document (chat-room).
     * @param stanza Additional data sent together with the leave request from the client.
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id, Stanza stanza) {
        super.onLeave(id, stanza);

        checkForDisposed();

        final Session session = MessageHelper.getServerSession(null, id);
        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "recentfile"), true);

        if ((null != session) && (null != jsonRequest)) {
            JSONObject recentFile = jsonRequest.optJSONObject("file");
            JSONObject performanceData = jsonRequest.optJSONObject("performanceData");
            ApplicationType app = ApplicationType.stringToEnum(jsonRequest.optString("app", null));

            if ((null != recentFile) && (ApplicationType.APP_NONE != app)) {
                DocumentMetaData currentMetaData = null;
                synchronized(m_syncAccess) {
                    currentMetaData = m_lastKnownMetaData;
                }
                recentFile = FileDescriptor.createJSONObject(currentMetaData, new java.util.Date());

                RecentFileListManager recentFileListManager = new RecentFileListManager(m_services, session);
                ArrayList<JSONObject> recentFileList = ConnectionHelper.getRecentFileList(recentFileListManager, app);
                if (null != recentFileList) {
                    recentFileListManager.addCurrentFile(recentFileList, recentFile, this.getLastModifiedTime());
                    recentFileListManager.writeRecentFileList(app, recentFileList);
                    recentFileListManager.flush();
                }
            }

            if (null != performanceData) {
                // if performance data is not null, this means, that performance logging is enabled -> only checking log path
                String loggingPath = ConfigurationHelper.getStringOfficeConfigurationValue(
                    m_services,
                    session,
                    "//module/performanceDataLogPath",
                    null);
                if (null != loggingPath) {
                    OperationHelper.logPerformanceData(performanceData, recentFile, loggingPath);
                }
            }
        }
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework. This method is called when the last client left the document
     * and the Realtime-Framework wants to dispose the GroupDispatcher instance. Attention: If getSignOffMessage or onLeave is marked with
     * the annotation "Asynchronous" onDispose is also called from the same background thread. Therefore this method must be thread-safe,
     * too.
     *
     * @param id The ID of the last client that left the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        super.onDispose(id);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onDispose called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Last client leaving => Flushing operations");
        }

        try {
            // Use a try/catch throwable block to prevent throwing low-level
            // exceptions. Make the most important calls first!
            this.m_loadRequestQueue.purgeDocumentRequests(m_fileHelper.getResourceId());

            long threadId = Thread.currentThread().getId();
            if ((threadId != m_threadId) && LOG.isDebugEnabled()) {
                LOG.debug("RT connection: onDispose called from different thread " + String.valueOf(threadId));
            }

            // The connection object will be disposed. therefore we don't have
            // any client which can be informed. We have to ignore the error
            // for this flushDocument call. The next client will get this
            // info loading the document.
            if (m_saveOnDispose.get()) {
                flushDocument(MessageHelper.getServerSession(null, id), null, true, true);
            }

            synchronized (m_connectionStatus) {
                cancelPrepareLosingEditRightsTimer();
                m_connectionStatus.setWantsEditRightsUser("", "");
            }
        } catch (Throwable e) {
            LOG.error("RT connection: Exception while cleanup connection instance", e);
        } finally {
            // Special handling destroying the calcengine document. In
            // case of an error this can throw low-level exceptions.
            if (m_calcClient != null) {
                ICalcEngineClient calcClient = m_calcClient;
                String calcDocHandle = m_calcDocumentHandle;
                m_calcClient = null;
                m_calcDocumentHandle = null;

                try {
                    if (calcDocHandle != null)
                        calcClient.destroyDocument(calcDocHandle);
                } catch (Throwable e) {
                    LOG.error("RT connection: Error deleting spreadsheet document", e);
                }
                try {
                    calcClient.dispose();
                } catch (Throwable e) {
                    LOG.error("RT connection: Error disposing spreadsheet client", e);
                }
            }

            // set instance to disposed
            m_disposed.set(true);
            m_resourceManager.lockResources(false);

            synchronized(m_syncAccess) {
                // Reset members which can reference big memory elements as RT-core
                // references our Connection object even after calling onDispose().
                m_uiOperations = null;
                m_messageChunkList = null;
                m_resourceManager = null;
            }
        }
    }

    /**
     * Determines which user ID should be used for the next
     * flushDocument call.
     *
     * @param context
     *  The ID of the client that request flushing the document.
     *
     * @param leaving
     *  Specifies, if the creator is needed for a leaving request or
     *  not.
     *
     * @return
     *  The ID of the creator for the next flushDocument or null, if the
     *  current
     */
    private ID impl_getCreatorForNextFlush(ID userId, boolean leaving) {
        ID creator = null;
        String lastModifyingUserId = "";
        String currentEditUserId = "";

        synchronized (m_connectionStatus) {
            lastModifyingUserId = m_connectionStatus.getLastModifyingUserId();
            currentEditUserId = m_connectionStatus.getCurrentEditingUserId();
        }

        if (leaving) {
            boolean editorLeaves = false;
            // Handle if the leaving user has edit rights
            if (StringUtils.isNotEmpty(currentEditUserId) && (currentEditUserId.equals(userId.toString()))) {
                // set state that editor is leaving
                editorLeaves = true;
            }

            if (!editorLeaves && StringUtils.isNotEmpty(lastModifyingUserId)) {
                // We want to use the session of the user who made the latest changes
                // to create temporary versions. Otherwise this can confuse
                // users if they look at the version history.
                creator = new ID(lastModifyingUserId);
            } else if (editorLeaves) {
                if (userId.toString().equals(lastModifyingUserId)) {
                    // Check if editor leaves that she also made the latest changes to the document.
                    // Otherwise we would change the creator to a virtual editor that just acquired
                    // edit rights without any modification.
                    creator = userId;
                }
            }
        } else {
            // there no client leaving the document connection -> use the last
            // modifying user as creator.
            if (StringUtils.isNotEmpty(lastModifyingUserId)) {
                creator = new ID(lastModifyingUserId);
            }
        }

        return creator;
    }

    /**
     * Handles the leaving of a client from the document connection.
     *
     * @param id
     *   The user id of the client leaving.
     */
    private void impl_handleClientLeaves(ID id) {
        int curClientCount = 0;
        boolean editorLeaves = false;
        ConnectionStatus connectionStatus = null;
        String lastModifyingUserId = "";
        String wantsEditRightsUser = "";
        String currentEditUser = "";

        synchronized (m_connectionStatus) {

            lastModifyingUserId = m_connectionStatus.getLastModifyingUserId();
            curClientCount = m_connectionStatus.getActiveClients();
            if (curClientCount > 0) {
                try {
                    boolean removeUserStatus = m_connectionStatus.removeActiveUser(id.toString());
                    if (LOG.isDebugEnabled() && removeUserStatus) {
                        LOG.debug("RT connection: onLeave removed this active user:" + id.toString());
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: onLeave catches JSON exception while removing active user", e);
                }

                curClientCount = m_connectionStatus.getActiveClients();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                    debugPrintActiveClients(m_connectionStatus);
                }
            }

            // Check if the leaving user wants the edit rights. If yes we have
            // to remove the user to enable others to acquire them.
            wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            currentEditUser = m_connectionStatus.getCurrentEditingUserId();
            if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                ID wantsEditRightsUserId = new ID(wantsEditRightsUser);
                if (wantsEditRightsUserId.equals(id)) {
                    // Reset wantsEditRightsUser from connectionStatus
                    m_connectionStatus.setWantsEditRightsUser("", "");
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Wants to edit client: " + wantsEditRightsUser + " leaving, reset wantsEditRightsUser");
                    }
                }
            }

            // Handle if the leaving user has edit rights
            if ((null != currentEditUser) && (currentEditUser.length() > 0)) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(id)) {
                    // Reset current editing user from connectionStatus
                    m_connectionStatus.setCurrentEditingUser("", "");
                    editorLeaves = true;

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Editor " + currentEditUser + " leaving, reset editor");
                    }

                    if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Editor " + currentEditUser + " left, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                        }
                        // Set wantsEditRightsUser as new editor if known
                        m_connectionStatus.setCurrentEditingUser(wantsEditRightsUser, m_connectionStatus.getWantsEditRightsUserName());
                        // Reset wantsEditRightsUser so others can acquire the edit rights
                        m_connectionStatus.setWantsEditRightsUser("", "");
                        // cancel timer as we already set the new editor
                        this.cancelPrepareLosingEditRightsTimer();
                    }
                }
            }
            connectionStatus = (ConnectionStatus) m_connectionStatus.clone();
        }

        if (curClientCount > 0) {
            updateClientsExceptSender(new MessageData(null, connectionStatus), id);
        }

        if (editorLeaves && (id.toString().equals(lastModifyingUserId))) {
            // Check if storage supports versions - if true we want to have a version for
            // every editor leaving the document. Otherwise we want to preserve the version
            // that existed when the first user opened the document.
            if (m_storageHelper.supportsFileVersions()) {
                // if editor leaves the next save must create a new version
                setCreateVersion(true);
            }
        }
    }

    /**
     * Implements the request processing for the asynchronous loading of documents. Currently there are two different requests: 1.
     * LoadRequest - initiates the loading of the document into the CalcEngine with chunks to circumvent problems with the string length. 2.
     * SyncRequest - sync any other clients that requests the same doc as a previous client to guarantee that the CalcEngine is ready to
     * process further requests.
     *
     * @param request The current request which must be processed by the Connection object.
     */
    @Override
    public void processRequest(Request request) throws OXException {

        switch (request.getType()) {
        case REQUEST_LOAD_REMAINING:
        case REQUEST_LOAD:
            Message loadMessage = null;
            ErrorCode errorCode = ErrorCode.NO_ERROR;

            try {

                synchronized (m_syncAccess) {
                    if (m_uiOperations == null) {
                        m_uiOperations = new JSONObject();
                    }
                }

                LOG.trace("RT connection: loading operations in chunks for document: " + request.getDocumentId());

                ID fromId = request.getId();
                JSONArray uiOperations = new JSONArray();

                JSONArray opArray = null;
                if (request.getOperations() != null) {
                    // for a normal load request we can pull the operations from the
                    // operations JSONObject.
                    opArray = request.getOperations().getJSONArray("operations");
                } else {
                    // in case of a remaining load request, we have to trigger the
                    // import of the remaining sheets and retrieve the operations from
                    // the response.
                    OXDocument docLoader = request.getDocumentLoader();
                    final JSONObject remainingOps = docLoader.getRemainingOperations(null);
                    if (remainingOps != null) {
                        opArray = remainingOps.getJSONArray("operations");
                    }
                }

                if (opArray.length() > MAX_OPERATIONS) {
                    int opCount = opArray.length();
                    int copyPosition = 0;

                    while (copyPosition < opCount) {
                        final JSONObject opPart = new JSONObject();
                        final JSONArray partArray = new JSONArray();
                        for (int op = 0; op < MAX_OPERATIONS && copyPosition < opCount; ++op) {
                            partArray.put(op, opArray.get(copyPosition));
                            ++copyPosition;
                        }

                        opPart.put("operations", partArray);
                        LOG.trace("RT connection: sending operations chunk to CalcEngine for document: " + request.getDocumentId());
                        boolean docDisposed = isDisposed();

                        LOG.trace("RT connection: check document is disposed (before executeSpreadsheetOperationFailsafe) disposed=" + docDisposed);

                        if (!docDisposed) {
                            final JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));
                            docDisposed = m_disposed.get();

                            if (!docDisposed) {
                                if (resultObject.has("uiOperations")) {
                                    try {
                                        JSONArray newUIOps = resultObject.getJSONArray("uiOperations");
                                        for (int i = 0; i < newUIOps.length(); i++) {
                                            uiOperations.put(newUIOps.get(i));
                                        }
                                    } catch (JSONException e) {
                                        LOG.error("RT connection: Error adding ui operations to JSON object", e);
                                    }
                                }
                            }
                        }

                        LOG.trace("RT connection: check document is disposed (after executeSpreadsheetOperationFailsafe) disposed=" + docDisposed);

                        if (docDisposed) {
                            // remove possible pending requests from clients already left the connection
                            this.m_loadRequestQueue.purgeDocumentRequests(request.getDocumentId());
                            break;
                        } else {
                            Thread.sleep(1000);
                        }
                    }

                    synchronized (m_syncAccess) {
                        if (!this.m_disposed.get()) {
                            LOG.trace("RT connection: put ui operations into m_uiOperations");
                            final JSONArray operations = m_uiOperations.optJSONArray("operations");
                            if (null != operations) {
                                JSONHelper.appendArray(operations, uiOperations);
                            } else {
                                m_uiOperations.put("operations", uiOperations);
                            }
                        }
                    }
                } else if (opArray.length() > 0) {
                    final JSONObject opPart = new JSONObject();
                    opPart.put("operations", opArray);

                    final JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));
                    if (resultObject.has("uiOperations")) {
                        try {
                            synchronized (m_syncAccess) {
                                if (!this.m_disposed.get()) {
                                    LOG.trace("RT connection: put complete ui operations into m_uiOperations");
                                    uiOperations = resultObject.getJSONArray("uiOperations");
                                    final JSONArray operations = m_uiOperations.optJSONArray("operations");
                                    if (null != operations) {
                                        JSONHelper.appendArray(operations, uiOperations);
                                    } else {
                                        m_uiOperations.put("operations", uiOperations);
                                    }
                                }
                            }
                        } catch (JSONException e) {
                            LOG.error("RT connection: Error adding ui operations to JSON object", e);
                        }
                    }
                    LOG.debug(resultObject.toString());
                }

                if (!m_disposed.get()) {
                    final JSONObject finishedResult = executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"loadFinished\"}]}");

                    if (finishedResult.has("uiOperations")) {
                        synchronized(m_syncAccess) {
                            if (!this.m_disposed.get()) {
                                JSONArray currUIOp = m_uiOperations.getJSONArray("operations");
                                JSONHelper.appendArray(currUIOp, finishedResult.getJSONArray("uiOperations"));
                                m_uiOperations.put("operations", currUIOp);
                            }
                        }
                    }
                    LOG.trace("RT connection: loading operations in chunks finished for document: " + request.getDocumentId());
                }

                loadMessage = impl_createWelcomeMessage(
                    request.getId(),
                    null,
                    request.getDocumentOSN(),
                    true,
                    null,
                    request.getPreviewUIOpsCount(),
                    request.getDocumentMetaData());
            } catch (NoClassDefFoundError e) {
                LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
                errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
            } catch (UnsatisfiedLinkError e) {
                LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
                errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
            } catch (CalcEngineException e) {
                // do nothing in case if shutdown
                if (isShutdown()) {
                    LOG.debug("RT connection: Shutdown detected during asynchronous spreadsheet operation processing!", e);
                    return;
                }
                LOG.error("RT connection: Error executing spreadsheet operations, calc engine notifies problems!", e);
                errorCode = ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR;
            } catch (InterruptedException e) {
                LOG.error("RT connection: Interrupted while executing spreadsheet operations - may be we are shutting down", e);
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                return;
            } catch (Exception e) {
                LOG.error("RT connection: Error executing spreadsheet operations", e);
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
            } catch (Throwable e) {
                LOG.error("RT connection: Error executing spreadsheet operations", e);
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
            } finally {
                // Set request to finish to enable new requests to be asynchronously
                // processed.
                if (!m_disposed.get()) {
                    this.setAsyncSpreadsheetLoaded(true);

                    if (errorCode.getCode() == ErrorCode.NO_ERROR.getCode()) {
                        // in case of no error we just sent the finalLoad update
                        try {
                            boolean isMember = false;
                            synchronized (m_connectionStatus) {
                                isMember = m_connectionStatus.hasActiveUser(request.getId().toString());
                            }
                            if (isMember) {
                                this.relayToID(loadMessage, request.getId());
                            }
                        } catch (Throwable t) {
                            LOG.debug("RT connection: Not able to send message to the document clients, may be client already left", t);
                        }
                    } else {
                        // in case of an error we have to sent a finalLoad update which
                        // has 'hasErrors' set and 'errorCode' includes more information
                        // about the root cause.
                        ConnectionStatus errorConnectionStatus = null;
                        synchronized (m_connectionStatus) {
                            errorConnectionStatus = (ConnectionStatus) m_connectionStatus.clone();
                            errorConnectionStatus.setFinalLoad(true);
                        }

                        try {
                            final MessageData messageData = new MessageData(null, errorConnectionStatus, MessageHelper.finalizeJSONResult(
                                errorCode,
                                null,
                                null));
                            this.updateClients(messageData, request.getId());
                        } catch (Throwable t) {
                            LOG.debug("RT connection: Not able to send message to the document clients, maybe client already left", t);
                        } finally {
                            // Purge the remaining sync requests from the queue
                            // as we already sent a 'finalLoad' update which
                            // includes an error code describing the root cause.
                            String documentId = request.getDocumentId();
                            this.m_loadRequestQueue.purgeDocumentRequests(documentId);
                        }
                    }
                }
                LOG.trace("RT connection: request finished for document: " + request.getDocumentId());
                request.finished();
            }
            break;

        case REQUEST_LOAD_SYNC:
            // Sync another client with asynchronous loading process
            // This code should only be reached in case of a successful loading
            request.finished();
            if (!m_disposed.get()) {
                LOG.trace("RT connection: additional client can now work on document: " + request.getDocumentId());
                Message welcomeMessage = impl_createWelcomeMessage(request.getId(), null, request.getDocumentOSN(), true, null, -1, request.getDocumentMetaData());
                try {
                    boolean isMember = false;
                    synchronized (m_connectionStatus) {
                        isMember = m_connectionStatus.hasActiveUser(request.getId().toString());
                    }
                    if (isMember) {
                        this.relayToID(welcomeMessage, request.getId());
                    }
                } catch (Throwable t) {
                    LOG.debug("RT connection: Not able to send message to the document clients", t);
                }
            }
            break;
        }
    }

    /**
     * Provides a stanza for the first message sent to the client joining a document (chat-room). The OX Documents client expects to find
     * the document operation within the stanza provided by this method.
     *
     * @param onBehalfOf The ID of the client which wants to join the document.
     * @return The stanza sent to the client which wants to join the document. The stanza includes the document operations and operations
     *         which are currently pending for the next save. The stanza also has the update state including the editor ID.
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(final ID onBehalfOf) {
        final ID fromId = getId();
        Message welcomeMessage = null;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            final ServerSession serverSession = MessageHelper.getServerSession(null, onBehalfOf);

            LOG.debug("RT connection: [getWelcomeMessge] called for document: " + getDocumentFolderAndFileId());

            // Check for the document type and optimize access in case the document has been loaded successfully
            if (m_fileHelper != null && (m_calcDocumentHandle != null) && (m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.XLSX || m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.ODS)) {
                DocumentMetaData currentMetaData = null;
                synchronized(m_syncAccess) {
                    currentMetaData = m_lastKnownMetaData;
                }

                // Optimizing the spreadsheet access for the second and other upcoming clients
                // No need to access the document and load it more than once.
                if (isAsyncSpreadsheetLoaded()) {
                    // In this case we already have finished loading and can therefore handle
                    // this by just sending operations synchronously
                    return impl_createWelcomeMessage(onBehalfOf, null, -1, false, null, -1, currentMetaData);
                } else {
                    // Loading of the document into the CalcEngine is in process, therefore
                    // we have to sync us with the first client that initiated the loading.
                    m_loadRequestQueue.putRequest(new Request(m_fileHelper.getFolderId(), m_fileHelper.getFileId(), onBehalfOf, this, -1, currentMetaData));
                    return impl_createAsyncLoadWelcomeMessage(fromId, onBehalfOf, null, 0);
                }
            }

            final OXDocument oxDocument = new OXDocument(serverSession, m_services, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper);
            final int documentOSN = oxDocument.getUniqueDocumentId();
            final JSONObject operations = new JSONObject();

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            ErrorCode errorCode = oxDocument.getLastError();

            // check error code after loading the document
            if (errorCode.getCode() != ErrorCode.NO_ERROR.getCode()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, m_fileHelper.getFolderId(), m_fileHelper.getFileId());
            }

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            String genericHtmlDocument = null;
            DocumentType docType = m_fileHelper.getDocumentType();
            JSONObject importerProps = new JSONObject();
            JSONObject previewData = null;

            // Set current meta data for the document
            synchronized(m_syncAccess) {
                m_lastKnownMetaData = oxDocument.getDocumentMetaData();
            }

            SystemInfoHelper.MemoryInfo memInfo = null;
            long maxHeapSizeForFilter = 0;

            // check error code before continuing to retrieve operations
            if (errorCode.getCode() == ErrorCode.NO_ERROR.getCode()) {
                // Initialize result object with operations from persistent document first.
                // Check for FilterExceptions that can occur due to problems with the file.
                try {
                    final IImporter importer = getImporter();

                    memInfo = SystemInfoHelper.getMemoryInfo();
                    if (null != memInfo) {
                        // The maximal heap size for the filter to load the document should not exceed
                        // half of the maximal heap size of the JVM.
                        maxHeapSizeForFilter = memInfo.maxHeapSize / 2;
                        importerProps.put("maxMemoryUsage", maxHeapSizeForFilter);
                        double percentage = ((double)(memInfo.maxHeapSize - SystemInfoHelper.MINIMAL_FREE_BACKEND_HEAPSPACE)) / (double)memInfo.maxHeapSize;
                        MemoryObserver memObserver = MemoryObserver.getMemoryObserver();
                        MemoryObserver.setPercentageUsageThreshold(percentage);
                        importerProps.put(DocumentProperties.PROP_MEMORYOBSERVER, memObserver);
                        importerProps.put(DocumentProperties.PROP_MAX_MEMORY_USAGE, maxHeapSizeForFilter);
                    }

                    if (OXDocument.supportsPreview(importer)) {
                        // If preview is supported use it to have a better user experience
                        oxDocument.preparePreviewOperations(importer, importerProps);
                        previewData = oxDocument.getPreviewOperations();
                    } else {
                        // load the whole document as a fall back for missing preview capabilities
                        final JSONObject documentOperations = oxDocument.getOperations(importer, importerProps, null);
                        if (docType == DocumentType.TEXT) {
                            UserConfigurationHelper userConfHelper = new UserConfigurationHelper(
                                m_services,
                                serverSession,
                                "io.ox/office",
                                Mode.WRITE_BACK);
                            final Boolean fastLoad = userConfHelper.getBoolean("text/documentFastLoad", true);
                            if (fastLoad != null && fastLoad) {
                                final TextTableLimits textTableLimits = ConfHelper.getTextTableLimits(userConfHelper);
                                //                                OperationReducer.reduceOperationCount(documentOperations);
                                genericHtmlDocument = GenericHtmlDocumentBuilder.buildHtmlDocument(
                                    documentOperations,
                                    oxDocument.getFileId(),
                                    oxDocument.getFolderId(),
                                    oxDocument.getVersion(),
                                    textTableLimits);
                            }
                        }
                        OperationHelper.appendJSON(operations, documentOperations);
                    }
                } catch (final FilterException e) {
                    errorCode = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;
                    LOG.debug("RT connection: getWelcomeMessage, Exception catched", e);

                    if (e.getErrorcode() == FilterException.ErrorCode.WRONG_PASSWORD || e.getErrorcode() == FilterException.ErrorCode.UNSUPPORTED_ENCRYPTION_USED) {
                        errorCode = ErrorCode.LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR;
                    } else if (e.getErrorcode() == FilterException.ErrorCode.COMPLEXITY_TOO_HIGH) {
                        errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR;
                    } else if ((e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_TOO_HIGH) || (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED)) {
                        // try to figure what happened with the memory usage of the filter as
                        // we want to provide two different error codes.
                        errorCode = ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR;
                        if (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED) {
                            memInfo = SystemInfoHelper.getMemoryInfo();
                            final String freeMemString = (memInfo == null) ? "unknown" : ((Long) ((memInfo.maxHeapSize - memInfo.usedHeapSize) / SystemInfoHelper.MBYTES)).toString();
                            LOG.info("Document could not be loaded, because current available heap space is not sufficient. Current free heap space: " + freeMemString + " MB");
                        }                    }
                } catch (final Exception e) {
                    errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                    LOG.debug("RT connection: getWelcomeMessage, Exception catched", e);
                }
            }

            // check error code again as the filter and other parts can throw errors, too
            if (errorCode.getCode() != ErrorCode.NO_ERROR.getCode()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, m_fileHelper.getFolderId(), m_fileHelper.getFileId());
            }

            if (m_fileHelper != null && (m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.XLSX || m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.ODS)) {
                final String folderId = oxDocument.getFolderId();
                final String fileId = oxDocument.getFileId();

                if (m_calcDocumentHandle == null && (operations.has("operations") || (null != previewData))) {
                    final CalcEngineClientFactory calcFactory = m_services.getService(CalcEngineClientFactory.class);

                    try {
                        m_calcClient = calcFactory.get();
                        m_calcDocumentHandle = m_calcClient.createDocument();
                        LOG.debug("context[file:" + getDocumentFolderAndFileId() + ",doc-handle:" + m_calcDocumentHandle + "] calc document created");

                        // return a asynchronous loading welcome message
                        boolean singleSheet = false;
                        JSONArray uiOperations = null;
                        JSONArray previewOperations = null;
                        JSONArray opArray = null;

                        if (operations.has("operations")) {
                            opArray = operations.getJSONArray("operations");
                        } else if (null != previewData) {
                            previewOperations = previewData.optJSONArray("operations");
                            singleSheet = (oxDocument.getSheetCount() == 1);
                        }

                        if (null != previewOperations) {
                            errorCode = impl_processPreviewRequest(fromId, onBehalfOf, previewOperations, singleSheet);
                            if (errorCode != ErrorCode.NO_ERROR) {
                                return impl_createErrorMessage4WelcomeMessage(onBehalfOf, errorCode, syncGetConnectionStatus());
                            }

                            synchronized (m_syncAccess) {
                                // Create a copy of the ui operations array, so we can be sure that
                                // we don't see changes by the parallel running thread loading the
                                // remaining sheets.
                                uiOperations = JSONHelper.shallowCopy(m_uiOperations.getJSONArray("operations"));
                            }
                        }

                        // create request and process load request in the calc load request queue asynchronously
                        if ((null != opArray) && (opArray.length() > 0)) {
                            // In case we don't have preview operations we have to process
                            // the whole operations asynchronously. The welcome message just
                            // provide the info that the loading is done asynchronously.
                            final Request request = new Request(folderId, fileId, onBehalfOf, opArray, this, documentOSN, oxDocument.getDocumentMetaData());
                            m_loadRequestQueue.putRequest(request);
                            welcomeMessage = impl_createAsyncLoadWelcomeMessage(getId(), onBehalfOf, uiOperations, -1);
                        } else if (null != uiOperations) {
                            if (singleSheet) {
                                // In case we just have one sheet and loaded the document synchronously
                                // we are done and return the final ui operations.
                                welcomeMessage = impl_createWelcomeMessage(onBehalfOf, null, -1, false, null, -1, oxDocument.getDocumentMetaData());
                            } else {
                                // In case we have preview operations we provide the OXDocument instance
                                // to complete the loading of the remaining sheets in a background thread.
                                final int activeSheetIndex = oxDocument.getActiveSheetIndex();
                                final Request request = new Request(
                                    folderId,
                                    fileId,
                                    onBehalfOf,
                                    oxDocument,
                                    this,
                                    documentOSN,
                                    uiOperations.length());
                                m_loadRequestQueue.putRequest(request);
                                welcomeMessage = impl_createAsyncLoadWelcomeMessage(getId(), onBehalfOf, uiOperations, activeSheetIndex);
                            }
                        } else {
                            LOG.error("RT connection: Impossible case trying to process spreadsheet request");
                            return impl_createErrorMessage4WelcomeMessage(
                                onBehalfOf,
                                ErrorCode.LOADDOCUMENT_FAILED_ERROR,
                                syncGetConnectionStatus());
                        }
                    } catch (NoClassDefFoundError e) {
                        LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
                        return impl_createErrorMessage4WelcomeMessage(
                            onBehalfOf,
                            ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR,
                            syncGetConnectionStatus());
                    } catch (UnsatisfiedLinkError e) {
                        LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
                        return impl_createErrorMessage4WelcomeMessage(
                            onBehalfOf,
                            ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR,
                            syncGetConnectionStatus());
                    } catch (Exception e) {
                        LOG.error("RT connection: Error executing spreadsheet operations", e);
                        return impl_createErrorMessage4WelcomeMessage(
                            onBehalfOf,
                            ErrorCode.LOADDOCUMENT_FAILED_ERROR,
                            syncGetConnectionStatus());
                    }
                }
            } else {
                welcomeMessage = impl_createWelcomeMessage(onBehalfOf, operations, documentOSN, false, genericHtmlDocument, -1, oxDocument.getDocumentMetaData());
            }
        } else {
            // update statistics with load error
            if (null != m_fileHelper) {
                DocumentEvent documentEvent = new DocumentEvent(m_fileHelper.getDocumentType(), ErrorType.LOAD);
                // update statistics
                Statistics.handleDocumentEvent(documentEvent);
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
            }
        }

        return welcomeMessage;
    }

    /**
     * Process the preview request and fill the global ui operations object.
     *
     * @param fromId The id of the client which requested the preview.
     * @param onBehalfOf The id of the client in which context this preview should be processed.
     * @param previewOperations The operations for the preview.
     * @param singleSheet Specifies if this is a single sheet document or not.
     */
    private ErrorCode impl_processPreviewRequest(final ID fromId, final ID onBehalfOf, final JSONArray previewOperations, boolean singleSheet) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        try {
            final String documentId = Request.buildDocumentId(m_fileHelper.getFolderId(), m_fileHelper.getFileId());

            synchronized (m_syncAccess) {
                m_uiOperations = new JSONObject();
            }

            LOG.debug("RT connection: loading preview operations in chunks for document: " + documentId);

            final JSONArray uiOperations = new JSONArray();
            if (previewOperations.length() > MAX_OPERATIONS) {
                int opCount = previewOperations.length();
                int copyPosition = 0;

                while (copyPosition < opCount) {
                    JSONObject opPart = new JSONObject();
                    JSONArray partArray = new JSONArray();
                    for (int op = 0; op < MAX_OPERATIONS && copyPosition < opCount; ++op) {
                        partArray.put(op, previewOperations.get(copyPosition));
                        ++copyPosition;
                    }

                    opPart.put("operations", partArray);
                    LOG.debug("RT connection: sending preview operations chunk to CalcEngine for document: " + documentId);

                    JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));

                    if (resultObject.has("uiOperations")) {
                        JSONHelper.appendArray(uiOperations, resultObject.getJSONArray("uiOperations"));
                    }
                }

                synchronized (m_syncAccess) {
                    m_uiOperations.put("operations", uiOperations);
                }
            } else {
                final JSONObject operations = new JSONObject();
                operations.put("operations", previewOperations);

                JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, operations.toString(false));
                if (resultObject.has("uiOperations")) {
                    synchronized (m_syncAccess) {
                        m_uiOperations.put("operations", resultObject.getJSONArray("uiOperations"));
                    }
                }
                LOG.debug(resultObject.toString());
            }

            if (singleSheet) {
                // single sheet needs a load finished - no more operations will follow
                JSONObject finishedResult = executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"loadFinished\"}]}");
                if (finishedResult.has("uiOperations")) {
                    JSONArray currUIOp = m_uiOperations.getJSONArray("operations");
                    JSONHelper.appendArray(currUIOp, finishedResult.getJSONArray("uiOperations"));
                    m_uiOperations.put("operations", currUIOp);
                }
            } else {
                executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"sheetLoaded\"}]}");
            }
            LOG.debug("RT connection: loading preview operations in chunks finished for document: " + documentId);
        } catch (NoClassDefFoundError e) {
            LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
            errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
        } catch (UnsatisfiedLinkError e) {
            LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
            errorCode = ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR;
        } catch (CalcEngineException e) {
            // do nothing in case if shutdown
            LOG.error("RT connection: Error executing spreadsheet operations, calc engine notifies problems!", e);
            errorCode = ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR;
        } catch (JSONException e) {
            LOG.error("RT connection: Error apply JSON operation", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        } catch (Exception e) {
            LOG.error("RT connection: Error executing spreadsheet operations", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        } catch (Throwable e) {
            LOG.error("RT connection: Error executing spreadsheet operations", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        }

        return errorCode;
    }

    /**
     * Creates a welcome message for the client where it's notified that an asynchronous load process is used on the backend side. That
     * means that the client must wait until the final load update is sent.
     *
     * @param fromId The id of this real-time connection instance.
     * @param onBehalfOf The id of the client that requested to join the connection instance.
     * @param previewUIOperations Optional operations for the preview mode.
     * @param previewActiveSheetIndex Optional preview active sheet index.
     * @return A welcome message prepared for asynchronous loading by the client.
     */
    private Message impl_createAsyncLoadWelcomeMessage(ID fromId, ID onBehalfOf, final JSONArray previewUIOperations, final int previewActiveSheetIndex) {
        Message welcomeMessage = null;
        // Provide a welcome message where the syncLoad attribute is set to false
        // which notifies the client that it has to wait for a specific update
        // message which has 'finalLoad' attribute set to true.
        final ConnectionStatus connectionStatus = syncGetConnectionStatus();
        final JSONObject actionData = new JSONObject();
        connectionStatus.setSyncLoad(false);

        if (LOG.isDebugEnabled()) {
            final String documentId = Request.buildDocumentId(m_fileHelper.getFolderId(), m_fileHelper.getFileId());
            LOG.debug("RT connection: Completed synchronous loading for document " + documentId + ", asynchronous loading started.");
        }

        // set preview operations
        JSONObject previewJSONData = null;
        try {
            if (null != previewUIOperations) {
                final JSONObject previewData = new JSONObject();
                previewJSONData = new JSONObject();
                previewData.put("operations", previewUIOperations);
                previewData.put("activeSheet", previewActiveSheetIndex);
                previewJSONData.put("preview", previewData);
            }
        } catch (JSONException e) {
            LOG.warn("Exception while creating preview object for asynchronous loading", e);
        }

        final MessageData messageData = new MessageData(
            new MessageChunk(actionData, getId()),
            connectionStatus,
            MessageHelper.finalizeJSONResult(ErrorCode.NO_ERROR, null, previewJSONData));

        // set empty actions as payload for the first welcome message
        welcomeMessage = new Message();
        welcomeMessage.setFrom(fromId);
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

        LOG.debug("RT connection: Sending preview [welcome message] to: " + onBehalfOf.toString());

        return welcomeMessage;
    }

    /**
     * Finalize load of the document for both use cases synchronous/asynchronous.
     *
     * @param onBehalfOf The ID of the client which initiated the join to the document group.
     * @param operations Operations of the document, must be null if the client receives special UI operations (in case of Spreadsheet).
     * @param documentOSN The document OSN retrieved directly from the document stream.
     * @param asyncLoad Specifies if the document was loaded by the asynchronous load request queue or not.
     * @param genericHtmlDocument Specifies an optional string containing the document as a generic HTML DOM.
     * @return The final welcome message to be sent to the client (first&last for synchronous loading and the second&last one for
     *         asynchronous loading).
     */
    private Message impl_createWelcomeMessage(ID onBehalfOf, JSONObject operations, int documentOSN, boolean asyncLoad, String genericHtmlDocument, int ignoreOpsCount, final DocumentMetaData docMetaData) {
        DocumentEvent documentEvent = null;
        Message welcomeMessage = null;
        int messageChunkListSize = 0;
        JSONObject lastOperation = null;
        JSONObject firstOperation = null;
        boolean opsFromDocument = true;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            // Determine the operation state number and append additional operations
            // applied to the document since the last save action.
            synchronized (m_syncAccess) {
                messageChunkListSize = (m_messageChunkList != null) ? m_messageChunkList.size() : 0;

                JSONObject opsObject = (m_uiOperations != null) ? m_uiOperations : operations;
                // set sync object with the current document OSN
                m_syncStatus.updateDocmentOSN(documentOSN);

                if (messageChunkListSize == 0) {
                    // If we don't have additional operations we have to send
                    // the current operation state number for the document. Therefore
                    // calculate or provide the current operation state number.
                    // Otherwise we don't provide the number from here but let
                    // the additional operations send it.
                    try {
                        int operationCount = opsObject.getJSONArray("operations").length();
                        lastOperation = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                        if (!lastOperation.has("osn")) {

                            if (m_connectionStatus.getOperationStateNumber() > 0) {
                                lastOperation.put("osn", m_connectionStatus.getOperationStateNumber() - 1);
                            } else {
                                // determine initial osn. we prefer the document OSN but if not available
                                // fallback to the operation count
                                int initialOSN = (documentOSN != -1) ? documentOSN : operationCount;
                                lastOperation.put("osn", initialOSN - 1);
                                // Set initial operation state number
                                m_connectionStatus.setOperationStateNumber(initialOSN);
                            }
                            lastOperation.put("opl", 1);
                        }

                    } catch (JSONException e) {
                        LOG.error("RT connection: Error setting operation state number to last operation", e);
                        return impl_createErrorMessage4WelcomeMessage(
                            onBehalfOf,
                            ErrorCode.LOADDOCUMENT_FAILED_ERROR,
                            syncGetConnectionStatus());
                    }
                } else if (messageChunkListSize > 0) {
                    opsFromDocument = false;
                    // Adding operations applied to the document since the last save action.
                    // These operations will provide the current operation state number, too.
                    OperationHelper.appendOperationChunks(operations, m_messageChunkList, false);

                    if (LOG.isDebugEnabled()) {
                        // Additional analyzing code to find root cause of missing operations
                        firstOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                        lastOperation = OperationHelper.debugGetLastOperationFromMessageChunkList(m_messageChunkList);
                    }
                }
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Connection.getWelcomeMessage, message chunk list size = " + Integer.toString(messageChunkListSize));
                if (null != firstOperation) {
                    LOG.debug("RT connection: Connection.getWelcomeMessage, first Operation in messsage chunk list: " + OperationHelper.operationToString(firstOperation));
                }
                if (null != lastOperation) {
                    if (opsFromDocument) {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in document: " + OperationHelper.operationToString(lastOperation));
                    } else {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in messsage chunk list: " + OperationHelper.operationToString(lastOperation));
                    }
                }
            }

            // check m_fileHelper which can be zero if ctor was called with
            // a invalid ID.
            if ((null != m_fileHelper) && (null != docMetaData)) {
                synchronized (m_connectionStatus) {
                    // set writeProtected status
                    m_connectionStatus.setWriteProtected(docMetaData.isWriteProtected());
                    // set lock state
                    m_connectionStatus.setLockState(docMetaData.getLockedUntil() != null);
                    // set locked by user
                    m_connectionStatus.setLockedByUser(docMetaData.getLockedByUser());
                }
            }

            // add additional client startup data
            final JSONObject jsonExtraData = new JSONObject();

            // add client id
            try {
                synchronized (m_syncAccess) {
                    jsonExtraData.put(MessagePropertyKey.KEY_CLIENT_ID, onBehalfOf.toString());
                    jsonExtraData.put(Sync.SYNC_INFO, m_syncStatus.toJSON());
                }
                if (null != genericHtmlDocument) {
                    jsonExtraData.put(MessagePropertyKey.KEY_HTMLDOCUMENT, genericHtmlDocument);
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Error adding client ID to JSON object.", e);
            }

            // putting all operations into an 'actions' object
            final JSONObject actionData = new JSONObject();
            try {
                synchronized (m_syncAccess) {
                    if (m_uiOperations != null) {
                        final JSONArray actions = new JSONArray();
                        if (ignoreOpsCount > 0) {
                            // If we have loaded with preview the preview operations
                            // should not be provided twice.
                            final JSONObject ops = new JSONObject();
                            JSONArray opsArray = m_uiOperations.getJSONArray("operations");
                            final int operationsCount = opsArray.length();

                            // cut out the old previewoperations to be not sent twice
                            opsArray = JSONHelper.subArray(opsArray, ignoreOpsCount);
                            if (operationsCount == ignoreOpsCount) {
                                // add a synchronization no-op into the empty operations list
                                opsArray.put(OperationHelper.createSyncNoOperation(m_connectionStatus.getOperationStateNumber()));
                            }
                            // put the adapted operations object into the actions array
                            ops.put("operations", opsArray);
                            actions.put(ops);
                        } else {
                            // Other clients need the whole UI operation list
                            actions.put(m_uiOperations);
                        }
                        actionData.put(MessagePropertyKey.KEY_ACTIONS, actions);
                    } else if (!operations.isEmpty()) {
                        actionData.put(MessagePropertyKey.KEY_ACTIONS, (new JSONArray()).put(operations));
                    } else {
                        jsonExtraData.put(MessagePropertyKey.KEY_HAS_ERRORS, true);

                        // update statistics with load error
                        if (null != m_fileHelper) {
                            documentEvent = new DocumentEvent(m_fileHelper.getDocumentType(), ErrorType.LOAD);
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Error adding operations to JSON", e);
                return impl_createErrorMessage4WelcomeMessage(onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR, syncGetConnectionStatus());
            }

            final ConnectionStatus connectionStatus = syncGetConnectionStatus();
            String answerString = "getactions";

            if (asyncLoad) {
                // message will be the final load update
                answerString = "update";
                connectionStatus.setFinalLoad(true);
            } else {
                // message will be the one and only welcome message including
                // document operations
                connectionStatus.setSyncLoad(true);
            }

            final MessageData messageData = new MessageData(
                new MessageChunk(actionData, getId()),
                connectionStatus,
                MessageHelper.finalizeJSONResult(ErrorCode.NO_ERROR, null, jsonExtraData));

            // set operations chunk of collected JSON objects as payload
            welcomeMessage = new Message();
            welcomeMessage.setFrom(onBehalfOf);
            welcomeMessage.setTo(onBehalfOf);
            welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", answerString)).build()));

            LOG.debug("RT connection: Sending [welcome message] to: " + onBehalfOf.toString());
        }

        // update statistics: if no (error) documentEvent has been
        // created by now, a new document has been successfully opened
        Statistics.handleDocumentEvent((null != documentEvent) ? documentEvent : getOpenEvent(docMetaData));

        return welcomeMessage;
    }

    /**
     * Handles an error code and updates the recent file according to the provided error code.
     *
     * @param onBehalfOf The ID of the client requesting the action.
     * @param serverSession The session of the client requesting the action.
     * @param errorCode The error code describing the problem of a recent action.
     * @param folderId The folder id of the document file.
     * @param fileId The file id of the document file.
     * @return The message to be sent to the client side. A recent list has been updated regarding the error code provided. If the error
     *         code specifies that the file is not found, the recent list will not list the file anymore.
     */
    private Message impl_handleError4Welcome(ID onBehalfOf, Session serverSession, ErrorCode errorCode, String folderId, String fileId) {
        // It's clear that we encountered on an error. Check for some
        // specific errors which need special handling.
        if (errorCode.getCode() == ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR.getCode()) {
            // file was not found => the recent file list must be updated
            RecentFileListManager recentFileListManager = new RecentFileListManager(m_services, serverSession);
            ConnectionHelper.removeFromRecentFile(recentFileListManager, m_appType, folderId, fileId);
        }

        final ConnectionStatus errConnectionStatus = syncGetConnectionStatus();
        errConnectionStatus.setSyncLoad(true);
        return impl_createErrorMessage4WelcomeMessage(onBehalfOf, errorCode, errConnectionStatus);
    }

    /**
     * Provides a message which can be used within the getWelcomeMessage() method to provide error states.
     *
     * @param onBehalfOf The ID of the client that triggered the welcomeMessage means that just have joined the Connection. Must be set
     *            otherwise the message won't be sent by the Realtime Framework.
     * @param errorCode The initialized ErrorCode to be sent as the result of the getWelcomeMessage() method. Note : This generates an error
     *            message usable as an answer for getWelcomeMessage() only. If you want to make it more generic you have to make sure the
     *            payload has to be more flexible extended ...
     */
    private Message impl_createErrorMessage4WelcomeMessage(ID onBehalfOf, ErrorCode errorCode, ConnectionStatus connectionStatus) {
        JSONObject jsonExtraData = new JSONObject();
        try {
            jsonExtraData.put(MessagePropertyKey.KEY_CLIENT_ID, onBehalfOf.toString());
        } catch (JSONException jsonEx) {
            LOG.error("RT connection: Error adding client ID to JSON object for error message", jsonEx);
        }
        final MessageData messageData = new MessageData(null, connectionStatus, MessageHelper.finalizeJSONResult(
            errorCode,
            null,
            jsonExtraData));

        // create error welcome message
        Message welcomeMessage = new Message();
        welcomeMessage.setFrom(getId());
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

        return welcomeMessage;
    }

    /**
     * Sends a "hangup" message to clients to switch to read-only mode and show an error message.
     *
     * @param fromId The ID of the client that sends the "hangup" message. Must not be null to ensure that the message is send.
     * @param statusToSend The connection status to be sent via the hangup message.
     * @param errorCode The error code to be sent via the hangup message. This should specify why the hangup has been sent to the client.
     *            Can be null if no error can be specified which triggers a generic error message on the client.
     * @param toAll Must be set to send the "hangup" message to all clients sharing the document otherwise the message is sent to the client
     *            with ID fromId.
     * @param logMessage The message to be added to the logging entry in debug mode only.
     */
    protected void impl_sendHangUp(ID fromId, ConnectionStatus statusToSend, ErrorCode errorCode, final boolean toAll, final String logMessage) throws OXException {
        if (fromId == null) {
            throw new RuntimeException("Without fromId no hang up !");
        }

        if (statusToSend == null) {
            statusToSend = syncGetConnectionStatus();
        }

        final MessageData messageData = new MessageData(null, statusToSend, MessageHelper.finalizeJSONResult(errorCode, null, null));

        final Message returnMessage = new Message();
        returnMessage.setFrom(fromId);
        returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), "office", "hangup")).build()));

        LOG.debug("RT connection: send hang up message: " + (logMessage == null ? "" : logMessage));

        if (toAll) {
            relayToAll(returnMessage);
        } else {
            relayToID(returnMessage, fromId);
        }
    }

    /**
     * Tries to handle calc engine errors in a general way.
     *
     * @param session The session of the client which requested the calc engine to process the operations that lead to a calc engine error.
     * @param e The calc engine error we received on operation processing.
     * @return TRUE if the error is fatal and the instance must be disposed, FALSE if not fatal. Attention: Currently EVERY error is treated
     *         as fatal.
     */
    private boolean impl_handleCalcEngineError(Session session, ECalcEngineError e) {
        final boolean result = true;

        // Set this instance to dispose as we encountered a calc engine error.
        // Dispose calls onDispose() which will save the document.
        LOG.error("RT connection: Due to the calc engine error: " + e + ", the connection instance is going to be disposed");
        this.dispose();

        return result;
    }

    /**
     * Resets the edit client to ensure that the client is not able to make any changes on the document anymore. This method is normally
     * used if a "hangup" message is sent to the specific client because it triggered an invalid state (e.g. sent incorrect operations,
     * failed to sent all operations in time while switching edit rights)
     *
     * @param fromId The ID of the editor to be reseted. It must be the current editor ID of this Connection instance otherwise the method
     *            won't do anything.
     * @return TRUE if the editor has been reseted or FALSE if not.
     */
    private boolean impl_resetEditor(ID fromId) {

        if (null == fromId) {
            return false;
        }

        boolean result = false;
        synchronized (m_connectionStatus) {
            String currentEditUser = m_connectionStatus.getCurrentEditingUserId();

            if (null != currentEditUser) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(fromId)) {
                    String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
                    if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                        // Set wantsToEditUser as new editor to speed up switching process in this case
                        m_connectionStatus.setCurrentEditingUser(
                            wantsEditRightsUser.toString(),
                            m_connectionStatus.getWantsEditRightsUserName());
                        // cancel a possible switching timeout timer
                        this.cancelPrepareLosingEditRightsTimer();
                        LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                    } else {
                        // Reset edit user
                        m_connectionStatus.setCurrentEditingUser("", "");
                        LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor.");
                    }
                    result = true;
                }
            }
        }

        return result;
    }

    /**
     * Provides a stanza for the last message sent to the client leaving a document (chat-room). ATTENTION: This method is called by a
     * background thread from the Realtime-Framework. Therefore this method must be implemented thread-safe.
     *
     * @param onBehalfOf The ID of the client which wants to leave the document.
     * @return The stanza sent to the client which wants to leave the document. (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#getSignOffMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    @Asynchronous
    public Stanza getSignOffMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message signoffMessage = null;

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: [signoff message] called for document: " + getDocumentFolderAndFileId());
        }

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            int activeClients = 0;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [signoff message] for: " + onBehalfOf.toString());
            }

            synchronized (m_connectionStatus) {
                activeClients = m_connectionStatus.getActiveClients();
            }

            final ID creator = impl_getCreatorForNextFlush(onBehalfOf, true);
            final ServerSession serverSession = MessageHelper.getServerSession(null, onBehalfOf);
            ErrorCode errorCode = flushDocument(serverSession, creator, false, (activeClients == 1));
            impl_sendFlushInfoOnError(errorCode, onBehalfOf, true);
            if ((activeClients == 1) && (errorCode.getCode() != ErrorCode.CODE_NO_ERROR)) {
                // Set save on dispose to false, which prevents an unnecessary save
                // in case the last client leaves and we encountered an error.
                m_saveOnDispose.set(false);
            }

            final JSONObject jsonExtraData = new JSONObject();
            try {
                synchronized (m_syncAccess) {
                    String fileName = "unknown";
                    if (null != m_lastKnownMetaData) {
                        fileName = m_lastKnownMetaData.getFileName();
                    }
                    m_syncStatus.setFileName(fileName);
                    jsonExtraData.put(Sync.SYNC_INFO, m_syncStatus.toJSON());
                }
            } catch (JSONException e) {
                // sync info is an optional feature used for local storage
                LOG.warn("RT connection: [getSignOffMessage] catches exception while trying to put sync info to JSON result. ", e);
            }

            // send back the JSON result object returned by flushDocument
            signoffMessage = new Message();
            signoffMessage.setFrom(fromId);
            signoffMessage.setTo(onBehalfOf);
            signoffMessage.addPayload(new PayloadTree(
                PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(errorCode, null, jsonExtraData), "json", "office", "closedocument")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [signoff message] to: " + onBehalfOf.toString());
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't send [signoff message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return signoffMessage;
    }

    /**
     * Handles the inactivity notice of the real-time framework for a client that is a member of this connection instance. Currently the
     * handler disables the remote selection of inactive clients.
     *
     * @param stanza The Stanza containing the inactive client identified by ElementPath 'com.openexchange.realtime.client' and the Duration
     *            of inactivity identified by 'com.openexchange.realtime.client.inactivity'.
     */
    @Override
    public void handleInactivityNotice(Stanza stanza) {
        Optional<ID> inactiveClient = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime", "client"), ID.class);
        Optional<Duration> inactivityDuration = stanza.getSinglePayload(
            new ElementPath("com.openexchange.realtime.client", "inactivity"),
            Duration.class);

        if (inactiveClient.isPresent() && inactivityDuration.isPresent()) {
            String id = inactiveClient.get().toString();
            Duration timeOfDuration = inactivityDuration.get();
            boolean enableActiveUser = false;

            LOG.debug("RT connection: User=" + inactiveClient.get() + " was inactive for " + inactivityDuration.get());

            if (timeOfDuration.equals(Duration.NONE)) {
                enableActiveUser = true;
            } else if (timeOfDuration.getValueInS() > 0) {
                enableActiveUser = false;
            }

            ConnectionStatus statusToSend = null;
            synchronized (m_connectionStatus) {

                boolean isUserEnabled = m_connectionStatus.isActiveUserEnabled(id);
                if (isUserEnabled != enableActiveUser) {
                    if (enableActiveUser) {
                        m_connectionStatus.enableActiveUser(id);
                    } else {
                        // Disable user to remove the remote selection and set
                        // the latest inactivity duration, too. This is used for
                        // the heuristic to give the same user the edit rights, if
                        // 1.) The same user has the edit rights
                        // 2.) All collaboration clients are from the same user
                        // 3.) The edit rights RT-user has at least an inactivity
                        // of 30 seconds.
                        m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                    }
                    LOG.debug("RT connection: User=" + id + " enabled=" + String.valueOf(enableActiveUser));
                } else if (!enableActiveUser) {
                    // update inactivity time
                    m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                }

                // The status must now be sent on any possible change
                // (update of collaboration clients).
                statusToSend = (ConnectionStatus) m_connectionStatus.clone();
                statusToSend.setInactivityChanged(id);
            }

            if (null != statusToSend) {
                updateClients(new MessageData(null, statusToSend, null), inactiveClient.get());
            }
        }
    }

    /**
     * merge the content of the 'contents' element of a local result to the global result
     *
     * @param globalChanged changed cells information of previous operations
     * @param localChanged changed cells information of current operations
     * @return information of all cells, older information is overwritten by new cells
     */
    JSONArray mergeContents(JSONArray globalChanged, JSONArray localChanged) {

        try {
            Set<Long> localHashes = new HashSet<Long>();

            Iterator<Object> localIt = localChanged.iterator();
            while (localIt.hasNext()) {
                JSONObject cell = (JSONObject) localIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                localHashes.add(new Long(posValue));
            }

            Iterator<Object> globalIt = globalChanged.iterator();
            while (globalIt.hasNext()) {
                JSONObject cell = (JSONObject) globalIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                if (!localHashes.contains(new Long(posValue))) {
                    localChanged.put(cell);
                }
            }
        } catch (JSONException e) {
            // should never occur
        }
        return localChanged;
    }

    /**
     * merge the content of the 'changed' element of a local result to the global result
     *
     * @param jsonResult global result object
     * @param localJsonResult local result object
     */
    void mergeChanged(JSONObject jsonResult, JSONObject localJsonResult) {
        try {
            if (localJsonResult.has("changed")) {
                if (!jsonResult.has("changed")) {
                    jsonResult.put("changed", new JSONObject());
                }
                JSONObject globalChanged = jsonResult.getJSONObject("changed");
                if (!globalChanged.has("type") || !globalChanged.getString("type").equals("all")) {
                    JSONObject localChanged = localJsonResult.getJSONObject("changed");
                    globalChanged.put("type", localChanged.getString("type"));
                    if (!globalChanged.getString("type").equals("all")) {
                        // if still not 'all' is changed merge the cell/column/row ranges
                        if (localChanged.has("sheets")) {
                            if (!globalChanged.has("sheets")) {
                                globalChanged.put("sheets", localChanged.getJSONObject("sheets"));
                            } else {
                                JSONObject globalSheets = globalChanged.getJSONObject("sheets");
                                JSONObject localSheets = localChanged.getJSONObject("sheets");
                                Set<String> localSheetSet = localSheets.keySet();

                                Iterator<String> localSheetIt = localSheetSet.iterator();
                                while (localSheetIt.hasNext()) {
                                    String localSheetKey = localSheetIt.next();
                                    JSONObject localSheet = localSheets.getJSONObject(localSheetKey);
                                    if (!globalSheets.has(localSheetKey)) {
                                        globalSheets.put(localSheetKey, localSheet);
                                    } else {
                                        JSONObject globalSheet = globalSheets.getJSONObject(localSheetKey);
                                        //
                                        if (localSheet.has("contents")) {
                                            if (globalSheet.has("contents")) {
                                                globalSheet.put(
                                                    "contents",
                                                    mergeContents(globalSheet.getJSONArray("contents"), localSheet.getJSONArray("contents")));
                                            } else {
                                                globalSheet.put("contents", localSheet.getJSONArray("contents"));
                                            }
                                        }

                                        if (localSheet.has("cells")) {
                                            if (!globalSheet.has("cells")) {
                                                globalSheet.put("cells", localSheet.getJSONArray("cells"));
                                            } else {
                                                // merge cells "cells:[{start:[1,1],end:[3,3]}, {start:[5,7]}]"
                                                JSONArray globalCells = globalSheet.getJSONArray("cells");
                                                JSONArray localCells = localSheet.getJSONArray("cells");
                                                CellsMerger cellsMerger = new CellsMerger(globalCells);
                                                cellsMerger.add(localCells);
                                                globalSheet.put("cells", cellsMerger.getCells());
                                            }
                                        }
                                        if (localSheet.has("rows")) {
                                            if (!globalSheet.has("rows")) {
                                                globalSheet.put("rows", localSheet.getJSONArray("rows"));
                                            } else {
                                                // merge rows "rows:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalRows = globalSheet.getJSONArray("rows");
                                                JSONArray localRows = localSheet.getJSONArray("rows");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalRows);
                                                rowColsMerger.add(localRows);
                                                globalSheet.put("rows", rowColsMerger.getRowCols());
                                            }
                                        }
                                        if (localSheet.has("columns")) {
                                            if (!globalSheet.has("columns")) {
                                                globalSheet.put("columns", localSheet.getJSONArray("columns"));
                                            } else {
                                                // merge columns "columns:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalColumns = globalSheet.getJSONArray("columns");
                                                JSONArray localColumns = localSheet.getJSONArray("columns");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalColumns);
                                                rowColsMerger.add(localColumns);
                                                globalSheet.put("cols", rowColsMerger.getRowCols());
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (JSONException e) {
            LOG.debug("Exception in RT connection: mergeChanged" + jsonResult.toString() + " / " + localJsonResult.toString());
        }

    }

    /**
     * Handles the request to apply new actions to the document. The method controls that the sender has the edit rights and it makes some
     * basic operation checking to protected the document consistency.
     *
     * @notice Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     *         element: "message", payloads: [{ element: "action", data: "applyactions" }, { namespace: "office", element: "actions", data:
     *         "[{operations: [...]}, ...]" }] }
     * @param stanza A Stanza which contain the operations to be applied to the current document.
     * @return The message
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {

        checkForDisposed();

        final ID fromId = stanza.getFrom();
        final String editor = m_connectionStatus.getCurrentEditingUserId();
        final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
        boolean stanzaFromEditor = (null != editor) && (null != fromId) && (fromId.equals(editorId));
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleApplyActions called from different thread " + String.valueOf(threadId));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: [applyActions] received for document: " + getDocumentFolderAndFileId());
        }

        // Check that we have a ID for the sender stanza and that the sender
        // has the edit rights.
        final Message returnMessage = new Message();
        if ((null != fromId) && isMember(fromId) && stanzaFromEditor) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(
                stanza,
                new ElementPath("office", MessagePropertyKey.KEY_ACTIONS),
                true);
            MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
            MessageChunk uiActionChunk = null;
            JSONObject jsonResult = new JSONObject();
            boolean emptyActionChunk = false;
            ServerSession session = MessageHelper.getServerSession(null, fromId);

            // increase number of received actions
            m_nActionsReceived++;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [applyActions] received, applied from: " + fromId.toString());
            }

            // Server must check the consistency of the operations sent by the current editor client.
            // If check fails the client must receive a 'hangup' message and be excluded from the
            // document group.
            if (!OperationHelper.hasValidOperations(actionChunk, syncGetConnectionStatus().getOperationStateNumber())) {
                // send hangup message and ignore operations
                impl_sendHangUp(
                    fromId,
                    null,
                    ErrorCode.HANGUP_INVALID_OPERATIONS_SENT_ERROR,
                    HANG_UP_ID,
                    "sent hangup due to inconsistent operations in message chunk.");
                if (impl_resetEditor(fromId)) {
                    // also send a update message with the new/reseted editor to all clients
                    updateClients(new MessageData(null, syncGetConnectionStatus()), fromId);
                }
                return returnMessage;
            }

            boolean sendToAll = false;
            boolean isPaste = false;
            boolean isSort = false;
            // update spreadsheet document
            if (m_calcClient != null && actionChunk.getOperations().has(MessagePropertyKey.KEY_ACTIONS)) {
                try {
                    JSONObject firstSourceOp = new JSONObject();
                    int sourceOperationLength = 0;
                    JSONArray chunkActions = actionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                    for (int chunkAction = 0; chunkAction < chunkActions.length(); ++chunkAction) {
                        JSONObject localJsonResult = new JSONObject();
                        JSONObject sheetOperations = chunkActions.getJSONObject(chunkAction);
                        JSONArray operations = sheetOperations.getJSONArray("operations");
                        sourceOperationLength += operations.length();
                        JSONObject firstChunkOperation = operations.getJSONObject(0);
                        // put locale into the first operation of _all_ chunks
                        if (!firstChunkOperation.has("parse")) {
                            String userLanguage = session.getUser().getPreferredLanguage();
                            firstChunkOperation.put("parse", userLanguage);
                        }

                        if (chunkAction == 0) {
                            firstSourceOp = firstChunkOperation;
                        }
                        if(operations.length() == 1) {
                            String opName = operations.getJSONObject(0).getString("name");
                            if(opName.equals("paste")){
                                isPaste = opName.equals("paste");
                            } else if(opName.equals("sort")) {
                                isSort = true;
                            }
                        }
                        if (isPaste) {
                            JSONObject pasteOp = operations.getJSONObject(0);
                            CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                            event.sUserSession = pasteOp.getString("UserSession");
                            event.sTargetSelectionJSON = pasteOp.getString("TargetSelection");
                            event.sClipboardHandle = pasteOp.getString("ClipboardHandle");
                            executeSpreadsheetPasteFailSafe(fromId, event);
                            localJsonResult = new JSONObject(event.sPasteOperationsJSON);
                        } else {
                            localJsonResult = executeSpreadsheetOperationFailsafe(fromId, sheetOperations.toString());
                        }
                        if (chunkAction == 0) {
                            jsonResult = localJsonResult;
                        } else {
                            // add result elements to jsonResult
                            if (localJsonResult.has("undoCount")) {
                                jsonResult.put("undoCount", localJsonResult.get("undoCount"));
                            }
                            if (localJsonResult.has("redoCount")) {
                                jsonResult.put("redoCount", localJsonResult.get("redoCount"));
                            }
                            if (localJsonResult.has("changedOperations")) {
                                JSONArray lclChangedOp = localJsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                                JSONArray resultChangedOp = jsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                                for (int op = 0; op < lclChangedOp.length(); ++op) {
                                    resultChangedOp.put(lclChangedOp.get(op));
                                }
                            }
                            mergeChanged(jsonResult, localJsonResult);
                        }
                    }
                    sendToAll = true;
                    LOG.debug(jsonResult.toString());
                    if (jsonResult.has("changedOperations")) {
                        JSONObject actions = new JSONObject();
                        JSONObject changedOperations = jsonResult.getJSONObject("changedOperations");
                        JSONArray changedOpsArray = new JSONArray();
                        JSONArray ops = changedOperations.optJSONArray("operations");

                        emptyActionChunk = (ops == null || ops.length() == 0);
                        changedOpsArray.put(0, changedOperations);
                        actions.put(MessagePropertyKey.KEY_ACTIONS, changedOpsArray);
                        actionChunk = new MessageChunk(actions, actionChunk.getSender());
                        jsonResult.remove("changedOperations");
                    }
                    JSONObject actions = new JSONObject();
                    JSONArray uiActionsArray = new JSONArray();
                    JSONObject uiOperations = new JSONObject();
                    if (jsonResult.has("uiOperations")) {
                        JSONArray uiOperationsArray = jsonResult.getJSONArray("uiOperations");
                        int osnDiff = sourceOperationLength - uiOperationsArray.length();
                        int opState = m_connectionStatus.getOperationStateNumber();
                        for (int op = 0; op < uiOperationsArray.length(); ++op) {
                            uiOperationsArray.getJSONObject(op).put("opl", 1).put("osn", opState + op);
                        }
                        if (osnDiff > 0) {
                            uiOperationsArray.getJSONObject(uiOperationsArray.length() - 1).put("opl", osnDiff + 1);
                        }
                        uiOperations.put("operations", uiOperationsArray);
                    } else {
                        // create a dummy operation to align osl/opn
                        int osn = firstSourceOp.getInt("osn");
                        JSONObject dummyOp = new JSONObject();
                        dummyOp.put("name", "noOp");
                        dummyOp.put("osn", osn);
                        dummyOp.put("opl", sourceOperationLength);
                        JSONArray operationsArray = new JSONArray();
                        operationsArray.put(0, dummyOp);
                        uiOperations.put("operations", operationsArray);
                    }
                    uiActionsArray.put(0, uiOperations);

                    actions.put(MessagePropertyKey.KEY_ACTIONS, uiActionsArray);
                    uiActionChunk = new MessageChunk(actions, fromId);

                } catch (JSONException e) {
                    LOG.debug("Error executing spreadsheet operation", e);
                } catch (CalcEngineException e) {
                    LOG.error("RT connection: Error executing spreadsheet operation. Calc engine worker notifies problems", e);
                    impl_handleCalcEngineError(session, e.getErrorcode());
                    throw new OXException(e);
                } catch (Exception e) {
                    LOG.error("RT connection: Error executing spreadsheet operation", e);
                    throw new OXException(e);
                }
            }

            // handling operation state number for operations from clients
            try {
                MessageChunk osnActionChunk = uiActionChunk != null ? uiActionChunk : actionChunk;
                if (osnActionChunk.getOperations().hasAndNotNull(MessagePropertyKey.KEY_ACTIONS)) {
                    JSONArray allActions = osnActionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                    JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                    if (lastAction.has("operations")) {
                        JSONArray allOps = lastAction.getJSONArray("operations");
                        JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                        // update statistics with incoming operations requests
                        if (null != m_fileHelper) {
                            Statistics.handleDocumentEvent(new OperationsEvent(m_fileHelper.getDocumentType(), OperationsType.INCOMING));
                        }

                        if (lastOp.has("osn") && lastOp.has("opl")) {
                            int osn = -1;
                            synchronized (m_syncAccess) {
                                osn = lastOp.getInt("osn") + lastOp.getInt("opl");
                                m_connectionStatus.setOperationStateNumber(osn);
                            }
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: handleApplyActions, lastOp received = " + OperationHelper.operationToString(lastOp));
                                LOG.debug("RT connection: handleApplyActions, current server osn = " + osn);
                            }
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.debug("RT connection: handleApplyActions: Error setting operation state number to operation", e);
            }

            // update the shared state
            int msgChunkListSize = 0;
            synchronized (m_syncAccess) {
                if (!emptyActionChunk) {
                    m_messageChunkList.add(actionChunk);
                }
                if (uiActionChunk != null) {
                    OperationHelper.appendJSON(m_uiOperations, uiActionChunk.toJSON());
                }
                msgChunkListSize = m_messageChunkList.size();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));
            }

            final MessageData messageData = new MessageData(
                uiActionChunk != null ? uiActionChunk : actionChunk,
                    syncGetConnectionStatus(),
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null));

            // Send the message to all participants (excluding the one who sent it originally)
            returnMessage.setFrom(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
            }

            if (isPaste||isSort) {
                relayToAll(returnMessage, stanza);
            } else {
                relayToAllExceptSender(returnMessage, stanza);
            }

            // update statistics with distributed operations count
            int otherClientCount = 0;

            synchronized (m_connectionStatus) {
                otherClientCount = m_connectionStatus.getActiveClients() - 1;
                m_connectionStatus.setLastModifyingUserId(fromId.toString());
            }

            if ((otherClientCount > 0) && (null != m_fileHelper)) {
                Statistics.handleDocumentEvent(new OperationsEvent(
                    m_fileHelper.getDocumentType(),
                    OperationsType.DISTRIBUTED,
                    otherClientCount));
            }

            // update further clients
            if (sendToAll) {
                // update all Clients with the result, too
                updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), fromId);
                m_nActionsReceived = 0;
            } else if (m_nActionsReceived >= ACTIONS_UNTIL_FORCE_UPDATE) {
                // send update to sender, too
                m_nActionsReceived = 0;

                final Message senderMessage = new Message();
                final MessageData senderMessageData = new MessageData(null, syncGetConnectionStatus(), null);
                senderMessage.setFrom(fromId);
                senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(senderMessageData, MessageData.class.getName(), "office", "update")).build()));
                if (stanza.getTracer() != null) {
                    senderMessage.setTracer(stanza.getTracer());
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [update] to editor for keep alive, applied from: " + fromId.toString());
                }
                relayToID(senderMessage, fromId);
            }

        } else {
            if (null != fromId && !stanzaFromEditor) {
                // The sender has no edit rights, therefore we send a hang up message
                // so the client will be notified and can reload the document.
                impl_sendHangUp(
                    fromId,
                    null,
                    ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR,
                    HANG_UP_ID,
                    "Relaying [hangup] due to missing edit rights, applied from: " + fromId.toString());
            }

            if (LOG.isDebugEnabled()) {
                if (!isMember(fromId)) {
                    LOG.debug("RT connection: Didn't relay [applyActions] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
                }
                if (!stanzaFromEditor) {
                    LOG.debug("RT connection: Didn't relay [applyActions] because the requester doesn't have the edit rights");
                }
            }
        }
        return returnMessage;
    }

    /**
     * Stanzas layout: AcquireEdit { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "acquireedit" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAcquireEdit(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAcquireEdit called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = MessageHelper.getServerSession(null, fromId);
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleAcquireEdit error retrieving osn from JSON object. ", e);
            }

            if (LOG.isDebugEnabled()) {
                String editor = syncGetConnectionStatus().getCurrentEditingUserId();
                LOG.debug("RT connection: Handling [acquireEdit], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
                LOG.debug("RT connection: Editor: " + ((editor != null) ? editor : "unknown"));
            }

            acquireEdit(serverSession, fromId, false, false);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [acquireEdit] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: CanLoseEditRights { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "canloseeditrights" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCanLoseEditRights(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleCanLoseEditRights called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = MessageHelper.getServerSession(null, fromId);
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleCanLoseEditRights cannot retrieve osn from JSON request. ", e);
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [canLoseEditRights], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
            }

            canLoseEditRights(serverSession, fromId, requesterOSN, false);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Didn't handle [canLoseEditRights] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: AddResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "addresource" }, { namespace: "office", element: "resource", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAddResource(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAddResource called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "resource"), false);
            final ServerSession serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = addResource(serverSession, jsonRequest);

            // send back the JSON result object returned by addResource
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "resource")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [addResource], called from: " + fromId.toString());
            }

            // send the message to the one who requested to add the resource
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [addResource] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: FlushDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "flushdocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleFlushDocument(Stanza stanza) throws OXException {
        ID fromId = MessageHelper.getIDFromStanza(stanza);

        checkForDisposed();
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleFlushDocument called from different thread " + String.valueOf(threadId));
        }

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final ID creator = this.impl_getCreatorForNextFlush(fromId, false);
            final ErrorCode errorCode = flushDocument(serverSession, creator, false, false);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [flushDocument], called from: " + fromId.toString());
            }

            // send back the JSON result object returned by flushDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(
                PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(errorCode, jsonRequest, null), "json", "office", "flushdocument")).build()));
            send(returnMessage);
            // send possible flushDocument() error code to other clients
            impl_sendFlushInfoOnError(errorCode, fromId, true);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [flushDocument] since the requester was no member");
        }
    }

    /**
     * @param stanza
     * @param isUndo
     * @throws OXException
     */
    private void processUndoRedo(Stanza stanza, boolean isUndo) throws OXException {
        ID fromId = MessageHelper.getIDFromStanza(stanza);

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "document"), false);
            Stanza applyStanza = new Message();
            applyStanza.setFrom(fromId);
            JSONObject undoAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject undo = new JSONObject();

            if (LOG.isDebugEnabled()) {
                if (isUndo) {
                    LOG.debug("RT connection: Handling [undo], called from: " + fromId.toString());
                } else {
                    LOG.debug("RT connection: Handling [redo], called from: " + fromId.toString());
                }
            }

            try {
                undo.put("name", isUndo ? "undo" : "redo");
                undo.put("osn", m_connectionStatus.getOperationStateNumber());
                undo.put("opl", 1);
                operations.put(0, undo);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                undoAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.debug("RT connection: processUndoRedo, error creating undoActions JSON object. ", e);
            }
            applyStanza.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(undoAction, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    MessagePropertyKey.KEY_ACTIONS)).build()));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path = new ElementPath("office", "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "undo")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "undo" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleUndo(Stanza stanza) throws OXException {
        checkForDisposed();
        processUndoRedo(stanza, true);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "redo" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRedo(Stanza stanza) throws OXException {
        checkForDisposed();
        processUndoRedo(stanza, false);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "replace" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleReplaceAll(Stanza stanza) throws OXException {
        checkForDisposed();

        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        final Session session = MessageHelper.getServerSession(null, fromId);

        try {
            final JSONArray operationsArray = new JSONArray();
            JSONObject operations = new JSONObject();
            try {
                jsonRequest.put("name", "replace");
                LOG.debug(jsonRequest.toString());
                operationsArray.put(jsonRequest);
                operations.put("operations", operationsArray);
            } catch (JSONException e) {
                LOG.error("JSONException on handleReplaceAll", e);
            }
            // return operations;
            JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
            jsonResult = MessageHelper.checkJSONResult(jsonResult);
            if (jsonResult.has("changedOperations")) {
                JSONObject actions = new JSONObject();
                JSONArray changedOperations = new JSONArray();
                changedOperations.put(0, jsonResult.getJSONObject("changedOperations"));
                actions.put(MessagePropertyKey.KEY_ACTIONS, changedOperations);
                MessageChunk actionChunk = new MessageChunk(actions, fromId);
                jsonResult.remove("changedOperations");

                // update the shared state
                synchronized (m_syncAccess) {
                    m_messageChunkList.add(actionChunk);
                }
            }
            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [replaceAll] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [replaceAll] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (JSONException e) {
            LOG.error("Exception on handleReplaceAll", e);
        } catch (CalcEngineException e) {
            LOG.error("Exception on handleReplaceAll", e);
            impl_handleCalcEngineError(session, e.getErrorcode());
            throw new OXException(e);
        } catch (Exception e) {
            LOG.error("Exception on handleReplaceAll", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "sort" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleSort(Stanza stanza) throws OXException {
        checkForDisposed();

        ID fromId = MessageHelper.getIDFromStanza(stanza);
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            Stanza applyStanza = new Message();
            final String editor = m_connectionStatus.getCurrentEditingUserId();
            final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
            applyStanza.setFrom(editorId);
            JSONObject sortAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject sort = new JSONObject();

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [sort], called from: " + fromId.toString());
            }

            try {
                //                final JSONArray operationsArray = new JSONArray();
                LOG.debug(jsonRequest.toString());
                //                operationsArray.put(jsonRequest);

                sort.put("name", "sort");
                //                sort.put("UserSession", jsonRequest.getJSONObject("session").getString("resource"));
                sort.put("osn", m_connectionStatus.getOperationStateNumber());
                sort.put("opl", 1);
                Iterator <Map.Entry<String, Object>> reqIt = jsonRequest.entrySet().iterator();
                while(reqIt.hasNext())
                {
                    Map.Entry<String, Object> nextElement = reqIt.next();
                    sort.put(nextElement.getKey(), nextElement.getValue());
                }
                operations.put(0, sort);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                sortAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.error("RT connection: handleSort catches exception while creating JSON actions object. ", e);
            }

            applyStanza.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(sortAction, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    MessagePropertyKey.KEY_ACTIONS)).build()));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path = new ElementPath("office", "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "spreadsheet")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "query" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleQuery(Stanza stanza) throws OXException {
        checkForDisposed();

        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        final Session session = MessageHelper.getServerSession(jsonRequest, fromId);

        LOG.debug("RT connection: Handling [query], called from: " + fromId.toString());

        try {
            final JSONArray operationsArray = new JSONArray();
            JSONObject operations = new JSONObject();
            jsonRequest.put("name", "query");
            LOG.debug(jsonRequest.toString());
            operationsArray.put(jsonRequest);
            operations.put("operations", operationsArray);

            // return operations;
            JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
            jsonResult = MessageHelper.checkJSONResult(jsonResult);
            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [query] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [query] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (CalcEngineException e) {
            LOG.error("Exception on handleQuery", e);
            impl_handleCalcEngineError(session, e.getErrorcode());
            throw new OXException(e);
        } catch (Exception e) {
            LOG.error("Exception on handleQuery", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "copy" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopy(Stanza stanza) throws OXException {
        checkForDisposed();

        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        final Session session = MessageHelper.getServerSession(jsonRequest, fromId);
        try {
            JSONObject jsonResult = new JSONObject();
            if (m_calcClient != null) {
                try {
                    CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                    event.sClipboardHandle = CalcEngineClipBoard.newHandle();
                    event.sUserSession = jsonRequest.getJSONObject("session").getString("resource");
                    JSONObject sourceSelection = new JSONObject();
                    sourceSelection.put("sheet", jsonRequest.get("sheet"));
                    sourceSelection.put("start", jsonRequest.get("start"));
                    if( jsonRequest.has("end")) {
                        sourceSelection.put("end", jsonRequest.get("end"));
                    }
                    event.sSourceSelectionJSON = sourceSelection.toString();
                    executeSpreadsheetCopyFailSafe(fromId, event);
                    jsonResult.put("handle", event.sClipboardHandle);
                } catch (CalcEngineException e) {
                    LOG.error("Exception on handleQuery", e);
                    impl_handleCalcEngineError(session, e.getErrorcode());
                    throw new OXException(e);
                }
            }

            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [query] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [query] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (Exception e) {
            LOG.error("Exception on handleQuery()", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "paste" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handlePaste(Stanza stanza) throws OXException {
        checkForDisposed();

        ID fromId = MessageHelper.getIDFromStanza(stanza);
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            Stanza applyStanza = new Message();
            final String editor = m_connectionStatus.getCurrentEditingUserId();
            final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
            applyStanza.setFrom(editorId);
            JSONObject pasteAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject paste = new JSONObject();

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [paste], called from: " + fromId.toString());
            }

            try {
                paste.put("name", "paste");
                paste.put("UserSession", jsonRequest.getJSONObject("session").getString("resource"));
                JSONObject targetSelection = new JSONObject();
                targetSelection.put("sheet", jsonRequest.get("sheet"));
                targetSelection.put("start", jsonRequest.get("start"));
                if( jsonRequest.has("end")) {
                    targetSelection.put("end", jsonRequest.get("end"));
                }
                paste.put("TargetSelection", targetSelection );
                paste.put("ClipboardHandle", jsonRequest.getString("handle").toString());
                paste.put("osn", m_connectionStatus.getOperationStateNumber());
                paste.put("opl", 1);
                operations.put(0, paste);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                pasteAction.put(MessagePropertyKey.KEY_ACTIONS, actions);
            } catch (JSONException e) {
                LOG.error("RT connection: handlePaste catches exception while creating JSON actions object. ", e);
            }

            applyStanza.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(pasteAction, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    MessagePropertyKey.KEY_ACTIONS)).build()));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path = new ElementPath("office", "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "spreadsheet")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "getdocument" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleGetDocument(Stanza stanza) throws OXException {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleGetDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final ServerSession serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getDocument(serverSession, jsonRequest);

            // send back the JSON result object returned by getDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    "office",
                    "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [getDocument], sending to: " + fromId.toString());
            }

            // send the message to the one who requested to get the document
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [getDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: RenameDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "renamedocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRenameDocument(Stanza stanza) throws OXException {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleRenameDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final RenameResult result = renameDocument(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(MessageHelper.finalizeJSONResult(result.jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [renameDocument] result to: " + stanza.getFrom().toString());
            }

            // always send the message to the one who requested to rename the document
            send(returnMessage);

            // if rename was successful update all clients with the result, too
            final ErrorCode errorCode = result.errorCode;
            if (!errorCode.isError()) {
                LOG.debug("RT connection: Relaying [renameDocument] result, called from: " + stanza.getFrom().toString());
                // update all Clients with the result, too
                updateClients(new MessageData(null, syncGetConnectionStatus(), result.jsonResult), null);
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [renameDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: CopyDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "copydocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopyDocument(Stanza stanza) throws OXException {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleCopyDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was sent as an
        // internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = copyDocument(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(MessageHelper.finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [handleCopyDocument] result to: " + stanza.getFrom().toString());
            }

            // send the return message to the one who requested to copy document
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [handleCopyDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: UpdateView { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "updateview" }, { namespace: "office", element: "spreadsheet", data: "{... : ...,}"
     * }] }
     *
     * @param stanza
     * @throws OXException
     */
    public Boolean handleUpdateView(Stanza stanza) throws OXException {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        Message returnMessage = null;
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleUpdateView called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getSpreadsheetViewUpdate(fromId, serverSession, jsonRequest);
            if (jsonResult != null && !jsonResult.has("error")) {
                returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(
                        MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json",
                        "office",
                        "spreadsheet")).build()));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Sending [updateView] result to: " + stanza.getFrom().toString());
                }

                // send the message to the one who requested to rename the document
                send(returnMessage);

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [updateView] result, called from: " + stanza.getFrom().toString());
                }

                // update all Clients with the result, too
                updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
            }
        }
        return new Boolean(returnMessage != null);
    }

    /**
     * Updates the user data of the provided client e.g. the current selection.
     *
     * @param stanza Stanzas layout: UpdateUserData { to:
     *            "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element: "message", payloads:
     *            [{ namespace: "office", element: "state", data: "{... : ...,}]" }] }
     * @throws OXException
     * @throws JSONException
     */
    public void handleUpdateUserData(Stanza stanza) {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleUpdateUserData called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            LOG.debug("RT connection: [updateUserData] received by " + fromId.toString());
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "state"), false);
            final String userId = fromId.toString();

            // update user data
            m_connectionStatus.updateUserData(userId, jsonRequest);

            ConnectionStatus statusToSend = (ConnectionStatus) m_connectionStatus.clone();
            statusToSend.setSelectionChanged(userId);

            LOG.debug("RT connection: relaying [updateUserData]");
            // updateClients(new MessageData(null, null, jsonResult), null);
            updateClientsExceptSender(new MessageData(null, statusToSend, null), fromId);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [updateUserData] since the requester was no member");
        }
    }

    /**
     * Writes a log message sent from a client to the sevrer log file.
     *
     * @param stanza Stanzas layout: LogMessage { to:
     *            "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element: "message", payloads:
     *            [{ namespace: "office", element: "state", data: "{... : ...,}]" }] }
     */
    public void handleLogMessage(Stanza stanza) {
        checkForDisposed();
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: [logMessage] called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            LOG.debug("RT connection: [logMessage] received from " + fromId.toString());
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath("office", "state"), false);
            final String message = (null != jsonRequest) ? jsonRequest.optString("message") : "no message from client";

            // write client message to log file for Connection
            if (null != message) {
                LOG.error("Client message: " + message);
            }
        } else {
            LOG.debug("RT connection: Didn't handle [logMessage] since the requester was no member");
        }
    }

    /**
     * Checks if a saving operation for this connection is currently in progress.
     *
     * @return TRUE if there is a saving in progress otherwise FALSE.
     */
    public boolean savingDocumentInProgress() {
        return (this.m_saveDocumentInProgress.get() > 0);
    }

    /**
     * Provides the time stamp of the latest save operation. If there have been no save operation the time is the same as the Connection
     * instance has been created (stored as m_startupTime).
     *
     * @return The date of the latest save operation.
     */
    public Date getSaveTimeStamp() {
        return (null != this.m_lastSaveTime) ? m_lastSaveTime : m_startupTime;
    }

    /**
     * Provides the time of modification or null if no modification has been made since loading the document.
     *
     * @return Time of the last modification or null if no modification has been made,
     */
    public Date getLastModifiedTime() {
        return m_lastSaveTime;
    }

    /**
     * Provides the last fail safe save error code that saving the document encountered.
     *
     * @return The error code of the last fail safe save process (in the background).
     */
    public synchronized ErrorCode getLastFailSafeSaveError() {
        return m_lastFailSafeSaveError;
    }

    /**
     * Sets the last fail safe save error code that saving the document encountered.
     *
     * @param latestErrorCode The error code of the last fail safe save process (in the background).
     */
    private synchronized void setLastFailSafeSaveError(ErrorCode latestErrorCode) {
        m_lastFailSafeSaveError = latestErrorCode;
    }

    /**
     * Provides the number of operations not applied to a saved revision of the document.
     *
     * @return The number of pending operations (not saved in the document file yet).
     */
    public long getNumberOfPendingOperations() {
        long num = 0;

        synchronized (m_syncAccess) {
            if (null != m_messageChunkList) {
                num = this.m_messageChunkList.size();
            }
        }
        return num;
    }

    /**
     * Tries to flush a modified document due to a dispose notification from the Realtime framework.
     *
     * @return TRUE if the saving the document was successful otherwise FALSE. The method also provides TRUE if there was no need to save
     *         the document as there are no pending operations.
     */
    public boolean saveDocumentOnDispose() {
        String editingUserId = null;
        boolean result = false;

        synchronized (m_syncAccess) {
            editingUserId = m_connectionStatus.getCurrentEditingUserId();
        }

        final boolean modified = (getNumberOfPendingOperations() > 0);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Save document on dispose initiated");
        }

        // Try to use the session information from the edit user.
        // If a client leaves the document it will be saved, therefore
        // there must be an editing user if we have operations in the
        // message chunk list.
        if (modified && !m_disposed.get() && (null != editingUserId)) {
            ID userId = null;
            try {
                userId = new ID(editingUserId);
            } catch (Exception e) {
                LOG.error("RT connection: Save document on dispose => no user for saving found!", e);
            }

            if ((null != userId) && (m_saveOnDispose.get())) {
                try {
                    Session session = userId.toSession();
                    // No need to check result - we cannot reach any client here
                    flushDocument(session, userId, true, true);
                } catch (OXException e) {
                    LOG.error("RT connection: Flushing document on dispose resulted in exception!", e);
                }
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Save document on dispose - no need to flush document");
            }
        }

        return result;
    }

    /**
     * Make a fail safe revision of the current document. This method is called by a timer
     * thread and therefore must be implemented thread-safe. Attention: There are members
     * which are set to null in onDispose() to return memory to the Java VM as soon as possible.
     * Therefore access to these member must be surrounded by a check for null!
     *
     * @return The error code describing if the fail safe save was successful or not it describes
     * what's the root cause of a possible failure.
     */
    public ErrorCode failSafeSaveDocument() {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        // Check current editing user, can happen that we lost
        // our editing user and then we cannot process the request.
        if ((editingUserId != null) && (!editingUserId.isEmpty())) {
            ID userId = new ID(editingUserId);
            Session session = null;

            // retrieve session information from the edit user
            // if there is no editing user we don't need to make a
            // fail safe save
            try {
                session = userId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: failSafeSaveDocument catches exception while retrieving session from user ID. ", e);
            }

            if (null != session) {

                // make sure we are the only method to trigger the saving process
                if (this.m_saveDocumentInProgress.getAndIncrement() > 0) {
                    // there is a saving in progress so just leave
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Fail safe save triggered: saving document in progress => nothing to do");
                    }
                    this.m_saveDocumentInProgress.decrementAndGet();
                    return errorCode;
                }

                final long startTime = System.currentTimeMillis();

                // close document in every case to write left over operations,
                // if nothing needs to be written, newFileVersion will be the empty string
                try {
                    int osn = -1;
                    int messageChunkListSize = 0;
                    JSONObject outputOperation = null;

                    ArrayList<MessageChunk> currentMessages = null;
                    ResourceManager resourceManager = null;
                    synchronized (m_syncAccess) {
                        resourceManager = m_resourceManager;
                        currentMessages = getMessageChunkClone();
                        osn = m_connectionStatus.getOperationStateNumber();
                    }

                    if ((null != currentMessages) && (null != resourceManager) && !currentMessages.isEmpty()) {
                        final boolean createVersion = this.getCreateVersion();
                        final String revisionless = createVersion ? "with new version" : "revisionless";


                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Fail safe save triggered: Trying to save document " + revisionless);
                        }

                        if (LOG.isDebugEnabled()) {
                            // Additional analyzing code to find root cause of missing operations
                            LOG.debug("RT connection: Fail safe save (before save): Current message chunk list size = " + currentMessages.size());
                            outputOperation = OperationHelper.debugGetLastOperationFromMessageChunkList(currentMessages);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Fail safe save (before save): Last Operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));
                                outputOperation = null;
                            }
                        }

                        final OXDocument oxDocument = new OXDocument(session, m_services, m_fileHelper, m_newDocLoaded, resourceManager, m_storageHelper);
                        oxDocument.setUniqueDocumentId(osn);
                        final SaveResult saveResult = oxDocument.save(getExporter(), resourceManager, currentMessages, userId, !createVersion, false);

                        // read out return values
                        errorCode = saveResult.errorCode;

                        // update Statistics with saved document count (either OPS100 or 15MINS type)
                        if (null != m_fileHelper) {
                            Statistics.handleDocumentEvent(new SaveEvent(
                                m_fileHelper.getDocumentType(),
                                (!createVersion) ? SaveType.OPS_100 : SaveType.OPS_15MINS));
                        }

                        // clean messageChunkList in case the messageChunkList has been flushed
                        if (!errorCode.isError()) {

                            if (createVersion) {
                                // after saving a new version we save revisionless
                                setCreateVersion(false);
                            }

                            int currSize = 0;
                            synchronized (m_syncAccess) {
                                if (null != m_messageChunkList) {
                                    currSize = m_messageChunkList.size();

                                    // Remove all operations which have been saved recently
                                    // leaving just the new ones added during the save process.
                                    int index = currentMessages.size() - 1;

                                    if (index == m_messageChunkList.size() - 1) {
                                        m_messageChunkList.clear(); // optimization
                                    } else if ((index > 0) && ((index + 1) < m_messageChunkList.size())) {
                                        // New operation list must only hold the new operations
                                        // therefore we need to remove the old ones.
                                        ArrayList<MessageChunk> copy = getMessageChunkClone();
                                        List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                        m_messageChunkList.clear();
                                        m_messageChunkList.addAll(newPart);
                                    }
                                    if (LOG.isDebugEnabled()) {
                                        messageChunkListSize = m_messageChunkList.size();
                                        outputOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                                    }
                                }
                            }

                            m_lastSaveTime = new Date();
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: Fail safe save (after save): Before adjustment, message chunk list size = " + currSize);
                                LOG.debug("RT connection: Fail safe save (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                                if (null != outputOperation) {
                                    LOG.debug("RT connection: Fail safe save (after save): First operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));
                                }
                            }
                        } else if (saveResult.errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                            throw new OXDocumentException("", saveResult.errorCode);
                        }
                    } else {
                        LOG.debug("RT connection: Fail safe save: No need to save document");
                    }

                    LOG.debug("RT connection: Fail safe  save: Flushing operations to native document succeeded");
                } catch (OXDocumentException e) {
                    errorCode = e.getErrorcode();
                    LOG.error("RT connection: Fail safe save failed to save document correctly: " + errorCode.getDescription(), e);
                    if (errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                        // In case of a fatal error we clear the message chunk list
                        // to prevent saving a backup file again and again
                        impl_clearMessageChunkList();
                    }
                } catch (Exception e) {
                    LOG.error("RT connection: Exception detected while doing a fail safe save", e);
                    errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                    // setErrorClass provides a clone with changed error class!
                    errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                } finally {
                    m_saveDocumentInProgress.decrementAndGet();

                    long saveDocumentTime = (System.currentTimeMillis() - startTime);
                    LOG.debug("RT connection: Fail safe save, TIME to flushDocument: " + saveDocumentTime + "ms");

                    // in case of an error we send a flush info error to all clients
                    this.impl_sendFlushInfoOnError(errorCode, userId, false);
                }

                // sets the latest error code for this fail safe save operation
                this.setLastFailSafeSaveError(errorCode);

                // check for result and trigger update if save was successful
                // which means (warning or no error detected)
                if (errorCode.getErrorClass() <= ErrorCode.ERRORCLASS_WARNING) {
                    // Trigger update message so clients can update files view
                    m_connectionStatus.setFailSafeSaveDone(true);
                    updateClients(new MessageData(null, syncGetConnectionStatus()), null);
                    m_connectionStatus.setFailSafeSaveDone(false);
                }
            }
        }

        return errorCode;
    }

    /**
     * Sends a update message to all clients of the document except the sender. This method can be used if a update should only be sent to
     * all other clients, because the sender already have all information available.
     *
     * @param messageData The message data to be sent to all clients.
     * @param senderId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     *            This client won't be part of the clients that will receive the update message.
     */
    private void updateClientsExceptSender(MessageData messageData, ID senderId) {
        impl_updateClients(messageData, senderId, true);
    }

    /**
     * Sends a update message to all clients of the document.
     *
     * @param messageData The message data to be sent to all clients.
     * @param fromId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     */
    private void updateClients(MessageData messageData, ID fromId) {
        impl_updateClients(messageData, fromId, false);
    }

    /**
     * Internal method to sends a update message to a set of clients of the document.
     *
     * @param messageData The message data to be sent to all clients.
     * @param fromId The client who sent a request which resulted in this update message. In the normal case this is the editor client.
     * @param exceptSender If set to TRUE the sender/fromId won't be part of the clients that will received the update message.
     */
    private void impl_updateClients(MessageData messageData, ID fromId, boolean exceptSender) {
        final Message updateMessage = new Message();
        final ID senderId = (null != fromId) ? fromId : getId();

        if (null != senderId) {
            updateMessage.setFrom(senderId);
            updateMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                String updateClients = exceptSender ? "updateClientsExceptSender" : "updateClients";
                LOG.debug("RT connection: Handling [" + updateClients + "], originated by: " + senderId.toString());
            }

            try {
                if (exceptSender) {
                    relayToAllExceptSender(updateMessage);
                } else {
                    relayToAll(updateMessage);
                }
            } catch (OXException e) {
                if (exceptSender) {
                    LOG.error("RT connection: [updateClients] catches exception for relayToAllExceptSender.", e);
                } else {
                    LOG.error("RT connection: [updateClients] catches exception for relayToAll.", e);
                }
            }
        }
    }

    /**
     * Checks and sends a flushErrorInfo message to all clients if the result JSON object contains an error, which describes problems on
     * flushing a document to OX Drive.
     *
     * @param errorCode The error code of the flushDocument operation, which could be zero or non-zero (that is the error code).
     * @param fromId The ID of the client that triggered the flushDocument. Must not be null, otherwise the flushErrorInfo message cannot be
     *            sent.
     * @param exceptSender Specifies if the client with fromId should get the flushErrorInfo message or not.
     */
    private void impl_sendFlushInfoOnError(ErrorCode errorCode, ID fromId, boolean exceptSender) {
        final ID senderId = (null != fromId) ? fromId : getId();

        if ((null != senderId) && (errorCode != ErrorCode.NO_ERROR)) {
            JSONObject result = new JSONObject();
            try {
                result.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
                impl_sendFlushInfo(result, senderId, exceptSender);
            } catch (JSONException e) {
                LOG.error("Exception while setting up result for flushErrorInfo notification", e);
            }
        }
    }

    /**
     * Sends a flushErrorInfo message to all clients using the provided JSON object that must contain the error code.
     *
     * @param result The JSONObject directly returned by the flushDocument() call containing the error code.
     * @param fromId The ID of the client that triggered the flushDocument. Must not be null, otherwise the flushErrorInfo message cannot be
     *            sent.
     * @param exceptSender Specifies if the client with fromId should get the flushErrorInfo message or not.
     */
    private void impl_sendFlushInfo(JSONObject result, ID senderId, boolean exceptSender) {
        try {
            final MessageData messageData = new MessageData(null, null, result);
            final Message flushErrorInfoMessage = new Message();

            flushErrorInfoMessage.setFrom(senderId);
            flushErrorInfoMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "flusherrorinfo")).build()));

            if (exceptSender) {
                relayToAllExceptSender(flushErrorInfoMessage);
            } else {
                relayToAll(flushErrorInfoMessage);
            }
        } catch (OXException e) {
            LOG.error("RT connection: [flushErrorInfo] catches exception for relayToAll.", e);
        }
    }

    /**
     * Clears the message chunk list in case of errors where saving a complete document is not possible anymore. E.g. when the filter
     * detects a problem with the operations.
     */
    private void impl_clearMessageChunkList() {
        synchronized (m_syncAccess) {
            if (null != m_messageChunkList) {
                m_messageChunkList.clear();
            }
        }
    }

    /**
     * Implementation method to process a "acquireedit" request from a client. Sending the required "preparelosingeditrights" message to the
     * current editor and setup a timeout thread to handle non-responsive clients.
     *
     * @param session The session of the user who wants to receive edit rights.
     * @param fromId The ID of the user who wants to receive edit rights.
     * @param silent If set to TRUE the method won't send any update nor preparelosingeditrights notification. Used by methods which already
     *            know that switching edit rights can be done immediately.
     * @param force If set to TRUE the method forces to set the provided client as the new editor regardless of other properties. Should be
     *            called carefully as this switches edit rights without any hint to the current editor.
     */
    private void acquireEdit(Session session, ID fromId, boolean silent, boolean forceSwitch) throws OXException {
        Session curSession = session;
        String displayUserName = "";

        if ((null == curSession) && (null != fromId)) {
            try {
                curSession = fromId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: acquireEdit catches exception while retrieving session from sender ID. ", e);
            }
        }

        if ((null != curSession) && (curSession instanceof ServerSession)) {
            final User curUser = ((ServerSession) curSession).getUser();

            if (null != curUser) {
                displayUserName = curUser.getDisplayName();
            }
        }

        if (null != fromId) {

            String wantsEditRightsUserId = "";
            String hasEditRightsUserId = "";

            synchronized (m_connectionStatus) {
                wantsEditRightsUserId = m_connectionStatus.getWantsEditRightsUserId();
                hasEditRightsUserId = m_connectionStatus.getCurrentEditingUserId();
            }

            // Check if user tries to acquire edit rights although she has it already.
            // Can be related to a slow connection and the user tries again and again
            // to acquire the rights.
            if ((null != hasEditRightsUserId) && (hasEditRightsUserId.length() > 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                if (fromId.equals(currentEditorId)) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Ignore [acquireedit] from editor");
                    }
                    return;
                }
            }

            if (((null == hasEditRightsUserId) || (hasEditRightsUserId.length() == 0)) || forceSwitch) {
                String wantsToEdit = "";
                ConnectionStatus connectionStatus = null;

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: acquireedit no current editor found, editor now: " + fromId.toString() + ", wants to edit:" + wantsEditRightsUserId);
                }

                synchronized (m_connectionStatus) {
                    // first acquire edit rights on a non-acquired document can be directly approved
                    m_connectionStatus.setCurrentEditingUser(fromId.toString(), displayUserName);
                    if (wantsEditRightsUserId.equals(fromId.toString())) {
                        // Special case: If user1 acquires edit rights and the editor leaves and afterwards
                        // the user1 again acquires edit rights, user1 will be editor and also the user who
                        // wants to get these rights. So we have to reset wantsToEditRightsUser!
                        m_connectionStatus.setWantsEditRightsUser("", "");
                    }
                    // cancels a possible prepare losing edit rights timer
                    cancelPrepareLosingEditRightsTimer();
                    wantsToEdit = m_connectionStatus.getWantsEditRightsUserId();
                    connectionStatus = (ConnectionStatus) m_connectionStatus.clone();
                }
                if (!silent) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [updateClients] to all clients, editor:" + fromId.toString() + ", wants to edit:" + wantsToEdit);
                    }
                    updateClients(new MessageData(null, connectionStatus), null);
                }
            } else if ((null == wantsEditRightsUserId) || (wantsEditRightsUserId.length() == 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                Message returnMessageLose = null;
                Message returnMessageGet = null;
                MessageData messageData = null;

                synchronized (m_connectionStatus) {
                    // Start "preparelosingeditrights" timeout timer to prevent that we have a document
                    // where the edit rights are pending forever (e.g. the current editor loses connection
                    // while we are switching the edit rights - waiting for canLoseEditRights)
                    cancelPrepareLosingEditRightsTimer();
                    startPrepareLosingEditRightsTimer(session, currentEditorId);

                    // store user who wants to receive the edit rights
                    m_connectionStatus.setWantsEditRightsUser(fromId.toString(), displayUserName);

                    if (!silent) {
                        // Send the message to the client who will lose the edit rights
                        returnMessageLose = new Message();
                        returnMessageGet = new Message();
                        messageData = new MessageData(null, (ConnectionStatus) m_connectionStatus.clone(), null);
                    }
                }

                if (!silent) {
                    if (null == returnMessageGet) {
                        returnMessageGet = new Message();
                    }

                    // send info notification to the client which will receive the edit rights
                    returnMessageGet.setFrom(fromId);
                    returnMessageGet.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "acceptedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [acceptedacquireeditrights] to client requested edit rights, applied from: " + fromId.toString());
                    }

                    relayToID(returnMessageGet, fromId);

                    if (null == returnMessageLose) {
                        returnMessageLose = new Message();
                    }

                    // send message to the client which will lose the edit rights for preparation
                    returnMessageLose.setFrom(currentEditorId);
                    returnMessageLose.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "preparelosingeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [preparelosingeditrights] to client with edit rights, to: " + currentEditorId.toString());
                    }

                    relayToID(returnMessageLose, currentEditorId);
                }
            } else {
                ID wantsEditorRightsId = new ID(wantsEditRightsUserId);

                if (!fromId.equals(wantsEditorRightsId) && !silent) {
                    // Someone else already wants to acquire the edit rights. Therefore we
                    // have to decline this request until the edit rights changed.

                    final Message returnMessage = new Message();
                    final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), null);

                    returnMessage.setFrom(fromId);
                    returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "declinedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [declinedacquireeditrights] to client requested edit rights while there is a switch in progress, applied from: " + fromId.toString());
                        LOG.debug("RT connection: Relaying [declinedacquireeditrights], wantsEditorRights=" + wantsEditRightsUserId + ", editor=" + hasEditRightsUserId);
                    }

                    relayToID(returnMessage, fromId);
                }
            }
        }
    }

    /**
     * Implementation method to handle the response from the current editor client to give up the edit rights. This method is called by the
     * "preparelosingeditrights" timer thread and the RT core. Therefore it must be thread-safe implemented.
     *
     * @param session The session of the user who was notified via "preparelosingeditrights"
     * @param fromId The id who was notified via "preparelosingeditrights".
     * @param clientOSN The osn of the client responding to the "preparelosingeditrights" notification.
     * @param timeout Specifies that the method is called from the timeout callback.
     */
    protected void canLoseEditRights(Session session, ID fromId, int clientOSN, boolean timeout) {

        ConnectionStatus statusToSend = null;
        int serverOSN = clientOSN;
        ID currentEditorId = null;

        synchronized (m_connectionStatus) {

            currentEditorId = new ID(this.m_connectionStatus.getCurrentEditingUserId());
            if (timeout || currentEditorId.equals(fromId)) {
                String newEditorId = m_connectionStatus.getWantsEditRightsUserId();

                this.cancelPrepareLosingEditRightsTimer();

                // Current editor completed to send her latest changes. It's now safe
                // to switch the edit rights.
                m_connectionStatus.setCurrentEditingUser(newEditorId, m_connectionStatus.getWantsEditRightsUserName());
                // Remove the requester so others can acquire the edit rights
                m_connectionStatus.setWantsEditRightsUser("", "");
                // Create copy of the connection status we can use updateClients
                // outside our synchronized and have a consistent state
                if (timeout) {
                    // set inactivity in case of a non-responding client losing edit rights
                    m_connectionStatus.disableActiveUser(fromId.toString(), m_nMaxTimeForSendingLastActions / 1000);
                }
                statusToSend = (ConnectionStatus) m_connectionStatus.clone();
                serverOSN = m_connectionStatus.getOperationStateNumber();

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Switching edit rights to client: " + newEditorId);
                }
            }
        }

        if (statusToSend != null) {
            updateClients(new MessageData(null, statusToSend), null);
        }

        // Check the current server osn and determine if the client losing
        // edit rights is out of sync. Do this only if we don't have a timeout
        // scenario.
        // Additional information:
        // Checking both client and server OSN with a less rigid
        // scheme, as it's possible that a client is behind the last edit client
        // and this should not block transferring edit rights. As the "real",
        // UI related client, edit rights are processed synchronized with the
        // operations this won't result in bad scenarios. Currently (7.4.2) this
        // can result in strange behaviour where the client is switching fast
        // between edit rights bubble messages (if the client catched up with
        // operation processing).
        // This could be fixed using snooping the "message queue" on the client side
        // and prevent to switch between "old" update states but just use the latest
        // one from the queue.
        // Bug fix: 30656 use less rigid check for "canloseeditrights" and client OSN
        if (!timeout && (clientOSN > serverOSN) && (null != currentEditorId)) {
            try {
                impl_sendHangUp(
                    fromId,
                    statusToSend,
                    ErrorCode.HANGUP_INVALID_OSN_DETECTED_ERROR,
                    HANG_UP_ID,
                    "Relaying [hangup] to client that has recently lost edit rights due to inconsitent osn, called from: " + fromId.toString() + " with client-osn: " + String.valueOf(clientOSN) + " server-osn: " + String.valueOf(serverOSN));
            } catch (OXException e) {
                LOG.error("RT connection: Exception while sending [hangup] to editor client due to inconsistent OSN", e);
            }
        }
    }

    /**
     * Adds a resource to the distributed managed file resource, so it can be used by the client via a media url.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The addResource request which includes the resourceid
     * @return A json object containing the result of the addResource request.
     */
    private JSONObject addResource(Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String managedResourceId = jsonRequest.optString("resourceid", "");

            // we got a file id to create a distributed managed file resource from
            if (managedResourceId.length() > 0) {
                final Resource managedResource = m_resourceManager.createManagedResource(managedResourceId);

                if (null != managedResource) {
                    m_resourceManager.addResource(managedResource);
                    LOG.debug("RT connection: addResource added resource " + managedResourceId + "to resourceManager.");

                    try {
                        jsonResult.put("resourceid", managedResource.getManagedId());
                    } catch (JSONException e) {
                        LOG.error("RT connection: addResource - could not retrieve managed resource id");
                    }
                } else {
                    LOG.error("RT connection: addResource - managed resource could not be retrieved from id: " + managedResourceId);
                }
            } else {
                // try to create the resource from the given data
                final String data64 = jsonRequest.optString("dataurl", "");
                byte[] data = null;

                if (null != data64) {
                    final String searchPattern = "base64,";
                    int pos = data64.indexOf(searchPattern);

                    if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                        data = Base64.decode(data64.substring(pos));
                    }
                }

                if ((data != null) && (data.length > 0)) {
                    final String fileName = jsonRequest.optString("filename", "");
                    final String fileExtension = FileHelper.getExtension(fileName);

                    String newExtension = jsonRequest.optString("add_ext", "");

                    // extension to add to resource fileName
                    if (newExtension.length() < 1) {
                        newExtension = fileExtension;
                    }

                    // crc32 of resource data
                    int crc32 = jsonRequest.optInt("add_crc32", 0);

                    if (0 == crc32) {
                        crc32 = Resource.getCRC32(data);
                    }

                    // resource id
                    final long uid = m_resourceManager.addResource(crc32, data);
                    LOG.debug("RT connection: addResource added resource " + Long.toString(uid) + "to resourceManager.");

                    try {
                        String resourceName = null;

                        switch (m_fileHelper.getDocumentFormat()) {
                        case DOCX: {
                            resourceName = "word/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case PPTX: {
                            resourceName = "ppt/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case XLSX: {
                            resourceName = "xl/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case ODT: {
                            resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case ODS: {
                            resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }

                        case NONE:
                        case ODG:
                        case ODP:
                        default: {
                            break;
                        }
                        }

                        if (null != resourceName) {
                            jsonResult.put("added_filename", resourceName);
                        }
                    } catch (Exception e) {
                        LOG.error("RT connection: addResource catches exception adding filename to JSON result. ", e);
                    }
                }
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Provides the stream and mimetype of a requested document.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The request containing the folder/file id and the target format.
     * @return A json object containing the stream and mimetype.
     */
    private JSONObject getDocument(ServerSession session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String documentFormat = jsonRequest.optString("documentformat", "native"); // ["native", "pdf"]
            final String returnType = jsonRequest.optString("returntype", "dataurl"); // ["dataurl", "resourceid"]
            final String source = jsonRequest.optString("source", "file"); //
            final String folder = jsonRequest.optString("folder");
            final String id = jsonRequest.optString("id");
            final String module = jsonRequest.optString("module");
            final String attachment = jsonRequest.optString("attachment");
            ArrayList<MessageChunk> currentMessages = null;
            MailServletInterface mailInterface = null;
            InputStream pdfDocumentStm = null;
            InputStream documentStm = null;

            synchronized (m_syncAccess) {
                currentMessages = getMessageChunkClone();
            }

            try {
                ErrorCode errorCode = ErrorCode.NO_ERROR;

                if (source.equals("mail")) {
                    try {
                        mailInterface = MailServletInterface.getInstance(session);
                        MailPart part = mailInterface.getMessageAttachment(folder, id, attachment, false);

                        if (part != null) {
                            documentStm = part.getInputStream();
                        }
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from mail part. ", e);
                    }
                } else if (source.equals("tasks") || source.equals("calendar") || source.equals("contacts")) {
                    try {
                        documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folder),
                            Integer.parseInt(attachment),
                            Integer.parseInt(module),
                            Integer.parseInt(id),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from task attachment. ", e);
                    }
                } else {
                    OXDocument oxDocument = new OXDocument(session, m_services, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper);
                    OXDocument.ResolvedStreamInfo docStreamInfo = oxDocument.getResolvedDocumentStream(
                        getExporter(),
                        m_resourceManager,
                        currentMessages,
                        null,
                        false,
                        false);

                    if (null != docStreamInfo) {
                        documentStm = (null != docStreamInfo) ? docStreamInfo.resolvedStream : null;
                        errorCode = docStreamInfo.errorCode;
                    } else {
                        errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                    }
                }

                // convert document stream content to PDF, if requested
                if (null != documentStm) {
                    String fileName = jsonRequest.optString("filename", "document");
                    String mimeType = jsonRequest.optString("mimetype", "");

                    if (documentFormat.equals("pdf") && (errorCode == ErrorCode.NO_ERROR)) {
                        final IManager dcManager = m_services.getService(IManager.class);

                        // convert content of document stream to PDF
                        if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                            final HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                            final HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);

                            jobProperties.put(Properties.PROP_INPUT_STREAM, documentStm);
                            jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);

                            try {
                                final CapabilityService capService = m_services.getService(CapabilityService.class);
                                final CapabilitySet capSet = (null != capService) ? capService.getCapabilities(session) : null;

                                // enable sharepoint conversion, depending on the user capability set
                                if ((null != capSet) && capSet.contains("sharepointconversion")) {
                                    jobProperties.put(Properties.PROP_FEATURES_ID, Integer.valueOf(Feature.SHAREPOINT_PDFCONVERSION.id()));
                                }
                            } catch (OXException e) {
                                if (LOG.isErrorEnabled()) {
                                    LOG.error(
                                        "RT connection: getDocument catches exception trying to retrieve 'sharepointconversion' capability.",
                                        e);
                                }
                            }

                            pdfDocumentStm = dcManager.convert("pdf", jobProperties, resultProperties);
                            IOUtils.closeQuietly(documentStm);
                            documentStm = pdfDocumentStm;
                            pdfDocumentStm = null;

                            fileName = fileName + ".pdf";
                            mimeType = "application/pdf";
                        }
                    }

                    if (mimeType.length() < 1) {
                        mimeType = DocFileHelper.getMimeType(fileName);
                    }

                    // set resulting data at JSONObject; document
                    // stream might be null after a conversion error
                    if (null != documentStm) {
                        // add resourceid or dataurl to the result object, depending on the value of returnType
                        OperationHelper.appendData(m_resourceManager, jsonResult, documentStm, returnType, mimeType);

                        // add filename and mimetype properties to the valid result object
                        if (!jsonResult.isEmpty()) {
                            try {
                                jsonResult.put("resourcename", fileName);
                                jsonResult.put("mimetype", mimeType);
                            } catch (JSONException e) {
                                LOG.error("RT connection: getDocument catches exception setting resource name/mimetype to JSON result. ", e);
                            }
                        }

                        IOUtils.closeQuietly(documentStm);
                    }
                }
            } finally {
                IOUtils.closeQuietly(mailInterface);
                IOUtils.closeQuietly(documentStm);
                IOUtils.closeQuietly(pdfDocumentStm);
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Renames the name of the document. This doesn't influence the folder&file id. ATTENTION: The assumption that folder & file id are not
     * influenced are not true. At least there are storage-plugins (e.g. Dropbox) which use a uri to the file/folder. Therefore changing the
     * name invalidate the file id!!!!
     *
     * @param session The session of the user who wants to rename a document.
     * @param jsonRequest The request containing the folder/file id and the new name.
     * @return A json object containing the result of the rename request.
     */
    private RenameResult renameDocument(Session session, JSONObject jsonRequest) {
        String requestedNewFileName = "";
        RenameResult result = new RenameResult();

        try {

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper) && jsonRequest.has(MessagePropertyKey.KEY_FILE_NAME)) {
                final StorageHelper storageHelper = new StorageHelper(m_services, session, m_fileHelper.getFolderId());
                requestedNewFileName = jsonRequest.getString(MessagePropertyKey.KEY_FILE_NAME);
                result = m_fileHelper.renameDocument(session, requestedNewFileName, storageHelper);
            }

            if ((null != result) && (result.errorCode == ErrorCode.NO_ERROR) && (null != result.metaData)) {
                result.jsonResult.put(MessagePropertyKey.KEY_FILE_NAME, result.metaData.getFileName());
                result.jsonResult.put(MessagePropertyKey.KEY_FILE_VERSION, result.metaData.getVersion());
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
                // update current document meta data for access
                synchronized(m_syncAccess) {
                    final String lockedByUser = m_lastKnownMetaData.getLockedByUser();
                    final boolean writeProtected = m_lastKnownMetaData.isWriteProtected();
                    m_lastKnownMetaData = new DocumentMetaData(result.metaData, lockedByUser, writeProtected);
                }
            } else {
                // In case of an error set at least the change result entry
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: renameDocument - failed to rename document to " + requestedNewFileName, e);

            // Fix for bug 29261: Don't return an empty jsonResult which results in a response
            // where the property "hasErrors" is set to "true". This forces the client to immediately
            // show an error bubble and stop accepting server messages.
            try {
                // provide change result to the client
                result.errorCode = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.RENAMEDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: renameDocument catches exception trying to set rename result to JSON object. ", je);
            }
        }

        MessageHelper.checkJSONResult(result.jsonResult);
        return result;
    }

    /**
     * Saves the current document including the latest changes in the message chunk list to a specified folder. The current state of the
     * connection including the message chunk list, editor etc. are not changed.
     *
     * @param session The session of the user that requested to copy the current document to a sepcified folder.
     * @param jsonRequest The request containing the folder/file id, the target folder id, the new file name. The method uses the folder of
     *            the document if the target folder id is missing, if this is not possible, the user folder is used. If the new file name is
     *            missing the original file name is used.
     * @return A json object containing the result of the copy document request.
     */
    private JSONObject copyDocument(final Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();
        String targetFolder = "";
        boolean asTemplate = false;

        try {
            String newFileName = null;

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper)) {
                boolean bCreateDocumentCopy = true;
                targetFolder = jsonRequest.optString(MessagePropertyKey.KEY_TARGET_FOLDER);
                newFileName = jsonRequest.optString(MessagePropertyKey.KEY_FILE_NAME);
                asTemplate = jsonRequest.optBoolean(MessagePropertyKey.KEY_AS_TEMPLATE);

                if ((null == targetFolder) || (targetFolder.length() == 0)) {
                    // There is no target folder
                    boolean bCanUseFolder = false;
                    if (asTemplate) {
                        // templates are always created in the user folder
                    } else {
                        // First we try to use the source folder and check the
                        // permissions. If we cannot write, we take the users
                        // default folder as fallback.
                        final IDBasedFolderAccessFactory folderFactory = m_services.getService(IDBasedFolderAccessFactory.class);
                        final IDBasedFolderAccess folderAccess = ((null != folderFactory) && (null != session)) ? folderFactory.createAccess(session) : null;

                        targetFolder = m_fileHelper.getFolderId();
                        int[] permissions = FolderHelper.getFolderPermissions(folderAccess, targetFolder);
                        if (null != permissions) {
                            // Check folder permissions that we can create a copy
                            // of the document file and write to it.
                            bCanUseFolder = (permissions[FolderHelper.FOLDER_PERMISSIONS_FOLDER] >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER) & (permissions[FolderHelper.FOLDER_PERMISSIONS_WRITE] > FileStoragePermission.NO_PERMISSIONS);
                        }
                    }

                    // Fallback use the user folder to create the document
                    // For template this is the default case!
                    if (!bCanUseFolder) {
                        targetFolder = DocFileHelper.getUserFolderId(session, m_services);
                        if (null == targetFolder) {
                            jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR.getAsJSON());
                            bCreateDocumentCopy = false;
                        }
                    }
                }

                if ((null == newFileName) || (newFileName.length() == 0)) {
                    // We don't have a new file name for the copy of the
                    // document. Therefore we retrieve the original name.
                    try {
                        newFileName = m_fileHelper.getFilenameFromFile(session);
                    } catch (OXException e) {
                        newFileName = null;
                    }
                    if (null == newFileName) {
                        jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_FILENAME_UNKNOWN_ERROR.getAsJSON());
                        bCreateDocumentCopy = false;
                    }
                }

                if (bCreateDocumentCopy) {
                    ErrorCode errorCode = impl_copyDocument(session, targetFolder, newFileName, asTemplate, jsonResult);
                    jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
                }
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: copyDocument - failed to copy document.", e);

            try {
                // provide change result to the client
                jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: copyDocument catches exception trying to set result to JSON object. ", je);
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Creates a copy of the current document including changes within the message chunk list.
     *
     * @param session The session of the user that requested to create a copy of the current document.
     * @param targetFolder The target folder where the document should be stored to. Must not be null or an empty string.
     * @param fileName The file name of the new document without extension. Must not be null or an empty string.
     * @param asTemplate Specifies if the new document should be a template.
     * @param jsonResult Contains the json part of the copy document method. If the result of the method is no error, these values can be
     *            used to retrieve the new file id, folder id, version, mime type and file name.
     * @return The error code of the request. Must be ErrorCode.NO_ERROR if no error occurred, otherwise it describes the problem that
     *         prevented to store the document in the target folder.
     */
    private ErrorCode impl_copyDocument(final Session session, String targetFolder, String fileName, boolean asTemplate, JSONObject jsonResult) {
        ErrorCode errorCode = ErrorCode.COPYDOCUMENT_FAILED_ERROR;

        if ((null != session) && (null != targetFolder) && (null != fileName)) {

            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
            InputStream documentStm = null;
            boolean rollback = false;
            boolean giveUp = false;

            if (null != fileAccess) {

                try {
                    ArrayList<MessageChunk> currentMessages = null;
                    int osn = -1;

                    // make sure that no save operation interrupts our copy document processing
                    this.m_saveDocumentInProgress.getAndIncrement();
                    synchronized (m_syncAccess) {
                        currentMessages = getMessageChunkClone();
                        osn = m_connectionStatus.getOperationStateNumber();
                    }

                    // set up the new file object used to create a new infostore item
                    final File targetFile = new DefaultFile();
                    final File sourceFile = m_fileHelper.getMetaDataFromFile(session, FileStorageFileAccess.CURRENT_VERSION);
                    String extension = FileHelper.getExtension(sourceFile.getFileName());
                    String mimeType = sourceFile.getFileMIMEType();

                    // We want to copy the document and store it as a template
                    // document. Make sure that the document is not already a
                    // docment template.
                    if (asTemplate && !DocumentFormatHelper.isSupportedTemplateFormat(mimeType, extension)) {
                        // the document should be copied and "transformed" into
                        // a document template. We have to find out what format
                        // must be used.
                        HashMap<String, String> templateFormatInfo = DocumentFormatHelper.getTemplateFormatInfoForDocument(
                            mimeType,
                            extension);
                        if (null != templateFormatInfo) {
                            mimeType = templateFormatInfo.get(Properties.PROP_MIME_TYPE);
                            extension = templateFormatInfo.get(Properties.PROP_INPUT_TYPE);
                        } else {
                            // We don't know that template format to "transform"
                            // the source document to. Therefore we have to give
                            // up here and provide an error to the requesting
                            // client.
                            errorCode = ErrorCode.COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR;
                            giveUp = true;
                        }
                    }

                    if (!giveUp) {
                        // use provided values to initialize file object
                        targetFile.setId(FileStorageFileAccess.NEW);
                        targetFile.setFileName(fileName + "." + extension);
                        targetFile.setFolderId(targetFolder);
                        targetFile.setFileMIMEType(mimeType);
                        targetFile.setDescription(sourceFile.getDescription());

                        // Create the new document stream containing the latest changes
                        // from the message chunk list.
                        final OXDocument oxDocument = new OXDocument(session, m_services, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper);
                        oxDocument.setUniqueDocumentId(osn);

                        final OXDocument.ResolvedStreamInfo docStreamInfo = oxDocument.getResolvedDocumentStream(
                            getExporter(),
                            m_resourceManager,
                            currentMessages,
                            null,
                            false,
                            asTemplate);

                        if ((null != docStreamInfo) && (null != docStreamInfo.resolvedStream) && (!docStreamInfo.debugStream)) {
                            documentStm = docStreamInfo.resolvedStream;
                            fileAccess.startTransaction();
                            rollback = true;
                            LOG.trace("RT connection: [copyDocument] saveDocument using meta data: " + ((null != targetFile) ? targetFile.toString() : "null"));
                            fileAccess.saveDocument(targetFile, documentStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                            fileAccess.commit();
                            rollback = false;

                            // return actual parameters of new file
                            jsonResult.put("id", targetFile.getId());
                            jsonResult.put("folder_id", targetFile.getFolderId());
                            jsonResult.put("version", targetFile.getVersion());
                            jsonResult.put("filename", targetFile.getFileName());
                            jsonResult.put("file_mimetype", targetFile.getFileMIMEType());
                            errorCode = ErrorCode.NO_ERROR;
                        } else {
                            // extract error code from the result object
                            errorCode = (null != docStreamInfo) ? docStreamInfo.errorCode : ErrorCode.COPYDOCUMENT_FAILED_ERROR;
                        }
                    }
                } catch (OXException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to copy document.", e);
                    errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR, true);
                } catch (JSONException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to setup result json object");
                } finally {
                    // Roll-back (if needed) and finish
                    if (rollback) {
                        TransactionAwares.rollbackSafe(fileAccess);
                    }

                    TransactionAwares.finishSafe(fileAccess);
                    IOUtils.closeQuietly(documentStm);

                    // decrement counter to enable saving again
                    this.m_saveDocumentInProgress.decrementAndGet();
                }
            }
        }

        return errorCode;
    }

    /**
     * Sends a client request to receive updated view data to the calc engine.
     *
     * @param fromId The ID of the client which requested the view update.
     * @param session The session of the client which requested the view update.
     * @param jsonRequest The request as JSON object containing the required data to process the view update.
     * @return The result of the view update request.
     * @throws OXException
     */
    private JSONObject getSpreadsheetViewUpdate(final ID fromId, Session session, JSONObject jsonRequest) throws OXException {
        JSONObject updateViewOperation = createUpdateViewOperation(jsonRequest);
        JSONObject resultObject = null;

        try {
            resultObject = executeSpreadsheetOperationFailsafe(fromId, updateViewOperation.toString());
        } catch (CalcEngineException e) {
            LOG.error("Exception on getSpreadsheetViewUpdate()", e);
            impl_handleCalcEngineError(session, e.getErrorcode());
            throw new OXException(e);
        } catch (Exception e) {
            LOG.debug("Exception on getSpreadsheetViewUpdate()", e);
        }

        return MessageHelper.checkJSONResult(resultObject);
    }

    /**
     * Tries to process a request for the calc engine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param operation The request to be processed by the calc engine.
     * @return The result of the calc engine.
     * @throws Exception
     */
    protected JSONObject executeSpreadsheetOperationFailsafe(final ID fromId, final String operation) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute operation"));

        final StringBuffer result = new StringBuffer(256);
        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            error = m_calcClient.executeOperation(m_calcDocumentHandle, operation, result);
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                impl_sendHangUp(fromId, null, ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HANG_UP_ALL, null);
            }
            throw new CalcEngineException("error [" + error + "] on execute operation '" + operation + "'.", error);
        }

        final String resultStr = result.toString();
        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_DEBUG).toLog("... result : '" + resultStr + "'"));
        JSONObject resultObject = new JSONObject(resultStr);
        return resultObject;
    }

    /**
     * Tries to process a copy request for the calcengine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param event The copy event to be processed by the calc engine.
     * @throws Exception in case anything went wrong
     */
    void executeSpreadsheetCopyFailSafe(final ID fromId, final CalcEngineClipBoardEvent event) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute copy"));

        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            error = m_calcClient.copy(m_calcDocumentHandle, event);
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                impl_sendHangUp(fromId, null, ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HANG_UP_ALL, null);
            }
            throw new CalcEngineException("error [" + error + "] on execute copy.'", error);
        }
    }

    /**
     * Tries to process a copy request for the calcengine in a fail safe way.
     *
     * @param fromId The ID of the client requested the spreadsheet operation.
     * @param event The copy event to be processed by the calc engine.
     * @throws Exception in case anything went wrong
     */
    void executeSpreadsheetPasteFailSafe(final ID fromId, final CalcEngineClipBoardEvent event) throws Exception {

        final ContextAwareLogHelp aLog = mem_LogHelp();
        aLog.enterContext("doc-handle", m_calcDocumentHandle);

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("execute paste"));

        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
            error = m_calcClient.paste(m_calcDocumentHandle, event);
        } catch (Throwable ex) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... got exception"), ex);
            sendHangUp = true;
        }

        LOG.debug(mem_LogHelp().forLevel(ELogLevel.E_INFO).toLog("... got state : " + error));

        // Currently we handle all errors as fatal therefore send a hangup
        // in case we detect an error case. In the future it would be nice
        // to handle certain errors as not fatal
        if (error != ECalcEngineError.E_NONE) {
            sendHangUp = true;
        }

        if (sendHangUp) {
            LOG.error(mem_LogHelp().forLevel(ELogLevel.E_ERROR).toLog("... hang up all clients"));
            // Don't send any messages to clients if we are in a middle of shutdown process
            if (!isShutdown()) {
                impl_sendHangUp(fromId, null, ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HANG_UP_ALL, null);
            }
            throw new CalcEngineException("error [" + error + "] on execute paste.'", error);
        }
    }

    // /* disabled !
    // * restore not possible by non well designed core ...
    // * refactor the underlying design and reactivate this method
    // */
    // private void restoreSpreadsheetFromOperationCache(final String lastFailedOperation,
    // final StringBuffer resultOfLastOperation) throws Exception {
    //
    // // let exception out !
    // // outside code rely on that .-)
    //
    // if (m_messageChunkList != null) {
    // final int c = m_messageChunkList.size();
    // int i = 0;
    // final String[] restoreOperations = new String[c];
    //
    // for (MessageChunk chunk : m_messageChunkList) {
    // final JSONObject jsonObj = chunk.getOperations();
    // final String jsonStr = jsonObj.toString();
    // restoreOperations[i++] = jsonStr;
    // }
    //
    // LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] start restore of document");
    // final ECalcEngineError error = m_calcClient.restoreDocument (m_calcDocumentHandle, restoreOperations);
    // if (error != ECalcEngineError.E_NONE)
    // throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] restore failed.");
    // }
    //
    // LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] apply last operation (after restore)");
    // resultOfLastOperation.setLength(0); // reset !
    // final ECalcEngineError error = m_calcClient.executeOperation (m_calcDocumentHandle, lastFailedOperation, resultOfLastOperation);
    // if (error != ECalcEngineError.E_NONE)
    // throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] apply of last operation (after restore) - failed.");
    // }

    /**
     * @param requestData
     * @return
     */
    private static JSONObject createUpdateViewOperation(JSONObject requestData) {
        final JSONArray operationsArray = new JSONArray();
        JSONObject operations = new JSONObject();
        try {
            requestData.put("name", "updateView");
            LOG.debug(requestData.toString());
            operationsArray.put(requestData);
            operations.put("operations", operationsArray);
        } catch (JSONException e) {
            LOG.error("Exception on createUpdateViewOperation", e);
        }
        return operations;
    }

    /**
     * Provides the importer for the current document.
     *
     * @return The Importer for the current file helper object
     */
    private IImporter getImporter() {
        IImporter importer = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentFormat()) {
            case DOCX: {
                importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                break;
            }

            case PPTX: {
                importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                break;
            }

            case XLSX: {
                importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                break;
            }

            case ODT:
            case ODS: {
                importer = m_services.getService(com.openexchange.office.odf.Importer.class);
                break;
            }

            case NONE:
            case ODG:
            case ODP:
            default: {
                break;
            }
            }
        }

        return importer;
    }

    /**
     * Provides the exporter for the current document depending on the format.
     *
     * @return The Exporter for the current file helper object
     */
    private IExporter getExporter() {
        IExporter exporter = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentFormat()) {

            case DOCX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.docx.Exporter.class);
                break;
            }

            case PPTX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.pptx.Exporter.class);
                break;
            }

            case XLSX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.xlsx.Exporter.class);
                break;
            }

            case ODT:
            case ODS: {
                exporter = m_services.getService(com.openexchange.office.odf.Exporter.class);
                break;
            }

            case NONE:
            case ODG:
            case ODP:
            default: {
                break;
            }
            }
        }

        return exporter;
    }

    /**
     * Provides thread-safe the current connection status as a shallow clone.
     *
     * @return The current connection status as a shallow clone.
     */
    private ConnectionStatus syncGetConnectionStatus() {
        synchronized (m_connectionStatus) {
            return (ConnectionStatus) m_connectionStatus.clone();
        }
    }

    /**
     * Save the latest changes of the document to the infostore.
     *
     * @param session
     *   The session of the client that requested to store the lastest changes.
     * @param creator
     *   The creator of the document to be written. Can be null which means that
     *   the default owner is not set.
     * @param forceSave
     *   Forces a save even if a concurrent save is in progress. This should only be used in special situations, like a
     *   shutdown and an emergency save is needed.
     * @param finalSave
     *   Specifies if this is the last flush so the filter can do cleanup work.
     * @return
     *   An error code describing if and why the method has failed. In case of success ErrorCode.NO_ERROR is returned.
     */
    private ErrorCode flushDocument(Session session, ID creator, boolean forceSave, boolean finalSave) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        long startTime = System.currentTimeMillis();

        if (null != session) {
            LOG.debug("RT connection: flushDocument called.");

            // We only want to have just ONE save document process with
            // only ONE exception: if forceSave is set (only if the connection
            // is going to be disposed) we have to write the latest state.
            int saveCount = this.m_saveDocumentInProgress.getAndIncrement();
            if ((saveCount > 0) && !forceSave) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: flushDocument/failSafeSave in progress, no need to flush");
                }
                this.m_saveDocumentInProgress.getAndDecrement();
                return errorCode;
            }

            // From here we can be sure that just one thread is active
            boolean pendingOperations = false;
            ArrayList<MessageChunk> currentMessages = null;
            int osn = -1;
            int currSize = 0;

            // make a shallow copy of the message chunk list to have a defined
            // point where we save the document. handleApplyActions can be
            // called from a different thread and manipulates the message chunk
            // list, too.
            synchronized (m_syncAccess) {
                pendingOperations = !m_messageChunkList.isEmpty();
                if (pendingOperations) {
                    currentMessages = getMessageChunkClone();
                }
                osn = m_connectionStatus.getOperationStateNumber();
            }

            if (!pendingOperations) {
                // no pending operations to save -> bail out early
                LOG.debug("RT connection: No changes in message chunk list found => no need to flush document");
                this.m_saveDocumentInProgress.getAndDecrement();
                return errorCode;
            }

            // for debugging purposes we log some data about the current
            // pending operations and server osn.
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Flushing operations, current message chunk list size = " + currentMessages.size());
                LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                JSONObject outputOperation = OperationHelper.debugGetLastOperationFromMessageChunkList(currentMessages);
                if (null != outputOperation) {
                    LOG.debug("RT connection: Flushing operations (before save): last operation in current message chunk list = " + OperationHelper.operationToString(outputOperation));
                }
            }

            // if we encountered an error bail out and provide the error code
            if (errorCode != ErrorCode.NO_ERROR) {
                this.m_saveDocumentInProgress.getAndDecrement();
                return errorCode;
            }

            // we now try to save the document using the copied pending operations
            final boolean createVersion = this.getCreateVersion();

            OXDocument oxDocument = null;
            SaveResult saveResult = null;
            try {
                oxDocument = new OXDocument(session, m_services, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper);
                oxDocument.setUniqueDocumentId(osn);
                errorCode = oxDocument.getLastError();

                if (errorCode == ErrorCode.NO_ERROR) {
                    saveResult = oxDocument.save(getExporter(), m_resourceManager, currentMessages, creator, !createVersion, finalSave);
                    if (saveResult.errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                        throw new OXDocumentException("Fatal error detected while saving document", saveResult.errorCode);
                    } else {
                        // take over non-fatal errors/warnings to deliver them to the clients
                        errorCode = saveResult.errorCode;
                    }
                } else {
                    // We were not able to read the document bail out throwing
                    // exception with the provided error code.
                    throw new OXDocumentException("Error reading document stream via OXDocuments", errorCode);
                }
            } catch (OXDocumentException e) {
                LOG.error("Exception catched while flushing document", e);
                // Document exception detected -> this is a fatal error
                errorCode = e.getErrorcode();
                if (errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                    // Decrement the save document counter although saving
                    // won't help
                    this.m_saveDocumentInProgress.getAndDecrement();
                    // Fatal error detected on save which means that a backup file
                    // has been created. Further saving won't help, therefore we
                    // clear the message chunk list
                    impl_clearMessageChunkList();
                    return errorCode;
                }
            } catch (Exception e) {
                LOG.error("Exception catched while flushing document", e);
                // Exception detected -> this is a fatal error
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                this.m_saveDocumentInProgress.getAndDecrement();
                // Fatal error detected on save which means that a backup file
                // has been created. Further saving won't help, therefore we
                // clear the message chunk list
                impl_clearMessageChunkList();
                return errorCode;
            } finally {
                // if we encountered an error just bail out here
                if (errorCode != ErrorCode.NO_ERROR) {
                    this.m_saveDocumentInProgress.getAndDecrement();
                    return errorCode;
                }
            }

            // saving was successfull - some more cleanup is needed
            try {
                // clean messageChunkList in case the messageChunkList has been flushed
                int messageChunkListSize = 0;
                JSONObject outputOperation = null;

                if (createVersion) {
                    // After a new version we want to save revisionless
                    // until an editor leaves the document.
                    setCreateVersion(false);
                }

                synchronized (m_syncAccess) {
                    currSize = m_messageChunkList.size();

                    // update sync status to the latest saved document
                    this.m_syncStatus.updateSyncInfo((saveResult != null) ? saveResult.version : null, osn);

                    // Remove all operations which have been saved recently
                    // leaving just the new ones added during the save process.
                    int index = currentMessages.size() - 1;

                    if (index == m_messageChunkList.size() - 1) {
                        m_messageChunkList.clear(); // optimization
                    } else if ((index >= 0) && ((index + 1) < m_messageChunkList.size())) {
                        // New operation list must only hold the new operations
                        // therefore we need to remove the old ones.
                        ArrayList<MessageChunk> copy = getMessageChunkClone();
                        List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                        m_messageChunkList.clear();
                        m_messageChunkList.addAll(newPart);
                    }
                    if (LOG.isDebugEnabled()) {
                        messageChunkListSize = m_messageChunkList.size();
                        outputOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                    }
                    m_lastSaveTime = new Date();
                    osn = m_connectionStatus.getOperationStateNumber();
                }

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Flushing operations completed.");
                    LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                    LOG.debug("RT connection: Flushing operations (after save): Before adjustment, message chunk list size = " + currSize);
                    LOG.debug("RT connection: Flushing operations (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                    if (null != outputOperation) {
                        LOG.debug("RT connection: Flushing operation (after save): First operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));
                    }
                }

                // update Statistics with saved document count (standard CLOSE type)
                if (null != m_fileHelper) {
                    Statistics.handleDocumentEvent(new SaveEvent(m_fileHelper.getDocumentType(), SaveType.CLOSE));
                }
            } catch (Throwable e) {
                // Exception detected while updating the message chunk list
                // This will likely result in a inconsistency between document file
                // and operations, therefore stop the clients to make further changes.
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                LOG.error("RT connection: flushDocument catched exception while cleaning up - state could be inconsistent!", e);
            }
        } else {
            // no session - therefore saving is not possible
            LOG.error("RT connection: flushDocument called without session - saving is not possible");
            return errorCode;
        }

        // decrement the atomic integer
        m_saveDocumentInProgress.decrementAndGet();

        // logging time for flushing the document
        long flushDocumentTime = (System.currentTimeMillis() - startTime);
        if (flushDocumentTime > MAX_TIME_FOR_FLUSHDOCUMENT) {
            LOG.warn("RT connection: TIME to flushDocument exceeded client timeout: " + flushDocumentTime + "ms");
        } else {
            LOG.debug("RT connection: TIME to flushDocument: " + flushDocumentTime + "ms");
        }

        return errorCode;
    }

    /**
     * Starts a new "preparelosingeditrights" timer thread which ensures that a non-responsive client won't block switchting edit rights.
     *
     * @param session The session of the current editor client
     * @param id The ID of the current editor client.
     */
    private void startPrepareLosingEditRightsTimer(Session session, ID id) {
        TimerService timerService = m_services.getService(TimerService.class);

        if ((timerService != null) && (m_nMaxTimeForSendingLastActions > 0)) {
            // zero or negative values mean there is no time out at all
            m_prepareLosingEditRightsTimer = timerService.schedule(
                new PrepareLosingEditRightsTimeoutRunnable(this, session, id),
                m_nMaxTimeForSendingLastActions);
        }
    }

    /**
     * Cancels a running "preparelosingeditrights" timer. Won't do anything if there is no preparelosingeditrights timer running.
     */
    private void cancelPrepareLosingEditRightsTimer() {
        // Cancel possible timeout timer
        if (m_prepareLosingEditRightsTimer != null) {
            m_prepareLosingEditRightsTimer.cancel();
            m_prepareLosingEditRightsTimer = null;
        }
    }

    /**
     * Print all client user ids for debug purpose.
     *
     * @param connectionStatus The connection status where the active clients are stored within.
     */
    private static void debugPrintActiveClients(ConnectionStatus connectionStatus) {
        if (null != connectionStatus) {
            JSONArray activeUsers = connectionStatus.getActiveUsers();
            for (int i = 0; i < activeUsers.length(); i++) {
                JSONObject activeUser = activeUsers.optJSONObject(i);
                if (null != activeUser) {
                    String userId = activeUser.optString("userId", null);
                    LOG.debug("RT connection: Member = " + ((null != userId) ? userId : "unknown"));
                }
            }
        }
    }

    /**
     * Provides the folder and file id that this connection is associated with.
     *
     * @return A string which contains the folder and file id separated by a colon. If an id is unknown "n/a" is used instead.
     */
    private String getDocumentFolderAndFileId() {
        String folderFileId = "n/a.n/a";

        if (null != m_fileHelper) {
            final String folderId = m_fileHelper.getFolderId();
            final String fileId = m_fileHelper.getFileId();

            folderFileId = (folderId == null) ? "n/a" : folderId;
            folderFileId += ".";
            folderFileId += (fileId == null) ? "n/a" : fileId;
        }

        return folderFileId;
    }

    /**
     * Provides an OpenEvent instance to be used for administrative purposes like VM surveillance.
     *
     * @return An OpenEvent instance describing the current document.
     */
    private OpenEvent getOpenEvent(final DocumentMetaData docMetaData) {
        OpenEvent openEvent = null;

        if (null != m_fileHelper) {
            final DocumentFormat documentFormat = m_fileHelper.getDocumentFormat();

            if (null != documentFormat) {
                OpenType documentOpenType = OpenType.NONE;
                final Date docCreated = docMetaData.getCreated();
                final long docExistsForMilliseconds = (null != docCreated) ? (System.currentTimeMillis() - docCreated.getTime()) : Long.MAX_VALUE;

                // mark document type as binary only if it has just been converted
                if (docMetaData.isBinaryConverted() && (docExistsForMilliseconds <= MAX_TIME_FOR_BINARY_CONVERSION_CONSIDERATION)) {
                    documentOpenType = OpenType.BINARY;
                } else {
                    switch (documentFormat) {
                    case DOCX:
                    case XLSX:
                    case PPTX: {
                        documentOpenType = OpenType.MSX;
                        break;
                    }

                    case ODT:
                    case ODS:
                    case ODP: {
                        documentOpenType = OpenType.ODF;
                        break;
                    }

                    case NONE:
                    case ODG:
                    default: {
                        documentOpenType = OpenType.NONE;
                        break;
                    }
                    }
                }

                if (OpenType.NONE != documentOpenType) {
                    openEvent = new OpenEvent(m_fileHelper.getDocumentType(), documentOpenType, docMetaData.getFileSize());
                }
            }
        }

        return openEvent;
    }

    /**
     * Provides a context aware logging instance to used for logging messages.
     *
     * @return The context aware logging instance.
     */
    private ContextAwareLogHelp mem_LogHelp() {
        try {
            if (m_logHelp == null) {
                m_logHelp = new ContextAwareLogHelp(LOG);
            }
            return m_logHelp;
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Sets the create version state to the provided value in a thread-safe way.
     *
     * @param newValue The new value of the create version state.
     * @return The old value of the create version state.
     */
    synchronized private boolean setCreateVersion(boolean newValue) {
        boolean oldValue = m_createVersion;
        m_createVersion = newValue;
        return oldValue;
    }

    /**
     * Retrieves the create version state in a thread-safe way.
     *
     * @return The current value of the create version state.
     */
    synchronized public boolean getCreateVersion() {
        return m_createVersion;
    }

    /**
     * Determines if the asynchronous loading has been completed or not.
     *
     * @return TRUE if the asynchronous loading has been completed otherwise FALSE.
     */
    synchronized public boolean isAsyncSpreadsheetLoaded() {
        return m_asyncSpreadsheetLoaded;
    }

    /**
     * Sets the asynchronous loading state.
     *
     * @param set Sets the asynchronous loading state accordingly.
     */
    synchronized void setAsyncSpreadsheetLoaded(boolean set) {
        m_asyncSpreadsheetLoaded = set;
    }

    /**
     * Returns if the connection instance has been disposed by the RT-framework or not.
     *
     * @return TRUE if RT called the onDispose() method or FALSE if not. The connection
     */
    public boolean isDisposed() {
        return this.m_disposed.get();
    }

    /**
     * Checks if the connection instance has been disposed and throws a RuntimeException if true, otherwise nothing happens.
     *
     * @throws RuntimeException In case of a disposed connection.
     */
    private void checkForDisposed() throws RuntimeException {
        if (isDisposed()) {
            throw new RuntimeException("Connection instance has already been disposed - access not allowed");
        }
    }

    /**
     * Determine whether we are in a shutdown process or not.
     *
     * @return TRUE if the instance was informed by the ConnectionComponent that a shutdown has been requested, FALSE if not.
     */
    public boolean isShutdown() {
        return this.m_shutdown.get();
    }

    /**
     * Sets the shutdown state indicating that we are in a shutdown process where nothing should be done.
     */
    public void setShutdown() {
        this.m_shutdown.getAndSet(true);
    }

    /**
     * Provides a clone of the current message chunk list.
     *
     * Attention: Must be synchronized with m_messageChunkList form outside.
     *
     * @return A clone of the current message chunk list.
     */
    @SuppressWarnings("unchecked")
    public ArrayList<MessageChunk> getMessageChunkClone() {
        ArrayList<MessageChunk> messageChunkList = null;

        if (null != m_messageChunkList) {
            messageChunkList = (ArrayList<MessageChunk>) m_messageChunkList.clone();
        }
        return messageChunkList;
    }

    // - Members ----------------------------------------------------------------

    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(Connection.class);

    static private final ActionHandler m_handler = new ActionHandler(Connection.class);

    static private final long ACTIONS_UNTIL_FORCE_UPDATE = 20;

    static private final long MAX_TIME_FOR_FLUSHDOCUMENT = 20000; // time in milliseconds

    static private final long MAX_TIME_FOR_BINARY_CONVERSION_CONSIDERATION = 30000; // time in milliseconds

    static private final int DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS = 20; // default in seconds

    static final boolean HANG_UP_ALL = true;

    static final boolean HANG_UP_ID = false;

    private ServiceLookup m_services = null;

    private DocFileHelper m_fileHelper = null;

    private DocumentMetaData m_lastKnownMetaData = null;

    private StorageHelper m_storageHelper = null;

    private ResourceManager m_resourceManager = null;

    private ArrayList<MessageChunk> m_messageChunkList = new ArrayList<MessageChunk>();

    private JSONObject m_uiOperations = null;

    private boolean m_asyncSpreadsheetLoaded = false;

    private boolean m_newDocLoaded = false;

    private String m_firstJoinedID = "";

    private final ConnectionStatus m_connectionStatus = new ConnectionStatus();

    private Sync m_syncStatus = null;

    private ICalcEngineClient m_calcClient = null;

    private String m_calcDocumentHandle;

    private boolean m_createVersion = true;

    private final AtomicInteger m_saveDocumentInProgress = new AtomicInteger();

    private final AtomicBoolean m_syncAccess = new AtomicBoolean();

    private final AtomicBoolean m_disposed = new AtomicBoolean();

    private final AtomicBoolean m_shutdown = new AtomicBoolean();

    private final AtomicBoolean m_saveOnDispose = new AtomicBoolean();

    private Date m_lastSaveTime = null;

    private ErrorCode m_lastFailSafeSaveError = ErrorCode.NO_ERROR;

    private Date m_startupTime = null;

    private ScheduledTimerTask m_prepareLosingEditRightsTimer = null;

    private long m_threadId = 0;

    private long m_nActionsReceived = 0;

    private int m_nMaxTimeForSendingLastActions = (DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS * 1000);

    private ContextAwareLogHelp m_logHelp = null;

    private AsyncLoadRequestQueue m_loadRequestQueue = null;

    private ApplicationType m_appType = ApplicationType.APP_NONE;

}
