/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.doc.api;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

  /**
   * Enumeration for ODF specific JSON-RPC request parameters.
   *
   */
  public enum ParameterType {

    // Commonly used parameters.
    WAVE_ID("waveId", String.class),
    WAVELET_ID("waveletId", String.class),
    BLIP_ID("blipId", String.class),
//
//    // Operation specific parameters.
//    ANNOTATION("annotation", Annotation.class),
//    BLIP_AUTHOR("blipAuthor", String.class),
//    BLIP_CREATION_TIME("blipCreationTime", Long.class),
//    BLIP_DATA("blipData", BlipData.class),
//    BLIPS("blips", Map.class),
//    CAPABILITIES_HASH("capabilitiesHash", String.class),
//    CHILD_BLIP_ID("childBlipId", String.class),
//    CONTENT("content", String.class),
//    DATADOC_NAME("datadocName", String.class),
//    DATADOC_VALUE("datadocValue", String.class),
//    DATADOC_WRITEBACK("datadocWriteback", String.class),
//    ELEMENT("element", Element.class),
//    FETCH_PROFILES_REQUEST("fetchProfilesRequest", FetchProfilesRequest.class),
//    FETCH_PROFILES_RESULT("fetchProfilesResult", FetchProfilesResult.class),
//    INDEX("index", Integer.class),
//    LANGUAGE("language", String.class),
//    MESSAGE("message", String.class),
//    MODIFY_ACTION("modifyAction", DocumentModifyAction.class),
//    MODIFY_HOW("modifyHow", String.class),
//    MODIFY_QUERY("modifyQuery", DocumentModifyQuery.class),
//    NAME("name", String.class),
//    NEW_BLIP_ID("newBlipId", String.class),
//    NUM_RESULTS("numResults", Integer.class),
//    PARTICIPANT_ID("participantId", String.class),
//    PARTICIPANT_PROFILE("participantProfile", ParticipantProfile.class),
//    PARTICIPANT_ROLE("participantRole", String.class),
//    PARTICIPANTS_ADDED("participantsAdded", List.class),
//    PARTICIPANTS_REMOVED("participantsRemoved", List.class),
//    PROTOCOL_VERSION("protocolVersion", String.class),
//    PROXYING_FOR("proxyingFor", String.class),
//    QUERY("query", String.class),
//    RANGE("range", Range.class),
//    SEARCH_RESULTS("searchResults", SearchResult.class),
//    STYLE_TYPE("styleType", String.class),
//    THREADS("threads", Map.class),
//    WAVELET_DATA("waveletData", WaveletData.class),
//    WAVELET_TITLE("waveletTitle", String.class),
//    RAW_DELTAS_FROM_VERSION("rawDeltasFromVersion", Long.class),
//    RETURN_RAW_SNAPSHOT("returnRawSnapshot", Boolean.class),
//    RAW_SNAPSHOT("rawSnapshot", List.class),
    RAW_DELTAS("rawDeltas", List.class);

    private static final Logger LOG = Logger.getLogger(ParameterType.class.getName());

    private static final Map<String, ParameterType> reverseLookupMap =
        new HashMap<String, ParameterType>();

    static {
      for (ParameterType property : ParameterType.values()) {
        if (reverseLookupMap.containsKey(property.key)) {
          LOG.warning("Parameter with key " + property.key + " already exist.");
        }
        reverseLookupMap.put(property.key, property);
      }
    }

    private final String key;
    private final Class<? extends Object> clazz;

    private ParameterType(String key, Class<? extends Object> clazz) {
      this.key = key;
      this.clazz = clazz;
    }

    /**
     * Returns the string key to access the property.
     *
     * @return A string key to access the property.
     */
    public String key() {
      return key;
    }

    /**
     * Returns the {@link Class} object that represents the type of this
     * property.
     *
     * @return A {@link Class} object that represents the type of this property.
     */
    public Class<? extends Object> clazz() {
      return clazz;
    }

    /**
     * Returns a {@link ParamsProperty} enumeration that has the given key.
     *
     * @param key The method name of a property.
     * @return An {@link ParamsProperty} that has the given key.
     */
    public static ParameterType fromKey(String key) {
      return reverseLookupMap.get(key);
    }
  }
