/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.logging.Logger;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElementBase;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderRowsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowsElement;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * A MultiCoomponent uses a single XML element to represent
 * multipmCurrentTablePartElementle components. This container can be used for
 * spreadsheet row and cell components using repeated elements via an attribute.
 *
 * @author svante.schubertATgmail.com
 */
class TableTraversal {

    /**
     * One traversal for each table
     */
    TableTraversal(TableTableElement tableElement) {
        mTableRootElement = tableElement;
        mFileDom = (OdfFileDom) tableElement.getOwnerDocument();
        resetTableChild();
    }
    private static final Logger LOG = Logger.getLogger(TableTraversal.class.getName());

    private int mColumnStartNo;
    private int mColumnEndNo;
    private int mColumnMaxNo;
    private int mCurrentColumnNo;
    private int mRowStartNo;
    private int mRowEndNo;
    private int mCurrentRowNo;
    private TableChange mTableChange;
    private OdfFileDom mFileDom;
    TableTableElement mTableRootElement;
    private OdfStylableElement mCurrentTablePartElement;
    private OdfStylableElement mNextTablePartElement;

//	private OdfStylableElement getFirstTablePart(TableTableElement tableElement) {
//		Node nextElement = tableElement.getFirstChildElement();
//		// ensure that mNextTablePartElement is an element
//		while (nextElement != null && !(nextElement instanceof OdfStylableElement)) {
//			nextElement = nextElement.getNextSibling();
//		}
//		return (OdfStylableElement) nextElement;
//	}
    private OdfStylableElement getNextTablePart(Node currentNode) {
        OdfStylableElement nextTablePart = null;
        Node lastNode = currentNode;
        Node nextNode = currentNode.getNextSibling();
        // ensure that mNextTablePartElement is an stylable element
        while (nextNode != null && !(nextNode instanceof OdfStylableElement)) {
            // traverse deeper through the grouping elements
            if (nextNode instanceof TableTableColumnGroupElement
                || nextNode instanceof TableTableHeaderColumnsElement
                || nextNode instanceof TableTableColumnsElement
                || nextNode instanceof TableTableRowGroupElement
                || nextNode instanceof TableTableHeaderRowsElement
                || nextNode instanceof TableTableRowsElement) {
                getNextTablePart(nextNode.getFirstChild());
            }
            lastNode = nextNode;
            nextNode = nextNode.getNextSibling();
        }
        if (nextNode == null) {
            Node parent = lastNode.getParentNode();
            // make sure that the end of the table is reached
            if (parent != null && !(parent instanceof TableTableElement)) {
                // otherwise go up from a grouping element
                getNextTablePart(parent);
            }
        }
        // make sure there is no null converted
        if (nextNode != null) {
            nextTablePart = (OdfStylableElement) nextNode;
        }
        return nextTablePart;
    }

    private void resetTableChild() {
        OdfElement tableChild = mTableRootElement.getFirstChildElement();
        while (tableChild != null && !(tableChild instanceof OdfStylableElement)) {
            tableChild = getNextTablePart(mTableRootElement.getFirstChildElement());
        }
        if (tableChild != null && tableChild instanceof OdfStylableElement) {
            mCurrentTablePartElement = (OdfStylableElement) tableChild;
        }
    }

    /**
     * Traverse a table/spreadsheet to dispatch each element of the demanded
     * range to the mTableChange function (if necessary expand the
     * table/spreadsheet)
     *
     * NOTE: Currently Group & Header elements are not being dispatched to the
     * change.. It would be easier to handle the ODF XML if those would have
     * been metadata instead of XML (e.g. attached via attributes)
     */
    void findRange(int rowStartNo, int rowEndNo, int columnStartNo, int columnEndNo, TableChange change) {
        mTableChange = change;
        mRowStartNo = rowStartNo;
        mRowEndNo = rowEndNo;
        mColumnStartNo = columnStartNo;
        mColumnEndNo = columnEndNo;
        resetTableChild();
        mCurrentRowNo = -1;
        mCurrentColumnNo = -1;
        findRange();
    }

    /**
     * Traverse a table/spreadsheet to dispatch each element of the demanded
     * range to the mTableChange function (if necessary expand the
     * table/spreadsheet)
     *
     * NOTE: Currently Group & Header elements are not being dispatched to the
     * change.. It would be easier to handle the ODF XML if those would have
     * been metadata instead of XML (e.g. attached via attributes)
     */
    private void findRange() {
        int maxColumnCount = 1;
        Integer cellsToExpand = null;
        boolean isColumnChangeFinished = false;
        OdfStylableElement previousTablePartElement = null;
        while (mCurrentTablePartElement != null) {
            mNextTablePartElement = this.getNextTablePart(mCurrentTablePartElement);
            // applies the action on the column (split if required)
            if (!isColumnChangeFinished) {
                if (mCurrentTablePartElement instanceof TableTableColumnElement) {
                    if (mTableChange.effectsColumns()) {
                        mCurrentColumnNo++;
                        mCurrentColumnNo = considerChange((TableTableColumnElement) mCurrentTablePartElement, mNextTablePartElement, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, mTableChange);
                        // traverses all column container
                    } else {
                        mCurrentColumnNo += ((TableTableColumnElement) mCurrentTablePartElement).getRepetition();
                    }
                } else if (mCurrentTablePartElement instanceof TableTableRowElement) { // the last column existing column was reached
                    // if the rows are reached, but not the columnStart, the gap has to be filled
                    if (mCurrentColumnNo >= 0 && mCurrentColumnNo < mColumnEndNo) {
                        // EXPAND TABLE COLUMNS
                        if (mTableChange.effectsColumns()) {
                            // if not even the start position was reached
                            if (mCurrentColumnNo + 1 < mColumnStartNo) {
                                int gapSize = mColumnStartNo - mCurrentColumnNo - 1;
                                // expand an existing repeating empty element...
                                if (previousTablePartElement instanceof TableTableColumnElement && isEmptyOrRepeating((TableTableColumnElement) previousTablePartElement)) {
                                    TableTableColumnElement existingColumn = (TableTableColumnElement) previousTablePartElement;
                                    existingColumn.setTableNumberColumnsRepeatedAttribute(existingColumn.getTableNumberColumnsRepeatedAttribute() + gapSize);
                                } else {
                                    // or create
                                    TableTableColumnElement columnGap = newColumnElement(gapSize);
                                    mTableRootElement.insertBefore(columnGap, mCurrentTablePartElement);
                                }
                                mCurrentColumnNo = mColumnStartNo - 1;
                            }
                            // if the currentposition is inbetween start and end position
                            TableTableColumnElement newColumnElement = newColumnElement(mColumnEndNo - mCurrentColumnNo);
                            mTableRootElement.insertBefore(newColumnElement, mCurrentTablePartElement);
                            mCurrentColumnNo++; // moved to the start
                            mCurrentColumnNo = considerChange(newColumnElement, mNextTablePartElement, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, mTableChange);
                        } else {
                            // if not even the start position was reached
                            if (mCurrentColumnNo + 1 < mColumnStartNo || mColumnEndNo - mCurrentColumnNo > 0) {
                                TableTableColumnElement newColumnElement = newColumnElement(mColumnEndNo - mCurrentColumnNo);
                                mTableRootElement.insertBefore(newColumnElement, mCurrentTablePartElement);
                                mCurrentColumnNo = mColumnEndNo;
                            }
                        }
                    }
                    // start counting with zero
                    this.setColumnMax(mCurrentColumnNo);
                    maxColumnCount = mCurrentColumnNo + 1;
                    isColumnChangeFinished = true;
                }
            }
            if (mCurrentTablePartElement instanceof TableTableRowElement) {
                mCurrentRowNo++;
                int additionalRows = ((TableTableRowElement) mCurrentTablePartElement).getRepetition() - 1;
                // REACHED = the start range was reached (or exceeded)..
                boolean withinRange = (mRowStartNo <= (mCurrentRowNo + additionalRows) && ((mCurrentRowNo <= mRowEndNo) || mRowEndNo < 0));
                // cell style being added in conjunction with column style will be added as default cell style to the column
                if (withinRange && (mTableChange.effectsRows() || (mTableChange.effectsCells() && (!mTableChange.effectsColumns() || mTableChange.isColumnDeletion())))) {
                    mCurrentColumnNo = 0; // start the row from the first column
                    mCurrentRowNo = considerChange((TableTableRowElement) mCurrentTablePartElement, mNextTablePartElement, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, mTableChange);
                } else {
                    // FILL UP CELLS IN ROW
                    Row rowComponent = (Row) ((TableTableRowElement) mCurrentTablePartElement).getComponent();
// ToDo: Optimistical PerformanceImprovent:
//                  if (cellsToExpand == null) {
                    cellsToExpand = maxColumnCount - rowComponent.size();
//                    }
                    if (cellsToExpand > 0) {
                        Node lastChild = mCurrentTablePartElement.getLastChild();

                        // expand an existing repeating empty element...
                        if (lastChild instanceof TableTableCellElement && isEmptyOrRepeating((TableTableCellElement) lastChild)) {
                            TableTableCellElement existingCell = (TableTableCellElement) lastChild;
                            existingCell.setTableNumberColumnsRepeatedAttribute(existingCell.getTableNumberColumnsRepeatedAttribute() + cellsToExpand);

                        } else {
                            TableTableCellElement cellContent = newCellElement(cellsToExpand);
                            Component.createComponent(rowComponent, cellContent);
                            mCurrentTablePartElement.appendChild(cellContent);
                        }

                    }
                    mCurrentRowNo += additionalRows; // as counting starts with 0 not 1
                }
            }
            previousTablePartElement = mCurrentTablePartElement;
            mCurrentTablePartElement = mNextTablePartElement;
        }
        // if the end of table is reached, but not the rowStart, the gap has to be filled
        if (mRowEndNo > 0 && mCurrentRowNo >= 0 && mCurrentRowNo < mRowEndNo && (mTableChange.effectsRows() || mTableChange.effectsCells())) {
            // EXPAND TABLE ROWS
            // if not even the start position was reached
            TableTableCellElement cellContent;
            if (mCurrentRowNo + 1 < mRowStartNo) {
                TableTableRowElement rowGap = newRowElement(mRowStartNo - mCurrentRowNo - 1);
                Component.createComponent(mTableRootElement.getComponent(), rowGap);
                cellContent = newCellElement(maxColumnCount);
                Component.createComponent(rowGap.getComponent(), cellContent);
                rowGap.appendChild(cellContent);
                mTableRootElement.appendChild(rowGap);
                mCurrentRowNo = mRowStartNo - 1;
            }
            // if the currentposition is inbetween start and end position
            TableTableRowElement newRowElement = newRowElement(mRowEndNo - mCurrentRowNo);
            Component.createComponent(mTableRootElement.getComponent(), newRowElement);
            cellContent = newCellElement(maxColumnCount);
            Component.createComponent(newRowElement.getComponent(), cellContent);
            newRowElement.appendChild(cellContent);
            mTableRootElement.appendChild(newRowElement);
            mCurrentRowNo++;
            // expand rows for default table cells, but do not apply any cell style to the cells within the rows
            if (!mTableChange.effectsColumns()) {
                considerChange(newRowElement, null, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, mTableChange);
            }
        }
    }

    private boolean isEmptyOrRepeating(OdfStylableElement emptyElement) {
        boolean isEmptyElement = false;
        NamedNodeMap attrs = emptyElement.getAttributes();
        // if there is no attribute..
        if (attrs.getLength() == 0) {
            isEmptyElement = true;
        } else if (attrs.getLength() == 1) {
            // if columns repeated is the only attribute
            isEmptyElement = (null != attrs.getNamedItemNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
        }
        if (emptyElement.hasChildNodes()) {
            isEmptyElement = false;
        }
        return isEmptyElement;
    }

    private void traverseRow(Component rowComponent, Node child, int mRowStartNo, int mRowEndNo, int mCurrentRowNo, int mColumnStartNo, int mColumnEndNo, int mCurrentColumnNo, TableChange change) {
        OdfElement lastElement = null;
        while (child != null) {
            Node nextElement = child.getNextSibling();
            // drop text nodes, go to the next element node
            while (nextElement != null && !(nextElement instanceof OdfElement)) {
                nextElement = nextElement.getNextSibling();
            }
            if (child instanceof OdfElement) {
                if (child instanceof TableTableCellElementBase) {
                    mCurrentColumnNo++;
                    mCurrentColumnNo = considerChange((TableTableCellElementBase) child, (OdfElement) nextElement, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, change);
                    if (change.effectsCells() && (mColumnEndNo <= mCurrentColumnNo && mColumnEndNo > -1)) {
                        break;
                    }
                }
                lastElement = (OdfElement) child;
            }
            child = nextElement;
        }
        // if the end of table is reached, but not the cellStart, the gap has to be filled
        if (lastElement != null && mCurrentColumnNo >= 0 && mCurrentColumnNo < mColumnEndNo) {
            // EXPAND TABLE CELLS
            // if not even the start position was reached
            if (mCurrentColumnNo + 1 < mColumnStartNo) {
                TableTableCellElementBase cellGap = newCellElement(mColumnStartNo - mCurrentColumnNo - 1);
                lastElement.getParentNode().appendChild(cellGap);
                Component.createComponent(rowComponent, cellGap);
                mCurrentColumnNo = mColumnStartNo - 1;
            }
            // if the currentposition is inbetween start and end position
            TableTableCellElementBase newCellElement = newCellElement(mColumnEndNo - mCurrentColumnNo);
            lastElement.getParentNode().appendChild(newCellElement);
            Component.createComponent(rowComponent, newCellElement);
            mCurrentColumnNo++;
            considerChange(newCellElement, null, mRowStartNo, mRowEndNo, mCurrentRowNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, change);
        }
//        // we expect that the appliations will not provide us with something bigger than possible
//        if(mColumnMaxNo > Table.MAX_COLUMN_NUMBER){
//            // minimize fineal row
//        }
    }

    /**
     * Considers if the element is in the given range and applies if necessary a
     * split for it due to repetition of a component, which overlaps the
     * selected range border.
     */
    private int considerChange(OdfStylableElement tableChildElement, OdfElement nextElement, int mRowStartNo, int mRowEndNo, int mCurrentRowNo, int mColumnStartNo, int mColumnEndNo, int mCurrentColumnNo, TableChange change) {
        int startNo;
        int endNo;
        int currentNo;
        if (tableChildElement instanceof TableTableRowElement) {
            startNo = mRowStartNo;
            endNo = mRowEndNo;
            currentNo = mCurrentRowNo;
        } else {
            startNo = mColumnStartNo;
            endNo = mColumnEndNo;
            currentNo = mCurrentColumnNo;
        }
        int repetition = tableChildElement.getRepetition();
        // MATCH: the START reached (or exceeded) and END not reached yet
        if (startNo <= (currentNo + repetition - 1) && ((currentNo <= endNo) || endNo < 0)) {
            NamedNodeMap attrs = tableChildElement.getAttributes();
            // EXPAND: ADD MISSING ELEMENTS TO GAP TILL END
            // Only expand current element by repetition if
            // A) There is a gap between START and END and..
            // B) Current element has no attributes NOR content
            if ((endNo - currentNo > 0) && (tableChildElement.getAttributes() == null && attrs == null || attrs.getLength() == 0) && !tableChildElement.hasChildNodes()) {
                // There is a gap if either there is
                // A) No next element
                // B) Or this element is a column BUT the next element is not a column (in this case it is a row and columns are missing), only column & rows are on the same level!
                if (nextElement == null || (tableChildElement instanceof TableTableColumnElement && !(nextElement instanceof TableTableColumnElement || nextElement instanceof TableTableColumnGroupElement
                    || nextElement instanceof TableTableHeaderColumnsElement
                    || nextElement instanceof TableTableColumnsElement))) {
                    repetition = endNo - currentNo - repetition + 2;
                    if (repetition > 1) {
                        tableChildElement.setRepetition(repetition);
                    }
                }
            }
            // FULL MATCH: current position is the start or is after it
            if (startNo < currentNo) {
                if (tableChildElement instanceof TableTableRowElement) {
                    change.execute(tableChildElement, nextElement, startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, change);
                    if (change.effectsCells()) {
                        traverseRow(((TableTableRowElement) tableChildElement).getComponent(), tableChildElement.getFirstChild(), startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, -1, change);
                    }
                } else {
                    change.execute(tableChildElement, nextElement, mRowStartNo, mRowEndNo, mCurrentRowNo, startNo, endNo, currentNo, change, startNo);
                }
                // NEXT ELEMENT OVERLAPS START: Component have to be split..
            } else if ((currentNo + repetition - 1) >= startNo) {
                // START POSITION SPLIT: Due to a repetition the startNo is higher than the currentNo.. (NO split if parameter is 0)
                OdfStylableElement secondHalf = (OdfStylableElement) tableChildElement.split(startNo - currentNo);
                if (currentNo + (repetition - 1) > endNo) {
                    // END POSITION SPLIT: Due to a repetition the endNo is higher than the startNo (endNo not negativ meaning infinite)
                    // Using the middle part the piece starts at startNo and end at endNo and is at least 1 long
                    nextElement = secondHalf.split(endNo - startNo + 1);
                    if (tableChildElement instanceof TableTableRowElement) {
                        change.execute(secondHalf, nextElement, startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, change);
                        if (change.effectsCells()) {
                            traverseRow(((TableTableRowElement) secondHalf).getComponent(), secondHalf.getFirstChild(), startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, -1, change);
                        }
                    } else {
                        change.execute(secondHalf, nextElement, mRowStartNo, mRowEndNo, mCurrentRowNo, startNo, endNo, currentNo, change, startNo);
                    }
                } else {
                    // NO END POSITION SPLIT: Using the complete second half..
                    if (tableChildElement instanceof TableTableRowElement) {
                        if (change.effectsRows()) {
                            
                        }
                        change.execute(secondHalf, nextElement, startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, mCurrentColumnNo, change);
                        if (change.effectsCells()) {
                            traverseRow(((TableTableRowElement) secondHalf).getComponent(), secondHalf.getFirstChild(), startNo, endNo, currentNo, mColumnStartNo, mColumnEndNo, -1, change);
                        }
                    } else {
                        change.execute(secondHalf, nextElement, mRowStartNo, mRowEndNo, mCurrentRowNo, startNo, endNo, currentNo, change, startNo);
                    }
                }
            }
        }
        // if there had been a repetition (over the usual default of 1) add it
        currentNo += repetition - 1;
        return currentNo;
    }

    /**
     * As the column end position may vary during the traversal of the tree
     */
    void setColumnEndPosition(int columnEndPosition) {
        mColumnEndNo = columnEndPosition;
    }

    /**
     * @return The last found column end position
     */
    int getColumnEndPosition() {
        return mColumnEndNo;
    }


    /**
     * As the column end position may vary during the traversal of the tree
     */
    void setColumnMax(int columnMax) {
        mColumnMaxNo = columnMax;
    }


    /**
     * @return The last found column end position
     */
    int getColumnMaxNo() {
        return mColumnMaxNo;
    }

    private TableTableColumnElement newColumnElement(int repeated) {
        TableTableColumnElement newColumnElement = mFileDom.newOdfElement(TableTableColumnElement.class);
        if (repeated != 1) {
            newColumnElement.setTableNumberColumnsRepeatedAttribute(repeated);
        }
        return newColumnElement;
    }

    private TableTableRowElement newRowElement(int repeated) {
        TableTableRowElement newRowElement = mFileDom.newOdfElement(TableTableRowElement.class);
        if (repeated != 1) {
            newRowElement.setTableNumberRowsRepeatedAttribute(repeated);
        }
        return newRowElement;
    }

    private TableTableCellElement newCellElement(int repeated) {
        TableTableCellElement newCellElement = mFileDom.newOdfElement(TableTableCellElement.class);
        if (repeated > 1) {
            newCellElement.setTableNumberColumnsRepeatedAttribute(repeated);
        }
        return newCellElement;
    }
}
