/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONException;
import org.json.JSONObject;
import static org.odftoolkit.odfdom.component.JsonOperationConsumer.addStyle;
import static org.odftoolkit.odfdom.component.JsonOperationConsumer.insertParagraph;
import static org.odftoolkit.odfdom.component.JsonOperationConsumer.insertText;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.element.text.TextParagraphElementBase;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * A MultiCoomponent uses a single XML element to represent multiple components.
 * This container can be used for spreadsheet row and cell components using
 * repeated elements via an attribute.
 *
 * @author svante.schubertATgmail.com
 * @param <T>
 */
public class Cell<T> extends Component {

    private static final String FORMULA_PREFIX = "of:";

    public Cell(OdfElement componentElement, Component parent) {
        super(componentElement, parent);
    }

    /**
     * A multiple components can be represented by a single XML element
     *
     * @return the number of components the elements represents
     */
    @Override
    public int repetition() {
        return mRootElement.getRepetition();
    }

// CELL ONLY
//	Map<String, Object> mInnerCellStyle = null;
//
//	/** The inner style of a cell will be temporary saved at the cell.
//	 Whenever the cell content is deleted, the style is being merged/applied to the cell style */
//	public Map<String, Object> getInternalCellStyle(){
//		return mInnerCellStyle;
//	}
//
//
//	/** The inner style of a cell will be temporary saved at the cell.
//	 Whenever the cell content is deleted, the style is being merged/applied to the cell style */
//	public void setInternalCellStyle(Map<String, Object> newStyles){
//		mInnerCellStyle = newStyles;
//	}
//
    /**
     * Adds the given component to the root element
     */
    @Override
    public void addChild(int index, Component c) {
        mRootElement.insert(c.getRootElement(), index);
// 2DO: Svante: ARE THE ABOVE AND THE BELOW EQUIVALENT?
//		OdfElement rootElement = c.getRootElement();
//		if (index >= 0) {
//			mRootElement.insertBefore(rootElement, ((OdfElement) mRootElement).receiveNode(index));
//		} else {
//			mRootElement.appendChild(rootElement);
//		}
    }

    /**
     * @return either a text node of size 1 or an element being the root element
     * of a component
     */
    @Override
    public Node getChildNode(int index) {
        return ((OdfElement) mRootElement).receiveNode(index);

    }

    /**
     * Removes a component from the text element container. Removes either an
     * element representing a component or text node of size 1
     */
    @Override
    public Node remove(int index) {
        Node removedNode = null;
        Node node = (Node) this.getChildNode(index);
        if (node != null) {
            removedNode = mRootElement.removeChild(node);
        }
        return removedNode;
    }

    /**
     * All children of the root element will be traversed. If it is a text node
     * the size is added, if it is an element and a component a size of one is
     * added, if it is a marker, for known text marker elements (text:span,
     * text:bookmark) the children are recursive checked
     *
     * @return the number of child components
     */
    @Override
    public int size() {
        return mRootElement.componentSize();
    }

    private static final String FLOAT = "float";
    private static final String STRING = "string";

    /**
     * Adding cell content: either as formula or paragraph and text content,
     * latter with default styles
     */
    public TableTableCellElement addCellStyleAndContent(Component rootComponent, Object value, JSONObject attrs) {
        // see if the cell is repeated
        TableTableCellElement cell = (TableTableCellElement) this.getRootElement();
        OdfFileDom ownerDoc = (OdfFileDom) rootComponent.getOwnerDocument();
        // save the URL as everyting else will be deleted
        String url = reuseCellHyperlink(cell, attrs);
        // exchanges the content if requested
        if (value != null) {
            cell.removeContent();
            // if there is new content..
            if (!value.equals(JSONObject.NULL)) {
                String valueString = value.toString();
                if (valueString.startsWith(OdfFileSaxHandler.EQUATION)) {
                    // How am I able to set the other values? What is the OOXML solution for this?
                    cell.setAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "office:value-type", FLOAT);
                    cell.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:formula", FORMULA_PREFIX.concat(valueString));
                } else {
                    // insert a paragraph to store the text within..
                    TextParagraphElementBase newParagraph = insertParagraph(this, 0, attrs);
                    if (url != null) {
                        attrs = addUrlToCharacterProps(attrs, url);
                    }
                    // if the formula was masked
                    if (value instanceof String && valueString.startsWith(OdfFileSaxHandler.APOSTROPHE_AND_EQUATION)){
                        // cut the first apostrophe
                        valueString = valueString.substring(1);
                    }
                    // addChild the text & removes existing values
                    insertText(newParagraph, 0, attrs, valueString);
                    if (value instanceof Integer || value instanceof Double || value instanceof Float) {
                        cell.setAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "office:value-type", FLOAT);
                        cell.setAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "office:value", valueString);
                    } else if (value instanceof String) {
                        cell.setAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "office:value-type", STRING);
                    }
                }
            }
        }
        if (attrs != null) {
            // Format: Adding Styles to the element
            addStyle(attrs, cell, ownerDoc);
            if (cell.hasChildNodes()) {
                NodeList children = cell.getChildNodes();
                for (int i = 0; i < children.getLength(); i++) {
                    Node child = children.item(i);
                    if (child instanceof OdfElement) {
                        ((OdfElement) child).markText(0, Integer.MAX_VALUE - 1, attrs);
                    }
                }
            } else {
                if (url != null) {
                    TextPElement containerElement1 = new TextPElement(ownerDoc);
                    TextAElement containerElement2 = new TextAElement(ownerDoc);
                    containerElement2.setXlinkHrefAttribute(url);
                    cell.appendChild(containerElement1);
                    containerElement1.appendChild(containerElement2);
                }
            }
        }
        return cell;
    }

    /**
     * To be able to reuse existing style on the full table, new cell hyperlinks
     * will be stored in the cell text style properties as @xlink:href attribute
     * and taken back when nothing new is set.
     */
    private static String reuseCellHyperlink(TableTableCellElement cell, JSONObject attrs) {
        String cellURL = null;
        if (attrs != null) { // apply style changes to the cell
            // apply new styles to the cell (modifiying not overwriting)
            if (attrs.has("character")) {
                JSONObject charProps = attrs.optJSONObject("character");
                if (charProps != null) {
                    if (charProps.hasAndNotNull("url")) {
                        cellURL = charProps.optString("url");
                    } else if (charProps.has("url")) {
                        //removeAnchors();
                    }
                }
            }
        }
        // if there is no new hyperlink given, check for an existing cached (in the properties)
        if (cellURL == null || cellURL.isEmpty()) {
            // check if there is still one given at the cell
            OdfStyle autoStyle = cell.getAutomaticStyle();
            if (autoStyle != null) {
                OdfElement textProps = autoStyle.getPropertiesElement(OdfStylePropertiesSet.TextProperties);
                if (textProps != null && textProps.hasAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href")) {
                    cellURL = textProps.getAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href");
                }
            }
        }
        return cellURL;
    }

    private static JSONObject addUrlToCharacterProps(JSONObject attrs, String cellURL) {
        JSONObject charProps = null;
        if (cellURL != null && !cellURL.isEmpty()) {
            if (attrs == null) {
                attrs = new JSONObject();
            }
            if (!attrs.has("character")) {
                charProps = new JSONObject();
                try {
                    attrs.put("character", charProps);
                } catch (JSONException ex) {
                    Logger.getLogger(JsonOperationConsumer.class.getName()).log(Level.SEVERE, null, ex);
                }
            } else {
                charProps = attrs.optJSONObject("character");
            }
            try {
                charProps.put("url", cellURL);
            } catch (JSONException ex) {
                Logger.getLogger(JsonOperationConsumer.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        return attrs;
    }
}
