/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.json.actions;

import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.IResourceProvider;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.ECalcEngineMode;
import com.openexchange.office.json.tools.EmptyDocumentCache;
import com.openexchange.office.json.tools.FilterExceptionToErrorCode;
import com.openexchange.office.json.tools.OXDocumentHelper;
import com.openexchange.office.json.tools.OXDocumentHelper.OfficeDocumentTypeProperties;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;
import com.openxchange.office_communication.cluster_management.core.ClusterManagerRemoteControl;
import com.openxchange.office_communication.cluster_management.core.IClusterManagerRemoteControl;


/**
 * {@link PrefetchOfficePartsAction}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class PrefetchOfficePartsAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(PrefetchOfficePartsAction.class);

    final private EmptyDocumentCache defaultDocCache;

    public PrefetchOfficePartsAction(ServiceLookup services, EmptyDocumentCache cache) {
        super(services);
        this.defaultDocCache = cache;
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {

        AJAXRequestResult requestResult = null;

        if ((null != requestData) && (null != session)) {
            ErrorCode errorCode = ErrorCode.NO_ERROR;

            try {
                final String part = requestData.getParameter("part");
                if (null != part) {
                    if (part.equalsIgnoreCase("text")) {
                        // OX Text
                        impl_prefetchOfficeParts(session, part);
                    } else if (part.equalsIgnoreCase("spreadsheet")) {
                        // OX Spreadsheet
                        impl_prefetchOfficeParts(session, part);
                        impl_prefetchSpreadsheetWorker();
                    } else if (part.equalsIgnoreCase("presentation")) {
                        // TODO: must be implemented if presentation is a available app
                    } else {
                        errorCode = ErrorCode.GENERAL_ARGUMENTS_ERROR;
                    }
                } else {
                    errorCode = ErrorCode.GENERAL_ARGUMENTS_ERROR;
                }
            } catch (FilterException e) {
                errorCode = FilterExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR);
            } catch (Exception e) {
                errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
            }

            requestResult = getAjaxRequestResult(errorCode.getAsJSONResultObject());
        }

        return requestResult;
    }

    /**
     * Prefetch the necessary OX Text office backend parts so documents can be
     * loaded as fast as possible. This includes the fiter code for the default
     * document format.
     *
     * @param session
     *  The session of the client that requested to prefetch OX Text code.
     *
     * @throws FilterException
     */
    private void impl_prefetchOfficeParts(ServerSession session, String part) throws FilterException {
        InputStream inputStream = null;

        try {
            OfficeDocumentTypeProperties docTypeProps = OXDocumentHelper.determineDefaultOfficeProperties(part);

            // look into the soft cache for stored empty default document data
            final IImporter importer = OXDocumentHelper.getImporter(m_services, docTypeProps.docType);
            final EmptyDocumentCache.EmptyDocData cacheData = defaultDocCache.getDocData(docTypeProps.module, null);
            if (null != cacheData) {
                // cache hit - nothing to do code has already been loaded
                LOG.debug("RT connection: cache hit for document type: " + docTypeProps.module + " data send from cache.");
            } else {
                // no cache hit - load the default document
                LOG.debug("RT connection: cache miss for document type: " + docTypeProps.module + " data must be retrieved from file.");

                final IResourceProvider resourceProvider = m_services.getService(com.openexchange.office.IResourceProvider.class);
                InputStream resourceInputStream = null;

                // try to get default document from resource first
                if (null != resourceProvider) {
                    // !!! resource leak warning has been checked:
                    // if resourceInputStream is returned unchanged as the result
                    // InputStream, it will be closed by the caller; otherwise it will
                    // be closed instantly after retrieving the real result InputStream from the
                    // resourceInputStream after calling importer.getDefaultDocument(...) !!!
                    resourceInputStream = resourceProvider.getResource(resourceProvider.getEntry(docTypeProps.module, "template", "Default"));
                }

                if (null != importer) {
                    final DocumentProperties docProps = new DocumentProperties();
                    inputStream = importer.getDefaultDocument(resourceInputStream, docProps);
                }

                if (resourceInputStream != inputStream) {
                    // Close resource input stream if default document stream
                    // was created by the import getDefaultDocument function. Don't
                    // do that if inputStm refers to resourceInputStream
                    IOUtils.closeQuietly(resourceInputStream);
                }

                // just load the filter part on importing the default empty document
                OXDocumentHelper oxDocHelper = new OXDocumentHelper(m_services, session, inputStream, new ResourceManager(m_services));
                oxDocHelper.getOperations(session, importer);
            }
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

    /** prefetch spreadsheet backend worker
     */
    private void impl_prefetchSpreadsheetWorker()
    {
    	try {
    		// prefect of worker functional in JMS env only ...
    		final CalcEngineConfig aCfg  = CalcEngineConfig.get();
    		final ECalcEngineMode  eMode = aCfg.getMode();
    		if (eMode != ECalcEngineMode.E_JMS)
    			return;
    		
    		LOG.info("::SPREADSHEET::BOOT 'prefetch calcengine worker - START'");
			final IClusterManagerRemoteControl aControl = ClusterManagerRemoteControl.getForJMSEnv();
			aControl.startApp("calcengine-worker");
    		LOG.info("::SPREADSHEET::BOOT 'prefetch calcengine worker - END'");
		} catch (Throwable ex) {
			LOG.error ("::SPREADSHEET::BOOT 'prefetch calcengine worker - END'\n"+ex.getMessage(), ex);
		}
    }
}
