/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.json.actions;

import java.awt.Image;
import java.awt.Transparency;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import com.openexchange.ajax.container.ByteArrayFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.DocFileHelper;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.StreamInfo;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GetFileAction}
 *
 * @author <a href="mailto:firstname.lastname@open-xchange.com">Firstname Lastname</a>
 */
/*
 * MH/KA compile fix for buildsystem
 * @Action(method = RequestMethod.GET, name = "getfile", description =
 * "Get the whole document as PDF file or get the substream of an infostore file in its naive format.", parameters = {
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "id", description = "Object ID of the requested infoitem."),
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "version", optional=true, description =
 * "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
 * @Parameter(name = "filter_format", optional=true, description =
 * "If this value is set to 'pdf', the whole document is as converted PDF file, in all other cases, the content of the document's substream, described by the parameter 'fragment, is returned as unmodified file."
 * ),
 * @Parameter(name = "fragment", optional=true, description =
 * "If this value is set and the filter_format is not set or not set to 'pdf', this parameter describes the substream name of the document internal file to be returned."
 * ),
 * @Parameter(name = "filename", optional=true, description =
 * "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."
 * ), }, responseDescription = "Response with timestamp: The content of the documents substream in its native format, or converted to PNG in
 * the case of EMF, WMF and SVM graphic file formats)
 */
@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetFileAction extends DocumentFilterAction {

    /**
     * Initializes a new {@link GetFileAction}.
     *
     * @param services
     * @param oqm
     */
    public GetFileAction(ServiceLookup services, ResourceManager resMgr) {
        super(services);
        resourceManager = resMgr;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        String resourceName = request.getParameter("get_filename");
        final String fileId = request.getParameter("id");
        final String folderId = request.getParameter("folder_id");

        if (StringUtils.isEmpty(fileId) || StringUtils.isEmpty(folderId)) {
            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(400);
            return result;
        }

        boolean canRead = false;
        try {
            final IDBasedFileAccessFactory fileAccessFactory = m_services.getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess        fileAccess        = fileAccessFactory.createAccess(session);
            final File                     metaData          = fileAccess.getFileMetadata(fileId, FileStorageFileAccess.CURRENT_VERSION);

            // we assume that we have at least read access for the file if we can get the meta data
            canRead = (metaData != null);
        } catch (Exception e) {
            // do nothing - an exception is interpreted as missing access rights
            LOG.error("AjaxRequest GetFileAction exception caught while trying to read meta file for requested document images", e);
        } finally {
            if (!canRead) {
                final AJAXRequestResult result = new AJAXRequestResult();
                result.setHttpStatusCode(403);
                return result;
            }
        }

        if ((null != resourceName) && (resourceName.length() > 0)) {
            // sanitize resource name
            if (resourceName.startsWith("./")) {
                resourceName = resourceName.substring(2);
            }

            final String filename = FileHelper.getBaseName(resourceName);
            LOG.debug("AjaxRequest GetFileAction request for " + filename);

            // try to get the graphic from the resource manager(s) first
            if (filename.startsWith("uid")) {
                // 1. try: resource is already contained within this nodes resource manager
                final long uid = Resource.getUIDFromResourceName(filename);
                final Resource nodeResource = resourceManager.getResource(uid);
                byte[] resdata = ((nodeResource != null) ? nodeResource.getBuffer() : null);

                LOG.trace("AjaxRequest GetFileAction resource data " + filename + " not found in resourceManager.");

                // 2. try: create a managed resource with the given uid
                if (null == resdata) {
                    try {
                        final String managedId = Resource.getManagedIdFromUID(uid);

                        if ((null != managedId) && (managedId.length() > 0)) {
                            final Resource managedResource = Resource.createFrom(m_services, managedId);

                            // add a new managed resource to this nodes resource manager for later speedup
                            if (null != (resdata = ((null != managedResource) ? managedResource.getBuffer() : null))) {
                                resourceManager.addResource(managedResource);
                            }
                        }
                    } catch (Exception e) {
                        LOG.error("AjaxRequest GetFileAction exception catched while trying to create managed resource.",  e);
                    }
                }

                // if we've got the resource data, set it as request result
                if (null != resdata) {
                    try {
                        final String mimeType = DocFileHelper.getMimeType(resourceName);
                        final ByteArrayFileHolder fileHolder = new ByteArrayFileHolder(resdata);

                        fileHolder.close();
                        fileHolder.setName(resourceName);
                        fileHolder.setContentType(mimeType);
                        requestResult = new AJAXRequestResult(fileHolder, "file");
                    } catch (Exception e) {
                        LOG.error("AjaxRequest GetFileAction exception catched while setting data to resultRequest.",  e);
                    }
                }
            }

            // try getting the graphic from the document's zip archive
            if (null == requestResult) {
                LOG.trace("AjaxRequest GetFileAction image must be retrieved from document stream");

                // always use the current version of the document
                request.putParameter("version", "0");
                final StorageHelper storageHelper = new StorageHelper(m_services, session, folderId);
                final StreamInfo streamInfo = DocFileHelper.getDocumentStream(m_services, session, request, storageHelper);
                final InputStream documentStm = streamInfo.getDocumentStream();

                if (null != documentStm) {
                    LOG.trace("AjaxRequest GetFileAction document stream successfully retrieved - try to get substream for resource data " + filename);
                    final String[] mimeType = { "" };
                    final InputStream resultStm = getZipEntryStream(documentStm, session, resourceName, mimeType);

                    if (null != resultStm) {
                        byte[] buffer = null;

                        try {
                            buffer = IOUtils.toByteArray(resultStm);
                        } catch(IOException e) {
                            LOG.error("AjaxRequest GetFileAction exception catched while reading data from document stream.",  e);
                        } finally {
                            IOUtils.closeQuietly(resultStm);
                        }

                        if (null != buffer) {
                            requestResult = createFileRequestResult(request, buffer, mimeType[0], null);
                            LOG.trace("AjaxRequest GetFileAction resource data " + filename + " successfully retrieved.");
                        } else {
                            LOG.warn("AjaxRequest GetFileAction resource data " + filename + " not found.");
                        }
                    }

                    IOUtils.closeQuietly(documentStm);

                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }
                } else {
                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }
                }
            }
        } else {
            LOG.warn("AjaxRequest GetFileAction request parameters are not correct - request cannot be processed!");
        }

        return requestResult;
    }

    /**
     * @param documentStm
     * @param resourceName
     * @param mimeType
     * @param closeDocumentStream
     * @return
     */
    final private InputStream getZipEntryStream(InputStream documentStm, ServerSession session, String resourceName, String[] mimeType) {
        InputStream resultStm = null;

        if (null != documentStm) {
            final ZipInputStream zipInputStm = new ZipInputStream(documentStm);
            final String filename = FileHelper.getBaseName(resourceName);
            String uidResourceName = null;

            // check for uid Resources within all entries
            if (filename.startsWith("uid") && (filename.length() > 3)) {
                uidResourceName = filename;
            }

            try {
                for (ZipEntry zipEntry = null; (null == resultStm) && ((zipEntry = zipInputStm.getNextEntry()) != null);) {
                    final String curEntryName = zipEntry.getName();

                    if (curEntryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                        // check for resources contained within the original rescue document
                        byte[] entryBuffer = null;

                        try {
                            entryBuffer = IOUtils.toByteArray(zipInputStm);
                        } catch (IOException e) {
                            //
                        }

                        if (null != entryBuffer) {
                            final InputStream entryInputStream = new ByteArrayInputStream(entryBuffer);
                            resultStm = getZipEntryStream(entryInputStream, session, resourceName, mimeType);
                            IOUtils.closeQuietly(entryInputStream);
                        }
                    } else if (curEntryName.equals(resourceName) || ((null != uidResourceName && curEntryName.endsWith(uidResourceName)))) {
                        // check for resources equal the the given name and for resources matching the uid resource manager naming schema
                        byte[] entryBuffer = null;

                        try {
                            entryBuffer = IOUtils.toByteArray(zipInputStm);
                        } catch (IOException e) {
                            LOG.error("AjaxRequest GetFileAction.getZipEntryStream exception catched while trying to read data from document stream.",  e);
                        }

                        if (null != entryBuffer) {
                            final String fileExtension = FilenameUtils.getExtension(resourceName).trim().toLowerCase();

                            // check if we create a temporary png file that can be displayed within the browser,
                            // we will do this for the formats "wmf and emf... TODO: the resulting graphic should
                            // somehow be part of the resourcemanager to prevent unnecessary graphic conversions
                            if (fileExtension.equals("emf") || fileExtension.equals("wmf") || (resourceName.indexOf("ObjectReplacements/") > -1)) {
                                final IManager dcManager = m_services.getService(IManager.class);

                                if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                    final HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                                    final HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                                    final InputStream inputStm = new ByteArrayInputStream(entryBuffer);

                                    jobProperties.put(Properties.PROP_INPUT_STREAM, inputStm);
                                    jobProperties.put(Properties.PROP_INPUT_TYPE, fileExtension);
                                    jobProperties.put(Properties.PROP_MIME_TYPE, "image/png");
                                    jobProperties.put(Properties.PROP_INFO_FILENAME, resourceName);

                                    resultStm = dcManager.convert("graphic", jobProperties, resultProperties);
                                    IOUtils.closeQuietly(inputStm);
                                    mimeType[0] = (String) resultProperties.get(Properties.PROP_RESULT_MIME_TYPE);
                                }
                            }
                            else if(fileExtension.equals("jpg") || fileExtension.equals("jpeg") || fileExtension.equals("png")) {

                                entryBuffer = ensureImageResolution(entryBuffer, session, fileExtension);
                            }
                            if (null == resultStm) {
                                resultStm = new ByteArrayInputStream(entryBuffer);
                                mimeType[0] = DocFileHelper.getMimeType(resourceName);
                            }
                        }
                    }

                    zipInputStm.closeEntry();
                }
            } catch (Throwable e) {
                LOG.error("AjaxRequest GetFileAction.getZipEntryStream exception catched while trying to read data from document stream.",  e);
            } finally {
                IOUtils.closeQuietly(zipInputStm);
                IOUtils.closeQuietly(documentStm);
            }
        }

        // !!! Possible resource leak warning has been checked:
        // returned resultStm is closed by caller of this method
        return resultStm;
    }

    final private byte[] ensureImageResolution(byte[] entryBuffer, ServerSession session, String fileExtension) {

        // we will check the image dimensions and scale down if necessary
        final int maxImageSize = ConfigurationHelper.getIntegerOfficeConfigurationValue(m_services, session, "//module/documentImageSize", 1280);
        if(maxImageSize>0) {

            final Iterator<ImageReader> iterReader = ImageIO.getImageReadersBySuffix(fileExtension);
            while(iterReader.hasNext()) {

                final ImageReader imageReader = iterReader.next();
                ImageWriter imageWriter = null;
                final ByteArrayInputStream inputStream = new ByteArrayInputStream(entryBuffer);
                ByteArrayOutputStream outputStream = null;
                ImageOutputStream imageOutputStream = null;
                ImageInputStream imageInputStream = null;
                BufferedImage source = null;
                BufferedImage dest = null;

                try{
                    imageInputStream = ImageIO.createImageInputStream(inputStream);
                    imageReader.setInput(imageInputStream);

                    final int width = imageReader.getWidth(imageReader.getMinIndex());
                    final int height = imageReader.getHeight(imageReader.getMinIndex());
                    if(width>0&&height>0) {
                        if(width>maxImageSize||height>maxImageSize) {
                            int newWidth = maxImageSize;
                            int newHeight = maxImageSize;
                            if(width>height) {
                                newHeight = (int)((((double)height/(double)width)*maxImageSize)+0.5);
                            }
                            else {
                                newWidth = (int)((((double)width/(double)height)*maxImageSize)+0.5);
                            }
                            source = imageReader.read(imageReader.getMinIndex());
                            if(source!=null) {
                                final Image scaledImage = source.getScaledInstance(newWidth, newHeight, BufferedImage.SCALE_DEFAULT);
                                if(scaledImage instanceof BufferedImage) {
                                    dest = (BufferedImage)scaledImage;
                                }
                                else {
                                    dest = new BufferedImage(newWidth, newHeight, source.getType());
                                    dest.getGraphics().drawImage(scaledImage, 0, 0, null);
                                }
                                final String destFormat = source.getTransparency() == Transparency.OPAQUE ? "jpeg" : "png";
                                final Iterator<ImageWriter> iterWriter = ImageIO.getImageWritersByFormatName(destFormat);
                                if(iterWriter.hasNext()) {
                                    imageWriter = iterWriter.next();
                                    ImageWriteParam iwp = null;
                                    if(destFormat.equals("jpeg")) {

                                        int jpegCompression = ConfigurationHelper.getIntegerOfficeConfigurationValue(m_services, session, "//module/documentImageQuality", 75);
                                        if(jpegCompression<0)
                                            jpegCompression = 0;
                                        else if(jpegCompression>100)
                                            jpegCompression = 100;
                                        iwp = imageWriter.getDefaultWriteParam();
                                        iwp.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
                                        iwp.setCompressionQuality(jpegCompression/100f);
                                    }
                                    outputStream = new ByteArrayOutputStream();
                                    imageOutputStream = ImageIO.createImageOutputStream(outputStream);
                                    imageWriter.setOutput(imageOutputStream);
                                    imageWriter.write(null, new IIOImage(dest, null, null), iwp);
                                    entryBuffer = outputStream.toByteArray();
                                    break;
                                }
                            }
                        }
                    }
                }
                catch(IOException e) {
                    LOG.error("AjaxRequest GetFileAction.ensureImageResolution exception catched while trying to create ImageInputStream.", e);
                }
                finally {
                    imageReader.dispose();
                    if(imageWriter!=null) {
                        imageWriter.dispose();
                    }
                    IOUtils.closeQuietly(inputStream);
                    IOUtils.closeQuietly(outputStream);
                    if(imageInputStream!=null) {
                        try {
                            imageInputStream.close();
                        } catch (IOException e)
                        {
                            LOG.error("AjaxRequest GetFileAction.ensureImageResolution exception catched while trying to create ImageInputStream.", e);
                        }
                    }
                    if(imageOutputStream!=null) {
                        try {
                            imageOutputStream.close();
                        } catch (IOException e)
                        {
                            LOG.error("AjaxRequest GetFileAction.ensureImageResolution exception catched while trying to create ImageInputStream.", e);
                        }
                    }
                    if(source!=null)
                        source.getGraphics().dispose();
                    if(dest!=null)
                        dest.getGraphics().dispose();
                }
            }
        }
        return entryBuffer;
    }

    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(GetFileAction.class);

    private ResourceManager resourceManager = null;
}
