/*
 *  Copyright 2007-2008, Plutext Pty Ltd.
 *
 *  This file is part of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.

    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.

 */

package org.docx4j.jaxb;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;

import javax.xml.bind.ValidationEvent;
import javax.xml.bind.ValidationEventHandler;
import javax.xml.bind.ValidationEventLocator;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.stream.StreamSource;

import org.docx4j.XmlUtils;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

public class JaxbValidationEventHandler implements ValidationEventHandler {

	private static Logger log = LoggerFactory.getLogger(JaxbValidationEventHandler.class);

	private boolean shouldContinue = true;

	private HashSet<String> ignorables = null;

	private HashSet<String> mceIgnorables = null;

	public void setContinue(boolean val) {
		shouldContinue = val;
	}

    public void setIgnorables(HashSet<String> v) {
        ignorables = v;
    }

	public final static String UNEXPECTED_MC_ALTERNATE_CONTENT = "unexpected element (uri:\"http://schemas.openxmlformats.org/markup-compatibility/2006\", local:\"AlternateContent\")";

	static Templates mcPreprocessorXslt;

	public static Templates getMcPreprocessor() throws IOException, TransformerConfigurationException {

		if (mcPreprocessorXslt==null) {
			try {
				Source xsltSource  = new StreamSource(
						org.docx4j.utils.ResourceUtils.getResource(
								"org/docx4j/jaxb/mc-preprocessor.xslt"));
				mcPreprocessorXslt = XmlUtils.getTransformerTemplate(xsltSource);
			} catch (IOException e) {
				log.error("Problem setting up  mc-preprocessor.xslt", e);
				throw(e);
			} catch (TransformerConfigurationException e) {
				log.error("Problem setting up  mc-preprocessor.xslt", e);
				throw(e);
			}
		}
		return mcPreprocessorXslt;
	}

	public boolean handleEvent(ValidationEvent ve) {

        if(org.docx4j.jaxb.Context.isLowMemoryAbort()) {
            throw new RuntimeException();
        }
	    if (ve.getSeverity()==ValidationEvent.FATAL_ERROR || ve.getSeverity()==ValidationEvent.ERROR) {

	        if(ve.getLinkedException()!=null) {
	            if(ve.getLinkedException().getCause() instanceof OutOfMemoryError) {
	                throw new OutOfMemoryError();
	            }
	            if (log.isDebugEnabled() ) {
	                ve.getLinkedException().printStackTrace();
	            }
	        }
	        else {
	            // check for ignorables like w:tblDescription, which is used only in Office 2010 documents
	            // and also check for mceIgnorables ... so we can continue without xslt precprocessing
	            final ValidationEventLocator locator = ve.getLocator();
	            if(locator!=null&&locator.getNode()!=null) {
	                if(ignorables!=null) {
    	                if(ignorables.contains(locator.getNode().getNodeName())) {
    	                    return true;
    	                }
	                }
	                if(mceIgnorables==null) {
	                    // try to get the mceIgnorables
	                    mceIgnorables = new HashSet<String>(3);
	                    final Document ownerDocument = locator.getNode().getOwnerDocument();
	                    if(ownerDocument!=null) {
	                        final Node firstChild = ownerDocument.getFirstChild();
	                        if(firstChild!=null) {
	                            final NamedNodeMap attributes = firstChild.getAttributes();
	                            if(attributes!=null) {
	                                final Node ignorableNode = attributes.getNamedItemNS(Namespaces.MARKUP_COMPATIBILITY, "Ignorable");
	                                if(ignorableNode!=null) {
	                                    final String ignorableNodeValue = ignorableNode.getNodeValue();
	                                    if(ignorableNodeValue!=null&&!ignorableNodeValue.isEmpty()) {
	                                        mceIgnorables.addAll(Arrays.asList(ignorableNodeValue.split(" ")));
	                                    }
	                                }
	                            }
	                        }
	                    }
	                }
	                final String prefix = locator.getNode().getPrefix();
	                if(prefix!=null&&!prefix.isEmpty()) {
	                    if(mceIgnorables.contains(prefix)) {
	                        // this node is part of the mceIgnorables and can be ignored,
	                        return true;
	                    }
                    }
	            }
	        }
	        // print message from validation event
	        if (log.isDebugEnabled() || ve.getMessage().length() < 120 ) {
	            log.warn( printSeverity(ve) + ": " + ve.getMessage() );
	        }
	        else {
	            log.warn( printSeverity(ve) + ": " + ve.getMessage().substring(0, 120));
	        }
	    }
	    else if (ve.getSeverity()==ve.WARNING) {
	        log.warn(printSeverity(ve) + "Message is " + ve.getMessage());
	    }
	    // JAXB provider should attempt to continue its current operation.
	    // (Marshalling, Unmarshalling, Validating)
	    log.info("continuing (with possible element/attribute loss)");
	    return shouldContinue;
    }

    public String printSeverity(ValidationEvent ve) {
    	String errorLevel;

    	switch (ve.getSeverity()) {
    		case ValidationEvent.FATAL_ERROR: {
              	// FATAL_ERROR is usually not actually fatal!
    			errorLevel="(non)FATAL_ERROR";
    			break; }
    		case ValidationEvent.ERROR: { errorLevel="ERROR"; break; }
    		case ValidationEvent.WARNING: { errorLevel="WARNING"; break; }
    		default: errorLevel = new Integer (ve.getSeverity()).toString() ;
    	}
    	return "[" + errorLevel + "] ";
    }
}
