/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.CalcEngineClipBoard;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
public class CalcEngineClientBound2ClipBoard implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientBound2ClipBoard.class);

    //-------------------------------------------------------------------------
    private CalcEngineClientBound2ClipBoard ()
    {}

    //-------------------------------------------------------------------------
    public static synchronized CalcEngineClientBound2ClipBoard create (final ICalcEngineClient iWrapped)
        throws Exception
    {
        final CalcEngineClientBound2ClipBoard aInst = new CalcEngineClientBound2ClipBoard ();
        aInst.m_iWrapped = iWrapped;
        return aInst;
    }

    //-------------------------------------------------------------------------
    @Override
    public String createDocument()
        throws Exception
    {
        return m_iWrapped.createDocument();
    };

    //-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sDocHandle)
        throws Exception
    {
        m_iWrapped.destroyDocument(sDocHandle);
    };

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sDocHandle    ,
                                             final String       sOperationJSON,
                                             final StringBuffer sResultJSON   )
        throws Exception
    {
        return m_iWrapped.executeOperation(sDocHandle, sOperationJSON, sResultJSON);
    };

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument (final String    sDocHandle          ,
                                             final String... lPagedOperationJSONs)
        throws Exception
    {
        return m_iWrapped.restoreDocument(sDocHandle, lPagedOperationJSONs);
    }

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError copy(final String                   sSourceDocHandle,
                                 final CalcEngineClipBoardEvent aEvent          )
        throws Exception
    {
    	final ContextAwareLogHelp aLog = new ContextAwareLogHelp (LOG);
    	aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sSourceDocHandle);
    	
        LOG.info (aLog.forLevel(ELogLevel.E_INFO ).toLog("copy request"                ));
        LOG.debug(aLog.forLevel(ELogLevel.E_DEBUG).toLog("copy event data:'"+aEvent+"'"));

        // a) do operation on calc document ...
        //    {@link CalcEngineClipBoard#sOperationJSON} entry is expected to be filled afterwards ...

        final ECalcEngineError eError = m_iWrapped.copy(sSourceDocHandle, aEvent);

        if (eError == ECalcEngineError.E_NONE)
        {
            // b) ... and {@link CalcEngineClipBoard#sOperationJSON} needs to be cached here ...
            //    so a following paste event can read and use it.

            LOG.info (aLog.enterContext(CalcEngineLogContextConst.CONTEXT_USER_SESSION, aEvent.sUserSession)
            		      .forLevel    (ELogLevel.E_INFO                                                   )
            		      .toLog       ("access user clipboard"                                            ));

            final CalcEngineClipBoard aUserClipBoard = CalcEngineClipBoard.forUser(aEvent.sUserSession);
           
            LOG.debug(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_CLIPDBOARD_HANDLE, aEvent.sClipboardHandle)
            		      .forLevel    (ELogLevel.E_DEBUG                                                           )
            		      .toLog       ("cache data:'"+aEvent+"'"                                                   ));

            aUserClipBoard.setData(aEvent.sClipboardHandle, aEvent);
        }
        else
        	LOG.error (aLog.forLevel(ELogLevel.E_ERROR).toLog("error:"+eError));

        // c) ensure 'private data' don't reach caller .-)
        aEvent.sCopyOperationsJSON = null;

        return eError;
    }

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError paste(final String                   sTargetDocHandle,
                                  final CalcEngineClipBoardEvent aEvent          )
        throws Exception
    {
    	final ContextAwareLogHelp aLog = new ContextAwareLogHelp (LOG);
    	aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sTargetDocHandle);
    	
        LOG.info (aLog.forLevel(ELogLevel.E_INFO ).toLog("paste request"                ));
        LOG.debug(aLog.forLevel(ELogLevel.E_DEBUG).toLog("paste event data:'"+aEvent+"'"));

        // a) prepare these paste event ...
        //    search for any cached copy event (done before) ...
        //    and put the value of {@link CalcEngineClipBoard#sOperationJSON} into these paste event ...

        LOG.info (aLog.enterContext(CalcEngineLogContextConst.CONTEXT_USER_SESSION     , aEvent.sUserSession    )
        		      .enterContext(CalcEngineLogContextConst.CONTEXT_CLIPDBOARD_HANDLE, aEvent.sClipboardHandle)
  		              .forLevel    (ELogLevel.E_INFO                                                            )
  		              .toLog       ("access user clipboard"                                                     ));

        final CalcEngineClipBoard      aUserClipBoard             = CalcEngineClipBoard.forUser(aEvent.sUserSession);
        final CalcEngineClipBoardEvent aCachedCopyEvent           = aUserClipBoard.getData(aEvent.sClipboardHandle);
                                       aEvent.sCopyOperationsJSON = aCachedCopyEvent.sCopyOperationsJSON;

        LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG                            )
                       .toLog   ("got cached data:"+aEvent.sCopyOperationsJSON));

        // b) ... forward those 'enriched' paste event to the calc engine and let it generate
        //    a suitable answer. Those answer can be routed to the caller of this method
        //    without any further investigation.

        final ECalcEngineError eError = m_iWrapped.paste(sTargetDocHandle, aEvent);

        if (eError == ECalcEngineError.E_NONE)
        	LOG.error (aLog.forLevel(ELogLevel.E_ERROR).toLog("error:"+eError));

        // c) ensure 'private data' don't reach caller of this method .-)
        aEvent.sCopyOperationsJSON = null;

        return eError;
    }

    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
        throws Exception
    {
        return m_iWrapped.getVersion ();
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
        m_iWrapped.setLogLevel (eLevel);
    }

    //-------------------------------------------------------------------------
    @Override
    public void dispose ()
    	throws Exception
    {
    	if (m_iWrapped != null)
    		m_iWrapped.dispose ();
    	m_iWrapped = null;
    }

    //-------------------------------------------------------------------------
    private ICalcEngineClient m_iWrapped = null;
}
