/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.jms;

import java.io.File;
import java.net.URI;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

public class JMSEnv
{
	//-------------------------------------------------------------------------
	public static final String CONFIG_FILE = "jms.properties";

	//-------------------------------------------------------------------------
	public static final String CFGPROP_JMS_SERVER_IP        = "jms.server.ip"       ;
	public static final String CFGPROP_JMS_SERVER_PORT      = "jms.server.port"     ;
	public static final String CFGPROP_JMS_SERVER_DATA_PATH = "jms.server.data.path";
	
	//-------------------------------------------------------------------------
	public static final String DEFAULT_SERVER_DATA_PATH = "/opt/open-xchange/jms/data";
	
	//-------------------------------------------------------------------------
	private JMSEnv ()
	{}
	
	//-------------------------------------------------------------------------
	public synchronized static JMSEnv create ()
	    throws Exception
	{
		if (m_gEnv == null)
			m_gEnv = new JMSEnv ();
		return m_gEnv;
	}
	
	//-------------------------------------------------------------------------
	public synchronized static JMSEnv get ()
	    throws Exception
	{
		Validate.notNull(m_gEnv, "No environment defined. Somewhere has to call JMSEnv.create() and initialize the given env before.");
		return m_gEnv;
	}

	//-------------------------------------------------------------------------
	public synchronized void init (final Properties aConfig)
	    throws Exception
	{
		Validate.notNull(aConfig, "Miss configuration file '"+CONFIG_FILE+"'.");
	
		// a) mandatory values
		
		final String sServerIP   = aConfig.getProperty (CFGPROP_JMS_SERVER_IP  );
		final String sServerPort = aConfig.getProperty (CFGPROP_JMS_SERVER_PORT);
		
		Validate.notEmpty(sServerIP  , "Miss configuration value for '"+CFGPROP_JMS_SERVER_IP  +"'.");
		Validate.notEmpty(sServerPort, "Miss configuration value for '"+CFGPROP_JMS_SERVER_PORT+"'.");
		
		m_sServerIP   = sServerIP;
		m_nServerPort = Integer.parseInt(sServerPort); // mandatory ! at least AMQ requires that .-)
		
		// b) optional values
		
		String sServerDataPath = aConfig.getProperty(CFGPROP_JMS_SERVER_DATA_PATH);
		if (StringUtils.isEmpty(sServerDataPath))
			sServerDataPath = impl_guessServerDataPath ();
		m_sServerDataPath = sServerDataPath;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void setServerIP (final String sIP)
	    throws Exception
	{
		m_sServerIP = sIP;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void setServerPort (final int nPort)
	    throws Exception
	{
		m_nServerPort = nPort;
	}

	//-------------------------------------------------------------------------
	public synchronized final URI getServerURI ()
	    throws Exception
	{
		final StringBuffer sURI = new StringBuffer (256);

		sURI.append ("tcp://"   );
		sURI.append (m_sServerIP);

		if (m_nServerPort != null)
		{
			sURI.append (":"          );
			sURI.append (m_nServerPort);
		}

		return new URI(sURI.toString ());
	}

	//-------------------------------------------------------------------------
	public synchronized final String getServerURIString ()
	    throws Exception
	{
		final URI    aURI = getServerURI ();
		final String sURI = aURI.toASCIIString ();
		return sURI;
	}
	
	//-------------------------------------------------------------------------
	public synchronized final String getContextFactoryName ()
	    throws Exception
	{
		// TODO configure me ... depending from used server implementation .-)
		return "org.apache.activemq.jndi.ActiveMQInitialContextFactory";
	}

	//-------------------------------------------------------------------------
	public synchronized String getLocalDataPath ()
	    throws Exception
	{
		return m_sServerDataPath;
	}
	
	//-------------------------------------------------------------------------
	private String impl_guessServerDataPath ()
	    throws Exception
	{
		// a) prefer default path of our product installation
		
		final File aDefaultPath = new File (DEFAULT_SERVER_DATA_PATH);
		if (aDefaultPath.isDirectory())
			return aDefaultPath.getAbsolutePath();
		
		// b) create temp. folder on local disc (which will be removed on shutdown automatically)
		
		final File aTempPath     = FileUtils.getTempDirectory();
		final File aTempDataPath = new File (aTempPath, "ox-jms-data");
		aTempDataPath.mkdirs      (    );
		aTempDataPath.setWritable (true); // TODO clarify : security issue ... but we need it .-) ?
		aTempDataPath.deleteOnExit(    );
		FileUtils.cleanDirectory(aTempDataPath);
		return aTempDataPath.getAbsolutePath();
	}
	
	//-------------------------------------------------------------------------
	private static JMSEnv m_gEnv = null;

	//-------------------------------------------------------------------------
	private String m_sServerIP = null;

	//-------------------------------------------------------------------------
	private Integer m_nServerPort = null;

	//-------------------------------------------------------------------------
	private String m_sServerDataPath = null;
}
